%{
/*
** Copyright (C) 2007-2012 by Carnegie Mellon University.
**
** @OPENSOURCE_HEADER_START@
**
** Use of the SILK system and related source code is subject to the terms
** of the following licenses:
**
** GNU Public License (GPL) Rights pursuant to Version 2, June 1991
** Government Purpose License Rights (GPLR) pursuant to DFARS 252.227.7013
**
** NO WARRANTY
**
** ANY INFORMATION, MATERIALS, SERVICES, INTELLECTUAL PROPERTY OR OTHER
** PROPERTY OR RIGHTS GRANTED OR PROVIDED BY CARNEGIE MELLON UNIVERSITY
** PURSUANT TO THIS LICENSE (HEREINAFTER THE "DELIVERABLES") ARE ON AN
** "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY
** KIND, EITHER EXPRESS OR IMPLIED AS TO ANY MATTER INCLUDING, BUT NOT
** LIMITED TO, WARRANTY OF FITNESS FOR A PARTICULAR PURPOSE,
** MERCHANTABILITY, INFORMATIONAL CONTENT, NONINFRINGEMENT, OR ERROR-FREE
** OPERATION. CARNEGIE MELLON UNIVERSITY SHALL NOT BE LIABLE FOR INDIRECT,
** SPECIAL OR CONSEQUENTIAL DAMAGES, SUCH AS LOSS OF PROFITS OR INABILITY
** TO USE SAID INTELLECTUAL PROPERTY, UNDER THIS LICENSE, REGARDLESS OF
** WHETHER SUCH PARTY WAS AWARE OF THE POSSIBILITY OF SUCH DAMAGES.
** LICENSEE AGREES THAT IT WILL NOT MAKE ANY WARRANTY ON BEHALF OF
** CARNEGIE MELLON UNIVERSITY, EXPRESS OR IMPLIED, TO ANY PERSON
** CONCERNING THE APPLICATION OF OR THE RESULTS TO BE OBTAINED WITH THE
** DELIVERABLES UNDER THIS LICENSE.
**
** Licensee hereby agrees to defend, indemnify, and hold harmless Carnegie
** Mellon University, its trustees, officers, employees, and agents from
** all claims or demands made against them (and any related losses,
** expenses, or attorney's fees) arising out of, or relating to Licensee's
** and/or its sub licensees' negligent use or willful misuse of or
** negligent conduct or willful misconduct regarding the Software,
** facilities, or other rights or assistance granted by Carnegie Mellon
** University under this License, including, but not limited to, any
** claims of product liability, personal injury, death, damage to
** property, or violation of any laws or regulations.
**
** Carnegie Mellon University Software Engineering Institute authored
** documents are sponsored by the U.S. Department of Defense under
** Contract FA8721-05-C-0003. Carnegie Mellon University retains
** copyrights in all material produced under this contract. The U.S.
** Government retains a non-exclusive, royalty-free license to publish or
** reproduce these documents, or allow others to do so, for U.S.
** Government purposes only pursuant to the copyright license under the
** contract clause at 252.227.7013.
**
** @OPENSOURCE_HEADER_END@
*/

/*
**  rwpmapbuild.l
**
**  Create a prefixmap (pmap) file from textual input.
**
**  Mark Thomas
**  September 2007
*/


#include <silk/silk.h>

RCSIDENT("$SiLK: rwpmapbuild.l 372a8bc31d8a 2012-02-10 21:55:28Z mthomas $");

#include <silk/skprefixmap.h>
#include <silk/skstream.h>
#include <silk/utils.h>


/* LOCAL DEFINES AND TYPEDEFS */

/* where to write --help output */
#define USAGE_FH stdout

/* label to use for unknown ranges if the user does not provide one */
#define DEFAULT_LABEL  "UNKNOWN"

/* default mode to use */
#define DEFAULT_MODE  SKPREFIXMAP_CONT_ADDR

/* types of statements that we can parse */
typedef enum {
    STMT_DEFAULT, STMT_LABEL, STMT_CIDR, STMT_IPS, STMT_NUMBERS, STMT_PROPORTS
} pmap_stmt_type_t;

/* information for the current statement */
typedef struct pmap_stmt_st {
    pmap_stmt_type_t   type;
    uint32_t           value;
    uint32_t           range_start;
    uint32_t           range_end;
} pmap_stmt_t;

/* To avoid warnings with flex */
#define YY_NO_UNPUT 1


/* LOCAL VARIABLE DEFINITIONS */

/* the pmap that we will create */
static skPrefixMap_t *map = NULL;

/* information for the current statement */
static pmap_stmt_t stmt;

/* which line of input we are reading */
static int linenum = 1;

/* number of errors encountered during processing */
static int error_count = 0;

/* whether in IP or proto-port mode (or not-yet-specified == -1) */
static int mode = -1;

/* input stream for reading the prefix map text */
static skstream_t *in_stream = NULL;

/* output stream for the resulting prefix map */
static skstream_t *out_stream = NULL;


/* OPTIONS SETUP */

typedef enum {
    OPT_INPUT_FILE,
    OPT_OUTPUT_FILE
} appOptionsEnum;

static struct option appOptions[] = {
    {"input-file",          REQUIRED_ARG, 0, OPT_INPUT_FILE},
    {"output-file",         REQUIRED_ARG, 0, OPT_OUTPUT_FILE},
    {0,0,0,0}               /* sentinel entry */
};


static const char *appHelp[] = {
    "Read prefix map definition from this file",
    "Write the binary prefix map to this file",
    (char *)NULL
};


/* LOCAL FUNCTION PROTOTYPES */

static void appUsageLong(void);
static void appTeardown(void);
static void appSetup(int argc, char **argv);
static int  appOptionsHandler(clientData cData, int opt_index, char *opt_arg);

static int stmtDefault(int state, char *string);
static int stmtMode(int state, skPrefixMapContent_t new_mode);
static int stmtLabel(int state, char *string);
static int stmtMapName(int state, char *string);
static int stmtCIDR(int state, char *string);
static int stmtIPs(int state, char *string);
static int stmtNumbers(int state, char *string);
static int stmtProPorts(int state, char *string);
static int gotLabel(char *string);

/* to keep gcc quiet  */
int yylex(void);

#if 0
/* Newer versions of flex define these functions.  Declare them here
 * to avoid gcc warnings, and just hope that their signatures don't
 * change. */
int   pmapbuild_get_leng(void);
char *pmapbuild_get_text(void);
int   pmapbuild_get_debug(void);
void  pmapbuild_set_debug(int bdebug);
int   pmapbuild_get_lineno(void);
void  pmapbuild_set_lineno(int line_number);
FILE *pmapbuild_get_in(void);
void  pmapbuild_set_in(FILE *in_str);
FILE *pmapbuild_get_out(void);
void  pmapbuild_set_out(FILE *out_str);
int   pmapbuild_lex_destroy(void);
#endif  /* #if 0 */

/*  *****  LEX INPUT FOLLOWS  *****  */

%}

%option noyywrap never-interactive nounput
%option prefix="pmapbuild_" outfile="lex.yy.c"


%s ST_EOL
%s ST_ERROR
%s ST_GETLABEL
%s ST_IPS
%s ST_LABEL
%s ST_MAPNAME
%s ST_NUMBERS
%s ST_PROPORTS


ws_opt  [ \t\r]*
ws      [ \t\r]+

nl      [ \t\r]*(#.*)?\n

number  [0-9]+

ip      [0-9]+\.[0-9]+\.[0-9]+\.[0-9]+

cidr    [0-9]+\.[0-9]+\.[0-9]+\.[0-9]+"/"[0-9]+

label   [^# \t\r\n]|[^# \t\r\n][^#\r\n]*[^# \t\r\n]

mapname [^#:, \t\r\n]+

%%

<INITIAL>mode{ws}ip{nl}           { if (stmtMode(0, SKPREFIXMAP_CONT_ADDR)) {
                                        ++error_count;
                                    }
                                    ++linenum; }

<INITIAL>mode{ws}proto-port{nl}   { if(stmtMode(0,SKPREFIXMAP_CONT_PROTO_PORT))
                                    {
                                        ++error_count;
                                    }
                                    ++linenum; }

<INITIAL>mode.*{nl}               { skAppPrintErr(("Missing or invalid mode"
                                                   " on line %d"),
                                                  linenum);
                                    ++error_count;
                                    ++linenum;
                                  }

<INITIAL>map-name{ws}             { BEGIN(ST_MAPNAME); }

<ST_MAPNAME>{mapname}             { if (stmtMapName(0, yytext))
                                    {
                                        ++error_count;
                                    }
                                    BEGIN(ST_EOL); }


<INITIAL>default{ws}              { stmtDefault(0, yytext);
                                    BEGIN(ST_GETLABEL); }

<INITIAL>label{ws}                { stmtLabel(0, yytext);
                                    BEGIN(ST_LABEL); }

<ST_LABEL>{number}{ws}            { if (stmtLabel(1, yytext)) {
                                        ++error_count;
                                        BEGIN(ST_ERROR);
                                    } else {
                                        BEGIN(ST_GETLABEL);
                                    } }

<INITIAL>{cidr}{ws}               { if (stmtCIDR(0, yytext)) {
                                        ++error_count;
                                        BEGIN(ST_ERROR);
                                    } else {
                                        BEGIN(ST_GETLABEL);
                                    } }

<INITIAL>{ip}{ws}                 { if (stmtIPs(0, yytext)) {
                                        ++error_count;
                                        BEGIN(ST_ERROR);
                                    } else {
                                        BEGIN(ST_IPS);
                                    } }

<ST_IPS>{ip}{ws}                  { if (stmtIPs(1, yytext)) {
                                        ++error_count;
                                        BEGIN(ST_ERROR);
                                    } else {
                                        BEGIN(ST_GETLABEL);
                                    } }

<INITIAL>{number}"/"{number}{ws}  { if (stmtProPorts(0, yytext)) {
                                        ++error_count;
                                        BEGIN(ST_ERROR);
                                    } else {
                                        BEGIN(ST_PROPORTS);
                                    } }

<ST_PROPORTS>{number}"/"{number}{ws}  { if (stmtProPorts(1, yytext)) {
                                            ++error_count;
                                            BEGIN(ST_ERROR);
                                        } else {
                                            BEGIN(ST_GETLABEL);
                                        } }

<INITIAL>{number}{ws}             { if (stmtNumbers(0, yytext)) {
                                        ++error_count;
                                        BEGIN(ST_ERROR);
                                    } else {
                                        BEGIN(ST_NUMBERS);
                                    } }

<ST_NUMBERS>{number}{ws}          { if (stmtNumbers(1, yytext)) {
                                        ++error_count;
                                        BEGIN(ST_ERROR);
                                    } else {
                                        BEGIN(ST_GETLABEL);
                                    } }

<ST_GETLABEL>{label}              { if (gotLabel(yytext)) {
                                        ++error_count;
                                    }
                                    BEGIN(ST_EOL); }

<INITIAL,ST_EOL>{nl}      { ++linenum; BEGIN(INITIAL); }

{ws}                      ;

<INITIAL>(mode|map-name|default|label)/{nl} {
                            skAppPrintErr("Incomplete %s statement on line %d",
                                          yytext, linenum);
                            ++error_count;
                            BEGIN(ST_ERROR); }

<ST_LABEL,ST_GETLABEL,ST_IPS,ST_PROPORTS,ST_NUMBERS,ST_MAPNAME>\n  {
                            skAppPrintErr("Incomplete statement on line %d",
                                          linenum);
                            ++error_count;
                            ++linenum;
                            BEGIN(INITIAL); }

.                         { skAppPrintErr("Unrecognized input on line %d",
                                          linenum);
                            ++error_count;
                            BEGIN(ST_ERROR); }

<ST_ERROR>.*\n            { ++linenum; BEGIN(INITIAL); }

%%

/* FUNCTION DEFINITIONS */

/*
 *  appUsageLong();
 *
 *    Print complete usage information to USAGE_FH.  Pass this
 *    function to skOptionsSetUsageCallback(); skOptionsParse() will
 *    call this funciton and then exit the program when the --help
 *    option is given.
 */
static void appUsageLong(void)
{
#define USAGE_MSG                                                             \
    ("[SWITCHES]\n"                                                           \
     "\tCreate a binary prefix map file from a textual input file.  The\n"    \
     "\tinput is read from the named input file or from the standard\n"       \
     "\tinput.  The prefix map file is written to the named location or\n"    \
     "\tto the standard output if stdout is not connected to a terminal.\n")

    FILE *fh = USAGE_FH;

    skAppStandardUsage(fh, USAGE_MSG, appOptions, appHelp);
    skOptionsNotesUsage(fh);
}


/*
 *  appTeardown()
 *
 *    Teardown all modules, close all files, and tidy up all
 *    application state.
 *
 *    This function is idempotent.
 */
static void appTeardown(void)
{
    static int teardownFlag = 0;

    if (teardownFlag) {
        return;
    }
    teardownFlag = 1;

    /* close streams; destroy prefix map */
    skStreamDestroy(&in_stream);
    skStreamDestroy(&out_stream);
    if (map) {
        skPrefixMapDelete(map);
    }

    skAppUnregister();
}


/*
 *  appSetup(argc, argv);
 *
 *    Perform all the setup for this application include setting up
 *    required modules, parsing options, etc.  This function should be
 *    passed the same arguments that were passed into main().
 *
 *    Returns to the caller if all setup succeeds.  If anything fails,
 *    this function will cause the application to exit with a FAILURE
 *    exit status.
 */
static void appSetup(int argc, char **argv)
{
    int arg_index;
    int rv;

    /* verify same number of options and help strings */
    assert((sizeof(appHelp)/sizeof(char *)) ==
           (sizeof(appOptions)/sizeof(struct option)));

    /* register the application */
    skAppRegister(argv[0]);
    skOptionsSetUsageCallback(&appUsageLong);

    /* register the options */
    if (skOptionsRegister(appOptions, &appOptionsHandler, NULL)
        || skOptionsNotesRegister(NULL))
    {
        skAppPrintErr("Unable to register options");
        exit(EXIT_FAILURE);
    }

    /* parse options */
    arg_index = skOptionsParse(argc, argv);
    if (arg_index < 0) {
        /* options parsing should print error */
        skAppUsage(); /* never returns */
    }

    /* Complain about extra args on command line */
    if (arg_index != argc) {
        skAppPrintErr("Too many or unrecognized argument specified: '%s'",
                      argv[arg_index]);
        exit(EXIT_FAILURE);
    }

    if (atexit(appTeardown) < 0) {
        skAppPrintErr("Unable to register appTeardown() with atexit()");
        appTeardown();
        exit(EXIT_FAILURE);
    }

    /* check for input; if none specified, use stdin */
    if (!in_stream) {
        if ((rv = skStreamCreate(&in_stream, SK_IO_READ, SK_CONTENT_TEXT))
            || (rv = skStreamBind(in_stream, "stdin")))
        {
            skStreamPrintLastErr(in_stream, rv, &skAppPrintErr);
            exit(EXIT_FAILURE);
        }
    }

    /* open input stream */
    rv = skStreamOpen(in_stream);
    if (rv) {
        skStreamPrintLastErr(in_stream, rv, &skAppPrintErr);
        exit(EXIT_FAILURE);
    }

    /* output is "stdout" if none specified */
    if (!out_stream) {
        if ((rv = skStreamCreate(&out_stream, SK_IO_WRITE, SK_CONTENT_SILK))
            || (rv = skStreamBind(out_stream, "stdout")))
        {
            skStreamPrintLastErr(out_stream, rv, &skAppPrintErr);
            exit(EXIT_FAILURE);
        }
    }

    return; /* OK */
}


/*
 *  status = appOptionsHandler(cData, opt_index, opt_arg);
 *
 *    Called by skOptionsParse(), this handles a user-specified switch
 *    that the application has registered, typically by setting global
 *    variables.  Returns 1 if the switch processing failed or 0 if it
 *    succeeded.  Returning a non-zero from from the handler causes
 *    skOptionsParse() to return a negative value.
 *
 *    The clientData in 'cData' is typically ignored; 'opt_index' is
 *    the index number that was specified as the last value for each
 *    struct option in appOptions[]; 'opt_arg' is the user's argument
 *    to the switch for options that have a REQUIRED_ARG or an
 *    OPTIONAL_ARG.
 */
static int appOptionsHandler(
    clientData  UNUSED(cData),
    int         opt_index,
    char       *opt_arg)
{
    int rv;

    switch ((appOptionsEnum)opt_index) {
      case OPT_INPUT_FILE:
        if (in_stream) {
            skAppPrintErr("May only specify --%s one time",
                          appOptions[opt_index].name);
            return 1;
        }
        if ((rv = skStreamCreate(&in_stream, SK_IO_READ, SK_CONTENT_TEXT))
            || (rv = skStreamBind(in_stream, opt_arg)))
        {
            skStreamPrintLastErr(in_stream, rv, &skAppPrintErr);
            skStreamDestroy(&in_stream);
            return 1;
        }
        break;

      case OPT_OUTPUT_FILE:
        if (out_stream) {
            skAppPrintErr("May only specify --%s one time",
                          appOptions[opt_index].name);
            return 1;
        }
        if ((rv = skStreamCreate(&out_stream, SK_IO_WRITE, SK_CONTENT_SILK))
            || (rv = skStreamBind(out_stream, opt_arg)))
        {
            skStreamPrintLastErr(out_stream, rv, &skAppPrintErr);
            skStreamDestroy(&out_stream);
            return 1;
        }
        break;
    }

    return 0; /* OK */
}


/*
 *  status = stmtCIDR(state, string);
 *
 *    Set the global 'stmt' to STMT_CIDR and parse the CIDR address in
 *    'string'.  The 'state' parameter is ignored.
 *
 *    Return 0 on success or -1 for failure.
 */
static int stmtCIDR(int UNUSED(state), char *string)
{
    skipaddr_t ip_min;
    skipaddr_t ip_max;
    uint32_t cidr;
    int rv;

    stmt.type = STMT_CIDR;

    /* parse the CIDR block */
    rv = skStringParseCIDR(&ip_min, &cidr, string);
    if (rv) {
        skAppPrintErr("Invalid CIDR block on line %d '%s': %s",
                      linenum, string, skStringParseStrerror(rv));
        return -1;
    }
#if SK_ENABLE_IPV6
    if (skipaddrIsV6(&ip_min)) {
        /* ignore IPv6 */
        return 0;
    }
#endif /* SK_ENABLE_IPV6 */

    skCIDR2IPRange(&ip_min, cidr, &ip_min, &ip_max);
    stmt.range_start = skipaddrGetV4(&ip_min);
    stmt.range_end = skipaddrGetV4(&ip_max);

    return 0;
}


/*
 *  status = stmtDefault(state, string);
 *
 *    Set the global 'stmt' to STMT_DEFAULT.  The paremters are
 *    ignored.  Return 0.
 */
static int stmtDefault(int UNUSED(state), char UNUSED(*string))
{
    stmt.type = STMT_DEFAULT;
    return 0;
}


/*
 *  status = stmtIPs(state, string);
 *
 *    Set the global 'stmt' to STMT_IPS and parse the IP address in
 *    'string'.  The 'state' parameter is 0 for the first IP on the
 *    line, and 1 for the second IP.
 *
 *    Return 0 on success or -1 for failure.
 */
static int stmtIPs(int state, char *string)
{
    skipaddr_t ip;
    int rv;

    assert(state == 0 || stmt.type == STMT_IPS);
    stmt.type = STMT_IPS;

    rv = skStringParseIP(&ip, string);
    if (rv) {
        skAppPrintErr("Invalid IP on line %d '%s': %s",
                      linenum, string, skStringParseStrerror(rv));
        return -1;
    }

    if (state == 0) {
        stmt.range_start = skipaddrGetV4(&ip);
    } else {
        stmt.range_end = skipaddrGetV4(&ip);
    }
    return 0;
}


/*
 *  status = stmtLabel(state, string);
 *
 *    Set the global 'stmt' to STMT_LABEL.  If 'state' is 0 return
 *    immediately; if state is 1, parse the label-id in 'string'.
 *
 *    Return 0 on success or -1 for failure.
 */
static int stmtLabel(int state, char *string)
{
    uint32_t val;
    int rv;

    assert(state == 0 || stmt.type == STMT_LABEL);
    stmt.type = STMT_LABEL;

    if (state == 0) {
        return 0;
    }

    rv = skStringParseUint32(&val, string, 0, SKPREFIXMAP_MAX_VALUE);
    if (rv) {
        skAppPrintErr("Invalid label on line %d '%s': %s",
                      linenum, string, skStringParseStrerror(rv));
        return -1;
    }

    stmt.value = val;
    return 0;
}


/*
 *  status = stmtMapName(state, string);
 *
 *    Set the prefix map name to 'string'.  Return 0 on success or -1
 *    for failure.
 */
static int stmtMapName(int UNUSED(state), char *string)
{
    skPrefixMapErr_t rv;

    if (skPrefixMapGetMapName(map) != NULL) {
        skAppPrintErr("Multiple map-names specified in input");
        return -1;
    }
    rv = skPrefixMapSetMapName(map, string);
    if (rv) {
        skAppPrintErr("Error setting prefix map name to '%s': %s",
                      string, skPrefixMapStrerror(rv));
        return -1;
    }
    return 0;
}


/*
 *  status = stmtMode(state, new_mode);
 *
 *    Set the global 'mode' to new_mode, unless the mode has
 *    previously been set to another value.
 *
 *    Return 0 on success or -1 for conflicting mode definitions.
 */
static int stmtMode(int UNUSED(state), skPrefixMapContent_t new_mode)
{
    if (mode == -1) {
        mode = new_mode;
        skPrefixMapSetContentType(map, mode);
    } else if (mode != (int)new_mode) {
        skAppPrintErr("Conflicting mode definition found on line %d", linenum);
        return -1;
    }

    return 0;
}


/*
 *  status = stmtNumbers(state, string);
 *
 *    Set the global 'stmt' to STMT_NUMBERS and, depending on the
 *    global 'mode', parse the text in 'string' as an IP address or as
 *    a protocol.  The 'state' parameter is 0 for the first number on
 *    the line, and 1 for the second number.
 *
 *    Return 0 on success or -1 for failure.
 */
static int stmtNumbers(int state, char *string)
{
    uint32_t proto;
    int rv;

    assert(state == 0 || stmt.type == STMT_NUMBERS);
    stmt.type = STMT_NUMBERS;

    if (mode == -1) {
        mode = DEFAULT_MODE;
    }

    if (mode == SKPREFIXMAP_CONT_ADDR) {
        skipaddr_t ip;

        rv = skStringParseIP(&ip, string);
        if (rv) {
            skAppPrintErr("Invalid IP on line %d '%s': %s",
                          linenum, string, skStringParseStrerror(rv));
            return -1;
        }

        if (state == 0) {
            stmt.range_start = skipaddrGetV4(&ip);
        } else {
            stmt.range_end = skipaddrGetV4(&ip);
        }
        return 0;
    }

    rv = skStringParseUint32(&proto, string, 0, 255);
    if (rv) {
        skAppPrintErr("Invalid protocol on line %d '%s': %s",
                      linenum, string, skStringParseStrerror(rv));
        return -1;
    }

    if (state == 0) {
        stmt.range_start = proto << 16;
    } else {
        stmt.range_end = (proto << 16) | UINT16_MAX;
    }
    return 0;
}


/*
 *  status = stmtIPs(state, string);
 *
 *    Set the global 'stmt' to STMT_PROPORTS and parse the proto/port
 *    pair in 'string'.  The 'state' parameter is 0 for the first
 *    proto/port pair on the line, and 1 for the second pair.
 *
 *    Return 0 on success or -1 for failure.
 */
static int stmtProPorts(int state, char *string)
{
    uint32_t proto;
    uint32_t port;
    int rv;

    assert(state == 0 || stmt.type == STMT_PROPORTS);
    stmt.type = STMT_PROPORTS;

    /* protocol */
    rv = skStringParseUint32(&proto, string, 0, 255);
    if (rv < 0) {
        skAppPrintErr("Invalid protocol on line %d '%s': %s",
                      linenum, string, skStringParseStrerror(rv));
        return -1;
    }

    if (rv == 0) {
        skAppPrintErr("Missing port value on line %d",
                      linenum);
        return -1;
    }

    /* rv is location of the '/' */
    string += rv + 1;

    /* port */
    rv = skStringParseUint32(&port, string, 0, UINT16_MAX);
    if (rv) {
        skAppPrintErr("Invalid port on line %d '%s': %s",
                      linenum, string, skStringParseStrerror(rv));
        return -1;
    }

    if (state == 0) {
        stmt.range_start = (proto << 16) | port;
    } else {
        stmt.range_end = (proto << 16) | port;
    }
    return 0;
}


/*
 *  status = gotLabel(label);
 *
 *    Use the global 'stmt' to decide what needs to be done with the
 *    label in 'label'.
 *
 *    Return 0 on success or non-zero for failure.
 */
static int gotLabel(char *string)
{
    static int default_set = 0;
    uint32_t dict_val;
    int rv = -1;

    /* Create an dictionary entry with a specific value */
    if (stmt.type == STMT_LABEL) {
        rv = skPrefixMapDictionaryInsert(map, stmt.value, string);
        if (rv) {
            skAppPrintErr(("Invalid label on line %d '%" PRIu32 " %s': %s"),
                          linenum, stmt.value, string,
                          skPrefixMapStrerror(rv));
        }
        return rv;
    }

    /* check if we need to initialize the mode and default */
    if (stmt.type != STMT_DEFAULT) {
        if (mode == -1) {
            mode = DEFAULT_MODE;
        }
        if (default_set == 0) {
            rv = skPrefixMapDictionarySearch(map, DEFAULT_LABEL, &dict_val);
            if (rv != SKPREFIXMAP_OK) {
                skAppPrintErr("Unable to create default value: %s",
                              skPrefixMapStrerror(rv));
                return rv;
            }
            rv = skPrefixMapSetDefaultVal(map, dict_val);
            if (rv != SKPREFIXMAP_OK) {
                skAppPrintErr("Unable to set default: %s",
                              skPrefixMapStrerror(rv));
                return rv;
            }
            default_set = 1;
        }
    }

    /* search-for or add the label */
    rv = skPrefixMapDictionarySearch(map, string, &dict_val);
    if (rv) {
        skAppPrintErr("Invalid label on line %d '%s': %s",
                      linenum, string, skPrefixMapStrerror(rv));
        return rv;
    }

    switch (stmt.type) {
      case STMT_DEFAULT:
        if (dict_val >= skPrefixMapDictionaryGetWordCount(map)) {
            skAppPrintErr("Default set to undefined index '%s'",
                          string);
            return -1;
        }
        rv = skPrefixMapSetDefaultVal(map, dict_val);
        if (rv) {
            skAppPrintErr("Invalid default on line %d: %s",
                          linenum, skPrefixMapStrerror(rv));
            return -1;
        }
        default_set = 1;
        break;

      case STMT_CIDR:
      case STMT_IPS:
        if (mode != SKPREFIXMAP_CONT_ADDR) {
            skAppPrintErr(("Invalid IP input on line %d in "
                           "proto-port prefix map input file"),
                          linenum);
            return -1;
        }
        /* FALLTHROUGH */

      case STMT_NUMBERS:
        rv = skPrefixMapAddRange(map, stmt.range_start, stmt.range_end,
                                 dict_val);
        if (rv) {
            skAppPrintErr("Cannot add entry on line %d to prefix map: %s",
                          linenum, skPrefixMapStrerror(rv));
            return -1;
        }
        break;

      case STMT_PROPORTS:
        if (mode != SKPREFIXMAP_CONT_PROTO_PORT) {
            skAppPrintErr(("Invalid proto/port input on line %d in "
                           "IP prefix map input file"),
                          linenum);
            return -1;
        }
        rv = skPrefixMapAddRange(map, stmt.range_start, stmt.range_end,
                                 dict_val);
        if (rv) {
            skAppPrintErr("Cannot add entry on line %d to prefix map: %s",
                          linenum, skPrefixMapStrerror(rv));
            return -1;
        }
        break;

      case STMT_LABEL:
        /* this was handled above; we can't get here */
        skAbortBadCase(stmt.type);
    }

    return 0;
}


int main(int argc, char **argv)
{
    int fd;
    int rv;

    appSetup(argc, argv);       /* never returns on error */

    /* Create the global prefix map */
    if (skPrefixMapCreate(&map) != SKPREFIXMAP_OK) {
        skAppPrintErr("Error creating prefix map");
        exit(EXIT_FAILURE);
    }

    /* Process input */
    fd = skStreamGetDescriptor(in_stream);
    assert(-1 != fd);
    yyin = fdopen(fd, "r");
    if (NULL == yyin) {
        skAppPrintSyserror("Error getting handle to stream '%s'",
                           skStreamGetPathname(in_stream));
        exit(EXIT_FAILURE);
    }
    yylex();

    if (error_count) {
        skAppPrintErr(("Encountered %d warnings during processing."
                       "  Output not written."),
                      error_count);
        exit(EXIT_FAILURE);
    }

    /* add notes if given */
    rv = skOptionsNotesAddToStream(out_stream);
    if (rv) {
        skStreamPrintLastErr(out_stream, rv, &skAppPrintErr);
        exit(EXIT_FAILURE);
    }
    skOptionsNotesTeardown();

    /* write output */
    rv = skStreamOpen(out_stream);
    if (rv) {
        skStreamPrintLastErr(out_stream, rv, &skAppPrintErr);
        exit(EXIT_FAILURE);
    }

    rv = skPrefixMapWrite(map, out_stream);
    if (rv != SKPREFIXMAP_OK) {
        skAppPrintErr("Error writing prefix map to '%s': %s",
                      skStreamGetPathname(out_stream),skPrefixMapStrerror(rv));
        exit(EXIT_FAILURE);
    }

    rv = skStreamClose(out_stream);
    if (rv) {
        skStreamPrintLastErr(out_stream, rv, &skAppPrintErr);
        exit(EXIT_FAILURE);
    }

    appTeardown();
    return 0;
}


/*
** Local Variables:
** mode:c
** indent-tabs-mode:nil
** c-basic-offset:4
** End:
*/
