/*
** Copyright (C) 2004-2012 by Carnegie Mellon University.
**
** @OPENSOURCE_HEADER_START@
**
** Use of the SILK system and related source code is subject to the terms
** of the following licenses:
**
** GNU Public License (GPL) Rights pursuant to Version 2, June 1991
** Government Purpose License Rights (GPLR) pursuant to DFARS 252.227.7013
**
** NO WARRANTY
**
** ANY INFORMATION, MATERIALS, SERVICES, INTELLECTUAL PROPERTY OR OTHER
** PROPERTY OR RIGHTS GRANTED OR PROVIDED BY CARNEGIE MELLON UNIVERSITY
** PURSUANT TO THIS LICENSE (HEREINAFTER THE "DELIVERABLES") ARE ON AN
** "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY
** KIND, EITHER EXPRESS OR IMPLIED AS TO ANY MATTER INCLUDING, BUT NOT
** LIMITED TO, WARRANTY OF FITNESS FOR A PARTICULAR PURPOSE,
** MERCHANTABILITY, INFORMATIONAL CONTENT, NONINFRINGEMENT, OR ERROR-FREE
** OPERATION. CARNEGIE MELLON UNIVERSITY SHALL NOT BE LIABLE FOR INDIRECT,
** SPECIAL OR CONSEQUENTIAL DAMAGES, SUCH AS LOSS OF PROFITS OR INABILITY
** TO USE SAID INTELLECTUAL PROPERTY, UNDER THIS LICENSE, REGARDLESS OF
** WHETHER SUCH PARTY WAS AWARE OF THE POSSIBILITY OF SUCH DAMAGES.
** LICENSEE AGREES THAT IT WILL NOT MAKE ANY WARRANTY ON BEHALF OF
** CARNEGIE MELLON UNIVERSITY, EXPRESS OR IMPLIED, TO ANY PERSON
** CONCERNING THE APPLICATION OF OR THE RESULTS TO BE OBTAINED WITH THE
** DELIVERABLES UNDER THIS LICENSE.
**
** Licensee hereby agrees to defend, indemnify, and hold harmless Carnegie
** Mellon University, its trustees, officers, employees, and agents from
** all claims or demands made against them (and any related losses,
** expenses, or attorney's fees) arising out of, or relating to Licensee's
** and/or its sub licensees' negligent use or willful misuse of or
** negligent conduct or willful misconduct regarding the Software,
** facilities, or other rights or assistance granted by Carnegie Mellon
** University under this License, including, but not limited to, any
** claims of product liability, personal injury, death, damage to
** property, or violation of any laws or regulations.
**
** Carnegie Mellon University Software Engineering Institute authored
** documents are sponsored by the U.S. Department of Defense under
** Contract FA8721-05-C-0003. Carnegie Mellon University retains
** copyrights in all material produced under this contract. The U.S.
** Government retains a non-exclusive, royalty-free license to publish or
** reproduce these documents, or allow others to do so, for U.S.
** Government purposes only pursuant to the copyright license under the
** contract clause at 252.227.7013.
**
** @OPENSOURCE_HEADER_END@
*/

/*
**  Functions to create and read from a UDP socket.
**
*/


#include <silk/silk.h>

RCSIDENT("$SiLK: udpsource.c 372a8bc31d8a 2012-02-10 21:55:28Z mthomas $");

#if SK_ENABLE_ZLIB
#include <zlib.h>
#endif
#include <silk/skvector.h>
#include <silk/sklog.h>
#include "circbuf.h"
#include "udpsource.h"



typedef struct udp_data_store_st {
    /* address in network byte order */
    in_addr_t       net_address;
    circBuf_t       data_buffer;
    void           *tmp_buffer;
    unsigned        stopped : 1;
} udp_data_store_t;


typedef struct udp_source_struct_st {
    sk_vector_t    *store;
    uint8_t        *file_buffer;
    pthread_t       thread;
    pthread_mutex_t mutex;
    pthread_cond_t  cond;
    size_t          data_size;
    uint32_t        bufsize;
#if SK_ENABLE_ZLIB
    gzFile          udpfile;
#else
    FILE           *udpfile;
#endif
    int             fd;
    int             read_pipe;
    int             write_pipe;
    uint8_t         ref;
    unsigned        file    : 1;
    unsigned        stopped : 1;
} udp_source_struct_t;
/* typedef struct udp_source_struct_st *udpSource_t;  // udpsource.h */


static void *udp_reader(void *vsource)
{
    udpSource_t source = (udpSource_t)vsource;
    sigset_t sigs;
    int maxfd;
    struct sockaddr_in addr;
    uint8_t i;
    void *data;
    udp_data_store_t *store;
    socklen_t len;

    assert(source != NULL);

    /* Lock for initialization */
    pthread_mutex_lock(&source->mutex);

    /* Don't handle signals */
    sigfillset(&sigs);
    pthread_sigmask(SIG_SETMASK, &sigs, NULL);

    /* Disable cancelling */
    pthread_setcancelstate(PTHREAD_CANCEL_DISABLE, NULL);

    /* Determine maximum file descriptor for future select */
    maxfd = ((source->fd > source->read_pipe) ?
             source->fd : source->read_pipe) + 1;

    data = (void *)malloc(source->data_size);
    assert(data != NULL);

    /* Signal completion of initialization */
    pthread_cond_signal(&source->cond);
    pthread_mutex_unlock(&source->mutex);

    /* Main loop */
    while (!source->stopped) {
        int count;
        fd_set readset;
        int rv;

        /* Set up select data (must be redone every loop) */
        FD_ZERO(&readset);
        FD_SET(source->fd, &readset);
        FD_SET(source->read_pipe, &readset);

        /* Wait for data */
        rv = select(maxfd, &readset, NULL, NULL, NULL);
        if (rv == -1) {
            if (errno == EINTR) {
                /* Interrupted by a signal, try again. */
                continue;
            }
            /* Error */
            ERRMSG("Select error (%d) [%s]", errno, strerror(errno));
            break;
        }

        /* See if we were requested to stop */
        if (FD_ISSET(source->read_pipe, &readset)) {
            continue;
        }

        /* We must have gotten real data */
        assert(FD_ISSET(source->fd, &readset));

        /* Read the data */
        len = sizeof(addr);
        rv = recvfrom(source->fd, data, source->data_size, 0,
                      (struct sockaddr *)&addr, &len);
        /* Check for error or recv from wrong address */
        if (rv == -1) {
            switch (errno) {
              case EINTR:
                /* Interrupted by a signal, try again. */
                continue;
              case EAGAIN:
                /* We should not be getting this, but have seen them
                 * in the field nonetheless.  Note and ignore them. */
                NOTICEMSG("Ignoring spurious EAGAIN from recvfrom() call");
                continue;
              default:
                ERRMSG("recvfrom error (%d) [%s]", errno, strerror(errno));
                goto BREAK_WHILE;
            }
        }

        pthread_mutex_lock(&source->mutex);
        /* Loop through, looking for the source with the address that
         * matches */
        count = skVectorGetCount(source->store);
        for (i = 0; i < count; i++) {
            skVectorGetValue(&store, source->store, i);
            if (!store->stopped &&
                (store->net_address == INADDR_ANY ||
                 store->net_address == addr.sin_addr.s_addr))
            {
                /* Copy the data */
                memcpy(store->tmp_buffer, data, source->data_size);

                /* Unlock before calling circBufNextHead(), since it
                 * can block */
                pthread_mutex_unlock(&source->mutex);

                /* Acquire the next location */
                store->tmp_buffer = (void*)circBufNextHead(store->data_buffer);
                if (store->tmp_buffer == NULL) {
                    NOTICEMSG("Non-existent data buffer,");
                    goto BREAK_WHILE;
                }
                break;
            }
        }

        if (i == count) {
            /* Unlock if we didn't a match for the address (otherwise
             * we will have already unlocked */
            pthread_mutex_unlock(&source->mutex);
        }

    } /* while */

  BREAK_WHILE:

    free(data);

    INFOMSG("UDP listener stopped");

    pthread_mutex_lock(&source->mutex);
    /* Wait to be explicitly stopped */
    while (!source->stopped) {
        pthread_cond_wait(&source->cond, &source->mutex);
    }
    /* Signal that we are finished */
    pthread_cond_signal(&source->cond);
    pthread_mutex_unlock(&source->mutex);

    return NULL;
}


udpSource_t udpSourceCreate(
    int         fd,
    in_addr_t   address,
    uint32_t    itemsize,
    uint32_t    bufsize)
{
    udpSource_t source = NULL;
    int pipefd[2] = {-1,-1};
    int flags;
    udp_data_store_t *store;

    /* Create thread structure */
    source = (udpSource_t)calloc(1, sizeof(udp_source_struct_t));
    if (source == NULL) {
        goto ERROR;
    }

    /* Fill the data structure */
    source->file = 0;             /* Not a file */
    source->fd = fd;
    source->stopped = 0;
    source->data_size = itemsize;
    source->bufsize = bufsize;
    source->ref = 1;
    pthread_mutex_init(&source->mutex, NULL);
    pthread_cond_init(&source->cond, NULL);

    if (pipe(pipefd) == -1) {
        pipefd[0] = -1;
        goto ERROR;
    }
    /* Make write pipe non-blocking */
    flags = fcntl(pipefd[1], F_GETFL, 0);
    fcntl(pipefd[1], F_SETFL, flags | O_NONBLOCK);

    source->store = skVectorNew(sizeof(udp_data_store_t *));
    if (source->store == NULL) {
        goto ERROR;
    }
    if (udpSourceAddAddress(source, address) == -1) {
        goto ERROR;
    }

    source->read_pipe = pipefd[0];
    source->write_pipe = pipefd[1];

    /* Start the thread */
    pthread_mutex_lock(&source->mutex);
    if (pthread_create(&source->thread, NULL, udp_reader, (void *)source)
        != 0)
    {
        pthread_mutex_unlock(&source->mutex);
        goto ERROR;
    }

    /* Wait for the thread to finish initializing before returning. */
    pthread_cond_wait(&source->cond, &source->mutex);
    pthread_mutex_unlock(&source->mutex);

    return source;

  ERROR:
    close(fd);
    if (source) {
        pthread_mutex_destroy(&source->mutex);
        pthread_cond_destroy(&source->cond);
        if (pipefd[0] != -1) {
            close(pipefd[0]);
            close(pipefd[1]);
        }
        if (source->store) {
            if (skVectorGetValue(&store, source->store, 0) == 0) {
                circBufDestroy(store->data_buffer);
            }
            skVectorDestroy(source->store);
        }
        free(source);
    }
    return NULL;
}


/* add 'address' to 'source'.  return its position in the vector */
int udpSourceAddAddress(udpSource_t source, in_addr_t address)
{
    udp_data_store_t *store;
    int retval;
    int rv;

    if (source == NULL || source->file != 0) {
        return -1;
    }

    store = (udp_data_store_t *)calloc(1, sizeof(*store));
    if (store == NULL) {
        return -1;
    }

    store->data_buffer = circBufCreate(source->data_size, source->bufsize);
    if (store->data_buffer == NULL) {
        free(store);
        return -1;
    }

    store->tmp_buffer = (void *)circBufNextHead(store->data_buffer);
    assert(store->tmp_buffer != NULL);

    store->net_address = htonl(address);

    pthread_mutex_lock(&source->mutex);
    rv = skVectorAppendValue(source->store, &store);
    if (rv == -1) {
        pthread_mutex_unlock(&source->mutex);
        circBufDestroy(store->data_buffer);
        free(store);
        return -1;
    }
    retval = skVectorGetCount(source->store) - 1;
    pthread_mutex_unlock(&source->mutex);

    return retval;
}


udpSource_t udpFileSourceCreate(
    const char *path,
    uint32_t    itemsize)
{
    udpSource_t source;

    /* Create thread structure */
    source = (udpSource_t)malloc(sizeof(udp_source_struct_t));
    if (source == NULL) {
        return NULL;
    }

    source->file = 1;             /* Source is a file */
    source->stopped = 0;
    source->data_size = itemsize;

#if SK_ENABLE_ZLIB
    source->udpfile = gzopen(path, "r");
#else
    source->udpfile = fopen(path, "r");
#endif

    if (source->udpfile == NULL) {
        ERRMSG("Unable to open file '%s'", path);
        free(source);
        return NULL;
    }

    source->file_buffer = (uint8_t *)malloc(source->data_size);
    if (source->file_buffer == NULL) {
#if SK_ENABLE_ZLIB
        gzclose(source->udpfile);
#else
        fclose(source->udpfile);
#endif
        free(source);
        return NULL;
    }

    pthread_mutex_init(&source->mutex, NULL);

    return source;
}


void udpSourceStop(udpSource_t source, int idx)
{
    udp_data_store_t *store;

    pthread_mutex_lock(&source->mutex);
    if (source->file) {
        source->stopped = 1;
    } else {
        skVectorGetValue(&store, source->store, idx);
        if (!store->stopped) {
            store->stopped = 1;
            circBufStop(store->data_buffer);
        }
        /* Break out of any blocking io */
        write(source->write_pipe, "", 1);
    }

    pthread_mutex_unlock(&source->mutex);
}


void udpSourceDestroy(udpSource_t source, int idx)
{
    uint8_t i;
    udp_data_store_t *store;

    if (!source->file) {
        skVectorGetValue(&store, source->store, idx);
        /* Break out of waits for buffer space */
        if (!store->stopped) {
            store->stopped = 1;
            circBufStop(store->data_buffer);
        }
        if (--source->ref > 0) {
            return;
        }
    }
    pthread_mutex_lock(&source->mutex);
    if (!source->stopped) {
        /* Signal that we are stopped */
        source->stopped = 1;
        if (!source->file) {
            /* Break out of any blocking io */
            write(source->write_pipe, "", 1);
        }
    }
    if (!source->file) {
        /* Wait for thread to end */
        pthread_cond_wait(&source->cond, &source->mutex);
        /* Close all handles */
        close(source->fd);
        close(source->write_pipe);
        close(source->read_pipe);
        /* Unlock and destroy mutex */
        for (i = 0; i < skVectorGetCount(source->store); i++) {
            skVectorGetValue(&store, source->store, i);
            circBufDestroy(store->data_buffer);
            free(store);
        }
        skVectorDestroy(source->store);
    } else {
        /* Close the file */
#if SK_ENABLE_ZLIB
        gzclose(source->udpfile);
#else
        fclose(source->udpfile);
#endif
        free(source->file_buffer);
    }
    pthread_mutex_unlock(&source->mutex);
    pthread_mutex_destroy(&source->mutex);
    if (!source->file) {
        pthread_cond_destroy(&source->cond);
    }
    /* Destroy source */
    free(source);
}


uint8_t *udpNextByIndex(udpSource_t source, int idx)
{
    uint8_t *data;

    assert(source);

    pthread_mutex_lock(&source->mutex);

    if (source->stopped) {
        data = NULL;
        goto END;
    }

    if (source->file) {
        int size;
#if SK_ENABLE_ZLIB
        size = gzread(source->udpfile, source->file_buffer, source->data_size);
#else
        size = (int)fread(source->file_buffer, 1, source->data_size,
                          source->udpfile);
#endif
        if (size <= 0 || (uint32_t)size < source->data_size) {
            data = NULL;
        } else {
            data = source->file_buffer;
        }
    } else {
        udp_data_store_t *store;
        int rv;

        rv = skVectorGetValue(&store, source->store, idx);
        if (rv == -1) {
            data = NULL;
        } else {
            pthread_mutex_unlock(&source->mutex);
            if (store->data_buffer) {
                return circBufNextTail(store->data_buffer);
            }
            return NULL;
        }
    }

  END:
    pthread_mutex_unlock(&source->mutex);

    return data;
}


uint8_t *udpNext(udpSource_t source)
{
    return udpNextByIndex(source, 0);
}


void udpSourceIncRef(udpSource_t source)
{
    source->ref++;
}

int udpSourceGetSocket(udpSource_t source)
{
    if (source->file) {
        return -1;
    }
    return source->fd;
}

/*
** Local Variables:
** mode:c
** indent-tabs-mode:nil
** c-basic-offset:4
** End:
*/
