/*
** Copyright (C) 2004-2012 by Carnegie Mellon University.
**
** @OPENSOURCE_HEADER_START@
**
** Use of the SILK system and related source code is subject to the terms
** of the following licenses:
**
** GNU Public License (GPL) Rights pursuant to Version 2, June 1991
** Government Purpose License Rights (GPLR) pursuant to DFARS 252.227.7013
**
** NO WARRANTY
**
** ANY INFORMATION, MATERIALS, SERVICES, INTELLECTUAL PROPERTY OR OTHER
** PROPERTY OR RIGHTS GRANTED OR PROVIDED BY CARNEGIE MELLON UNIVERSITY
** PURSUANT TO THIS LICENSE (HEREINAFTER THE "DELIVERABLES") ARE ON AN
** "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY
** KIND, EITHER EXPRESS OR IMPLIED AS TO ANY MATTER INCLUDING, BUT NOT
** LIMITED TO, WARRANTY OF FITNESS FOR A PARTICULAR PURPOSE,
** MERCHANTABILITY, INFORMATIONAL CONTENT, NONINFRINGEMENT, OR ERROR-FREE
** OPERATION. CARNEGIE MELLON UNIVERSITY SHALL NOT BE LIABLE FOR INDIRECT,
** SPECIAL OR CONSEQUENTIAL DAMAGES, SUCH AS LOSS OF PROFITS OR INABILITY
** TO USE SAID INTELLECTUAL PROPERTY, UNDER THIS LICENSE, REGARDLESS OF
** WHETHER SUCH PARTY WAS AWARE OF THE POSSIBILITY OF SUCH DAMAGES.
** LICENSEE AGREES THAT IT WILL NOT MAKE ANY WARRANTY ON BEHALF OF
** CARNEGIE MELLON UNIVERSITY, EXPRESS OR IMPLIED, TO ANY PERSON
** CONCERNING THE APPLICATION OF OR THE RESULTS TO BE OBTAINED WITH THE
** DELIVERABLES UNDER THIS LICENSE.
**
** Licensee hereby agrees to defend, indemnify, and hold harmless Carnegie
** Mellon University, its trustees, officers, employees, and agents from
** all claims or demands made against them (and any related losses,
** expenses, or attorney's fees) arising out of, or relating to Licensee's
** and/or its sub licensees' negligent use or willful misuse of or
** negligent conduct or willful misconduct regarding the Software,
** facilities, or other rights or assistance granted by Carnegie Mellon
** University under this License, including, but not limited to, any
** claims of product liability, personal injury, death, damage to
** property, or violation of any laws or regulations.
**
** Carnegie Mellon University Software Engineering Institute authored
** documents are sponsored by the U.S. Department of Defense under
** Contract FA8721-05-C-0003. Carnegie Mellon University retains
** copyrights in all material produced under this contract. The U.S.
** Government retains a non-exclusive, royalty-free license to publish or
** reproduce these documents, or allow others to do so, for U.S.
** Government purposes only pursuant to the copyright license under the
** contract clause at 252.227.7013.
**
** @OPENSOURCE_HEADER_END@
*/

/*
**  Interface to pull a single flow from a NetFlow v5 PDU
**
*/


#include <silk/silk.h>

RCSIDENT("$SiLK: pdusource.c 372a8bc31d8a 2012-02-10 21:55:28Z mthomas $");

#include <silk/utils.h>
#include <silk/libflowsource.h>
#include <silk/redblack.h>
#include <silk/sklog.h>
#include "v5pdu.h"
#include "udpsource.h"

#ifdef PDUSOURCE_TRACE_LEVEL
  #define TRACEMSG_LEVEL 1
  #define SAVE_LAST_TIMESTAMP 1
#else
  #define SAVE_LAST_TIMESTAMP 0
#endif
#define TRACEMSG(msg) TRACEMSG_TO_TRACEMSGLVL(1, msg)
#include <silk/sktracemsg.h>


/* Number of milliseconds the calculated router boot time for a PDU
 * packet must be beyond the last packet in order to consider the
 * router to have rebooted. */
#define ROUTER_BOOT_FUZZ 1e3

/* Per-engine data structures for a Netflow v5 stream */
typedef struct pdu_engine_info_st {
    uint16_t id;
    uint32_t flow_sequence;
    /* router boot time as milliseconds since the UNIX epoch */
    intmax_t router_boot;
    /* milliseconds since the router booted */
    intmax_t sysUptime;
#if SAVE_LAST_TIMESTAMP
    /* Timestamp of last pdu */
    sktime_t last_timestamp;
#endif
} pdu_engine_info_t;

typedef struct pdu_source_st {
    flowsource_stats_t  statistics;
    pthread_mutex_t     stats_mutex;

    udpSource_t         source;
    int                 source_index;

    v5PDU              *pdu;    /* Current pdu */

    struct rbtree      *engine_info_tree; /* Per-engine data */
    pdu_engine_info_t  *engine_info;      /* Per-engine data for most
                                           * recent engine */

    int                 sockbufsize;
    in_addr_t           listen_addr;
    int                 port;

    uint8_t             count;  /* Number of recs left in pdu */

    uint8_t             logopt;

    unsigned            file    : 1; /* File-based source */
    unsigned            stopped : 1;
} pdu_source_t;
/* typedef struct pdu_source_st *pduSource_t;   // libflowsource.h */


typedef struct pdu_source_pool_item_st {
    const skpc_probe_t *probe;
    udpSource_t         source;
    int                 index;
} pdu_source_pool_item_t;


#define COUNT_BAD_PACKET(source, pdu)                   \
    {                                                   \
        pthread_mutex_lock(&((source)->stats_mutex));   \
        (source)->statistics.badPkts++;                 \
        pthread_mutex_unlock(&((source)->stats_mutex)); \
    }

#define COUNT_BAD_RECORD(source, pdu)                   \
    {                                                   \
        pthread_mutex_lock(&((source)->stats_mutex));   \
        (source)->statistics.badRecs++;                 \
        pthread_mutex_unlock(&((source)->stats_mutex)); \
    }


/*
 *  TIME VALUES IN THE NETFLOW V5 PDU
 *
 *  The naive ordering of events with respect to time in the router
 *  would be to collect the flows and generate the PDU.  Thus, one
 *  would expect:
 *
 *      flow.Start  <  flow.End  <  hdr.sysUptime
 *
 *  where all values are given as milliseconds since the router's
 *  interface was booted, and hdr.sysUptime is advertised as the
 *  "current" time.
 *
 *  However, since values are given as 32bit numbers, the values will
 *  roll-over after about 49.7 days.  If the values roll-over in the
 *  middle of writing the PDU, we will see one of these two
 *  conditions:
 *
 *      hdr.sysUptime  <<  flow.Start  <  flow.End
 *
 *      flow.End  <  hdr.sysUptime  <<  flow.Start
 *
 *  Thus, if flow.End less than flow.Start, we need to account for the
 *  roll-over when computing the flow's duration.
 *
 *  In practice, the PDU's header gets filled in before flows are
 *  added, making the hdr.sysUptime not have any true time ordering
 *  with respect to the flow.Start and flow.End, and we have seen
 *  cases in real NetFlow data where hdr.sysUptime is slightly less
 *  than flow.End:
 *
 *      flow.Start  <  hdr.sysUptime  <  flow.End
 *
 *  Moreover, some naive NetFlow PDU generators simply pin the
 *  hdr.sysUptime to zero, and don't account for rollover at all.
 *  This can make hdr.sysUptime much less than flow.Start.
 *
 *  In order to make the determination whether the flow.Start or
 *  hdr.sysUptime values have overflown their values and rolled-over,
 *  we look at the difference between them.  If the absolute value of
 *  the difference is greater than some very large maximum defined in
 *  maximumFlowTimeDeviation (currently 45 days), we assume that one
 *  of the two has rolled over, and adjust based on that assumption.
 */
static const intmax_t maximumFlowTimeDeviation =
    (intmax_t)45 * 24 * 3600 * 1000; /* 45 days */

static const intmax_t maximumSequenceDeviation =
    (intmax_t)3600 * 1000; /* 1 hour of flows at 1k flows/sec */

#define ROLLOVER32 ((intmax_t)UINT32_MAX + 1)


/* FUNCTION PROTOTYPES */

static udpSource_t pduUdpSourceCreate(
    int             port,
    in_addr_t       from_address,
    in_addr_t       listen_address,
    uint32_t        max_pkts,
    int             sockbufsize);


/* FUNCTION DEFINITIONS */


static int pdu_engine_compare(
    const void *va,
    const void *vb,
    const void *UNUSED(ctx))
{
    const pdu_engine_info_t *a = va;
    const pdu_engine_info_t *b = vb;

    if (a->id < b->id) {
        return -1;
    }
    return (a->id > b->id);
}


pduSource_t pduSourceCreate(
    int             port,
    in_addr_t       from_address,
    in_addr_t       listen_address,
    uint32_t        max_pkts,
    int             sockbufsize)
{
    udpSource_t udpsource;
    pduSource_t source;

    udpsource = pduUdpSourceCreate(port,
                                   from_address,
                                   listen_address,
                                   max_pkts,
                                   sockbufsize);
    if (udpsource == NULL) {
        return NULL;
    }

    source = (pduSource_t)calloc(1, sizeof(*source));
    if (source == NULL) {
        udpSourceDestroy(udpsource, 0);
        return NULL;
    }

    source->engine_info_tree = rbinit(pdu_engine_compare, NULL);
    if (source->engine_info_tree == NULL) {
        free(source);
        udpSourceDestroy(udpsource, 0);
        return NULL;
    }

    source->source = udpsource;
    source->listen_addr = listen_address;
    source->sockbufsize = sockbufsize;
    source->port = port;
    pthread_mutex_init(&source->stats_mutex, NULL);
    source->logopt = SOURCE_LOG_ALL;

    return source;
}


pduSource_t pduFileSourceCreate(
    const char     *path)
{
    udpSource_t udpsource;
    pduSource_t source;

    udpsource = udpFileSourceCreate(path, V5PDU_LEN);
    if (udpsource == NULL) {
        return NULL;
    }

    source = (pduSource_t)calloc(1, sizeof(*source));
    if (source == NULL) {
        udpSourceDestroy(udpsource, 0);
        return NULL;
    }

    source->engine_info_tree = rbinit(pdu_engine_compare, NULL);
    if (source->engine_info_tree == NULL) {
        free(source);
        udpSourceDestroy(udpsource, 0);
        return NULL;
    }

    source->file = 1;
    source->source = udpsource;

    return source;
}


void pduSourceStop(pduSource_t source)
{
    source->stopped = 1;
    udpSourceStop(source->source, source->source_index);
}


void pduSourceDestroy(pduSource_t source)
{
    RBLIST *iter;
    pdu_engine_info_t *engine_info;

    source->stopped = 1;
    udpSourceDestroy(source->source, source->source_index);
    pthread_mutex_destroy(&source->stats_mutex);
    iter = rbopenlist(source->engine_info_tree);
    if (iter != NULL) {
        while ((engine_info = (pdu_engine_info_t *)rbreadlist(iter))) {
            free(engine_info);
        }
    }
    rbcloselist(iter);
    rbdestroy(source->engine_info_tree);
    free(source);
}


static v5PDU *pduNext(pduSource_t source)
{
    /* keep this array and the enum in sync.  we use these and the
     * macro to count the number of invalid PDUs we saw consecutively,
     * and we print a log message for the bad PDUs as a group.  */
    static const char *err_msgs[] = {
        "No Error",
        "not marked as version 5",
        "reporting more than " V5PDU_MAX_RECS_STR " records",
        "reporting zero records"
    };
    enum err_status_en {
        PDU_OK = 0,
        PDU_BAD_VERSION,
        PDU_ZERO_RECORDS,
        PDU_OVERFLOW_RECORDS
    } err_status = PDU_OK;
    uint32_t err_count = 0;

#define LOG_BAD_PDUS(new_err_status)                            \
    if (new_err_status == err_status) {                         \
        ++err_count;                                            \
    } else {                                                    \
        if (err_count) {                                        \
            assert(PDU_OK != err_status);                       \
            NOTICEMSG(("Rejected %" PRIu32 " PDU record%s %s"), \
                      err_count, ((err_count == 1) ? "" : "s"), \
                      err_msgs[err_status]);                    \
        }                                                       \
        err_count = 1;                                          \
        err_status = new_err_status;                            \
    }

    assert(source != NULL);

    /* Infloop; exit by return only */
    for (;;) {
        v5PDU             *pdu;
        intmax_t           now;
        uint16_t           count;
        uint32_t           flow_sequence;
        intmax_t           router_boot;
        intmax_t           sysUptime;
        pdu_engine_info_t *engine;
        pdu_engine_info_t  target;

        pdu = (v5PDU *)udpNextByIndex(source->source, source->source_index);
        if (pdu == NULL) {
            /* if we saw any bad PDUs, print message before returning */
            LOG_BAD_PDUS(PDU_OK);
            return NULL;
        }

        pthread_mutex_lock(&source->stats_mutex);
        source->statistics.procPkts++;
        pthread_mutex_unlock(&source->stats_mutex);

        if (ntohs(pdu->hdr.version) != 5) {
            /* reject packet */
            LOG_BAD_PDUS(PDU_BAD_VERSION);
            COUNT_BAD_PACKET(source, pdu);
            continue;
        }

        count = ntohs(pdu->hdr.count);

        if (count > V5PDU_MAX_RECS) {
            /* reject packet */
            LOG_BAD_PDUS(PDU_OVERFLOW_RECORDS);
            COUNT_BAD_PACKET(source, pdu);
            continue;
        }

        if (count == 0) {
            /* reject packet */
            LOG_BAD_PDUS(PDU_ZERO_RECORDS);
            COUNT_BAD_PACKET(source, pdu);
            continue;
        }

        /* at this point we are guaranteed to return from this
         * function.  Use the PDU_OK "error" to log about any bad PDUs
         * we saw previously */
        LOG_BAD_PDUS(PDU_OK);

        /* use the PDU header to get the "current" time as
         * milliseconds since the UNIX epoch; round nanoseconds by
         * adding 5e5 before dividing. */
        now = ((intmax_t)1000 * ntohl(pdu->hdr.unix_secs)
               + ((ntohl(pdu->hdr.unix_nsecs) + 5e5L) / 1e6L));

        /* get sysUptime, which is the "current" time in milliseconds
         * since the export device booted */
        sysUptime = ntohl(pdu->hdr.SysUptime);

        /* subtract sysUptime from current-time to get router boot
         * time as milliseconds since UNIX epoch */
        router_boot = now - sysUptime;

        /* get the sequence number */
        flow_sequence = ntohl(pdu->hdr.flow_sequence);

        /* Determine the current engine */
        target.id = ((uint16_t)pdu->hdr.engine_type << 8) | pdu->hdr.engine_id;
        engine = source->engine_info;
        if (engine == NULL || engine->id != target.id) {
            /* Current engine info must be updated */
            engine = (pdu_engine_info_t *)
                     rbfind(&target, source->engine_info_tree);
            if (engine == NULL) {
                /* There's no entry for this engine.  Add one */
                TRACEMSG(("New engine %" PRIu16 " noticed", target.id));
                engine = calloc(1, sizeof(*engine));
                if (engine == NULL) {
                    ERRMSG("Memory allocation error allocating PDU engine."
                           "  Aborting.");
                    exit(EXIT_FAILURE);
                }
                engine->id = target.id;
                engine->router_boot = router_boot;
                engine->sysUptime = sysUptime;
                engine->flow_sequence = flow_sequence;
                rbsearch(engine, source->engine_info_tree);
            }
            source->engine_info = engine;
        }

        /* check for router reboot.  Determine whether the absolute
         * value of
         *   (router_boot - engine->last_router_boot)
         * is greater than ROUTER_BOOT_FUZZ.  If so, assume router
         * rebooted and reset the engine values. */
        if (((router_boot > engine->router_boot)
             && ((router_boot - engine->router_boot) > ROUTER_BOOT_FUZZ))
            || ((router_boot - engine->router_boot) < -ROUTER_BOOT_FUZZ))
        {
            DEBUGMSG(("Router reboot for engine %" PRIu16 ":"
                      " Last time %" PRIdMAX
                      " Current time %" PRIdMAX),
                     target.id,
                     engine->router_boot,
                     router_boot);
            engine->flow_sequence = flow_sequence;
        }
        engine->router_boot = router_boot;
        engine->sysUptime = sysUptime;

        /* handle seq numbers here */
        if (flow_sequence == engine->flow_sequence) {
            /* This packet is in sequence.  Update the next expected seq */
            engine->flow_sequence = flow_sequence + count;
        } else {
            intmax_t expected;
            intmax_t actual;

            pthread_mutex_lock(&source->stats_mutex);

            actual = flow_sequence;
            expected = engine->flow_sequence;

            /* Try to account for sequence number rollover */
            if (actual > expected) {
                if ((actual - expected) > maximumSequenceDeviation) {
                    expected += ROLLOVER32;
                }
            } else if ((expected - actual) > maximumSequenceDeviation) {
                actual += ROLLOVER32;
            }

            if (actual < expected) {
                /*
                 * Out of sequence packet.  Reduce missing flow count.
                 * However, do not change the expected seq num.
                 */
                source->statistics.missingRecs -= count;
            } else {
                /* Increase missing flow count */
                source->statistics.missingRecs += actual - expected;
                if (source->logopt & SOURCE_LOG_MISSING) {
                    uint64_t allrecs = (source->statistics.goodRecs +
                                        source->statistics.badRecs +
                                        source->statistics.missingRecs);
                    INFOMSG(("Missing netflow records: "
                             "%" PRId64 "/%" PRIu64 " == %7.4g%%"),
                            source->statistics.missingRecs,
                            allrecs,
                            ((float)source->statistics.missingRecs /
                             (float)allrecs * 100.0));
                }

                TRACEMSG((("Engine: %" PRIu16
                           " Sequence numbers: Expected %" PRIu32
                           " Actual %" PRIu32
                           " Difference %" PRId64
                           " Time difference: %f seconds"),
                          target.id,
                          engine->flow_sequence,
                          flow_sequence,
                          ((int64_t)flow_sequence - engine->flow_sequence),
                          (double)(now - engine->last_timestamp) / 1000.0));

                /* Update the next expected seq */
                engine->flow_sequence = flow_sequence + count;
            }

            pthread_mutex_unlock(&source->stats_mutex);
        }

#if SAVE_LAST_TIMESTAMP
        engine->last_timestamp = (sktime_t)now;
#endif

        return pdu;
    }
}


static v5Record *pduSourceGetNextRec(
    pduSource_t source)
{
    assert(source != NULL);

    /* Infloop; exit by return only */
    for (;;) {
        v5Record *v5RPtr;
        intmax_t  difference;

        if (source->stopped) {
            return NULL;
        }

        /* If we need a pdu, get a new one, otherwise we are not finished
           with the last. */
        if (source->count == 0) {
            source->pdu = pduNext(source);
            if (source->pdu == NULL) {
                return NULL;
            }
            source->count = ntohs(source->pdu->hdr.count);
        }

        /* Get next record, and decrement counter*/
        v5RPtr = &source->pdu->data[ntohs(source->pdu->hdr.count)
                                    - source->count--];

        /* Check for zero packets or bytes.  No need for byteswapping
           when checking zero. */
        if (v5RPtr->dPkts == 0 || v5RPtr->dOctets == 0) {
            if (source->logopt & SOURCE_LOG_BAD) {
                NOTICEMSG("Netflow record has zero packets or bytes.");
            }
            COUNT_BAD_RECORD(source, source->pdu);
            continue;
        }

        /* Check to see if more packets them bytes. */
        if (ntohl(v5RPtr->dPkts) > ntohl(v5RPtr->dOctets)) {
            if (source->logopt & SOURCE_LOG_BAD) {
                NOTICEMSG("Netflow record has more packets them bytes.");
            }
            COUNT_BAD_RECORD(source, source->pdu);
            continue;
        }

        /* Check to see if the First and Last timestamps for the flow
         * record are reasonable, accounting for rollover.  If the
         * absolute value of the difference is greater than
         * maximumFlowTimeDeviation, we assume it has rolled over. */
        difference = (intmax_t)ntohl(v5RPtr->Last) - ntohl(v5RPtr->First);
        if ((difference > maximumFlowTimeDeviation)
            || ((difference < 0)
                && (difference > (-maximumFlowTimeDeviation))))
        {
            if (source->logopt & SOURCE_LOG_BAD) {
                NOTICEMSG(("Netflow record has earlier end time"
                           " than start time."));
            }
            COUNT_BAD_RECORD(source, source->pdu);
            continue;
        }

        /* Check for bogosities in how the ICMP type/code are set.  It
           should be in dest port, but sometimes it is backwards in src
           port. */
        if (v5RPtr->prot == 1 &&  /* ICMP */
            v5RPtr->dstport == 0) /* No byteswapping for check against 0 */
        {
            uint32_t *ports = (uint32_t *)&v5RPtr->srcport;
            *ports = BSWAP32(*ports); /* This will swap src into dest,
                                         while byteswapping. */
        }

        pthread_mutex_lock(&source->stats_mutex);
        source->statistics.goodRecs++;
        pthread_mutex_unlock(&source->stats_mutex);

        return v5RPtr;
    }
}


int pduSourceGetGeneric(
    pduSource_t     source,
    rwRec          *rwrec)
{
    const v5Record *v5RPtr;
    intmax_t v5_first, v5_last;
    intmax_t sTime;
    intmax_t difference;

    v5RPtr = pduSourceGetNextRec(source);
    if (v5RPtr == NULL) {
        return -1;
    }
    /* Setup start and duration */
    v5_first = ntohl(v5RPtr->First);
    v5_last = ntohl(v5RPtr->Last);
    if (v5_first > v5_last) {
        /* End has rolled over, while start has not.  Adjust end by
         * 2^32 msecs in order to allow us to subtract start from end
         * and get a correct value for the duration. */
        v5_last += ROLLOVER32;
    }

    /* v5_first is milliseconds since the router booted.  To get UNIX
     * epoch milliseconds, add the router's boot time. */
    sTime = v5_first + source->engine_info->router_boot;

    /* Check to see if the difference between the 32bit start time and
     * the sysUptime is overly large.  If it is, one of the two has
     * more than likely rolled over.  We need to adjust based on
     * this. */
    difference = source->engine_info->sysUptime - v5_first;
    if (difference > maximumFlowTimeDeviation) {
        /* sTime rollover */
        sTime += ROLLOVER32;
    } else if (difference < (-maximumFlowTimeDeviation)) {
        /* sysUptime rollover */
        sTime -= ROLLOVER32;
    }

    RWREC_CLEAR(rwrec);

    /* Convert NetFlow v5 to SiLK */
    rwRecSetSIPv4(rwrec, ntohl(v5RPtr->srcaddr));
    rwRecSetDIPv4(rwrec, ntohl(v5RPtr->dstaddr));
    rwRecSetSPort(rwrec, ntohs(v5RPtr->srcport));
    rwRecSetDPort(rwrec, ntohs(v5RPtr->dstport));
    rwRecSetProto(rwrec, v5RPtr->prot);
    rwRecSetFlags(rwrec, v5RPtr->tcp_flags);
    rwRecSetInput(rwrec, ntohs(v5RPtr->input));
    rwRecSetOutput(rwrec, ntohs(v5RPtr->output));
    rwRecSetNhIPv4(rwrec, ntohl(v5RPtr->nexthop));
    rwRecSetStartTime(rwrec, (sktime_t)sTime);
    rwRecSetPkts(rwrec, ntohl(v5RPtr->dPkts));
    rwRecSetBytes(rwrec, ntohl(v5RPtr->dOctets));
    rwRecSetElapsed(rwrec, (uint32_t)(v5_last - v5_first));
    rwRecSetRestFlags(rwrec, 0);
    rwRecSetTcpState(rwrec, SK_TCPSTATE_NO_INFO);

    rwRecSetApplication(rwrec, 0);

    return 0;
}


static udpSource_t pduUdpSourceCreate(
    int             port,
    in_addr_t       from_address,
    in_addr_t       listen_address,
    uint32_t        max_pkts,
    int             UNUSED(sockbufsize))
{
    struct sockaddr_in addr;
    int sock;

    /* Get a socket */
    sock = socket(AF_INET, SOCK_DGRAM, 0);
    if (sock == -1) {
        ERRMSG("Failed to allocate socket: %s", strerror(errno));
        return NULL;
    }

#if 0
    /* Make the socket receiver buffer size as close as we can to
       sockbufsize. */
    skGrowSocketBuffer(sock, SO_RCVBUF, sockbufsize);
#endif

    /* Set up the address structure */
    memset(&addr, 0, sizeof(addr));
    addr.sin_family = AF_INET;
    addr.sin_addr.s_addr = htonl(listen_address);
    addr.sin_port = htons(port);

    /* Bind socket to port */
    if (bind(sock, (struct sockaddr *)&addr, sizeof(addr)) == -1) {
        ERRMSG("Failed to bind address: %s", strerror(errno));
        close(sock);
        return NULL;
    }

    return udpSourceCreate(sock, from_address, V5PDU_LEN, max_pkts);
}


pduSourcePool_t pduSourcePoolCreate(void)
{
    return skVectorNew(sizeof(pdu_source_pool_item_t));
}


void pduSourcePoolDestroy(pduSourcePool_t pool)
{
    skVectorDestroy(pool);
}


pduSource_t pduSourceCreateFromProbeDef(
    pduSourcePool_t     pool,
    const skpc_probe_t *probe,
    uint32_t            max_pkts,
    int                 sockbufsize)
{
    uint8_t i;
    in_addr_t paddr;
    in_addr_t haddr;
    uint16_t pport;
    pduSource_t source;
    pdu_source_pool_item_t current;
    int rv;
    uint8_t flags;
    int sbufmin = SOCKETBUFFER_MINIMUM;
    int sbufnominaltotal = SOCKETBUFFER_NOMINAL_TOTAL;
    int sbufsize;
    char *env;
    char *end;

    assert(pool);
    assert(probe);

    flags = skpcProbeGetLogFlags(probe);

    rv = skpcProbeGetListenAsHost(probe, &paddr, &pport);
    if (rv == -1) {
        return NULL;
    }
    skpcProbeGetAcceptFromHost(probe, &haddr);

    /* Loop through the udpSources to see if one matches the probe. */
    for (i = 0; i < skVectorGetCount(pool); i++)
    {
        in_addr_t addr;
        uint16_t port;

        skVectorGetValue(&current, pool, i);

        rv = skpcProbeGetListenAsHost(current.probe, &addr, &port);
        if (rv == -1) {
            continue;
        }

        if (port == pport && addr == paddr) {
            in_addr_t from_addr;
            skpcProbeGetAcceptFromHost(current.probe, &from_addr);

            if (from_addr == haddr) {
                /* Matches in all respects.  Create a duplicate. */
                source = (pduSource_t)calloc(1, sizeof(pdu_source_t));
                if (source == NULL) {
                    return NULL;
                }

                source->source = current.source;
                udpSourceIncRef(current.source);
                source->source_index = current.index;
                pduSourceSetLogopt(source, flags);

                return source;
            }
            if (from_addr == INADDR_ANY ||
                haddr == INADDR_ANY) {
                /* Can't mix INADDR_ANY and specific ips */
                return NULL;
            }

            /* Matches in all but from_addr.  Add that addr. */
            source = (pduSource_t)calloc(1, sizeof(pdu_source_t));
            if (source == NULL) {
                return NULL;
            }

            source->source = current.source;
            udpSourceIncRef(current.source);
            source->source_index = udpSourceAddAddress(source->source, haddr);
            pduSourceSetLogopt(source, flags);

            return source;
        }
    }

    /* No match.  Add new source. */
    source = pduSourceCreate(pport, haddr, paddr, max_pkts, sockbufsize);
    if (source == NULL) {
        return NULL;
    }
    pduSourceSetLogopt(source, flags);
    current.source = source->source;
    current.index = source->source_index;
    current.probe = probe;

    skVectorAppendValue(pool, &current);

    /* Adjust socket buffer sizes */
    env = getenv(SOCKETBUFFER_NOMINAL_TOTAL_ENV);
    if (env) {
        long int val = strtol(env, &end, 0);
        if (end != env && *end == '\0') {
            if (val > INT_MAX) {
                val = INT_MAX;
            }
            sbufnominaltotal = val;
        }
    }
    env = getenv(SOCKETBUFFER_MINIMUM_ENV);
    if (env) {
        long int val = strtol(env, &end, 0);
        if (end != env && *end == '\0') {
            if (val > INT_MAX) {
                val = INT_MAX;
            }
            sbufmin = val;
        }
    }
    sbufsize = sbufnominaltotal / skVectorGetCount(pool);
    if (sbufsize < sbufmin) {
        sbufsize = sbufmin;
    }
    for (i = 0; i < skVectorGetCount(pool); i++) {
        int sock;
        skVectorGetValue(&current, pool, i);
        sock = udpSourceGetSocket(current.source);

        /* Make the socket receiver buffer size as close as we can to
           sbufsize. */
        skGrowSocketBuffer(sock, SO_RCVBUF, sbufsize);
    }

    return source;
}


/* Get statistics associated with a pdu source. */
void pduSourceGetStats(pduSource_t source, flowStats_t stats)
{
    pthread_mutex_lock(&source->stats_mutex);
    *stats = source->statistics;
    pthread_mutex_unlock(&source->stats_mutex);
}


static void pduSourceLogStatsBase(
    pduSource_t source,
    const char *name)
{
    INFOMSG(("'%s': Pkts %" PRIu64 "/%-10" PRIu64
             " Recs %-10" PRIu64
             " MissRecs %-10" PRId64
             " BadRecs %-10" PRIu64),
            name,
            (source->statistics.procPkts - source->statistics.badPkts),
            source->statistics.procPkts,
            source->statistics.goodRecs,
            source->statistics.missingRecs,
            source->statistics.badRecs);
}

/* Log statistics associated with a pdu source. */
void pduSourceLogStats(
    pduSource_t source,
    const char *name)
{
    pthread_mutex_lock(&source->stats_mutex);
    pduSourceLogStatsBase(source, name);
    pthread_mutex_unlock(&source->stats_mutex);
}


/* Log statistics associated with a pdu source, and then clear the
 * statistics. */
void pduSourceLogStatsAndClear(
    pduSource_t source,
    const char *name)
{
    pthread_mutex_lock(&source->stats_mutex);
    pduSourceLogStatsBase(source, name);
    memset(&source->statistics, 0, sizeof(source->statistics));
    pthread_mutex_unlock(&source->stats_mutex);
}


/* Clear out current statistics */
void pduSourceClearStats(pduSource_t source)
{
    pthread_mutex_lock(&source->stats_mutex);
    memset(&source->statistics, 0, sizeof(source->statistics));
    pthread_mutex_unlock(&source->stats_mutex);
}


/* Set logging options */
void pduSourceSetLogopt(pduSource_t source, uint8_t opt)
{
    source->logopt = opt;
}


/*
** Local Variables:
** mode:c
** indent-tabs-mode:nil
** c-basic-offset:4
** End:
*/
