/*
** Copyright (C) 2006-2012 by Carnegie Mellon University.
**
** @OPENSOURCE_HEADER_START@
**
** Use of the SILK system and related source code is subject to the terms
** of the following licenses:
**
** GNU Public License (GPL) Rights pursuant to Version 2, June 1991
** Government Purpose License Rights (GPLR) pursuant to DFARS 252.227.7013
**
** NO WARRANTY
**
** ANY INFORMATION, MATERIALS, SERVICES, INTELLECTUAL PROPERTY OR OTHER
** PROPERTY OR RIGHTS GRANTED OR PROVIDED BY CARNEGIE MELLON UNIVERSITY
** PURSUANT TO THIS LICENSE (HEREINAFTER THE "DELIVERABLES") ARE ON AN
** "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY
** KIND, EITHER EXPRESS OR IMPLIED AS TO ANY MATTER INCLUDING, BUT NOT
** LIMITED TO, WARRANTY OF FITNESS FOR A PARTICULAR PURPOSE,
** MERCHANTABILITY, INFORMATIONAL CONTENT, NONINFRINGEMENT, OR ERROR-FREE
** OPERATION. CARNEGIE MELLON UNIVERSITY SHALL NOT BE LIABLE FOR INDIRECT,
** SPECIAL OR CONSEQUENTIAL DAMAGES, SUCH AS LOSS OF PROFITS OR INABILITY
** TO USE SAID INTELLECTUAL PROPERTY, UNDER THIS LICENSE, REGARDLESS OF
** WHETHER SUCH PARTY WAS AWARE OF THE POSSIBILITY OF SUCH DAMAGES.
** LICENSEE AGREES THAT IT WILL NOT MAKE ANY WARRANTY ON BEHALF OF
** CARNEGIE MELLON UNIVERSITY, EXPRESS OR IMPLIED, TO ANY PERSON
** CONCERNING THE APPLICATION OF OR THE RESULTS TO BE OBTAINED WITH THE
** DELIVERABLES UNDER THIS LICENSE.
**
** Licensee hereby agrees to defend, indemnify, and hold harmless Carnegie
** Mellon University, its trustees, officers, employees, and agents from
** all claims or demands made against them (and any related losses,
** expenses, or attorney's fees) arising out of, or relating to Licensee's
** and/or its sub licensees' negligent use or willful misuse of or
** negligent conduct or willful misconduct regarding the Software,
** facilities, or other rights or assistance granted by Carnegie Mellon
** University under this License, including, but not limited to, any
** claims of product liability, personal injury, death, damage to
** property, or violation of any laws or regulations.
**
** Carnegie Mellon University Software Engineering Institute authored
** documents are sponsored by the U.S. Department of Defense under
** Contract FA8721-05-C-0003. Carnegie Mellon University retains
** copyrights in all material produced under this contract. The U.S.
** Government retains a non-exclusive, royalty-free license to publish or
** reproduce these documents, or allow others to do so, for U.S.
** Government purposes only pursuant to the copyright license under the
** contract clause at 252.227.7013.
**
** @OPENSOURCE_HEADER_END@
*/

/*
**  Interface to pull a single flow from IPFIX
**
*/


#include <silk/silk.h>

RCSIDENT("$SiLK: ipfixsource.c a1a0c305e641 2012-06-26 20:39:33Z mthomas $");

#include <silk/utils.h>
#include <silk/libflowsource.h>
#include <silk/skipfix.h>
#include <silk/sklog.h>
#include "circbuf.h"


struct ipfix_source_st {
    /* count of records read from this source and mutex to protect it */
    flowsource_stats_t  statistics;
    pthread_mutex_t     stats_mutex;

    /* host and port on which to listen when reading from the network */
    in_addr_t           listen_addr;
    int                 port;

    /* the listener and connection objects from libfixbuf */
    fbListener_t       *listener;
    fbConnSpec_t       *connspec;

    /* the libfixbuf object when reading from a file */
    fBuf_t             *readbuf;

    /* the file handle when reading from a file */
    FILE               *fileptr;

    /* if we get both a forward and reverse record from libfixbuf, we
     * temporarily cache the reverse record here. */
    rwRec               rvbuf;

    pthread_t           thread;
    uint32_t            bufsize;
    circBuf_t           data_buffer;
    void               *tmp_buffer;
    pthread_mutex_t     mutex;
    pthread_cond_t      cond;

    /* whether the source is in the process of being destroyed */
    unsigned            destroyed : 1;

    /* Whether the reading thread is currently running */
    unsigned            running : 1;

    /* whether the 'rvbuf' field holds a valid record (1==yes) */
    unsigned            reverse : 1;

    /* the type of data to put into input/output fields
     * (0==snmp,1==vlan) */
    unsigned            get_vlan : 1;
};
/* typedef struct ipfix_source_st *ipfixSource_t;  // libflowsource.h */


/* Mutex around calls to skiCreateListener. */
static pthread_mutex_t create_listener_mutex = PTHREAD_MUTEX_INITIALIZER;

#if 0
static gboolean fixbufConnect(
    fbListener_t               *UNUSED(listener),
    void                      **UNUSED(ctx),
    int                         UNUSED(fd),
    struct sockaddr            *peer,
    size_t                      peerlen,
    GError                    **UNUSED(err))
{

}
#endif


static void *ipfix_reader(void *vsource)
{
    ipfixSource_t source = (ipfixSource_t)vsource;
    GError *err = NULL;
    fBuf_t *ipfix_buf = NULL;
    int circbuf_advanced = 0;

    pthread_mutex_lock(&source->mutex);
    pthread_cond_signal(&source->cond);
    source->running = 1;
    pthread_mutex_unlock(&source->mutex);

    while (!source->destroyed) {
        /* wait for a new connection */
        ipfix_buf = fbListenerWait(source->listener, &err);
        if (NULL == ipfix_buf) {
            if (NULL == err) {
                /* got an unknown error---treat as fatal */
                NOTICEMSG("fbListenerWait unknown error");
                break;
            }
            DEBUGMSG(("fbListenerWait returned FALSE;"
                      " error domain/code =%" PRIu32 "/%" PRId32),
                     (uint32_t)err->domain, (int32_t)err->code);

            /* this error code indicates interrupted read, either
             * because the socket received EINTR or because
             * fbListenerInterrupt() was called. */
            if (g_error_matches(err, FB_ERROR_DOMAIN, FB_ERROR_NLREAD)) {
                g_clear_error(&err);
                continue;
            }

            /* treat any other error as fatal */
            NOTICEMSG("fbListenerWait: %s", err->message);
            g_clear_error(&err);
            break;
        }

        /* treat input as an endless stream of records */
        fBufSetAutomaticMode(ipfix_buf, TRUE);

        while (!source->destroyed) {
            rwRec reverse;
            gboolean rb;

            /* Advance the circular buffer */
            if (!circbuf_advanced) {
                source->tmp_buffer =
                    (void *)circBufNextHead(source->data_buffer);
                circbuf_advanced = 1;
            }
            if (source->tmp_buffer == NULL) {
                assert(source->destroyed);
                break;
            }

            /* Get the next SiLK record. */
            /* There is an issue here: Currently (2008-08), this does
             * not unblock even when fbListenerInterrupt() is
             * called. */
            rb = skiRwNextRecord(ipfix_buf, (rwRec *)source->tmp_buffer,
                                 &reverse, &err, source->get_vlan);
            if (rb == 0) {
                DEBUGMSG(("fBufNext returned FALSE;"
                          " error domain/code =%" PRIu32 "/%" PRId32),
                         (uint32_t)err->domain, (int32_t)err->code);

                /* this error code indicates interrupted read */
                if (g_error_matches(err, FB_ERROR_DOMAIN, FB_ERROR_NLREAD)) {
                    g_clear_error(&err);
                    continue;
                }

                /* for any other error, log the error and create a new
                 * listener */
                NOTICEMSG("fBufNext: %s", err->message);

                /* FB_ERROR_NETFLOWV9 indicates an anomalous netflow
                 * v9 record; these do not disturb fixbuf state, and
                 * so should be ignored. */
                if (g_error_matches(err, FB_ERROR_DOMAIN, FB_ERROR_NETFLOWV9)) {
                    g_clear_error(&err);
                    continue;
                }
                g_clear_error(&err);

                /* do not create a new listener for the UDP case,
                 * since destroying the fBuf destroys the collector
                 * object and the underlying socket. */
                if (source->connspec->transport != FB_UDP) {
                    fBufFree(ipfix_buf);
                    ipfix_buf = NULL;
                }
                break;
            }

            circbuf_advanced = 0;

            /* Deal with the reverse flow */
            if (rwRecGetPkts(&reverse) != 0) {
                source->tmp_buffer =
                    (void *)circBufNextHead(source->data_buffer);
                if (source->tmp_buffer == NULL) {
                    assert(source->destroyed);
                    break;
                }

                memcpy(source->tmp_buffer, &reverse, sizeof(reverse));
            }
        }
    }

    /* do we want the UPD check here? */
    if (ipfix_buf && source->connspec->transport != FB_UDP) {
        fBufFree(ipfix_buf);
    }

    pthread_mutex_lock(&source->mutex);
    while (!source->destroyed) {
        pthread_cond_wait(&source->cond, &source->mutex);
    }
    pthread_cond_signal(&source->cond);

    fbListenerFree(source->listener);
    source->listener = NULL;

    source->running = 0;

    pthread_mutex_unlock(&source->mutex);

    return NULL;
}


static void free_conspec(fbConnSpec_t *conspec)
{
    if (conspec->host) {
        free(conspec->host);
    }
    if (conspec->svc) {
        free(conspec->svc);
    }
    free(conspec);
}


/*
 *    Handler to print log messages.  This will be invoked by g_log()
 *    and the other logging functions from GLib2.
 */
static void ipfixGLogHandler(
    const gchar     UNUSED(*log_domain),
    GLogLevelFlags          log_level,
    const gchar            *message,
    gpointer         UNUSED(user_data))
{
    /* In syslog, CRIT is worse than ERR; in Glib2 ERROR is worse than
     * CRITICAL. */

    switch (log_level & G_LOG_LEVEL_MASK) {
      case G_LOG_LEVEL_CRITICAL:
        ERRMSG("%s", message);
        break;
      case G_LOG_LEVEL_WARNING:
        WARNINGMSG("%s", message);
        break;
      case G_LOG_LEVEL_MESSAGE:
        NOTICEMSG("%s", message);
        break;
      case G_LOG_LEVEL_INFO:
        INFOMSG("%s", message);
        break;
      case G_LOG_LEVEL_DEBUG:
        DEBUGMSG("%s", message);
        break;
      default:
        CRITMSG("%s", message);
        break;
    }
}


/*
 *    Performs any initialization required prior to creating the IPFIX
 *    sources.  Returns 0 on success, or -1 on failure.
 */
int ipfixSourcesSetup(void)
{
    int log_levels = (G_LOG_LEVEL_CRITICAL
                      | G_LOG_LEVEL_WARNING
                      | G_LOG_LEVEL_MESSAGE
                      | G_LOG_LEVEL_INFO
                      | G_LOG_LEVEL_DEBUG);

    /* tell fixbuf (glib) we are a threaded program.  this will abort
     * if glib does not have thread support */
    if (!g_thread_supported()) {
        g_thread_init(NULL);
    }

    /* set a log handler for messages from fixbuf and glib.
     * http://developer.gnome.org/glib/2.30/glib-Message-Logging.html */
    g_log_set_handler(NULL, log_levels, &ipfixGLogHandler, NULL);
    g_log_set_handler("GLib", log_levels, &ipfixGLogHandler, NULL);

    return 0;
}


ipfixSource_t ipfixFileSourceCreate(
    const skpc_probe_t *probe,
    const char         *path)
{
    ipfixSource_t  source;
    GError        *err = NULL;

    /* Create thread structure */
    source = (ipfixSource_t)calloc(1, sizeof(*source));
    if (source == NULL) {
        return NULL;
    }

    source->fileptr = fopen(path, "r");

    if (source->fileptr == NULL) {
        ERRMSG("Unable to open file '%s'", path);
        free(source);
        return NULL;
    }

    source->readbuf = skiCreateReadBufferForFP(source->fileptr, &err);

    if (source->readbuf == NULL) {
        if (err) {
            ERRMSG("%s: %s", "skiCreateReadBuffer", err->message);
        }
        g_clear_error(&err);
        fclose(source->fileptr);
        free(source);
        return NULL;
    }

    if (probe && SKPC_IFVALUE_VLAN == skpcProbeGetInterfaceValueType(probe)) {
        source->get_vlan = 1;
    }

    pthread_mutex_init(&source->mutex, NULL);
    pthread_mutex_init(&source->stats_mutex, NULL);

    return source;
}


ipfixSource_t ipfixSourceCreate(
    int             port,
    in_addr_t       listen_on_address,
    skpc_proto_t    protocol,
    uint32_t        max_flows,
    int             nfv9)
{
    ipfixSource_t source;
    GError *err = NULL;
    const char *errfn;
    char port_string[7];
    char ipname[SK_NUM2DOT_STRLEN];
    int rv;

#if !SK_ENABLE_NETFLOW9
    if (nfv9) {
        return NULL;
    }
#endif

    source = (ipfixSource_t)calloc(1, sizeof(*source));
    if (source == NULL) {
        return NULL;
    }
    source->listen_addr = listen_on_address;
    source->port = port;

    source->connspec = (fbConnSpec_t *)calloc(1, sizeof(*source->connspec));
    if (source->connspec == NULL) {
        free(source);
        return NULL;
    }
    switch (protocol) {
      case SKPC_PROTO_SCTP:
        source->connspec->transport = FB_SCTP;
        break;
      case SKPC_PROTO_TCP:
        source->connspec->transport = FB_TCP;
        break;
      case SKPC_PROTO_UDP:
        source->connspec->transport = FB_UDP;
        break;
      default:
        skAbortBadCase(protocol);
    }
    if (source->listen_addr != INADDR_ANY) {
        source->connspec->host = strdup(num2dot_r(source->listen_addr,
                                                  ipname));
        if (source->connspec->host == NULL) {
            free_conspec(source->connspec);
            free(source);
            return NULL;
        }
    }
    rv = snprintf(port_string, sizeof(port_string), "%i", source->port);
    assert(rv < (int)sizeof(port_string));
    source->connspec->svc = strdup(port_string);
    if (source->connspec->svc == NULL) {
        free_conspec(source->connspec);
        free(source);
        return NULL;
    }

    pthread_mutex_lock(&create_listener_mutex);
    errfn = "skiCreateListener";
    source->listener = skiCreateListener(source->connspec, NULL, NULL, &err);
#if SK_ENABLE_NETFLOW9
    if (source->listener && nfv9) {
        fbCollector_t *collector;
        if (!fbListenerGetCollector(source->listener, &collector, &err)) {
            errfn = "fbListenerGetCollector";
            fbListenerFree(source->listener);
            source->listener = NULL;
        } else if (!fbCollectorSetNetflowV9Translator(collector, &err)) {
            errfn = "fbCollectorSetNetflowV9Translator";
            fbListenerFree(source->listener);
            source->listener = NULL;
        }
    }
#endif  /* SK_ENABLE_NETFLOW9 */
    pthread_mutex_unlock(&create_listener_mutex);
    if (source->listener == NULL) {
        if (err) {
            ERRMSG("%s: %s", errfn, err->message);
        }
        g_clear_error(&err);
        free_conspec(source->connspec);
        free(source);
        return NULL;
    }

    pthread_mutex_init(&source->stats_mutex, NULL);
    pthread_mutex_init(&source->mutex, NULL);
    pthread_cond_init(&source->cond, NULL);

    source->data_buffer = circBufCreate(sizeof(rwRec), max_flows);
    if (source->data_buffer == NULL) {
        fbListenerFree(source->listener);
        free_conspec(source->connspec);
        free(source);
        return NULL;
    }

    source->bufsize = max_flows;

    pthread_mutex_lock(&source->mutex);
    rv = pthread_create(&source->thread, NULL, ipfix_reader, (void *)source);
    if (rv != 0) {
        circBufDestroy(source->data_buffer);
        free_conspec(source->connspec);
        fbListenerFree(source->listener);
        free(source);
        return NULL;
    }
    pthread_cond_wait(&source->cond, &source->mutex);
    if (source->destroyed) {
        circBufDestroy(source->data_buffer);
        pthread_mutex_unlock(&source->mutex);
        pthread_join(source->thread, NULL);
        fbListenerFree(source->listener);
        free_conspec(source->connspec);
        free(source);
        return NULL;
    }
    pthread_mutex_unlock(&source->mutex);

    return source;
}


ipfixSource_t ipfixSourceCreateFromProbeDef(
    const skpc_probe_t *probe,
    uint32_t            max_flows)
{
    ipfixSource_t ipfix_src;
    in_addr_t bind_addr = 0;
    uint16_t  bind_port = 0;
    skpc_proto_t protocol;
    int nfv9;
    int rv;

    rv = skpcProbeGetListenAsHost(probe, &bind_addr, &bind_port);
    if (rv == -1) {
        return NULL;
    }
    /* no way to restrict FIXBUF to particular host */
    /* skpcProbeGetAcceptFromHost(probe, &haddr); */

    protocol = skpcProbeGetProtocol(probe);
    if (protocol == SKPC_PROTO_UNSET) {
        return NULL;
    }

    nfv9 = (PROBE_ENUM_NETFLOW_V9 == skpcProbeGetType(probe));

    /* create the source */
    ipfix_src = ipfixSourceCreate(bind_port, bind_addr, protocol,
                                  max_flows, nfv9);
    if (ipfix_src) {
        if (SKPC_IFVALUE_VLAN == skpcProbeGetInterfaceValueType(probe)) {
            ipfix_src->get_vlan = 1;
        }
    }

    return ipfix_src;
}


void ipfixSourceDestroy(
    ipfixSource_t source)
{
    assert(source);

    pthread_mutex_lock(&source->mutex);
    source->destroyed = 1;

    if (source->listener) {
        fbListenerInterrupt(source->listener);
    }

    if (source->data_buffer) {
        circBufStop(source->data_buffer);

        pthread_cond_broadcast(&source->cond);
        while (source->running) {
            pthread_cond_wait(&source->cond, &source->mutex);
        }
        pthread_join(source->thread, NULL);

        /* Don't destroy the circular buffer until we know the reading
           thread has ended. */
        circBufDestroy(source->data_buffer);
    }

    if (source->data_buffer) {
        pthread_cond_destroy(&source->cond);
    }

    if (source->connspec) {
        free_conspec(source->connspec);
    }

    if (source->readbuf) {
        fBufFree(source->readbuf);
    }
    if (source->fileptr) {
        fclose(source->fileptr);
    }

    pthread_mutex_unlock(&source->mutex);
    pthread_mutex_destroy(&source->mutex);

    free(source);
}


int ipfixSourceGetGeneric(
    ipfixSource_t   source,
    rwRec          *rwrec)
{
    rwRec *rec;
    GError *err = NULL;

    assert(source);
    assert(rwrec);

    if (source->readbuf) {
        /* Reading from a file */

        pthread_mutex_lock(&source->mutex);

        if (source->reverse) {
            /* A reverse record exists from the last flow */
            RWREC_COPY(rwrec, &source->rvbuf);
            source->reverse = 0;
        } else if (skiRwNextRecord(source->readbuf, rwrec,
                                   &source->rvbuf, &err, source->get_vlan))
        {
            /* We have the next flow.  Set reverse if there is a
             * reverse record.  */
            if (rwRecGetBytes(&source->rvbuf)) {
                source->reverse = 1;
            }
        } else {
            /* End of file or other problem */
            g_clear_error(&err);
            pthread_mutex_unlock(&source->mutex);
            return -1;
        }

        pthread_mutex_unlock(&source->mutex);

    } else {
        /* Reading from the circular buffer */

        rec = (rwRec *)circBufNextTail(source->data_buffer);
        if (rec == NULL) {
            return -1;
        }

        RWREC_COPY(rwrec, rec);
    }

    return 0;
}


/*
** Local Variables:
** mode:c
** indent-tabs-mode:nil
** c-basic-offset:4
** End:
*/
