/*
** Copyright (C) 2007-2012 by Carnegie Mellon University.
**
** @OPENSOURCE_HEADER_START@
**
** Use of the SILK system and related source code is subject to the terms
** of the following licenses:
**
** GNU Public License (GPL) Rights pursuant to Version 2, June 1991
** Government Purpose License Rights (GPLR) pursuant to DFARS 252.227.7013
**
** NO WARRANTY
**
** ANY INFORMATION, MATERIALS, SERVICES, INTELLECTUAL PROPERTY OR OTHER
** PROPERTY OR RIGHTS GRANTED OR PROVIDED BY CARNEGIE MELLON UNIVERSITY
** PURSUANT TO THIS LICENSE (HEREINAFTER THE "DELIVERABLES") ARE ON AN
** "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY
** KIND, EITHER EXPRESS OR IMPLIED AS TO ANY MATTER INCLUDING, BUT NOT
** LIMITED TO, WARRANTY OF FITNESS FOR A PARTICULAR PURPOSE,
** MERCHANTABILITY, INFORMATIONAL CONTENT, NONINFRINGEMENT, OR ERROR-FREE
** OPERATION. CARNEGIE MELLON UNIVERSITY SHALL NOT BE LIABLE FOR INDIRECT,
** SPECIAL OR CONSEQUENTIAL DAMAGES, SUCH AS LOSS OF PROFITS OR INABILITY
** TO USE SAID INTELLECTUAL PROPERTY, UNDER THIS LICENSE, REGARDLESS OF
** WHETHER SUCH PARTY WAS AWARE OF THE POSSIBILITY OF SUCH DAMAGES.
** LICENSEE AGREES THAT IT WILL NOT MAKE ANY WARRANTY ON BEHALF OF
** CARNEGIE MELLON UNIVERSITY, EXPRESS OR IMPLIED, TO ANY PERSON
** CONCERNING THE APPLICATION OF OR THE RESULTS TO BE OBTAINED WITH THE
** DELIVERABLES UNDER THIS LICENSE.
**
** Licensee hereby agrees to defend, indemnify, and hold harmless Carnegie
** Mellon University, its trustees, officers, employees, and agents from
** all claims or demands made against them (and any related losses,
** expenses, or attorney's fees) arising out of, or relating to Licensee's
** and/or its sub licensees' negligent use or willful misuse of or
** negligent conduct or willful misconduct regarding the Software,
** facilities, or other rights or assistance granted by Carnegie Mellon
** University under this License, including, but not limited to, any
** claims of product liability, personal injury, death, damage to
** property, or violation of any laws or regulations.
**
** Carnegie Mellon University Software Engineering Institute authored
** documents are sponsored by the U.S. Department of Defense under
** Contract FA8721-05-C-0003. Carnegie Mellon University retains
** copyrights in all material produced under this contract. The U.S.
** Government retains a non-exclusive, royalty-free license to publish or
** reproduce these documents, or allow others to do so, for U.S.
** Government purposes only pursuant to the copyright license under the
** contract clause at 252.227.7013.
**
** @OPENSOURCE_HEADER_END@
*/
#ifndef _SKIPFIX_H
#define _SKIPFIX_H

#include <silk/silk.h>

RCSIDENTVAR(rcsID_SKIPFIX_H, "$SiLK: skipfix.h 372a8bc31d8a 2012-02-10 21:55:28Z mthomas $");

#include <silk/rwrec.h>
#include <fixbuf/public.h>

/*
**  skipfix.h
**
**  Converts between IPFIX and SiLK Flow Recs
**
*/

/**
 * @file
 *
 * SiLK Flow record (rwRec) interface to fixbuf. Supports the reading
 * and writing of IPFIX Files by SiLK applications, and the creation
 * of IPFIX Collecting and Exporting Processes using the SiLK flow
 * format.
 *
 * This library supports the creation of fbListener_t and fBuf_t
 * instances configured to read any IPFIX record containing the
 * appropriate information elements as a SiLK rwRec, and the creation
 * of fBuf_t instances configured to write SiLK rwRecs as IPFIX
 * records.
 *
 * To read SiLK Flow records from an IPFIX file, create a buffer using
 * skiCreateReadBufferForFP(), then iterate over records with
 * skiRwNextRecord().  To write SiLK Flow records to an IPFIX file,
 * create a buffer using skiCreateWriteBufferForFP(), then write each
 * record with skiRwAppendRecord(). Use fBufFree() to free the
 * resulting buffers.
 *
 * This library uses the GError facility from glib for reporting
 * errors.  Pass a pointer to a NULL GError * on any call taking an
 * err parameter; if an error occurs, the function will return NULL or
 * FALSE as appropriate, and allocate a GError structure describing
 * the error. If an error occurs, you can use the g_error_matches()
 * macro to match it against FB_ERROR_* constants defined by
 * libfixbuf, and the err->message field to get a human-readable error
 * message. After handling an error, use the g_clear_error() macro to
 * free the error description. See the glib documentation for more
 * details on this facility.
 *
 * See the documentation for libfixbuf for details on the fBuf_t,
 * fbListener_t, fbCollector_t, fbExporter_t, and fbConnSpec_t types,
 * and the fbListenerAppInit_fn and fbListenerAppFree_fn callbacks.
 */


/**
 * IPFIX Template ID for SiLK flows. Used as the external template ID
 * for SiLK Flow records written to IPFIX buffers.
 */
#define SKI_FLOW_TID 0xAFEA
/** IPFIX internal Template ID for extended SiLK flows. */
#define SKI_EXTFLOW_TID 0xAFEB


/**
 * Print to the named file handle information about the internal data
 * structures.  This can be used for debugging to ensure that the data
 * model does not contain holes or mis-aligned members.
 *
 * @param fh      the file handle on which to print the information
 */
void skiCheckDataStructure(FILE *fh);


/**
 * Create a IPFIX Collecting Process session listener for reading SiLK
 * Flow records. Used to wait for connections from IPFIX exporting
 * processes; use fbListenerWait() to accept connections and access
 * buffers suitable for use with skiRwNextRecord(). See the fixbuf
 * documentation for fbListenerAlloc() and fbListenerWait() for
 * details.
 *
 * @param spec    local endpoint connection specifier; specifies a port
 *                and transport protocol to use, and optionally the
 *                address of an interface to listen on.
 * @param appinit application connection initiation function. Called on each
 *                collection attempt; vetoes connection attempts and creates
 *                application context.
 * @param appfree application context free function.
 * @param err     an error description
 * @return a new listener, or NULL on failure.
 */
fbListener_t *skiCreateListener(
    fbConnSpec_t           *spec,
    fbListenerAppInit_fn    appinit,
    fbListenerAppFree_fn    appfree,
    GError                **err);


/**
 * Create a buffer attached to a collecting process endpoint suitable
 * for use with skiRwNextRecord(). See the fixbuf documentation for
 * fbCollectorAllocFile() and/or fbCollectorAllocFP() for details on
 * the fbCollector_t type.
 *
 * @param collector a collecting process endpoint
 * @param err       an error description
 * @return a new buffer, or NULL on failure.
 */
fBuf_t *skiCreateReadBuffer(
    fbCollector_t  *collector,
    GError        **err);


/**
 * Create a buffer attached to a standard IO file pointer suitable for
 * use for skiRwNextRecord(). The file pointer must be opened for
 * reading. This call is a wrapper around fbCollectorAllocFP() and
 * skiCreateReadBuffer().
 *
 * @param fp a file pointer
 * @param err an error description
 * @return a new buffer, or NULL on failure.
 */
fBuf_t *skiCreateReadBufferForFP(
    FILE           *fp,
    GError        **err);


/**
 * Read the next IPFIX record from a buffer and convert it to a SiLK
 * Flow record.  Overwrites the rwRec buffer pointed to by rec with
 * the converted record. If the next IPFIX record is a biflow record,
 * and revRec is not NULL, overwrites the rwRec buffer pointed to by
 * revRec with the reverse direction record; if revRec is NULL, the
 * reverse direction is discarded silently.
 *
 * Returns TRUE if the records are valid, FALSE on failure. On
 * failure, check the error description against FB_ERROR_EOF in domain
 * FB_ERROR_DOMAIN for end of file or connection normally closed by
 * the remote end; FB_ERROR_EOM for end of message (if the buffer was
 * returned by fbListenerWait() or was made manual by
 * fBufSetAutomaticMode()); or FB_ERROR_NLREAD to check for "no
 * packet" (which is possible on timeout over certain transport
 * layers, or if a blocking read call is interrupted).
 *
 * @param fbuf             an IPFIX Message buffer.
 * @param rec              pointer to forward direction SiLK Flow record.
 * @param revRec           pointer to reverse direction SiLK Flow record,
 *                         or NULL to ignore.
 * @param err              an error description
 * @param vlan_interfaces  if non-zero, put the vlanId and postVlanId into
 *                         the input and output fields, respectively, on
 *                         the rwRecs.  If zero, store the ingressInterface
 *                         and egressInterface in input and output.
 * @return TRUE on success, FALSE on failure.
 */
gboolean skiRwNextRecord(
    fBuf_t         *fbuf,
    rwRec          *rec,
    rwRec          *revRec,
    GError        **err,
    int             vlan_interfaces);


/**
 * Create a buffer attached to an exporting process endpoint suitable
 * for use with skiRwAppendRecord(). Exports messages within the
 * specified IPFIX Observation Domain. See the fixbuf documentation
 * for fbExporterAllocNet(), fbExporterAllocFile(), and/or
 * fbExporterAllocFP() for details on the fbCollector_t type.
 *
 * @param exporter  an exporting process endpoint
 * @param domain    IPFIX Observation Domain to export messages within.
 * @param err       an error description
 * @return a new buffer, or NULL on failure.
 */
fBuf_t *skiCreateWriteBuffer(
    fbExporter_t   *exporter,
    uint32_t        domain,
    GError        **err);


/**
 * Create a buffer attached to a standard IO file pointer suitable for
 * use for skiRwAppendRecord(). The file pointer must be opened for
 * writing. This call is a wrapper around fbExporterAllocFP() and
 * skiCreateWriteBuffer().
 *
 * @param fp     a file pointer
 * @param domain IPFIX Observation Domain to export messages within.
 * @param err    an error description
 * @return a new buffer, or NULL on failure.
 */
fBuf_t *skiCreateWriteBufferForFP(
    FILE           *fp,
    uint32_t        domain,
    GError        **err);


/**
 * Free the memory associated with the Info Model---note that doing so
 * is not tread safe.
 */
void skiTeardown(void);


/**
 * Convert an SiLK Flow record and append it to an IPFIX message
 * buffer.  next IPFIX record from a buffer and convert it to a SiLK
 * Flow record.
 *
 * Returns TRUE if the record was written successfully, FALSE on
 * failure. On failure, check the error description against
 * FB_ERROR_NLWRITE in domain FB_ERROR_DOMAIN for connection
 * abnormally closed by the remote end.
 *
 * @param fbuf   an IPFIX Message buffer.
 * @param rec    pointer to SiLK Flow record.
 * @param err    an error description
 * @return TRUE on success, FALSE on failure.
 */
gboolean skiRwAppendRecord(
    fBuf_t         *fbuf,
    const rwRec    *rec,
    GError        **err);

#endif /* _SKIPFIX_H */

/*
** Local Variables:
** mode:c
** indent-tabs-mode:nil
** c-basic-offset:4
** End:
*/
