/*
** Copyright (C) 2004-2012 by Carnegie Mellon University.
**
** @OPENSOURCE_HEADER_START@
**
** Use of the SILK system and related source code is subject to the terms
** of the following licenses:
**
** GNU Public License (GPL) Rights pursuant to Version 2, June 1991
** Government Purpose License Rights (GPLR) pursuant to DFARS 252.227.7013
**
** NO WARRANTY
**
** ANY INFORMATION, MATERIALS, SERVICES, INTELLECTUAL PROPERTY OR OTHER
** PROPERTY OR RIGHTS GRANTED OR PROVIDED BY CARNEGIE MELLON UNIVERSITY
** PURSUANT TO THIS LICENSE (HEREINAFTER THE "DELIVERABLES") ARE ON AN
** "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY
** KIND, EITHER EXPRESS OR IMPLIED AS TO ANY MATTER INCLUDING, BUT NOT
** LIMITED TO, WARRANTY OF FITNESS FOR A PARTICULAR PURPOSE,
** MERCHANTABILITY, INFORMATIONAL CONTENT, NONINFRINGEMENT, OR ERROR-FREE
** OPERATION. CARNEGIE MELLON UNIVERSITY SHALL NOT BE LIABLE FOR INDIRECT,
** SPECIAL OR CONSEQUENTIAL DAMAGES, SUCH AS LOSS OF PROFITS OR INABILITY
** TO USE SAID INTELLECTUAL PROPERTY, UNDER THIS LICENSE, REGARDLESS OF
** WHETHER SUCH PARTY WAS AWARE OF THE POSSIBILITY OF SUCH DAMAGES.
** LICENSEE AGREES THAT IT WILL NOT MAKE ANY WARRANTY ON BEHALF OF
** CARNEGIE MELLON UNIVERSITY, EXPRESS OR IMPLIED, TO ANY PERSON
** CONCERNING THE APPLICATION OF OR THE RESULTS TO BE OBTAINED WITH THE
** DELIVERABLES UNDER THIS LICENSE.
**
** Licensee hereby agrees to defend, indemnify, and hold harmless Carnegie
** Mellon University, its trustees, officers, employees, and agents from
** all claims or demands made against them (and any related losses,
** expenses, or attorney's fees) arising out of, or relating to Licensee's
** and/or its sub licensees' negligent use or willful misuse of or
** negligent conduct or willful misconduct regarding the Software,
** facilities, or other rights or assistance granted by Carnegie Mellon
** University under this License, including, but not limited to, any
** claims of product liability, personal injury, death, damage to
** property, or violation of any laws or regulations.
**
** Carnegie Mellon University Software Engineering Institute authored
** documents are sponsored by the U.S. Department of Defense under
** Contract FA8721-05-C-0003. Carnegie Mellon University retains
** copyrights in all material produced under this contract. The U.S.
** Government retains a non-exclusive, royalty-free license to publish or
** reproduce these documents, or allow others to do so, for U.S.
** Government purposes only pursuant to the copyright license under the
** contract clause at 252.227.7013.
**
** @OPENSOURCE_HEADER_END@
*/
#ifndef _LIBFLOWSOURCE_H
#define _LIBFLOWSOURCE_H

#include <silk/silk.h>

RCSIDENTVAR(rcsID_LIBFLOWSOURCE_H, "$SiLK: libflowsource.h 372a8bc31d8a 2012-02-10 21:55:28Z mthomas $");

/*
**  libflowsource.h
**
**  Definitions/Declaration for the libflowsource library and all the
**  possible external flow sources (IPFIX, PDU, etc).
**
*/

#include <silk/rwrec.h>
#include <silk/skvector.h>
#include <silk/probeconf.h>


#define SOURCE_LOG_NONE     0
#define SOURCE_LOG_MISSING (1 << 0)
#define SOURCE_LOG_BAD     (1 << 1)
#define SOURCE_LOG_ALL      0xff


/* Number of bytes we want to split between socket buffers */
#define SOCKETBUFFER_NOMINAL_TOTAL 0x800000 /* 8M */

/* Environment variable to modify SOCKETBUFFER_NOMINAL_TOTAL */
#define SOCKETBUFFER_NOMINAL_TOTAL_ENV "SK_SOCKETBUFFER_TOTAL"

/* Minimum number of bytes to attempt to allocate to a socket buffer */
#define SOCKETBUFFER_MINIMUM       0x20000 /* 128K */

/* Environment variable to modify SOCKETBUFFER_MINIMUM */
#define SOCKETBUFFER_MINIMUM_ENV "SK_SOCKETBUFFER_MINIMUM"



typedef struct flowsource_stats_st {
    /* Number of processed packets */
    uint64_t    procPkts;
    /* Number of completely bad packets */
    uint64_t    badPkts;
    /* Number of good records processed */
    uint64_t    goodRecs;
    /* Number of records with bad data */
    uint64_t    badRecs;
    /* Number of missing records; NOTE: signed int to allow for out of
     * seq pkts */
    int64_t     missingRecs;
} flowsource_stats_t;

typedef flowsource_stats_t *flowStats_t;


/***  PDU SOURCES  ********************************************************/

/*
 *    A PDU source is a flow record source based on NetFlow v5 PDU
 *    records.  Once created, records can be requested of it via a
 *    pull mechanism.
 */

/*    The type of PDU sources. */
typedef struct pdu_source_st *pduSource_t;

/*    A source-pool allows multiple pdu-sources to listen on a single
 *    port. */
typedef sk_vector_t *pduSourcePool_t;


/*
 *    Creates a pdusource_t object.
 *      port:                  The local port to bind to
 *      netflow_source_address:The host we expect to receive NetFlow from
 *                             (can be INADDR_ANY)
 *      listen_on_address:     The local address to bind to
 *                             (can be INADDR_ANY)
 *      max_pdus:              How many NetFlow PDUs to buffer
 *      sockbufsize:           How many bytes to allocate to the receiving
 *                             socket
 *
 *    Returns pduSource_t on success, NULL on error.
 */
pduSource_t pduSourceCreate(
    int             port,
    in_addr_t       netflow_source_address,
    in_addr_t       listen_on_address,
    uint32_t        max_pdus,
    int             sockbufsize);


/*
 *   Creates a pdusource_t object.
 *      path:       The path of the file containing NetFlow PDUs
 *
 *    Returns pduSource_t on success, NULL on error.
 */
pduSource_t pduFileSourceCreate(
    const char      *path);


/*
 *    Stops processing of pdus.  This will cause a call to any
 *    pduSourceGet function to stop blocking.  Meant to be used as a
 *    prelude to pduSourceDestroy in threaded code.
 */
void pduSourceStop(pduSource_t source);


/*
 *    Use to destroy a pduSource_t.  This will also cause a call to
 *    any pduSourceGet() function to stop blocking.
 */
void pduSourceDestroy(pduSource_t source);


/*
 *    Fills an rwRec from a pduSource.  Returns 0 on success, -1 on
 *    error.
 */
int pduSourceGetGeneric(
    pduSource_t     source,
    rwRec          *rwrec);


/*
 *    Creates a pduSourcePool_t object
 */
pduSourcePool_t pduSourcePoolCreate(void);


/*
 *    Creates a pduSourcePool_t object
 */
void pduSourcePoolDestroy(pduSourcePool_t pool);


/*
 *    Creates a PDU source based on a skpc_probe_t.  The source will
 *    be created in the context of the current pduSourcePool_t.  This
 *    pool saves state between calls to pduSourceCreateFromProbeDef()
 *    so probes with the same receive-as address and port can share a
 *    single underlying thread.
 */
pduSource_t pduSourceCreateFromProbeDef(
    pduSourcePool_t     pool,
    const skpc_probe_t *probe,
    uint32_t            max_pdus,
    int                 sockbufsize);


/*
 *    Get statistics associated with a pdu source.
 */
void pduSourceGetStats(pduSource_t source, flowStats_t stats);


/*
 *    Log statistics associated with a pdu source.
 */
void pduSourceLogStats(
    pduSource_t source,
    const char *name);


 /*
 *    Log statistics associated with a pdu source, and then clear the
 *    stats.
 */
void pduSourceLogStatsAndClear(
    pduSource_t source,
    const char *name);


/*
 *    Clear out current statistics
 */
void pduSourceClearStats(pduSource_t source);


/*
 *    Set logging options
 */
void pduSourceSetLogopt(pduSource_t source, uint8_t opt);


/***  IPFIX SOURCES  ******************************************************/

/*
 *    An IPFIX source is a flow record source based on IPFIX or
 *    NetFlow V9 records.  Once created, records can be requested of
 *    it via a pull mechanism.
 */

/*    The type of IPFIX sources. */
typedef struct ipfix_source_st *ipfixSource_t;


/*
 *    Performs any initialization required prior to creating the IPFIX
 *    sources.  Returns 0 on success, or -1 on failure.
 */
int ipfixSourcesSetup(void);


/*
 *    Creates a IPFIX source.
 *
 *    'port' is the tcp port to listen upon for IPFIX flows.
 *    'listen_on_address' is the address to bind to.  'protocol'
 *    determines whether the port listens for UDP or TCP traffic.
 *    'max_flows' is the number of IPFIX flows the created source can
 *    buffer in memory.
 *
 *    If 'nfv9' is non-zero, assume NetFlow v9.
 *
 *    Returns a IPFIX source on success, or NULL on failure.
 */
ipfixSource_t ipfixSourceCreate(
    int             port,
    in_addr_t       listen_on_address,
    skpc_proto_t    protocol,
    uint32_t        max_flows,
    int             nfv9);


/*
 *    Creates a IPFIX source based on an skpc_probe_t that listens for
 *    flows on the network.
 *
 *    'probe' contains the network information required to create the
 *    flow source (port, address, protocol).  'max_flows' is the
 *    number of IPFIX flows the created source can buffer in memory.
 */
ipfixSource_t ipfixSourceCreateFromProbeDef(
    const skpc_probe_t *probe,
    uint32_t            max_flows);


/*
 *    Creates a IPFIX source based on an skpc_probe_t that polls a
 *    directory.
 *
 *    'probe' is the probe that was polling the directory, and 'path'
 *    is a file that the directory poller returned.  'path' should be
 *    the filename of an IPFIX file.
 *
 *    Returns a IPFIX source on success, or NULL on failure.
 */
ipfixSource_t ipfixFileSourceCreate(
    const skpc_probe_t *probe,
    const char         *path);


/*
 *    Destroys a IPFIX source.
 */
void ipfixSourceDestroy(
    ipfixSource_t source);


/*
 *    Requests a SiLK Flow record from the IPFIX source 'source'.
 *
 *    This function will block if there are no IPFIX flows available
 *    from which to create a SiLK Flow record.
 *
 *    Returns 0 on success, -1 on failure.
 */
int ipfixSourceGetGeneric(
    ipfixSource_t   source,
    rwRec          *rwrec);


#endif /* _LIBFLOWSOURCE_H */

/*
** Local Variables:
** mode:c
** indent-tabs-mode:nil
** c-basic-offset:4
** End:
*/
