/*
** Copyright (C) 2001-2012 by Carnegie Mellon University.
**
** @OPENSOURCE_HEADER_START@
**
** Use of the SILK system and related source code is subject to the terms
** of the following licenses:
**
** GNU Public License (GPL) Rights pursuant to Version 2, June 1991
** Government Purpose License Rights (GPLR) pursuant to DFARS 252.227.7013
**
** NO WARRANTY
**
** ANY INFORMATION, MATERIALS, SERVICES, INTELLECTUAL PROPERTY OR OTHER
** PROPERTY OR RIGHTS GRANTED OR PROVIDED BY CARNEGIE MELLON UNIVERSITY
** PURSUANT TO THIS LICENSE (HEREINAFTER THE "DELIVERABLES") ARE ON AN
** "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY
** KIND, EITHER EXPRESS OR IMPLIED AS TO ANY MATTER INCLUDING, BUT NOT
** LIMITED TO, WARRANTY OF FITNESS FOR A PARTICULAR PURPOSE,
** MERCHANTABILITY, INFORMATIONAL CONTENT, NONINFRINGEMENT, OR ERROR-FREE
** OPERATION. CARNEGIE MELLON UNIVERSITY SHALL NOT BE LIABLE FOR INDIRECT,
** SPECIAL OR CONSEQUENTIAL DAMAGES, SUCH AS LOSS OF PROFITS OR INABILITY
** TO USE SAID INTELLECTUAL PROPERTY, UNDER THIS LICENSE, REGARDLESS OF
** WHETHER SUCH PARTY WAS AWARE OF THE POSSIBILITY OF SUCH DAMAGES.
** LICENSEE AGREES THAT IT WILL NOT MAKE ANY WARRANTY ON BEHALF OF
** CARNEGIE MELLON UNIVERSITY, EXPRESS OR IMPLIED, TO ANY PERSON
** CONCERNING THE APPLICATION OF OR THE RESULTS TO BE OBTAINED WITH THE
** DELIVERABLES UNDER THIS LICENSE.
**
** Licensee hereby agrees to defend, indemnify, and hold harmless Carnegie
** Mellon University, its trustees, officers, employees, and agents from
** all claims or demands made against them (and any related losses,
** expenses, or attorney's fees) arising out of, or relating to Licensee's
** and/or its sub licensees' negligent use or willful misuse of or
** negligent conduct or willful misconduct regarding the Software,
** facilities, or other rights or assistance granted by Carnegie Mellon
** University under this License, including, but not limited to, any
** claims of product liability, personal injury, death, damage to
** property, or violation of any laws or regulations.
**
** Carnegie Mellon University Software Engineering Institute authored
** documents are sponsored by the U.S. Department of Defense under
** Contract FA8721-05-C-0003. Carnegie Mellon University retains
** copyrights in all material produced under this contract. The U.S.
** Government retains a non-exclusive, royalty-free license to publish or
** reproduce these documents, or allow others to do so, for U.S.
** Government purposes only pursuant to the copyright license under the
** contract clause at 252.227.7013.
**
** @OPENSOURCE_HEADER_END@
*/
#ifndef _IOCHECKS_H
#define _IOCHECKS_H

#include <silk/silk.h>

RCSIDENTVAR(rcsID_IOCHECKS_H, "$SiLK: iochecks.h 372a8bc31d8a 2012-02-10 21:55:28Z mthomas $");

#include <silk/skstream.h>


#define MAX_PASS_DESTINATIONS 2
#define MAX_FAIL_DESTINATIONS 2
#define MAX_ALL_DESTINATIONS  1


typedef struct iochecksInfoStruct_st {
    uint8_t         devnullUsed;
    uint8_t         passCount;
    uint8_t         failCount;
    uint8_t         stdinUsed;
    uint8_t         stderrUsed;
    uint8_t         stdoutUsed;
    uint8_t         maxPassDestinations;
    uint8_t         maxFailDestinations;
    int             firstFile;
    int             fileCount;
    FILE           *passFD[MAX_PASS_DESTINATIONS];
    char           *passFPath[MAX_PASS_DESTINATIONS];
    int             passIsPipe[MAX_PASS_DESTINATIONS];
    FILE           *failFD[MAX_FAIL_DESTINATIONS];
    char           *failFPath[MAX_FAIL_DESTINATIONS];
    int             failIsPipe[MAX_FAIL_DESTINATIONS];
    int             inputPipeFlag;
    skstream_t     *inputCopyFD;
    const char     *inputCopyFPath;
    char          **fnArray;
    char          **argv;
    int             argc;
} iochecksInfoStruct_t;


/*
 *  ioISP = iochecksInit(max_pass, max_fail, argc, argv);
 *
 *    Allocates an iochecksInfoStruct_t, initializes it, and returns it.
 *    max_pass is the number of "pass" destinations to use; max_fail
 *    is the maximum number of "fail" destinations to use; argc and
 *    argv are the argc and argv that were handed to main().  Returns NULL
 */
iochecksInfoStruct_t *iochecksSetup(
    uint8_t     maxPass,
    uint8_t     maxFail,
    int         argc,
    char      **argv);


/*
 *  ok = iochecksPassDestinations(ioISP, file_path, tty_ok);
 *  ok = iochecksFailDestinations(ioISP, file_path, tty_ok);
 *
 *    Opens 'file_path' and adds it to the list of pass or fail
 *    destinations on 'ioISP'.  If 'tty_ok' is non-zero, then
 *    'file_path' can be a terminal.  Function returns 0 for normal
 *    exit, or non-zero for these error conditions:
 *
 *    -- The maximum pass or fail destinations have already been
 *       specified.
 *    -- The 'file_path' is connected to a terminal and tty_ok is 0.
 *    -- More than one stream wants to use stdout or stderr.
 *    -- The output 'file_path' is a file (not a FIFO), it exists, and
 *       it has a non-zero size.
 *    -- The 'file_path' cannot be opened.
 *    -- Memory allocation error.
 */
int iochecksPassDestinations(
    iochecksInfoStruct_t   *ioISP,
    const char             *file_path,
    int                     ttyOK);
int iochecksFailDestinations(
    iochecksInfoStruct_t   *ioISP,
    const char             *file_path,
    int                     ttyOK);


/*
 *  ok = iochecksAllDestinations(ioISP, file_path);
 *
 *    Similar to iochecksPassDestinations(), except this stream is not
 *    allowed to be a tty.
 */
int iochecksAllDestinations(
    iochecksInfoStruct_t   *ioISP,
    const char             *file_path);


/*
 *  ok = iochecksInputSource(ioISP, stream_name);
 *
 *    Verify that 'stream_name' makes sense as an input pipe.
 *    'stream_name' can be the path to a named pipe (FIFO), or the
 *    string "stdin", in which case input is read from the standard
 *    input.
 *
 *    Returns 0 and updates the input list in 'ioISP' if 'stream_name'
 *    is a named pipe or is "stdin" and the standard input is not
 *    connected to a terminal.  Returns non-zero for these error
 *    conditions:
 *
 *    -- 'stream_name' is "stdin" and stdin is a tty.
 *    -- 'stream_name' names a file that does not exist or that is not
 *       a FIFO.
 *    -- This function has been previously called to set-up an input
 *       source.
 *    -- Memory allocation errors.
 *
 *    The 'stream_name' is not opened; that must be handled by the
 *    caller.
 */
int iochecksInputSource(
    iochecksInfoStruct_t   *ioISP,
    const char             *stream_name);


/*
 *  ok = iochecksAcceptFromStdin(ioISP);
 *
 *    Checks to see if the standard input is a terminal; if it is, the
 *    functions returns 0.  Otherwise, the function calls
 *    iochecksInputSource() with "stdin" as the 'stream_name' argument
 *    and returns the status of that function.
 */
int iochecksAcceptFromStdin(iochecksInfoStruct_t *ioISP);


/*
 *  ok = iochecksInputs(ioISP, zero_files_ok);
 *
 *    Verify that no more than one source of input records has been
 *    specified; if 'zero_files_ok' is zero, verify that exactly one
 *    source is specified.  The source can be the files listed on the
 *    command line or a pipe (stdin or a FIFO), but not both.
 */
int iochecksInputs(
    iochecksInfoStruct_t   *ioISP,
    int                     zero_files_ok);


/*
 *  ok = iochecksTeardown(ioISP);
 *
 *    Deallocate all memory used by the ioISP and close the
 *    copy-destination if it is open.  This function does not close
 *    any other files; that is the responsibility of the caller.
 */
void iochecksTeardown(iochecksInfoStruct_t *ioISP);


/*
 *  os = iochecksOpenCopyDest(ioISP);
 *
 *    Open the copy-destination (all-destination) and return 0 if it
 *    was successfully opened.  Returns nonzero if there was an error
 *    opening the copy-destination.
 *
 *    Also return 0 if no copy-destination was specified.
 */
int iochecksOpenCopyDest(iochecksInfoStruct_t *ioISP);


#endif /* _IOCHECKS_H */

/*
** Local Variables:
** mode:c
** indent-tabs-mode:nil
** c-basic-offset:4
** End:
*/
