# (c) 1999 Matthias Urlichs, smurf@noris.de

package RT::ui::mail;

use utf8;
use warnings; no warnings qw(once redefine uninitialized);
use strict;
use Cf qw($RT_MAIL_OK $RT_MAX_SIZE $MAILDOM $WDESCR $TICKET_DOMAIN);
use Dbase::Globals qw(find_descr get_kunde get_person test_flag sendmail);
use Dbase::Help qw(DoFn Do DoN qquote DoTrans);
use Dbase::OTRS qw(TicketID);
use RT::database qw(req_in);
use RT::database::manipulate qw(
  add_comment
  add_correspondence
  add_new_request
  change_date_due
  change_date_fdue
  give
  reopen
  resolve
  stall
);

use RT::support::mail qw(template_mail);
use RT::support::utils qw(normalize_sn adr_filter);
use RT::database::config qw(is_a_queue);
use CGI;
use Getopt::Std;
use Umlaut qw(binmodus);

use constant { EX_HARDFAIL => 100, EX_TEMPFAIL => 111 };

use vars qw(@ISA @EXPORT_OK);
require Exporter;
@ISA = qw(Exporter);
@EXPORT_OK = qw(activate);

my %ids;

use vars qw(%queues %req);

BEGIN {
	*queues = \%RT::queues;
	*req = \%RT::req;
}


sub _activate {
	%ids = ();

	my($rc,$header,$body,@adr)=@_;
	return $rc if defined $rc;

	if ((my $size = length $body) > $RT_MAX_SIZE) {
		print STDERR "Diese Mail ist zu lang ($size > $RT_MAX_SIZE Bytes)!\n";

		# Solche Mails sollten eigentlich schon vorher von exim umgeleitet werden,
		# d. h. wenn sie hier landen, ist wohl was faul.
		return EX_TEMPFAIL;
	}

	my($queue,$in_action,$ser);
	if(@ARGV) {
		$queue=$ARGV[0];
		$in_action=$ARGV[1];
		#open(LOG,">/dev/null");
		#print LOG "\n\n*** New ***\n$header";
		#print LOG "Queue $queue, Action $in_action\n";
	} else {
		$in_action="comment"; # correspond
		# open(STDERR,">&STDOUT");
		#open(LOG,">>/tmp/ta.log");
		#print LOG "\n\n*** New ***\n$header";
		#print LOG "No Queue, Action $in_action\n";
	}
	if( $header =~ /^(?:X-)?RT-Queue:\s+(?:$WDESCR\.)?([\w-]+)\s*$/im ) {
		$queue = $1;
	}
	my %param;
	if ( $header =~ /^(?:X-)?RT-Kunde:\s+(\S+)/im ) {
		$param{kunde} = $1;
	}
	if ( $header =~ /^(?:X-)?RT-Info:\s+(.*)/im ) {
		$param{info} = $1;
	}
	if ( $header =~ /^(?:X-)?RT-Priority:\s+(\d\d?)(?!\d)/im ) {
		$param{prio} = $1;
	}
	if ( $header =~ /^(?:X-)?RT-Area:\s+(\S+)/im ) { $param{area} = $1 }
	if ( $header =~ /^(?:X-)?RT-Status:\s+(\S+)/im ) { $param{status} = $1 }
	if ( $header =~ /^(?:X-)?RT-Owner:\s+(\S+)/im ) {
		$param{owner} = $1;
	}

	if ( $header =~ /^(?:X-)RT-Autoreply:\s+(y|yes|n|no)\b/im ) {
		$param{autoreply} = $1 =~ /^y/i;
	}
	elsif ( $header=~ /^X-Cron-Env:\s+/im ) { $param{autoreply} = '' }

	my $otrs;

	if(defined $queue and $queue =~ /^(?:rt-)?(\d+)$/) {
		my $iticket = $1;
		$otrs = TicketID($iticket);
		if ($otrs) {
			print STDERR "Ticket #$iticket ist im OTRS! (intern #$otrs)\n";
			$otrs = $iticket;
		} else {
			my($ticket,$gticket) = DoFn("select id,ticket from ticket where id=$iticket");
			$otrs = TicketID($gticket) if $gticket and not $otrs and $gticket != $iticket;
			if($otrs) {
				print STDERR "Ticket #$gticket ist jetzt im OTRS! (intern #$otrs)\n";
				$otrs = $gticket;
			} elsif (DoFn("select id from ticket where id = $1")) {
				$ser = $1;
				$queue = DoFn("select queue.name from queue,ticket where ticket.id = $ser and queue.id = ticket.queue");
			} else {
				print STDERR "Ticket #$1 nicht gefunden.\n";
				return EX_HARDFAIL;
			}
		}
	}

	my $qid;
	unless($otrs) {
		if(not defined $queue or not ($qid = is_a_queue($queue))) {
			print STDERR "Unbekannte Queue!\n";
			return EX_HARDFAIL;
		}
		if($queues{$queue}{'otrs'}) {
			print STDERR "Queue #$qid:$queue ist im OTRS!\n";
			$otrs = $queue;
		}
	}

	$in_action='correspond' unless $in_action;

	my($serial_num,$subject,$current_user,$auth_user,$return);
	($rc,$serial_num,$subject,$current_user,$auth_user,$return) = parse_headers($header,$body,$queue); 
	return $rc if defined $rc;

	if(defined $otrs) {
		sendmail($body, $return, "$otrs\@$TICKET_DOMAIN") and return $RT_MAIL_OK;
		print STDERR "... konnte dort aber nicht hingemailt werden ??\n";
		return EX_HARDFAIL;
	};

	$serial_num = $ser if $ser;
	$serial_num = normalize_sn($serial_num);
	#print LOG "Ser.4 $serial_num\n" if $serial_num;

#take all those actions

	($header,$body,$current_user,$auth_user)=parse_actions($current_user,$auth_user,$header,$body);

	return $RT_MAIL_OK if $in_action eq 'actions';
	my($transaction,$message);

	$param{kunde} ||= $current_user;

	my $extern;
	if ( $header =~ /^(?:X-)?RT-Extern:\s+(y|yes|n|no)\b/im ) {
		$extern = lc substr $1, 0, 1;
	} else {
		{
			( my $kunde = $param{kunde} ) && $queues{$queue}{rtweb} or last;
			( $kunde = get_kunde $kunde ) && $kunde != 1 or last;
			if ($serial_num) {
				req_in($serial_num, '_rt_system');
				last unless $kunde == $req{$serial_num}{kunde_id};
			}
			$extern = 'y';
		}
	}

	$body = '' unless defined $body;
	if ($in_action eq 'comment' and $serial_num) {
		my $req = new CGI({requestors=>join(",",@adr),queue=>$queue,serial_num=>$serial_num,subject=>$subject,status=>"open",from=>$current_user,auth_user=>$auth_user,via_mail=>1,extern=>$extern, %param});
		($transaction,$message)=add_comment($req,$body);
		foreach my $id(keys %ids) {
			DoN("insert into ticketid set id=${\qquote $id},ticket=$serial_num") unless $id =~ /\@ticket.$MAILDOM\>/mi;
		}
	} elsif ($in_action eq 'correspond' or $in_action eq 'comment') {

		{
			my $kunde = get_kunde( $param{kunde} ) || 1;
			$serial_num = DoFn(<<_) and !$ENV{RTMAIL_TERSE} and print "(#$serial_num)\n"
	SELECT   id
	FROM     ticket
	WHERE    id      = ticket
	     AND kunde   = $kunde
	     AND queue   = $queues{$queue}{id}
	     AND status IN (${\ find_descr( tickets => open => 1 ) }, ${\ find_descr( tickets => stalled => 1 ) })
	     AND subject = ${\ qquote($subject) }
	ORDER BY id DESC
	LIMIT    1
_
			  unless $serial_num
				  || $subject eq 'Aus einem Xerox WorkCentre Pro scannen'
				  && $kunde == 1;
		}

		unless($serial_num) {

			my $due = $queues{$queue}{'default_due'} || 0;
			$due = $RT::time+$due*3600*24 if $due != 0;

			my $auth_user = $auth_user || '_rt_system'; # Workaround für RT#234829, sollte man evtl. nochmal überdenken
			my $req = new CGI({requestors=>$current_user,queue=>$queue,serial_num=>$serial_num,subject=>$subject,status=>"open",from=>$current_user,auth_user=>$auth_user,via_mail=>1,date_due=>$due,prio=>$queues{$queue}{'default_prio'},extern=>$extern, %param});

			($serial_num,$transaction, $message)=add_new_request($req,$body);
		} else {
			my $req = new CGI({requestors=>$current_user,queue=>$queue,serial_num=>$serial_num,subject=>$subject,status=>"open",from=>$current_user,auth_user=>$auth_user,via_mail=>1,extern=>$extern, %param});
			($transaction,$message)=add_correspondence($req,$body);
		}
	}
	if($transaction == 0) {
		#print LOG "* Fehler! $message\n";
		my $edited_content = "There has been an error with your request:\n" . $message . "\n\nYour message is reproduced below\n\n".$body;
			
		template_mail ('error','_rt_system',$auth_user,"","",0,0,"RT Error",$current_user,$edited_content) if @ARGV;
		print "-$message\n";
	} else {
		# Ausgabeformat nicht ändern, da sich notify_rt & Co. darauf verlassen!
		printf "%d-%d\n",$transaction->[0],$transaction->[1];
	}
	return $RT_MAIL_OK;
}

sub activate {
	my $rc;
	my @res = read_mail_from_stdin(@ARGV == 0);
	DoTrans { $rc = _activate(@res); } 3;
	exit $rc;
}

sub read_mail_from_stdin($) {
	my($smtpmode) = @_;
	use MIME::Head;
	use IO::String;

	binmodus(\*STDIN);
	if ((my $size = -s STDIN) > $RT_MAX_SIZE) {
		print STDERR "Mail zu groß ($size > $RT_MAX_SIZE Bytes)!\n";
		return 0;
	}
	local $/ = "\n\n";
	my $content = <STDIN>;
	my $data = IO::String->new(\$content);
	my $hdr = MIME::Head->read($data);
	$hdr->decode();

	my %req;
	foreach my $a(adr_filter($hdr->get("from"))) {
		$req{$a}++;
	}
	foreach my $a(adr_filter($hdr->get("reply-to"))) {
		$req{$a}++;
	}

	$/ = "\n";
	while (<STDIN>){
		if($smtpmode) {
			last if $_ =~ /^\.\r?\n/;
			$_ =~ s/^\.//;
		}
		$_ =~ s/\r\n/\n/;
		$content .= $_;
	}
	return (undef,$hdr->as_string,$content,keys %req);
}

sub CheckSN($$;$) {
	my ($ser,$queue,$log) = @_;
	my $id = $queues{$queue}{id};
	#print LOG "CheckSN ".(defined $ser ? $ser : "<?>")." in $queue $id :: $log\n";
	if($ser and $ser > 0) {
		my $q = DoFn("select queue from ticket where id = $ser");
		if($q == 0) {
			Do("update ticket set queue = $id where id = $ser");
			#print LOG "CheckSN SET $queue\n";
		} elsif($q != $id) {
			$ser = undef;
			#print LOG "CheckSN QUEUES $q $queue\n";
		}
	}
	$ser;
}

sub parse_headers($$) {
	# use Date::Parse;
	my ($content,$body,$queue) = @_;
	$queue = $queues{$queue}{name};
	$content =~ s/\n\s+/ /g;
	## ?? ## $current_user = $ENV{"SENDER"};
	my($msgid, $subject, $serial_num, $current_user);

	if ($content =~ /^Subject:(.*)\[${\join ' ?', split \/\/, $WDESCR}\s*# ?((?:\d ?)+)\][^\S\n]*(?:\n[^\S\n]|[^\S\n])*(.*)/imo) {
		($serial_num=$2) =~ y/ //d;
		$subject="$1 $3";
		$subject =~ s/\($queue\)//i;
	} elsif (($content =~ /^Subject: (.*)/mi) and (!$subject)){
		$subject=$1;
	} else {
		$subject="<no subject>";
	}
	$serial_num=$1 if $content =~ /^X-RT: $WDESCR (\d+)$/mi;

	$subject =~ s/^\s*((Re|AW|Antwort)(\^\d+)?\:\s*)+//i;
	# $time_in_text = str2time($1) if ($content =~ /^Date:\s+(.*)/mi);
	# $time_in_text = $RT::time if $time_in_text > $RT::time;

	if ($content =~ s/^(?:Resent-)?Message-Id:\s+\<.*\@ticket.$MAILDOM\>//mi) {
		## Loop detected...
		return $RT_MAIL_OK;
	}
	$serial_num = $2 if !$serial_num and $content =~ /^(References|In-Reply-To):\s+\<(\d+)(-\d+)?(%\S+)\@ticket.$MAILDOM\>/mi;

	while($content =~ s/^((?:Resent-)?Message-ID|References|In-Reply-To):\s[^\n]*?\<(\S+)\>/$1: /mi) {
		my $key = $1;
		my $id = substr $2,0,255; # ticketid.id kann nur 255 Zeichen fassen
		my($ser,$seq) = DoFn("select ticket,seq from ticketid where ticketid.id=${\qquote $id}");
		$serial_num = CheckSN($ser,$queue,"at $id") unless $serial_num;

		if ( $seq and $key =~ /\bmessage-id$/i ) { # Mail ist schon im RT
			printf "($ser-$seq)\n";
			return $RT_MAIL_OK;
		}
		$queues{$queue}{allow_user_create}=1 if $key =~ /resent-/i;
		$ids{$id}=1;
	}

	if ($subject) { $subject =~ s/\s+/ /g }
	else { $subject = "[No Subject Given]" }

	my($return) = ($content =~ /^Return-Path:\s+(.*)/mi);
	my $cu;
	if (
		$content =~ /^(?:Resent-)?From:\s+(.*)/mi ||
		$content =~ /^Sender:\s+(.*)/mi           ||
		$content =~ /^Return-Path:\s+(.*)/mi
	) {
		$current_user = $1;
		$current_user = $1
			if $current_user =~ /<(\S*\@\S*)>/ ||
			   $current_user =~ /(\S*\@\S*)/   ||
			   $current_user =~ /<(\S*)>/;

		$cu = get_person( $current_user, 'pop,?email', 1 )
 		  unless $current_user =~ /^(?:mailer-daemon|postmaster)\b/i;
	}

	while(not $serial_num and $body =~ s/^((?:Resent-)?Message-ID|References|In-Reply-To):\s[^\n]*?\<(\S+)\>/$1: /m) {
		$serial_num = CheckSN(DoFn("select ticket from ticketid where ticketid.id = ${\qquote $2}"),$queue," at $2");
	}
	(undef,$serial_num,$subject,$current_user,$cu,$return);
}


sub parse_actions($$$$) {
	my ($real_current_user, $auth_user, $header,$body) = @_;

	my ($trans, $message, $serial_num, $line, $original_line, $current_user);
	my ($username, $password, $parsed_body, $response);
	$current_user = $real_current_user;

	$body = "Content-Type: text/plain; charset=utf-8\n".$body
		unless $header =~ /^Content-Type:/mi;
	$body = "Content-Transfer-Encoding: 8bit\n".$body
		unless $header =~ /^Content-Transfer-Encoding:/mi;

	foreach $line (split(/\n/,$body)) {
		my $count;
		$original_line = $line;

		#if it's a line with an rt action, deal with it.
		if ($line =~ /^\%rt (.*)/i) {
			use Text::ParseWords;
			my @arg = shellwords($1);
	
			if ($arg[0] =~ /^user/i) {
				$username = $arg[1];
				$message = "Username $username noticed.";
			}
	
			elsif ($arg[0] =~ /help/i) {
				$message = "
Mail Mode for RT $RT::rtversion by jesse vincent <jesse\@fsck.com>
*** heavily hacked by Matthias Urlichs <urlichs\@noris.net> ***
Command Summary

RT commands are prefixed by %RT and are case insensitive.  RTMail evaluates
statements in the order you enter them.

%RT USER <username> 
	will tell RT who you really are. 

%RT PASS <password>
	will authenticate you to RT, provided you've already executed a USER 
	command.

%RT TAKE <num>
	will take request <num>

%RT UNTAKE <num>
	will give away request <num>, provided you own it.

%RT STEAL <num>
	will take request <num>, provided someone else owns it.

%RT RESOLVE <num>
	will resolve request <num>.

%RT RESOLVE <num>
	will resolve request <num>.

%RT OPEN <num>
	will open request <num>.

%RT STALL <num>
	will stall request <num>.

%RT KILL <num> yes
	will kill request <num>.

%RT MERGE <num1> [INTO] <num2>
	will merge request <num2> into request <num2>.

%RT SET owner <num> <user>
	will set request <num>'s owner to <user>.

%RT SET queue <num> <queue>
	will set request <num>'s queue to <queue>.

%RT SET area <num> <area>
	will set request <num>'s area to <aera>.

%RT SET due <num> <date>
	will set request <num>'s re-open date to <date>. <date> should probably 
	be in the form MM/DD/YY. 

%RT SET fdue <num> <date>
	will set request <num>'s due date to <date>. <date> should probably 
	be in the form MM/DD/YY. 

%RT SET prio <num> <prio>
	will set request <num>'s priority to <prio>.

%RT SET final <num> <prio>
	will set request <num>'s final priority to <prio>.

%RT SET status <num> (open|closed|stalled|dead yes)
	will set request <num>'s status to (open|closed|stalled|dead).

%RT SET user <num> <email>
	will set request <num>'s requestor(s) to the comma-delineated, 
	quote-enclosed string <email>.";
			}

			elsif ($arg[0] =~ /^pass/i) {
				$password = $arg[1];
				if ($username) {
					#check the authentication state
					if (not is_password($username, $password)) {
						$message = "Bad Login for $username.";
						$trans = 0;
					} else {
						$message = "You are now authenticated as $username.";
						$current_user = $username;
						$auth_user = get_person($username);
					}
				}
			}

			elsif ($arg[0] =~ /stall/i) {
				$serial_num=$arg[1];
				($trans,  $message)=stall($serial_num, $current_user);
			}

			elsif ($arg[0] =~ /open/i) {
				$serial_num=$arg[1];
				($trans,  $message)=reopen($serial_num, $current_user);
			}

			elsif ($arg[0] =~ /resolv/i)  {
				$serial_num=$arg[1];
				($trans,  $message)=resolve($serial_num, $current_user);
			}
		
			elsif (($arg[0] =~ /kill/i) and ($arg[2] =~ /^yes/)){
				$serial_num=int($arg[1]);
				($trans,  $message)=kill($serial_num, $current_user);
			}
	
			elsif ($arg[0] =~ /merg/i){
				$serial_num=int($arg[1]);
				my $into;
				if ($arg[2] =~ /in/i) {
					$into = $arg[3];
				} else {
					$into = $arg[2];
				}
				($trans,  $message)=merge($serial_num, $into, $current_user);
			}
	
			elsif ($arg[0] =~ /^take/i) {
				$serial_num=$arg[1];
				($trans,  $message)=take($serial_num, $current_user);
			}
		
			elsif ($arg[0] =~ /^untake/i) {
				$serial_num=$arg[1];
				($trans,  $message)=untake($serial_num, $current_user);
			}
		
			elsif ($arg[0] =~ /steal/i) {
				$serial_num=$arg[1];
				($trans,  $message)=steal($serial_num, $current_user);
			}

			elsif ($arg[0] =~ /^set/i) {
				
				if ($arg[1] =~ /^own/) {
					$serial_num=int($arg[2]);
					my $owner=$arg[3];
					($trans,  $message)=give($serial_num, $owner, $current_user);
				}
				
				elsif (($arg[1] =~ /^user/) or ($arg[1] =~ /^requestor/)) {
					$serial_num=int($arg[2]);
					my $new_user=$arg[3];
					($trans,  $message)=change_requestors($serial_num, $new_user, $current_user);
				}

				elsif ($arg[1] =~ /^sub/) {
					$serial_num=int($arg[2]);
					my $subject=$arg[3];
					($trans,  $message)=change_subject ($serial_num, $subject, $current_user);
				}
				
				elsif ($arg[1] =~ /^queue/) {
					$serial_num=int($arg[2]);
					my $queue=$arg[3];
					($trans,  $message)=change_queue ($serial_num, $queue, $current_user);
				}
				
				elsif ($arg[1] =~ /^area/) {
					$serial_num=int($arg[2]);
					my $area=$arg[3];
					($trans,  $message)=change_area ($serial_num, $area, $current_user);
				}

				elsif ($arg[1] =~ /^prio/) {
					$serial_num=int($arg[2]);
					my $prio=$arg[3];
					($trans,  $message)=change_priority ($serial_num, $prio, $current_user);
				}
				
				elsif ($arg[1] =~ /^due/) {
					$serial_num=int($arg[2]);
					my $due_string=$arg[3];
					
					my $due_date = date_parse($due_string);
					
					($trans,$message)=change_date_due($serial_num, $due_date, $current_user);
				}

				elsif ($arg[1] =~ /^fdue/) {
					$serial_num=int($arg[2]);
					my $fdue_string=$arg[3];
					
					my $fdue_date = date_parse($fdue_string);
					
					($trans,$message)=change_date_fdue($serial_num, $fdue_date, $current_user);
				}

				elsif ($arg[1] =~ /^status/) {
					$serial_num=int($arg[2]);
					my $status=$arg[3];
					my $confirmation=$arg[4];

					

					if ($status =~ /stall/i) {
						($trans,  $message)=stall($serial_num, $current_user);
					} elsif ($status =~ /open/i) {
						($trans,  $message)=reopen($serial_num, $current_user);
					} elsif ($status =~ /resolv/i)  {
						($trans,  $message)=resolve($serial_num, $current_user);
					} elsif (($status =~ /dead/i) and ($confirmation =~ /^yes/)){
						($trans,  $message)=kill($serial_num, $current_user);
					}
				}
			} elsif ($arg[0] =~ /^pass/) {
				$response .= "> $arg[0] ***** ";
			} else {
				$response .= "> " . $original_line . "\n";
			}
			if ($message) {
				$response .= "RT: $message ";
				$response .= "($trans)" if $trans;
				$response .="\n";
			}
		} else { 
			#if the line doesn't start with %rt, don't discard it.
			# $response .= $original_line . "\n";
			$parsed_body .= $original_line . "\n";
		}
	}

	# RESPONSE HERE
	
	if ($response) {
		($message)=template_mail ('act_response','_rt_system',$real_current_user,"","",0,0,"RT Actions Complete",$real_current_user,$response);
	}
	($header,$parsed_body, $current_user, $auth_user);
}

1;
