package RT::AddOn::Billing;

use utf8;
use warnings; no warnings qw(once redefine uninitialized);
use strict;
use base 'RT::AddOn';

use Cf qw($RT_BILLING_DIENSTE);
use Date::Format qw(time2str);
use Dbase::Globals qw(add_acct aufzaehlung get_kunde);
use Dbase::Help qw(Do DoSelect DoTrans in_list qquote);
use HTML::Entities qw(encode_entities);
use RT::database qw(req_in);
use RT::support::utils qw(normalize_sn);
use Time::DaysInMonth qw(days_in);

use constant MAXLENGTH4STRINGS => ( 1 << 16 ) - 1;

my @dienste = split ' ', lc( my $dummy = $RT_BILLING_DIENSTE )
  or die "Keine \$RT_BILLING_DIENSTE!?\n";
my ( %id4dienst, %info4dienst );
DoSelect {
	my ( $id, $name, $info ) = @_;
	$id4dienst{$name} = $id;
	$info4dienst{$name} = $info;
  }
  'SELECT id, LOWER(name), info FROM dienst WHERE '
  . in_list( name => '', @dienste );
die "Ich habe keinen der \$RT_BILLING_DIENSTE "
  . aufzaehlung(@dienste)
  . " gefunden.\n"
  unless keys %info4dienst;
my ($default_dienst) = @dienste = grep exists $info4dienst{$_}, @dienste;

sub _jjmmtt() {
	my ( $tt, $mm, $jj ) = (localtime)[ 3 .. 5 ];
	$jj += 1900;
	$mm = sprintf '%02d', $mm + 1;
	$tt = sprintf '%02d', $tt;
	if (wantarray) { $jj, $mm, $tt }
	else { "$jj$mm$tt" }
}

sub change_kunde {
	my ( $package, $serial, $kunde ) = @_;
	die qq(Invalid serial "$serial"; caller: ), map "<$_>", caller
	  unless $serial =~ /^\d+\z/;
	my ( $jj, $mm ) = _jjmmtt;
	if ( my @ids = map "[$_->{hash}-$_->{seq}]", grep $_->{jjmm} < "$jj$mm",
		my @rows = $package->init_ticket($serial) )
	{
		'Der Kunde dieses Tickets kann nicht geändert werden, da '
		  . (
			@ids == 1
			? "der zugehörige Accounting-Datensatz $ids[0] in einem bereits abgeschlossenen Monat liegt"
			: 'die zugehörigen Accounting-Datensätze '
			  . join( ' und ', @ids )
			  . ' in bereits abgeschlossenen Monaten liegen.'
		  );
	}
	elsif ( !( my $kunde_id = get_kunde $kunde) ) {
		qq(Kunde "$kunde" unbekannt);
	}
	elsif (@rows) {
		if (
			my $changed = Do(
				"UPDATE acct SET kunde=$kunde_id WHERE " . join ' OR ',
				map "`hash` = $_->{hash} AND seq = $_->{seq}",
				@rows
			)
		  )
		{
			$package->add_transaction(
				$serial, undef,
				'Zugehörige'
				  . (
					$changed == 1
					? 'n Accounting-Datensatz'
					: ' Accounting-Datensätze'
				  )
				  . qq( auf "$kunde" umgebucht)
			);
			();
		}
	}
}

sub change_queue {
	my ( $package, $serial ) = @_;
	die qq(Invalid serial "$serial"; caller: ), map "<$_>", caller
	  unless $serial =~ /^\d+\z/;
	my @ids = $package->init_ticket($serial) or return;
	'Die Queue dieses Tickets darf nicht geändert werden, da ihm '
	  . (
		@ids == 1
		? 'ein Accounting-Datensatz zugeordnet ist'
		: @ids . ' Accounting-Datensätze zugeordnet sind'
	  )
	  . '.';
}

sub init_ticket {
	my ( $package, $serial ) = @_;
	die qq(Invalid serial "$serial"; caller: ), map "<$_>", caller
	  unless $serial =~ /^\d+\z/;
	my @rows;
	DoSelect {
		push @rows,
		  {
			hash    => shift,
			seq     => shift,
			dienst  => shift,
			txt     => shift,
			seconds => shift,
			jjmm    => shift,
			tt      => shift
		  };
	  }
	  <<_;
	SELECT    rt_billing.`hash`, rt_billing.seq, dienst.name, acctassoc.info,
	          acct.bytes, acct.jjmm, LPAD(acct.tt,2,'0')
	FROM      ( acct, dienst, rt_billing )
	LEFT JOIN acctassoc
	       ON acct.`hash`       = acctassoc.`hash`
	      AND acct.seq          = acctassoc.seq
	WHERE     rt_billing.ticket = $serial
	      AND rt_billing.`hash` = acct.`hash`
	      AND rt_billing.seq    = acct.seq
	      AND acct.dienst       = dienst.id
	ORDER BY  acct.jjmm, acct.tt, acct.`hash`, acct.seq
_
	if (wantarray) { @rows }
	else { \@rows }
}

sub kill {
	my ( $package, $serial ) = @_;
	die qq(Invalid serial "$serial"; caller: ), map "<$_>", caller
	  unless $serial =~ /^\d+\z/;
	my @ids = $package->init_ticket($serial) or return;
	'Dieses Ticket darf nicht gekillt werden, da ihm '
	  . (
		@ids == 1
		? 'ein Accounting-Datensatz zugeordnet ist'
		: @ids . ' Accounting-Datensätze zugeordnet sind'
	  )
	  . '.';
}

sub check_merge {
	my ( $package, $serial, $into ) = @_;
	die qq(Invalid serial "$serial"; caller: ), map "<$_>", caller
	  unless $serial =~ /^\d+\z/;
	my @ids = $package->init_ticket($serial) or return;
	'Dieses Ticket darf nicht in ein anderes gesteckt werden, da ihm '
	  . (
		@ids == 1
		? 'ein Accounting-Datensatz zugeordnet ist'
		: @ids . ' Accounting-Datensätze zugeordnet sind'
	  )
	  . '.';
}

sub show_form {
	my ( $package, $req ) = @_;
	( my $serial = $req->{serial_num} ) =~ /^\d+\z/ or die;
	my @rows = $package->init_ticket($serial);
	my ( $jj, $mm, $tt ) = _jjmmtt;
	require RT::CGI4AddOn;
	my $cgi = RT::CGI4AddOn->new;
	print $cgi->start_form, $cgi->table(
		{ bgcolor => '#ccffcc', cellspacing => 16 },
		map( {
				my $hours = int +( $_->{seconds} ||= 0 ) / 3600;
				  my $minutes = sprintf '%02d', int $_->{seconds} % 3600 / 60;
				  die
"Kann mit $_->{seconds} Sekunden gebuchter Arbeitszeit aus Datensatz $_->{hash}-$_->{seq} nicht umgehen.\n"
				  unless grep $_ == $minutes, 0, 15, 30, 45;
				  my ( $_jj, $_mm ) = $_->{jjmm} =~ /^(\d+)(\d\d)\z/
				  or die "Unbekanntes Datumsformat: $_->{jjmm}"
				  if defined $_->{jjmm};
				  my $id = defined $_->{hash}
				  && defined $_->{seq} ? "[$_->{hash}-$_->{seq}]" : '[neu]';
				  $cgi->Tr(
					$cgi->td(
						$cgi->table(
							defined $_->{jjmm}
							  && $_->{jjmm} < "$jj$mm"
							? $cgi->caption(
"Bereits abgeschlossener Accounting-Datensatz $id:"
							  )
							  . $cgi->Tr(
								[
									$cgi->td("Datum: $_jj-$_mm-$_->{tt}")
									  . $cgi->td( { align => 'center' },
										"Arbeitszeit: $hours:$minutes" )
									  . $cgi->td(
										{ align => 'right' },
										'Dienst: '
										  . encode_entities(
											$info4dienst{ $_->{dienst} }
										  )
									  ),
									$cgi->td(
										{ colspan => 2 },
										$cgi->tt(
											join $cgi->br,
											split /\n/,
											encode_entities( $_->{txt} )
										)
									)
								]
							  )
							: $cgi->caption("Accounting-Datensatz $id:")
							  . $cgi->Tr(
								[
									$cgi->td(
										'Datum: ',
										$cgi->popup_menu(
											-default => $_jj || $jj,
											-name   => "jj$id",
											-values => [ $jj, $jj + 1 ]
										  )
										  . '-'
										  . $cgi->popup_menu(
											-default => $_mm || $mm,
											-name   => "mm$id",
											-values => [ '01' .. '12' ]
										  )
										  . '-'
										  . $cgi->popup_menu(
											-default => $_->{tt} || $tt,
											-name => "tt$id",
											-values => [ '01' .. '31' ]
										  )
									  )
									  . $cgi->td(
										{ align => 'center' },
										'Arbeitszeit: '
										  . $cgi->popup_menu(
											-default => $hours,
											-name    => "hours$id",
											-values  => [
												0 .. (
													$hours < 5 ? 9 : $hours << 1
												)
											]
										  )
										  . ':'
										  . $cgi->popup_menu(
											-default => $minutes,
											-name    => "minutes$id",
											-values  => [qw(00 15 30 45)]
										  )
									  )
									  . $cgi->td(
										{ align => 'right' },
										'Dienst: '
										  . $cgi->popup_menu(
											-default => $_->{dienst},
											-name    => "dienst$id",
											-values  => \@dienste,
											-labels  => \%info4dienst
										  )
									  ),
									$cgi->td(
										{ colspan => 2 },
										$cgi->textarea(
											-cols    => 35,
											-default => $_->{txt},
											-name    => "txt$id",
											-rows    => 5,
											-wrap    => 'virtual'
										)
									)
								]
							  )
						)
					)
				  )
			} @rows,
			{
				txt => "RT#$serial (\u$req->{queue}{name}"
				  . ( defined $req->{area} && "/$req->{area}" ) . "):\n"
			} ),
		$cgi->Tr(
			$cgi->td(
				{ align => 'center' },
				$req->{kunde_id} ? $cgi->hidden(
					-name  => 'kunde_id',
					-value => $req->{kunde_id}
				  ) : (),
				$cgi->save_data($serial)
			)
		)
	  ),
	  $cgi->end_form;
}

sub save_data {
	my ( $package, $cgi ) = @_;
	unless ( my $kunde_id = $cgi->param('kunde_id') ) {
'Bevor was accountet werden kann, bitte erst den Kunden des Tickets setzen!';
	}
	elsif ( $kunde_id !~ /^\d+\z/ ) {
		"ungültige kunde_id: $kunde_id";
	}
	else {
		my ( @errors, @messages );
		( my $serial = $cgi->param('serial_num') ) =~ /^\d+\z/ or die;
		$serial = normalize_sn($serial);
		my ( $_jj, $_mm, $_tt ) = _jjmmtt;
		for ( $package->init_ticket($serial), {} ) {
			my %update;
			next if keys %$_ && $_->{jjmm} < "$_jj$_mm";
			my $id = keys %$_ ? "[$_->{hash}-$_->{seq}]" : '[neu]';
			my ( @e, @m );

			unless ( defined( my $txt = $cgi->param("txt$id") ) ) {
				push @e, qq(Parameter "txt" nicht gefunden);
			}
			else {
				$txt =~ s/[\cM\cJ]+/\cJ/g;
				if ( length $txt > MAXLENGTH4STRINGS ) {
					push @e,
					  "Der eingegebene Text ist um "
					  . ( length $txt - MAXLENGTH4STRINGS )
					  . ' Zeichen zu lang.';
				}
				elsif ( $txt =~ y/|// ) {
					push @e,
'Pipe-Zeichen ("|") sind in Rechnungstexten leider nicht möglich.';
				}
				elsif ( $txt !~ /\S/ || $txt =~ /^RT#\d+(?: \(.*\))?:\s*\z/ ) {
					next unless keys %$_;

					DoTrans {
						Do($_) for <<1, <<2, <<3 };
	DELETE FROM acctassoc WHERE `hash` = $_->{hash} AND seq = $_->{seq}
1
	DELETE FROM rt_billing WHERE `hash` = $_->{hash} AND seq = $_->{seq}
2
	DELETE FROM acct WHERE `hash` = $_->{hash} AND seq = $_->{seq}
3
					push @messages, qq(Accounting-Datensatz $id gelöscht);
					next;
				}
				elsif ( !exists $_->{txt} || $txt ne $_->{txt} ) {
					$update{txt} = $txt;
					push @m, qq(Rechnungstext geändert) if exists $_->{txt};
				}
			}

			unless ( defined( my $dienst = $cgi->param("dienst$id") ) ) {
				push @e, qq(Parameter "dienst" nicht gefunden);
			}
			elsif ( !exists( $info4dienst{$dienst} ) ) {
				push @e, qq(Unbekannter Dienst "$dienst");
			}
			elsif ( $dienst ne $_->{dienst} ) {
                defined( $update{dienst} = $id4dienst{$dienst} )
                  or die qq(Unbekannter Dienst "$dienst".\n);
                push @m, qq(Dienst auf "$info4dienst{$dienst}" geändert);
			}

			unless ( defined( my $hours = $cgi->param("hours$id") ) ) {
				push @e, qq(Parameter "hours" nicht gefunden);
			}
			elsif ( !defined( my $minutes = $cgi->param("minutes$id") ) ) {
				push @e, qq(Parameter "minutes" nicht gefunden);
			}
			elsif ( $hours !~ /^\d+\z/ ) {
				push @e, qq(ungültiger Stunden-Wert "$hours");
			}
			elsif ( $minutes !~ /^(00|15|30|45)\z/ ) {
				push @e, qq(ungültiger Minuten-Wert "$minutes");
			}
			else {
				my $seconds = 3600 * $hours + 60 * $minutes;
				unless ( keys %$_ && $seconds == $_->{seconds} ) {
					$update{bytes} = $seconds;
					push @m,
					  sprintf 'zu verrechnende Arbeitszeit auf %d:%02d gesetzt',
					  $hours, $minutes;
				}
			}

			unless ( defined( my $jj = $cgi->param("jj$id") ) ) {
				push @e, qq(Parameter "jj" nicht gefunden);
			}
			elsif ( $jj !~ /^\d{4}\z/ ) {
				push @e, qq(ungültiger Jahres-Wert "$jj");
			}
			elsif ( $jj < $_jj ) {
				push @e,
				  qq(auf vergangene Jahre kann nichts mehr gebucht werden);
			}
			elsif ( !defined( my $mm = $cgi->param("mm$id") ) ) {
				push @e, qq(Parameter "mm" nicht gefunden);
			}
			elsif ( $mm !~ /^(?:0[0-9]|1[012])\z/ || $mm < 1 || $mm > 12 ) {
				push @e, qq(ungültiger Monats-Wert "$mm");
			}
			elsif ( $jj == $_jj && $mm < $_mm ) {
				push @e,
				  qq(auf vergangene Monate kann nichts mehr gebucht werden);
			}
			elsif ( !defined( my $tt = $cgi->param("tt$id") ) ) {
				push @e, qq(Parameter "tt" nicht gefunden");
			}
			elsif ( $tt !~ /^\d\d\z/ ) {
				push @e, qq(ungültiger TT-Wert "$tt");
			}
			elsif ( $tt > days_in $jj, $mm ) {
				push @e, qq(im Monat $jj-$mm gibt es keinen $tt.);
			}
			else {
				if ( !defined $_->{jjmm} || "$jj$mm" != $_->{jjmm} ) {
					$update{jjmm} = "$jj$mm";
					push @m, "Buchungsmonat auf $jj/$mm gesetzt";
				}
				if ( !defined $_->{tt} || $tt != $_->{tt} ) {
					$update{tt} = $tt;
					push @m, "Buchungstag auf $tt gesetzt";
				}
			}

			my ( $hash, $seq ) = ( $_->{hash}, $_->{seq} );
			if ( defined $hash and keys %$_ ) {
				Do(<<_) if defined $update{txt};
	UPDATE acctassoc SET
		info = ${\ qquote( delete $update{txt} ) }
	WHERE `hash` = $hash AND seq = $seq
_
				Do(     'UPDATE acct SET '
					  . join( ', ', map "$_=$update{$_}", keys %update )
					  . " WHERE `hash` = $hash AND seq = $seq" )
				  if keys %update;
				push @messages, "$id: " . join ', ', @m if @m;
				push @errors,   "$id: " . join ', ', @e if @e;
			}
			elsif (@e) {
				push @errors,
				  "Neuer Accounting-Datensatz konnte nicht angelegt werden: "
				  . join( ', ', @e );
			}
			elsif (
				!(
					( $hash, $seq ) = add_acct(
						$RT::req{$serial}{kunde_id}, $update{dienst},
						undef,                       $update{jjmm},
						$update{tt},                 0,
						$update{bytes},              $serial,
						$update{txt},                2
					)
				)
			  )
			{
				push @errors,
"Unbekannter Fehler beim Eintragen des Accounting-Datensatzes $id";
			}
			elsif ( !Do
"INSERT INTO rt_billing (ticket,`hash`,seq) VALUES ($serial,$hash,$seq)"
			  )
			{
				push @errors,
"Unbekannter Fehler beim Verlinken des Accounting-Datensatzes $id";
			}
			else {
				push @messages,
				  "Accounting-Datensatz [$hash-$seq] angelegt und dabei "
				  . join( ', ', @m );
			}
		}
		$package->add_transaction( $serial, undef, $_ ) for @messages;
		join( '; ', @errors, @messages ) || '(keine Änderungen)';
	}
}

1;
