#!/usr/bin/perl

use utf8;
use strict;
use warnings;

BEGIN {
	unshift @INC, ( $ENV{'POPHOME'} || '@POPHOME@' ) . '/lib'
		unless $ENV{'KUNDE_NO_PERLPATH'};
}

use Cf qw($BGCOLOR $RT_DOMAIN);
use CGI qw(:cgi);
use CGI::Carp qw(fatalsToBrowser);
use HTML::Entities qw(encode_entities);
use Carp qw(carp cluck croak confess);
use noris::MIME::Words qw(encode_mimewords);
use Text::Wrap qw(wrap);

use noris::Kunde qw(kundeid_get_all kundeid_get_unterkunden kundeid_get_by_flags);
use noris::Person qw(personid_get_by_kundeid);
use noris::TroubleTicket qw(tt_get_incomplete tt_get_descriptor_list);
use noris::ServiceWeb::Globals qw(print_time_select get_param_time);
use noris::CreateTicket ();
use noris::KundenListen ('%KundenListen');

use Dbase::Help qw(DoSelect in_list qquote);
use Dbase::Globals qw(find_descr);
use UTFkram qw(decode_anything);
use Umlaut qw(utf8modus);

utf8modus(\*STDOUT);
use HTTPerror qw(run);

use constant MINUTES => [ map sprintf('%02d', $_*5), 0..11 ];

our $MySelf;
{
	no warnings 'once';
	$MySelf = defined $ENV{SCRIPT_NAME} ? $ENV{SCRIPT_NAME} : $FindBin::Script;
}

our $CurrentTT;
our $KundenAdditional = {};
our $KundenExceptions = {};
our $ErrorCount = 0;

our $ActionFuncs =
{
	form    => [\&action_select_incomplete, \&action_print_form],
	load    => [\&action_tt_load, \&action_print_form],
	update  => [\&action_tt_read, \&action_print_form],
	preview => [\&action_tt_read, sub { if ($ErrorCount) { action_print_form (); } else { action_print_preview (); } }],
	view	=> [\&action_tt_load, \&action_print_preview],
	save    => [\&action_tt_read, sub { if ($ErrorCount) { action_print_form (); } else { action_save (); } }],
	confirm => [\&action_tt_load, \&action_confirm],
	remove  => [\&action_tt_load, \&action_delete]
};

our $ActionAlias =
{
	Aktualisieren		=> 'update',
	Editieren		=> 'update',
	Vorschau		=> 'preview',
	Speichern		=> 'save',
	Ansehen			=> 'view',
	'Bestätigen und Versenden'	=> 'confirm',
	'Löschen'		=> 'remove'
};

our $Action;

run {
$Action = param ('action');
$Action = 'form' if (!$Action);

# Work around an ugly charset problem, see RT#444067.  -octo
if ($Action =~ m/^Best..?.?tigen und Versenden$/)
{
  $Action = 'confirm';
}
elsif ($Action =~ m/^L..?.?schen$/)
{
  $Action = 'remove';
}

while (defined ($ActionAlias->{$Action}))
{
	$Action = $ActionAlias->{$Action};
}

$Action = 'form' if (!defined ($ActionFuncs->{$Action}));

delete ($ENV{'USER'}) if (exists $ENV{'USER'});
$ENV{'LOGNAME'} = $ENV{'REMOTE_USER'} if (defined ($ENV{'REMOTE_USER'}));

print <<HTTP;
Content-Type: text/html; charset="utf-8";
Cache-Control: no-cache

HTTP

print <<HTML;
<html>
  <head>
    <title>Problemkarten-Generierungseinheit</title>
    <style type="text/css">
    th { vertical-align: top; text-align: left; }
    fieldset { position: relative; background-color: white; }
    form > fieldset { background-color: rgb(240,240,240); }
    fieldset.textfield > textarea { background-color: rgb(246,246,246); }
    fieldset.textfield > textarea:focus { background-color: white; }
    .error { color: red; background-color: yellow; border: 3px groove red; padding: 1ex; }
    option.lev0 { background-color:#cfc;  }
    option.lev1 { background-color:#fff; padding-left:1em; }
    </style>
  </head>

  <body bgcolor="$BGCOLOR">
    <h1>noris network AG Trouble-Ticket-Generator</h1>
HTML

for (@{$ActionFuncs->{$Action}})
{
	$_->();
}

print <<HTML;
  </body>
</html>
HTML
};

exit (0);

sub action_select_incomplete
{
	my @incomplete = tt_get_incomplete ();

	return if (!@incomplete);

	print <<HTML;
    <form action="$MySelf" method="post">
      <fieldset>
        <legend>Unvollst&auml;ndige Trouble-Tickets</legend>
	<select name="id">
HTML
	for (@incomplete)
	{
		my $id = $_;
		my $tt = noris::TroubleTicket->load ($id);
		my $subject = $tt->get ('subject');
		my $ticket = $tt->get ('ticket');

		$ticket = $ticket ? "[Ticket #$ticket] " : '';

		print qq(\t  <option value="$id">$ticket) . encode_entities (decode_anything $subject) . "</option>\n";
	}

	print <<HTML;
	</select>
	<input type="submit" name="action" value="Ansehen" />
      </fieldset>
    </form>

HTML
}

sub action_tt_read
{
	my $select_kunden;
	my $kunden;

	$CurrentTT = noris::TroubleTicket->new ();

	if (param ('id'))
	{
		my $id = param ('id');
		$CurrentTT->error ();
		$CurrentTT->set (id => $id);
		if (my $error = $CurrentTT->error ())
		{
			$ErrorCount++;
			print qq(    <div class="error">$error</div>\n);
		}
	}

	for (qw(ticket subject type priority select_kunden description affected comment progress resolve))
	{
		$CurrentTT->set ($_ => decode_anything(param ($_)));
		if (my $error = $CurrentTT->error ())
		{
			$ErrorCount++;
			print qq(    <div class="error">$error</div>\n);
		}
	}

	for (kundeid_get_all ())
	{
		$KundenAdditional->{$_} = 0;
		$KundenExceptions->{$_} = 0;
	}

	$select_kunden = param ('select_kunden');

	if ($select_kunden eq 'flags')
	{
		$CurrentTT->set (select_kunden => $select_kunden);
		$CurrentTT->set (flag_kunde  => [param ('flag_kunde')]);
		$CurrentTT->set (flag_person => [param ('flag_person')]);
	}
	elsif ($select_kunden eq 'all')
	{
		$CurrentTT->set (select_kunden => $select_kunden);
		$CurrentTT->add_kunde ($_) for (kundeid_get_all ());
	}
	elsif ($select_kunden eq 'none')
	{
		$CurrentTT->set (select_kunden => $select_kunden);
		$CurrentTT->remove_kunde ($_) for (kundeid_get_all ())
	}
	else
	{
		my %aktive_kunden;

		die ("Invalid value for key `select_kunden'")
			if (!exists ($KundenListen{$select_kunden}));

		$aktive_kunden{$_} = 1 for ($KundenListen{$select_kunden}->());

		$CurrentTT->set (select_kunden => 'none');

		for (kundeid_get_all ())
		{
			my $id = $_;
			if (exists ($aktive_kunden{$id}))
			{
				$KundenAdditional->{$id} = 1;
				$CurrentTT->add_kunde ($id);
			}
		} # for (kundeid_get_all ())
	}

	$kunden = $CurrentTT->get_kunden ();

	if ($select_kunden ne 'all')
	{
		for (param ('kunden_additional'))
		{
			my $kundeid = $_;
			next if (defined ($kunden->{$kundeid}));

			$KundenAdditional->{$kundeid} = 1;
			$CurrentTT->add_kunde ($kundeid);
		}
	}

	if ($select_kunden ne 'none')
	{
		for (param ('kunden_exceptions'))
		{
			my $kundeid = $_;
			next if (!defined ($kunden->{$kundeid}));

			$KundenExceptions->{$kundeid} = 1;
			$CurrentTT->remove_kunde ($kundeid);
		}
	}

	if (param ('additional_rcpt'))
	{
		my $string = param ('additional_rcpt');
		my @list   = split (m/[\r\n]+/, $string);

		if ($CurrentTT->set ('additional_rcpt', \@list))
		{
			$ErrorCount++;
			print qq(    <div class="error">) . encode_entities (decode_anything $CurrentTT->error ()) . "</div>\n";
		}
	}

	if (param ('beginn_year'))
	{
		my $beginn = get_param_time ('beginn_', time ());
		my $ende   = get_param_time ('ende_',   time () - 10);

		$CurrentTT->set ('beginn', $beginn);
		$CurrentTT->set ('ende', $ende > $beginn ? $ende : undef);
	}

	if (param ('ende_unbekannt'))
	{
		$CurrentTT->set ('ende', undef);
	}

	if ($CurrentTT->error ())
	{
		$ErrorCount++;
	}

	# vgl. RT#306668
	if ($CurrentTT->get ('resolve') && !$CurrentTT->get ('ende'))
	{
		$ErrorCount++;
		print qq(    <div class="error">L&ouml;sungstext aber kein Endzeitpunkt angegeben.</div>\n);
	}
}

sub action_tt_load
{
	my $id = param ('id') or die;

	$CurrentTT = noris::TroubleTicket->load ($id);

	my $kunden = $CurrentTT->get_kunden ();

	for (kundeid_get_all ())
	{
		$KundenAdditional->{$_} = 0;
		$KundenExceptions->{$_} = 0;
	}

	if ($CurrentTT->get ('select_kunden') eq 'flags')
	{
		my $flag_kunde  = $CurrentTT->get ('flag_kunde');
		my $flag_person = $CurrentTT->get ('flag_person');

		for (kundeid_get_by_flags (kunde => $flag_kunde, person => $flag_person, invert => 1))
		{
			my $kundeid = $_;
			$KundenAdditional->{$kundeid} = 1 if (defined ($kunden->{$kundeid}));
		}

		for (kundeid_get_by_flags (kunde => $flag_kunde, person => $flag_person, invert => 0))
		{
			my $kundeid = $_;
			$KundenExceptions->{$kundeid} = 1 if (!defined ($kunden->{$kundeid}));
		}
	}
	elsif ($CurrentTT->get ('select_kunden') eq 'all')
	{
		for (keys %$KundenExceptions)
		{
			$KundenExceptions->{$_} = 1 if (!defined ($kunden->{$_}));
		}
	}
	elsif ($CurrentTT->get ('select_kunden') eq 'none')
	{
		for (keys %$KundenAdditional)
		{
			$KundenAdditional->{$_} = 1 if (defined ($kunden->{$_}));
		}
	}
}

{my $cache;
sub _format_kunde
{
	my ($kundeid, $kundename, $without_id ) = @_;
	my @personen = ();
	my $num;

	if (!$cache)
	{
		$cache = {};
		DoSelect {
			my($kundeid,$num) = @_;
			$cache->{$kundeid} = $num;
		} <<SQL;
	SELECT   kundemail.kunde, COUNT(*)
	FROM     kundemail, person
      WHERE  kundemail.dienst = ${\find_descr('dienst','trouble')}
	     AND kundemail.person = person.id
	     AND person.email IS NOT NULL
	GROUP BY kundemail.kunde
SQL
	}

	$num = $cache->{$kundeid} || 0;
	my $output = '';
	$output .= ( $without_id ? "#$kundeid:" : '' );
	$output .= encode_entities (decode_anything $kundename) . " ($num Empf&auml;nger)";
	return $output;
}}

sub action_print_form
{
	my $id       = 0;
	my $ticket   = '';
	my $subject  = '';
	my $type     = '';
	my $priority = '';
	my $flag_kunde = [];
	my $flag_person = [];

	my $beginn = time ();
	my $ende   = time () + 3600;

	my $this_year = (localtime ())[5] + 1900;

	my $kunden = {};

	my $select_kunden = param ('select_kunden') || 'flags';
	if ($select_kunden and $select_kunden =~ m/(all|none|flags)/) { $select_kunden = $1; }
	else { $select_kunden = "flags"; }

	if (defined ($CurrentTT))
	{
		$id          = $CurrentTT->get ('id');
		$ticket      = $CurrentTT->get ('ticket');
		$subject     = $CurrentTT->get ('subject');
		$type        = $CurrentTT->get ('type');
		$priority    = $CurrentTT->get ('priority');
		$flag_kunde  = $CurrentTT->get ('flag_kunde');
		$flag_person = $CurrentTT->get ('flag_person');

		$beginn      = $CurrentTT->get ('beginn');
		$ende        = $CurrentTT->get ('ende');

		$select_kunden = $CurrentTT->get ('select_kunden');
		$kunden = $CurrentTT->get_kunden ();

		#print "<!-- \$CurrentTT->get_kunden ():\n";
		#printf (" \%4i: \%s\n", $_, $kunden->{$_}) for (sort { $a <=> $b } (keys (%$kunden)));
		#print "-->\n";
	}
	else
	{
		$CurrentTT = noris::TroubleTicket->new ();
	}

	print <<HTML;
    <form action="$MySelf" method="post">
      <input type="hidden" name="id" value="$id" />
      <fieldset style="background-color: rgb(240,240,240);">
        <legend>Trouble-Ticket-Meta-Informationen</legend>
	<table>
	  <tr>
	    <th>Ticket:</th>
HTML
	if ($ticket)
	{
		print qq(\t    <td>#$ticket\n)
		. qq(\t      <input type="hidden" name="ticket" value="$ticket" /></td>\n);
	}
	else
	{
		print qq(\t    <td>benutze Ticket #<input type="text" name="ticket" value="$ticket" size="9" maxlength="9" /></td>\n);
	}

	print <<HTML;
	  </tr>
	  <tr>
	    <th>Betreff:</th>
	    <td><input type="text" name="subject" value="${\encode_entities (decode_anything $subject)}" size="72" /></td>
	  </tr>
	  <tr>
	    <th>St&ouml;rungsklasse:</th>
	    <td><select name="priority">
HTML
	{
		my $descriptoren = tt_get_descriptor_list ('priority');

		for (sort keys %$descriptoren)
		{
			my $id = $_;
			my $name = $descriptoren->{$id}{'name'};
			my $info = $descriptoren->{$id}{'info'};
			my $selected = $name eq $priority ? ' selected="selected"' : '';

			print qq(\t      <option value=")
			. encode_entities (decode_anything $name) . qq("$selected>)
			. encode_entities (decode_anything $info) . "</option>\n";
		}
	}
	print <<HTML;
	    </select></td>
	  </tr>
	  <tr>
	    <th>Ticket-Typ:</th>
	    <td><select name="type">
HTML
	{
		my $descriptoren = tt_get_descriptor_list ('type');

		for (sort keys %$descriptoren)
		{
			my $id = $_;
			my $name = $descriptoren->{$id}{'name'};
			my $info = $descriptoren->{$id}{'info'};
			my $selected = $name eq $type ? ' selected="selected"' : '';

			print qq(\t      <option value=")
			. encode_entities (decode_anything $name) . qq("$selected>)
			. encode_entities (decode_anything $info) . "</option>\n";
		}
	}
	print <<HTML;
	    </select></td>
	  </tr>
	  <tr>
	    <th>Beginn:</th>
	    <td>
HTML
		print_time_select ('beginn_', {values_minute => MINUTES,
				values_year => [$this_year - 2, $this_year - 1, $this_year, $this_year + 1],
				default => $beginn});
	print <<HTML;
	    </td>
	  </tr>
	  <tr>
	    <th>Ende:</th>
	    <td>
HTML
	{
		my $checked = $ende <= 0 ? ' checked="checked"' : '';
		print_time_select ('ende_', {values_minute => MINUTES,
				values_year => [$this_year - 2, $this_year - 1, $this_year, $this_year + 1],
				default => $ende});
		print qq(\t      <input type="checkbox" id="ende_unbekannt" name="ende_unbekannt" value="1"$checked />\n),
		      qq(\t      <label for="ende_unbekannt">Das Ende ist noch nicht bekannt</label>\n);
	}
	print <<HTML;
	    </td>
	  </tr>
	</table>
      </fieldset>

HTML

	{
		my @changes = $CurrentTT->get_history ();
		if (@changes)
		{
			print <<HTML;
      <fieldset>
        <legend>History</legend>
	<table>
	  <tr>
	    <th>Zeit</th>
	    <th>&Auml;nderungen</th>
	    <th>Username</th>
	  </tr>
HTML
			for (@changes)
			{
				my $time = localtime ($_->{'time'});
				my $num  = $_->{'num'};
				my $user = encode_entities (decode_anything $_->{'user'});

				print <<HTML;
	  <tr>
	    <td>$time</td>
	    <td style="text-align: right;">$num</td>
	    <td>$user</td>
	  </tr>
HTML
			}
			
			print <<HTML;
	</table>
      </fieldset>
HTML
		}
	}

	for (qw(description affected comment progress resolve))
	{
		my $field = $_;
		my $desc =
		{
			description => 'Beschreibung',
			affected    => 'Betroffen',
			comment     => 'Anmerkungen',
			progress    => 'Fortschritt',
			resolve     => 'L&ouml;sung'
		}->{$field};
		$CurrentTT->error (); # clear errors
		my $text = $CurrentTT->get ($field);

		if (my $error = $CurrentTT->error ())
		{
			print qq(      <div class="error">field = $field, error = $error</div>\n);
		}

		my $class = 'textfield empty';

		$text = '' if (!defined ($text));
		$text = encode_entities (decode_anything $text);

		$class = "textfield filled" if ($text);

		print <<HTML;
      <fieldset class="$class">
        <legend>$desc</legend>
	<textarea name="$field" cols="80" rows="5">$text</textarea>
      </fieldset>

HTML
	}

	print <<HTML;
      <fieldset>
        <legend>Kunden-Auswahl</legend>
HTML
	{
		my $i = 0;

		my $map =
		{
			'0all'   => 'Alle Kunden',
			'1none'  => 'nur ausgew&auml;hlte Kunden',
			'2flags' => 'Auswahl nach Flags'
		};

		for (sort (keys %KundenListen))
		{
			my $pi = encode_entities (decode_anything $_);
			$map->{'9' . $pi} = $pi;
		}

		for (sort (keys %$map))
		{
			my $type = $_;
			my $text = $map->{$type};
			my $checked;
			
			$type =~ s/^[0-9]//;
			$checked = $type eq $select_kunden ? ' checked="checked"' : '';

			print <<HTML;
	<div><input type="radio" name="select_kunden" value="$type"$checked id="select_kunden_$i" />
	  <label for="select_kunden_$i">$text</label></div>
HTML
			$i++;
		}
	}

	print <<HTML;
      </fieldset>

HTML
	if ($select_kunden eq 'flags')
	{
		print <<HTML;
      <fieldset>
        <legend>Kunden-Flags</legend>
	<select name="flag_kunde" multiple="multiple" size="10">
HTML
		{
			my $desc = tt_get_descriptor_list ('flag_kunde');
			for (sort { $desc->{$a}{'name'} cmp $desc->{$b}{'name'} } (keys %$desc))
			{
				my $id = $_;
				my $name = $desc->{$id}{'name'};
				my $info = $desc->{$id}{'info'};
				my $selected = '';
				$selected = ' selected="selected"' if (grep { $_ eq $name } (@$flag_kunde));

				print qq(\t  <option value=")
					. encode_entities (decode_anything $name) . qq("$selected>)
					. encode_entities (decode_anything "$name ($info)")  . "</option>\n";
			}
		}

		print <<HTML;
	</select>
      </fieldset>

      <fieldset>
        <legend>Personen-Flags</legend>
	<select name="flag_person" multiple="multiple" size="10">
HTML
		{
			my $desc = tt_get_descriptor_list ('flag_person');
			for (sort { $desc->{$a}{'name'} cmp $desc->{$b}{'name'} } (keys %$desc))
			{
				my $id = $_;
				my $name = $desc->{$id}{'name'};
				my $info = $desc->{$id}{'info'};
				my $selected = '';
				$selected = ' selected="selected"' if (grep { $_ eq $name } (@$flag_person));

				print qq(\t  <option value=")
					. encode_entities (decode_anything $name) . qq("$selected>)
					. encode_entities (decode_anything "$name ($info)")  . "</option>\n";
			}
		}

		print <<HTML;
	</select>
      </fieldset>

HTML
	} # if ($select_kunden eq 'flags')
	print <<HTML;
      <fieldset>
        <legend>Ausnahmen</legend>
HTML
	if ($select_kunden ne 'all')
	{
		print <<HTML;
	<fieldset style="float: left;">
          <legend>Zus&auml;tzliche Empf&auml;nger</legend>
	  <select name="kunden_additional" multiple="multiple" size="20">
HTML
		{
			
			my $all_kunden;

			if ($select_kunden eq 'flags')
			{
				$all_kunden = kundeid_get_by_flags(
					kunde  => $flag_kunde,
					person => $flag_person,
					invert => 1,
					hauptkunden => 1,
				);
			}
			elsif ($select_kunden eq 'none')
			{
				$all_kunden = kundeid_get_all (1);
			}

			print kunden_select_field($all_kunden, 0, 1);
		}
		print <<HTML;
	  </select>
        </fieldset>

HTML
	}

	if ($select_kunden ne 'none')
	{
		print <<HTML;
        <fieldset style="float: left;">
          <legend>Auszuschlie&szlig;ende Empf&auml;nger</legend>
	  <select name="kunden_exceptions" multiple="multiple" size="20">
HTML
		{
			my $all_kunden;

			if ($select_kunden eq 'flags')
			{
				$all_kunden = kundeid_get_by_flags(
					kunde       => $flag_kunde,
					person      => $flag_person,
					invert      => 0,
					hauptkunden => 0,
				);
			}
			elsif ($select_kunden eq 'all')
			{
				$all_kunden = kundeid_get_all (0);
			}

			print kunden_select_field($all_kunden, 0, 0);
		}
		print <<HTML;
	  </select>
        </fieldset>

HTML
	}

	print <<HTML;
	<fieldset style="float: left;">
	  <legend>Zus&auml;tzliche Empf&auml;nger (E-Mail eingeben)</legend>
HTML
	{
		my $additional = join ("\n", $CurrentTT->get ('additional_rcpt'));
		print qq(\t  <textarea name="additional_rcpt" rows="20" cols="30">)
		. encode_entities (decode_anything $additional) . "</textarea>\n";
	}
	print <<HTML;
	</fieldset>
      </fieldset>

      <fieldset>
        <legend>Aktionen</legend>
        <input type="submit" name="action" value="Aktualisieren" />
HTML
	print qq(\t<input type="submit" name="action" value="Vorschau" />\n) if ($Action ne 'form');
	print <<HTML;
      </fieldset>
    </form>
HTML
}

sub action_print_preview
{
	my $kunden = $CurrentTT->get_kunden ();	
	my $select_kunden = $CurrentTT->get ('select_kunden');

	my $subject  = $CurrentTT->get ('subject') || '';
	if ($CurrentTT->get ('ticket'))
	{
		$subject .= ' [noris #' . $CurrentTT->get ('ticket') . ']';
	}

	print <<HTML;
    <h2>${\encode_entities (decode_anything $subject)}</h2>

    <form action="$MySelf" method="post">
HTML
	for (qw(id ticket subject type priority description affected comment
				progress resolve select_kunden))
	{
		my $key = $_;
		my $val = $CurrentTT->get ($key);

		next if (!defined ($val));

		$val = encode_entities (decode_anything $val);

		print qq(      <input type="hidden" name="$key" value="$val" />\n);
	}

	for (qw(beginn ende))
	{
		my $key = $_;
		my $time = $CurrentTT->get ($key);

		if ($key eq 'ende' and !defined ($time) or !$time)
		{
			print qq(      <input type="hidden" name="ende_unbekannt" value="1" />\n);
			next;
		}
		
		my @time = (localtime ($time))[1..5];
		$time[3]++;
		$time[4] += 1900;

		my @fields = qw(minute hour day month year);

		for (my $i = 0; $i < @fields; $i++)
		{
			my $subkey = $fields[$i];
			my $val = encode_entities (decode_anything $time[$i]);
			print qq(      <input type="hidden" name="${key}_${subkey}" value="$val" />\n);
		}
	}

	if ($select_kunden eq 'flags')
	{
		for ($CurrentTT->get ('flag_kunde'))
		{
			my $flag = encode_entities (decode_anything $_);
			print qq(      <input type="hidden" name="flag_kunde" value="$flag" />\n);
		}
		for ($CurrentTT->get ('flag_person'))
		{
			my $flag = encode_entities (decode_anything $_);
			print qq(      <input type="hidden" name="flag_person" value="$flag" />\n);
		}
	}

	if ($select_kunden ne 'all')
	{
		for (keys %$KundenAdditional)
		{
			my $kundeid = $_;
			next if (!$KundenAdditional->{$kundeid});
			print qq(      <input type="hidden" name="kunden_additional" value=")
			. encode_entities (decode_anything $kundeid) . "\" />\n";
		}
	}

	if ($select_kunden ne 'none')
	{
		for (keys %$KundenExceptions)
		{
			my $kundeid = $_;
			next if (!$KundenExceptions->{$kundeid});
			print qq(      <input type="hidden" name="kunden_exceptions" value=")
			. encode_entities (decode_anything $kundeid) . "\" />\n";
		}
	}

	if ($CurrentTT->get ('additional_rcpt'))
	{
		my $email = join ("\n", $CurrentTT->get ('additional_rcpt'));
		$email = encode_entities (decode_anything $email);
		print qq(      <input type="hidden" name="additional_rcpt" value="$email" />\n);
	}

	print <<HTML;
      <fieldset>
        <legend>Trouble-Ticket-Mail</legend>
HTML

	$CurrentTT->error (); # clear errors
	my $mail = $CurrentTT->to_mail ();

	if (my $error = $CurrentTT->error ())
	{
		print qq(\t  <div class="error">$error</div>\n);
		action_print_form ();
		return;
	}

	$mail = encode_entities (decode_anything $mail);

	print qq(\t  <pre>$mail</pre>\n);

	my @buttons = ();
	push (@buttons, qq(<input type="submit" name="action" value="Editieren" />));
	if ($Action eq 'preview')
	{
		push (@buttons, qq(<input type="submit" name="action" value="Speichern" />))
	}
	if ($Action eq 'view')
	{
		push (@buttons, qq(<input type="submit" name="action" value="L&ouml;schen" />)) if ($CurrentTT->is_deletable ());
		push (@buttons, qq(<input type="submit" name="action" value="Best&auml;tigen und Versenden" />));
		# push (@buttons, qq(<input type="checkbox" id="send_email" name="send_email" value="1" checked="1" /> <label for="send_email">E-Mails verschicken</label>));
	}

	print <<HTML;
      </fieldset>

      <fieldset>
        <legend>Aktionen</legend>
HTML
	print "\t$_\n" for (@buttons);
	print <<HTML;
      </fieldset>

      <fieldset>
        <legend>Empf&auml;nger</legend>
	<ul>
HTML
	# my $kunden = $CurrentTT->get_kunden ();
	for (sort { $kunden->{$a} cmp $kunden->{$b} } (keys %$kunden))
	{
		my $kundeid = $_;
		my $kundename = $kunden->{$kundeid};

		print "\t  <li>" . _format_kunde ($kundeid, $kundename) . "</li>\n";
	}
	for ($CurrentTT->get ('additional_rcpt'))
	{
		my $rcpt = $_;
		print "\t  <li>" . encode_entities (decode_anything $rcpt) . "</li>\n";
	}

	print <<HTML;
	</ul>
      </fieldset>
    </form>

HTML
}

sub action_save
{
	$CurrentTT->error (); # clear errors..
	$CurrentTT->save ();
	if (my $error = $CurrentTT->error ())
	{
		print qq(    <div class="error">$error</div>\n);
		return;
	}

	my $ticket = $CurrentTT->get ('ticket');

	if ($ticket)
    {
        require noris::Ticket::API
          and noris::Ticket::API->import('get_pooled_connection')
          unless defined &get_pooled_connection;
        my ($ticket_url) =
          get_pooled_connection( $ENV{REMOTE_USER} )
          ->get_ticket( $ticket, [qw(ticket_url)] );
        my $ticket = "Ticket #$ticket";
        $ticket = '<a href="' . encode_entities($ticket_url) . qq(">$ticket</a>)
          if defined $ticket_url;
        print <<_;
    <div>Das Trouble-Ticket wurde gespeichert. Das zugeh&ouml;rige Ticket ist $ticket.</div>
_
    }
	else
	{
		print "    <div>Das Trouble-Ticket wurde gespeichert. Ein Ticket existiert noch nicht, vermutlich wurde das TT noch nicht best&auml;tigt.</div>\n";
	}

	print qq#    <div>Falls es noch mehr zu tun gibt, geht es hier <a href="$MySelf">zur Startseite des Trouble-Ticket-Generators</a>. <span style="font-size: smaller;">(Sam genie&szlig;t seinen Ruhestand.)</span></div>\n#;
}

sub action_confirm
{
	my $username = $ENV{'LOGNAME'};
	my $person = noris::Person->load (user => $username);

	my $mail = $CurrentTT->to_mail ();
	my $ticket = $CurrentTT->get ('ticket');

	# my $send_email = param ('send_email') || 0;

	if (!$person)
	{
		print <<HTML;
    <div class="error">
      Es konnte keine Person mit dem Usernamen &quot;${\encode_entities (decode_anything $username)}&quot;
      in der Datenbank gefunden werden.
    </div>
HTML
		return;
	}

	my $person_id = $person->get ('id');
	my $person_name = $person->get ('name');
	my $person_mail = $person->get ('email') || 'support@noris.net';

	if (!$ticket)
	{
		my $ticket = noris::CreateTicket->new( queue => 'incident', text => $mail, owner => $username );

		$ticket = $ticket->create ();
		$CurrentTT->set ('ticket', $ticket);
		if (my $error = $CurrentTT->error ())
		{
			print qq(    <div class="error">) . encode_entities (decode_anything $error) . " (noris::CreateTicket::create returned $ticket)</div>\n";
			return;
		}
		$CurrentTT->save ();
		print "    <div>Das Ticket #$ticket wurde generiert.</div>\n";
		$mail = $CurrentTT->to_mail ();
	}
	else
	{
		my $from = encode_mimewords (decode_anything "$person_name <$person_mail>");
		my $mail = $mail;
		$mail =~ s/^To: .*$/To: $ticket\@$RT_DOMAIN/mi;
		$mail =~ s/^From: .*$/From: $from/mi;

		my $fh = sendmail();
		print $fh $mail;
		close $fh or die "Fehler beim Mail-Versand ans RT: $!\n";
		print "    <div>Eine Mail wurde in das Ticket #$ticket geschickt.</div>\n";
		print "    <pre>" . encode_entities (decode_anything $mail) . "</pre>\n";
	}

	# confirm (and save) the TT.
	$CurrentTT->confirm ($person_id);
	if (my $error = $CurrentTT->error ())
	{
		print qq(    <div class="error">) . encode_entities (decode_anything $error) . "</div>\n";
		return;
	}

	# if ($send_email)
	{
		my ($kunde_header, $kunde_body) = split (m/\r?\n\r?\n/s, $mail, 2);
		my @kunde_id = $CurrentTT->get_kunden ();
		my @addresses = $CurrentTT->get ('additional_rcpt');

		DoSelect { push @addresses, @_ } <<_;
	SELECT DISTINCT person.email
	FROM   kundemail, person
	WHERE  kundemail.dienst = ${\find_descr('dienst','trouble')}
	   AND kundemail.person = person.id
	   AND ${\ in_list( 'kundemail.kunde', '', 1, @kunde_id ) }
	   AND person.email IS NOT NULL
_

		$Text::Wrap::columns = 72;
		my ($bcc) = wrap ('', "\t", "Bcc: " . join (', ', @addresses));

		my $kunde_mail = <<MAIL;
$kunde_header
$bcc

$kunde_body
MAIL
		my $fh = sendmail();
		print $fh $kunde_mail;
		close $fh or die "Fehler beim Mail-Versand an die Kunden: $!\n";

		print "    <div>Eine Mail wurde an die Kunden geschickt.</div>\n";
		print "    <pre>" . encode_entities (decode_anything $kunde_mail) . "</pre>\n";
	}	

	print "    <div>All done</div>\n";
}

sub action_delete
{
	if (!$CurrentTT->is_deletable ())
	{
		my $error = $CurrentTT->error () || 'Unknown error';
		print qq(    <div>Dieses Trouble-Ticket ist nicht l&ouml;schbar:</div>\n);
		print qq(    <div class="error">$error</div>\n);
	}
	else
	{
		my $id       = $CurrentTT->get ('id');
		my $ticket   = $CurrentTT->get ('ticket');
		my $subject  = $CurrentTT->get ('subject');

		$CurrentTT->delete ();
		if (my $error = $CurrentTT->error ())
		{
			print qq(    <div>Das L&ouml;schen dieses Trouble-Tickets ist fehlgeschlagen:</div>\n);
			print qq(    <div class="error">$error</div>\n);
		}
		else
		{
			print qq(    <div>Das Trouble-Ticket )
			. '&quot;' . encode_entities (decode_anything $subject) . '&quot;'
			. ($ticket ? " [RT #$ticket]" : '')
			. ", Trouble-Ticket-ID $id, wurde erfolgreich gel&ouml;scht.</div>\n";
		}
	}

	print <<HTML
    <div>Auf der <a href="$MySelf">Startseite des Trouble-Ticket-Generators</a> kann man weiterarbeiten, wenn man m&ouml;chte.</div>
HTML
}

sub sendmail() {
	open my $fh, '|-', '/usr/sbin/sendmail', '-f', 'support@noris.net', '-t'
	  or die "open(/usr/sbin/sendmail): $!\n";
	utf8modus($fh);
	$fh;
}

sub kunden_select_field {
  my ($all_kunden, $level, $mit_unterkunden) = @_;
 
  my $select_field = '';
  for (sort { $all_kunden->{$a} cmp $all_kunden->{$b} } (keys %$all_kunden)) {
		my $id = $_;
		my $name = $all_kunden->{$id};
		my $output = _format_kunde ($id, $name, 0);
		my $selected = $KundenAdditional->{$id} ? ' selected="selected"' : '';

		$select_field .= qq#\t  <option class="lev$level" value="$id"$selected>$output</option>\n#;
		if ($mit_unterkunden && ( my $unterkunden = kundeid_get_unterkunden($id))) {
			$level++;
			$select_field .= kunden_select_field($unterkunden, $level);
			$level--;
		}
  }
  return $select_field;
}

# vim:background=light:hlsearch:incsearch
