#!/usr/bin/perl

use utf8;
use strict;
use warnings;

BEGIN {
	unshift(@INC, ($ENV{'POPHOME'}||'@POPHOME@').'/lib')
		unless $ENV{'KUNDE_NO_PERLPATH'};
}

use Cf qw($BGCOLOR);
use CGI qw(:cgi);
use CGI::Carp qw(fatalsToBrowser);
use Data::Dumper;
use Carp qw(carp cluck croak confess);
use HTML::Entities ('encode_entities');
use HTTPerror qw(run);

use IO::String;

use Dbase::Help qw(DoSelect DoTime);
use Dbase::Globals;
use Dbase::Globals qw(content);

use Loader qw(list_hardware lister);

use HousingDB::RZ;
use HousingDB::Rack;
use HousingDB::Hardware;

use Umlaut; binmode(STDOUT,":raw :encoding(utf8ize)");

our $MySelf = $ENV{'SCRIPT_NAME'};

our %Actions =
(
	_default_	=> [\&form_select_rz, \&form_edit_rz],
	edit_rz		=> \&form_edit_rz,
	save_rz		=> \&action_save_rz,
	del_rz		=> \&action_del_rz,
	show_rz		=> \&action_show_rz,
	add_rack	=> \&form_edit_rack,
	del_rack	=> \&action_del_rack,
	edit_rack	=> \&form_edit_rack,
	save_rack	=> \&action_save_rack,
	show_rack	=> \&action_show_rack,
	add_hw		=> \&form_edit_hardware,
	edit_hw		=> \&form_edit_hardware,
	save_hw		=> \&action_save_hardware,
	search_form	=> \&form_search,
	search		=> [\&form_search, \&action_search]
);

run {
my $act = param ('action');
$act = '_default_' unless (defined ($act) and defined ($Actions{$act}));

print <<EOF;
Content-Type: text/html; charset="utf-8";
Cache-Control: no-cache

<html>
	<head>
		<title>Housing Datenbank der noris network AG</title>
		<style type="text/css">
		<!--
			address { color: black; background: white url(http://as12337.net/octopus/envelope.png) no-repeat center center; padding: 1ex; margin: 1ex; text-align: center; }
			
			div.error { color: red; background-color: yellow; border: 2px dashed red; padding: 1em; font-weight: bold; }
			div.error strong { text-decoration: underline; }

			table { border-collapse: collapse; }
			td, th { border: 1px solid black; }
			form td, form th { border: none; vertical-align: top; text-align: left; }
			form td { padding-right: 2em; }

			td.empty { color: black; background-color: #DDFFDD; }
			td.noris { color: black; background-color: #DDDDFF; }
			td.customer { color: black; background-color: FFEECC; }

			table.rack, table.rz { margin-left: auto; margin-right: auto; }
			table.rack td { font-size: x-small; padding: 0px; vertical-align: center; }
			table.rack td a { display: block; width: 100%; height: 100%; }
			table.rack td.empty a { color: silver; background-color: transparent; text-decoration: none; }
			table.rack td.empty a:hover { color: black; background-color: transparent; text-decoration: underline; }
			table.rack td.he a { color: inherit; text-decoration: none; }
			table.rz td.rack { width: 5ex; height: 5ex; padding: 0px; }
			table.rz td.rack.none a { text-decoration: none; }
			table.rz td.rack a { display: block; width: 100%; height: 100%; margin: 0px; text-align: center; }

			table.result a { color: navy; background-color: transparent; text-decoration: none; display: block; }

			ul.topmenu { color: black; background-color: rgb(240,240,240); margin: 1ex; padding: 0.5ex; }
			ul.topmenu li { display: inline; color: black; background-color: rgb(240,240,240); padding: 0px; margin: 0px; border: 1px solid gray; }
			ul.topmenu li:hover { color: black; background-color: rgb(220,220,220); border-color: black; }
			ul.topmenu li a { padding: 0.5ex; color: inherit; background-color: transparent; text-decoration: none; }
		-->
		</style>
	</head>

	<body bgcolor="$BGCOLOR">
		<h1>Housing-Datenbank</h1>
EOF

if (ref ($Actions{$act}) eq 'CODE')
{
	$Actions{$act}->();
}
elsif (ref ($Actions{$act}) eq 'ARRAY')
{
	$_->() for (@{$Actions{$act}});
}

print <<EOF;
	</body>
</html>
EOF
};

exit (0);

#
# RZ Stuff
#

sub form_edit_rz
{
	my $rz_id = param ('rz') || 0;
	my $rz_name = '';
	my $rz_standort = '';

	if ($rz_id)
	{
		my $rz = HousingDB::RZ->retrieve ($rz_id);
		my $dummy;

		($dummy, $rz_standort) = $rz->standort ();
		print STDERR "form_edit_rz: dummy = $dummy; rz_standort = $rz_standort;\n";

		$rz_name = html_escape ($rz->name ());
		$rz_standort = html_escape ($rz_standort);
	}

	print qq(\t\t<h2>RZ editieren</h2>\n) if ($rz_id);
	print qq(\t\t<h2>RZ hinzuf&uuml;gen</h2>\n) if (!$rz_id);
	print qq(\t\t<form action="$MySelf" method="post">\n);
	print qq(\t\t<input type="hidden" name="rz" value="$rz_id" />\n) if ($rz_id);

	print <<EOF;
			<input type="hidden" name="action" value="save_rz" />
			<table>
				<tr>
					<td>Name:</td>
					<td><input type="text" name="rz_name" value="$rz_name" /></td>
					<td></td>
				</tr>
				<tr>
					<td>Standort:</td>
					<td><input type="text" name="standort" value="$rz_standort" /></td>
					<td>(Person, der dieses RZ zugeordnet wird)</td>
				</tr>
				<tr>
					<td colspan="2" style="text-align: right;"><input type="submit" name="button" value="Ok" /></td>
					<td></td>
				</tr>
			</table>
		</form>
EOF
}

sub form_select_rz
{
	print <<EOF;
		<ul class="topmenu">
			<li><a href="$MySelf?action=search_form" title="Die Housing-Datenbank durchsuchen">Suchen</a></li>
		</ul>
		<h2>RZ ausw&auml;hlen</h2>
		<ul class="topmenu">
EOF

	for (sort { $a->name () cmp $b->name () } (HousingDB::RZ->retrieve_all ()))
	{
		my $rz = $_;
		my $id = $rz->id ();
		my $name = html_escape ($rz->name ());

		print qq(\t\t\t\t<li><a href="$MySelf?action=show_rz&rz=$id">$name</a></li>\n);
	}

	print <<EOF;
		</ul>
EOF
}

sub action_save_rz
{
	my $rz;
	my $rz_id = param ('rz') || 0;
	my $rz_name = param ('rz_name') or confess;
	my $rz_standort  = param ('standort')  or confess;

	if ($rz_id)
	{
		$rz = HousingDB::RZ->retrieve ($rz_id);
		$rz->name ($rz_name);
		$rz->standort ($rz_standort);
	}
	else
	{
		$rz = HousingDB::RZ->create ($rz_name, $rz_standort);
	}

	if ($rz)
	{
		param ('rz', $rz->id ());
		action_show_rz ();
	}
	else
	{
		$rz_standort = html_escape ($rz_standort);
		print STDOUT <<ERROR;
		<div class="error">
			Das RZ konnte <strong>nicht</strong> angelegt werden.
			Vermutlich konnte der angegebene Standort
			&quot;$rz_standort&quot; nicht in der Datenbank gefunden
			werden.
		</div>
ERROR
		form_select_rz ();
		form_edit_rz ();
	}
}

sub action_del_rz
{
	my $rz_id = param ('rz') or confess;
	my @racks = HousingDB::Rack->search (rz => $rz_id);
	my $rz = HousingDB::RZ->retrieve ($rz_id);
	my $name = html_escape ($rz->name ());

	if (@racks)
	{
		print <<EOF;
		<div class="error">
			Im RZ &quot;$name&quot; stehen noch Racks. Bitte l&ouml;schen Sie die zuerst.
		</div>
EOF
		action_show_rz ($rz_id);
	}
	else
	{
		$rz->extinguish ();
		form_select_rz ();
		form_edit_rz ();
	}
}

sub action_show_rz
{
	my $rz_id = param ('rz');
	my $rz = HousingDB::RZ->retrieve ($rz_id);
	my $name = html_escape ($rz->name ());
	my @racks = HousingDB::Rack->search (rz => $rz_id);

	$rz_id = shift if (@_);

	print <<EOF;
		<ul class="topmenu">
			<li><a href="$MySelf?action=search_form&rz=$rz_id" title="Die Housing-Datenbank durchsuchen">Suchen</a></li>
			<li><a href="$MySelf?action=edit_rz&rz=$rz_id" title="Das RZ editieren">Editieren</a></li>
EOF
	print qq(\t\t\t<li><a href="$MySelf?action=del_rz&rz=$rz_id" title="Das RZ l&ouml;schen">L&ouml;schen</a></li>\n) if (!@racks);
	print <<EOF;
			<li><a href="$MySelf?action=sel_rz" title="Zur&uuml;ck zur RZ Auswahl">Zur&uuml;ck</a></li>
		</ul>
		<h2>RZ-&Uuml;bersicht f&uuml;r $name</h2>
EOF
	print STDOUT "\t\t<address>"
	. join ('<br />', html_escape (map { split (m/\|/, $_) } ($rz->standort ())))
	. "</address>\n";

	_print_rz_map ($rz_id, { none => 'add_rack', empty => 'show_rack', noris => 'show_rack', customer => 'show_rack' });

	print <<LEGEND;
		<table style="margin-left: auto; margin-right: auto; margin-top: 1em;">
			<tr>
				<td class="empty" style="width: 33%; vertical-align: top;">
					<strong>leeres Rack</strong><br />
					Dieses Rack geh&ouml;rt noris, es ist
					aber keine Hardware eingebaut,
					zumindest keine Hardware mit
					Hardware-ID.
				</td>
				<td class="noris" style="width: 34%; vertical-align: top;">
					<strong>noris-Rack</strong><br />
					Dieses Rack geh&ouml;rt noris, und es
					ist Hardware eingebaut.
				</td>
				<td class="customer" style="width: 33%; vertical-align: top;">
					<strong>Kunden-Rack</strong><br />
					Dieses Rack ist an einen Kunden vermietet.
				</td>
			</tr>
		</table>
LEGEND
}

# 
# Rack stuff
#
sub action_show_rack
{
	my $rack_id = param ('rack');
	$rack_id = shift if (@_);
	my %opts = @_;
	
	my $rack = HousingDB::Rack->retrieve ($rack_id);
	confess ("Unable to load rack $rack_id from database") unless (defined ($rack) and ref ($rack));
	my $rack_name = html_escape ($rack->name ());
	my $rack_info = html_escape ($rack->info ());
	my $rz_id = $rack->rz ();
	my ($x, $y) = $rack->position ();
	my @hw = HousingDB::Hardware->search (rack => $rack_id);
	
	print <<EOF;
		<ul class="topmenu">
			<li><a href="$MySelf?action=search_form&rz=$rz_id" title="Die Housing-Datenbank durchsuchen">suchen</a></li>
			<li><a href="$MySelf?action=edit_rack&rack=$rack_id" title="Das Rack editieren">editieren</a></li>
EOF
	print qq(\t\t\t<li><a href="$MySelf?action=del_rack&rack=$rack_id" title="Das Rack l&ouml;schen">l&ouml;schen</a></li>\n) if (!@hw);
	print <<EOF;
			<li><a href="$MySelf?action=show_rz&rz=$rz_id" title="Zur&uuml;ck zur RZ-&Uuml;bersicht">zur&uuml;ck</a></li>
		</ul>
EOF

	print "\t\t<h2>Rack-Ansicht &quot;$rack_name&quot;</h2>\n";
	if ($rack->kunde != 1)
	{
		print <<HTML;
		<div>Dieses Rack geh&ouml;rt dem Kunden #${\( $rack->kunde )}:${\html_escape(name_kunde($rack->kunde))}.</div>
HTML
	}
	print "\t\t<div>Info: $rack_info</div>\n" if ($rack_info);

	_print_rack ($rack_id, { empty => 'add_hw', noris => 'edit_hw', customer => 'edit_hw' });
}

sub form_edit_rack
{
	my %opts = @_;

	my $rack_id = param ('rack');

	my $rack_rz    = param ('rz');
	my $rack_name  = param ('name');
	my $rack_info  = param ('info');
	my $rack_he    = param ('he');
	my $rack_kunde = param ('kunde');
	my $x = param ('x');
	my $y = param ('y');

	$rack_id = $opts{'id'} if (defined ($opts{'id'}));
	$rack_id ||= 0;

	if ($rack_id)
	{
		my $rack =  HousingDB::Rack->retrieve ($rack_id);
		$rack_rz    = $rack->rz ();
		$rack_name  = $rack->name ();
		$rack_info  = $rack->info ();
		$rack_he    = $rack->he ();
		$rack_kunde = $rack->kunde ();
		($x, $y)    = $rack->position ();
	}

	$rack_rz    = $opts{'rz'}    if (defined ($opts{'rz'}));
	$rack_name  = $opts{'name'}  if (defined ($opts{'name'}));
	$rack_info  = $opts{'info'}  if (defined ($opts{'info'}));
	$rack_he    = $opts{'he'}    if (defined ($opts{'he'}));
	$rack_kunde = $opts{'kunde'} if (defined ($opts{'kunde'}));
	$x = $opts{'x'} if (defined ($opts{'x'}));
	$y = $opts{'y'} if (defined ($opts{'y'}));

	my @rzs = HousingDB::RZ->retrieve_all ();
	
	$rack_rz    ||= 0;
	$rack_name  ||= '';
	$rack_info  ||= '';
	$rack_he    ||= 46;
	$rack_kunde ||= 1;

	my $kunden = _get_all_customers ();

	if ($rack_id)
	{
		print "\t\t<h2>Rack bearbeiten</h2>\n";
	}
	else
	{
		print "\t\t<h2>Rack hinzuf&uuml;gen</h2>\n";
	}

	$x = html_escape ($x);
	$y = html_escape ($y);

	$rack_name = html_escape ($rack_name);
	$rack_info = html_escape ($rack_info);
	$rack_he   = html_escape ($rack_he);

	print <<EOF;
		<form action="$MySelf" method="get">
			<input type="hidden" name="action" value="save_rack" />
			<input type="hidden" name="rack" value="$rack_id" />
			<input type="hidden" name="x" value="$x" />
			<input type="hidden" name="y" value="$y" />
			<table>
				<tr>
					<td>Name:</td>
					<td><input type="text" name="name" value="$rack_name" /></td>
				</tr>
				<tr>
					<td>Info:</td>
					<td><input type="text" name="info" value="$rack_info" /></td>
				</tr>
				<tr>
					<td>HE:</td>
					<td><input type="text" name="he" value="$rack_he" /></td>
				</tr>
				<tr>
					<td>Kunde:</td>
					<td><select name="kunde">
EOF
	for (sort { lc ($a) cmp lc ($b) } (keys %$kunden))
	{
		my $name = $_;
		my $kid = $kunden->{$name};
		my $selected = '';

		$name = html_escape ($name);
		$selected = ' selected="selected"' if ($rack_kunde == $kid);
		
		print qq(\t\t\t\t\t\t<option value="$kid"$selected>$name</option>\n);
	}
	print <<EOF;
					</select></td>
				</tr>
				<tr>
					<td>RZ</td>
					<td><select name="rz">
EOF
	for (@rzs)
	{
		my $rz = $_;
		my $id = $rz->id ();
		my $name = html_escape ($rz->name ());
		my $selected = ($id == $rack_rz ? ' selected="selected"' : '');

		print qq(\t\t\t\t\t\t<option value="$id"$selected>$name</option>\n);
	}
	print <<EOF;
					</select></td>
				</tr>
				<tr>
					</select></td>
					<td colspan="2"><input type="submit" name="button" value="Ok" /></td>
				</tr>
			</table>
		</form>
EOF
}

sub action_save_rack
{
	my $rack_id = param ('rack');

	my $rack_rz    = param ('rz');
	my $rack_name  = param ('name');
	my $rack_info  = param ('info');
	my $rack_he    = param ('he');
	my $rack_kunde = param ('kunde');
	my $x = param ('x');
	my $y = param ('y');

	$rack_id ||= 0;

	if ($rack_id)
	{
		my $rack =  HousingDB::Rack->retrieve ($rack_id);
		$rack->rz    ($rack_rz)    if ($rack_rz);
		$rack->name  ($rack_name)  if ($rack_name);
		$rack->info  ($rack_info)  if (defined ($rack_info));
		$rack->he    ($rack_he)    if ($rack_he);
		$rack->kunde ($rack_kunde) if ($rack_kunde);
		$rack->position ($x, $y)   if ($x and $y);
	}
	else
	{
		my $rack = HousingDB::Rack->create (rz => $rack_rz,
			name => $rack_name, info => $rack_info,
			he => $rack_he, kunde => $rack_kunde,
			x => $x, y => $y);

		if (!$rack)
		{
			print qq(\t<div class="error">Das Rack konnte nicht angelegt werden. Bitte &uuml;berpr&uuml;fen Sie ihre Eingabe.</div>\n);
			form_edit_rack ();
			return ();
		}

		$rack_id = $rack->id ();
	}

	action_show_rack ($rack_id);
}

sub action_del_rack
{
	my $rack_id = param ('rack') or confess;
	my $rack = HousingDB::Rack->retrieve ($rack_id);
	my $name = html_escape ($rack->name ());
	my @hw = HousingDB::Hardware->search (rack => $rack_id);
	param ('rz', $rack->rz ());

	if (@hw)
	{
		print <<EOF;
		<div class="error">
			Im Rack ist noch Hardware eingebaut. Bitte entfernen Sie die Hardware zuerst.
		</div>
EOF
		action_show_rack ($rack_id);
	}
	else
	{
		$rack->extinguish ();
		action_show_rz ();
	}
}

# 
# Hardware stuff
#
sub form_edit_hardware
{
	my %opts = @_;
	my @hw_list;

	my $hw;
	my $hw_id;

	my $hw_rack;
	my $hw_uhe;
	my $hw_he;
	my $hw_name;
	my $hw_hardware_id = '';
	my $hw_eigentuemer = '';
	my @hw_eigentuemer_info = ();
	my $hw_status = '';

	# Opera doesn't handle <optgroup> correctly, so disable them for Opera.
	# RT#234777-6
	my $browser_opera = $ENV{'HTTP_USER_AGENT'} =~ m/opera/i ? 1 : 0;

	$hw_id = param ('hw');
	$hw_id = $opts{'hw'} if (defined ($opts{'hw'}));
	$hw_id ||= 0;

	if ($hw_id)
	{
		$hw = HousingDB::Hardware->retrieve ($hw_id);
		$hw_rack = $hw->rack ();
		($hw_he, $hw_uhe) = $hw->he ();
		$hw_name = $hw->name ();
		$hw_hardware_id = $hw->hardware_id ();
	}

	$hw_rack = param ('rack') if (param ('rack'));
	$hw_he   = param ('he')   if (param ('he'));
	$hw_uhe  = param ('unterste_he') if (param ('unterste_he'));
	$hw_name = param ('name') if (param ('name'));

	$hw_rack = $opts{'rack'} if (defined ($opts{'rack'}));
	$hw_he   = $opts{'he'}   if (defined ($opts{'he'}));
	$hw_uhe  = $opts{'unterste_he'} if (defined ($opts{'unterste_he'}));
	$hw_name = $opts{'name'} if (defined ($opts{'name'}));

	$hw_rack ||= '0';
	$hw_uhe  = html_escape ($hw_uhe || '1');
	$hw_he   = html_escape ($hw_he  || '1');
	$hw_name = html_escape ($hw_name || 'Name der Hardware');

	print <<EOF;
		<ul class="topmenu">
			<li><a href="$MySelf?action=search_form" title="Die Housing-Datenbank durchsuchen">suchen</a></li>
EOF
	if ($hw_id)
	{
		my $args_ausbauen  = html_escape ("action=save_hw&hw=$hw_id&rack=0&he=$hw_he&unterste_he=$hw_uhe");
		my $args_beenden   = html_escape ("action=save_hw&hw=$hw_id&rack=-1&he=$hw_he&unterste_he=$hw_uhe");
		if ($hw_hardware_id)
		{
			print <<EOF;
			<li><a href="$MySelf?$args_ausbauen" title="Die Hardware nur ausbauen">nur ausbauen</a></li>
			<li><a href="$MySelf?$args_beenden" title="Die Hardware beenden">beenden</a></li>
EOF
		}
		else
		{
			print <<EOF;
			<li><a href="$MySelf?$args_ausbauen" title="Die Hardware l&ouml;schen">L&ouml;schen</a></li>
EOF
		}
	}
	print <<EOF;
			<li><a href="$MySelf?action=show_rack&amp;rack=$hw_rack" title="Zur&uuml;ck zur Rack-&Uuml;bersicht">Zur&uuml;ck</a></li>
		</ul>

EOF

	print "\t\t<h2>Hardware bearbeiten</h2>\n" if ($hw_id);
	print "\t\t<h2>Hardware hinzuf&uuml;gen</h2>\n" if (!$hw_id);

	print <<EOF;
		<form action="$MySelf" method="post">
			<input type="hidden" name="action" value="save_hw" />
EOF
	print qq(\t\t\t<input type="hidden" name="hw" value="$hw_id" />\n) if ($hw_id);
	print <<EOF;
			<table>
EOF

	if ($hw_hardware_id)
	{
		print <<EOF;
				<tr>
					<td>Hardware-ID</td>
					<td>$hw_hardware_id</td>
				</tr>
EOF
	}

	if (defined ($hw))
	{
		my $status = $hw->status ();
		my $ip = $hw->ip ();
		my $name = $hw->realname ();
		my $eigentuemer = $hw->eigentuemer ();
		my $kundeid = $hw->kunde ();
		my $kundename = name_kunde ($kundeid);

		print <<EOF if ($status);
				<tr>
					<td>Status:</td>
					<td>$status</td>
				</tr>
EOF
		print <<EOF if ($kundeid);
				<tr>
					<td>Kunde:</td>
					<td>#$kundeid: $kundename</td>
				</tr>
EOF
		print <<EOF if ($eigentuemer);
				<tr>
					<td>Eigent&uuml;mer:</td>
					<td>$eigentuemer</td>
				</tr>
EOF
		print <<EOF if ($name);
				<tr>
					<td>Ger&auml;tename/Typ:</td>
					<td>$name</td>
				</tr>
EOF
		print <<EOF if ($ip);
				<tr>
					<td>IP-Adresse:</td>
					<td>$ip</td>
				</tr>
EOF
	}

	if ($hw_id)
	{
		my $id = $hw->id ();
		my $name = html_escape ($hw->name ());
		my $hardware_id = html_escape ($hw->hardware_id ());
		my $rowspan = $hardware_id ? '1' : '2';

#		if ($hardware_id)
#		{
#			print <<EOF;
#					<td>
#						$name
#						<input type="hidden" name="hw" value="$id" />
#					</td>
#				</tr>
#EOF
#		}
#		else
		if (!$hardware_id)
		{
			print <<EOF;
				<tr>
					<td>Name:</td>
					<td>
						<input type="text" name="name" value="$name" />
						<input type="hidden" name="hw" value="$id" />
					</td>
				</tr>
EOF
		}
	}
	else
	{
		print <<EOH;
				<tr>
					<td rowspan="2">Name:</td>
					<td><select name="hw">
						<option value="0" style="color: black; background: rgb(230,230,255);">Hardware ohne Hardware-ID</option>
EOH
		for (HousingDB::Hardware->search (rack => undef, {order_by => 'hardware_id'}))
		{
			my $hw = $_;
			my $id = $hw->id ();
			my $name = html_escape ($hw->name ());
			my $hardware_id = html_escape ($hw->hardware_id ());

			next unless ($hardware_id); # not displayed, due to RT#220089

			print qq(\t\t\t\t\t\t<option value="$id">[$hardware_id] $name</option>\n);
		}
		print <<EOF;
					</select></td>
				</tr>
				<tr>
					<td>
						<input type="text" name="name" value="$hw_name" />
						(Nur f&uuml;r Hardware <strong>ohne</strong> Hardware-ID!)
					</td>
				</tr>
EOF
	}
	print <<EOF;
				<tr>
					<td>Rack:</td>
					<td><select name="rack">
EOF
	for (HousingDB::RZ->search ({order_by => 'name'}))
	{
		my $rz = $_;
		my $rz_id = $rz->id ();
		my $rz_name = html_escape ($rz->name ());

		print qq(\t\t\t\t\t\t<optgroup label="$rz_name">\n) if (!$browser_opera);

		for (HousingDB::Rack->search (rz => $rz_id, {order_by => 'name'}))
		{
			my $rack = $_;
			my $rack_id = $rack->id ();
			my $rack_name = html_escape ($rack->name ());
			my $selected = $rack_id eq $hw_rack ? ' selected="selected"' : '';

			print qq(\t\t\t\t\t\t\t<option value="$rack_id"$selected>$rz_name-$rack_name</option>\n);
		}
		print qq(\t\t\t\t\t\t</optgroup>\n) if (!$browser_opera);
	}

	print qq(\t\t\t\t\t\t<optgroup label="Ausbauen">\n) if (!$browser_opera);
	print qq(\t\t\t\t\t\t\t<option value="0">kein Rack</option>\n);
	print qq(\t\t\t\t\t\t</optgroup>\n) if (!$browser_opera);

	print <<EOF;
					</select></td>
				</tr>
				<tr>
					<td>HE:</td>
					<td><input type="text" name="he" value="$hw_he" /></td>
				</tr>
				<tr>
					<td>Unterste HE:</td>
					<td><input type="text" name="unterste_he" value="$hw_uhe" /></td>
				</tr>
				<tr>
					<td colspan="2" style="text-align: right; padding-top: 0.5ex;">
						<!-- <input type="submit" name="button" value="Abbrechen" /> -->
						<input type="submit" name="button" value="Ok" />
					</td>
				</tr>
			</table>
		</form>

EOF

	_print_hardware_children ($hw->children (), "\t\t") if ($hw);

	if ($hw_id)
	{
		# Dieser merkwuerdige Umweg ueber IO::String sorgt *irgendwie*
		# dafuer, dass die Umlaute die `list_hardware' ausspuckt auch
		# mit dem richtigen Zeichensatz ankommen. -octo

		no warnings 'once';

		print qq(\t\t<h2>Kunde-Info</h2>\n);
		print qq(\t\t<div style="white-space: pre; padding: 1ex; border: 2px inset silver; font-family: monospace;">);

		lister(5);
		# prints to $Db::pr_fh
		list_hardware ($hw_id);
		lister(0);
		print STDOUT html_escape ($Db::output);
		print "</div>\n";
	}
}

sub action_save_hardware
{
	my $hw_id = param ('hw');
	my $hw;
	my $rack;

	my $hw_rack = param ('rack');
	my $hw_he   = param ('he');
	my $hw_uhe  = param ('unterste_he');
	my $hw_name = param ('name');

	my $rack_he;

	if ($hw_id)
	{
		if (!($hw = HousingDB::Hardware->retrieve ($hw_id)))
		{
			print qq(<div class="error">Die ID $hw_id wurde in der <code>hardware</code>-Tabelle nicht gefunden.</div>\n);
			form_edit_hardware (id => 0);
			return;
		}
	}
	else
	{
		if (!$hw_name)
		{
			print qq(<div class="error">F&uuml;r Hardware ohne Hardware-ID muss ein Name angegeben werden.</div>\n);
			form_edit_hardware (id => 0, name => '');
			return;
		}

		if (!($hw = HousingDB::Hardware->create (name => $hw_name, kunde => 1)))
		{
			print qq(<div class="error">Die Hardware konnte nicht angelegt werden.</div>\n);
			form_edit_hardware (id => 0);
			return;
		}
	}

	$hw->name($hw_name) if content($hw_name) && !$hw->hardware_id;

	if (defined ($hw_rack) and ($hw_rack == 0))
	{
		my $old_rack = $hw->rack ();
		if (!$hw->hardware_id ())
		{
			$hw->ende( DoTime() );
		}
		else
		{
			# Setzt
			# - hardware.rack auf NULL,
			# - hardware.unterste_he auf NULL und
			# - hardware.standort auf den aktuellen Benutzer
			$hw->remove_from_rack ();
		}

		action_show_rack ($old_rack);
		return;
	}
	elsif (defined ($hw_rack) and ($hw_rack == -1))
	{
		my $old_rack = $hw->rack ();

		# Beende Hardware
		if (!$hw->hardware_id ())
		{
			die ("Hardware ohne Hardware-ID kann nicht beendet werden!");
		}
		$hw->ende( DoTime() );

		action_show_rack ($old_rack);
		return;
	}
	
	if (!($rack = HousingDB::Rack->retrieve ($hw_rack)))
	{
		print qq(<div class="error">Das angegebene Rack konnte nicht geladen werden.</div>\n);
		form_edit_hardware (rack => 0);
		return;
	}

	$rack_he = $rack->he ();

	if (!$hw_he or $hw_he < 1 or !$hw_uhe or $hw_uhe < 1 or ($hw_uhe + $hw_he - 1) > $rack_he)
	{
		print qq(<div class="error">Die angegebene H&ouml;he (in HE) bzw. Position ist ung&uuml;ltig.</div>\n);
		form_edit_hardware (he => 1, unterste_he => 1);
		return;
	}

	$hw->rack ($hw_rack);
	$hw->he ($hw_he, $hw_uhe);

	action_show_rack ($hw_rack);
}

sub form_search
{
	my @cust  = html_escape (param ('customer'));
	my @rz    = html_escape (param ('rz'));
	my $name  = html_escape (param ('name') || '');
	my $fqdn  = html_escape (param ('fqdn') || '');
	my $invnr = html_escape (param ('invnr') || '');
	my $hid   = html_escape (param ('hid') || '');
	my $customers = _get_all_customers ();
	my $show_all = param ('show_all') ? ' checked="checked"' : '';

	print <<EOF;
	<ul class="topmenu">
		<li><a href="$MySelf" title="Zur&uuml;ck zur Startseite">Zur&uuml;ck</a></li>
	</ul>
	<form action="$MySelf" method="post">
		<input type="hidden" name="action" value="search" />
		<fieldset>
			<legend>Such-Kriterien</legend>
			<table>
				<tr>
					<th>Name:</th>
					<td><input type="text" name="name" value="$name" /></td>
					<th rowspan="4">Kunde:</th>
					<td rowspan="4"><select name="customer" multiple="multiple" size="5">
EOF
	if (!@cust)
	{
		if (@rz or $name or $invnr or $hid)
		{
			@cust = ();
		}
		else
		{
			@cust = (1);
		}
	}

	for (sort { lc ($a) cmp lc ($b) } (keys %$customers))
	{
		my $name = $_;
		my $id = $customers->{$name};
		my $selected = (grep { $id == $_ } (@cust)) ? ' selected="selected"' : '';
		$name = html_escape ($name);

		print qq(\t\t\t\t\t\t<option value="$id"$selected>$name</option>\n);
	}

	print <<EOF;
					</select></td>
					<th rowspan="4">RZ:</th>
					<td rowspan="4"><select name="rz" multiple="multiple" size="5">
EOF
	for (sort { $a->name () cmp $b->name () } (HousingDB::RZ->retrieve_all ()))
	{
		my $rz = $_;
		my $name = $rz->name ();
		my $id = $rz->id ();
		my $selected = (grep { $id == $_ } (@rz)) ? ' selected="selected"' : '';

		print qq(\t\t\t\t\t\t<option value="$id"$selected>$name</option>\n);
	}

	print <<EOF;
					</select></td>
					<td><input type="checkbox" name="show_all" value="true"$show_all /> Nicht eingebaute Ger&auml;te anzeigen</td>
				</tr>
				<tr>
					<th><acronym title="fully qualified domain name">FQDN</acronym>:</th>
					<td><input type="text" name="fqdn" value="$fqdn" /></td>
					<td rowspan="3" style="vertical-align: bottom;"><input type="submit" name="button" value="Suchen" /></td>
				</tr>
				<tr>
					<th>Inventarnummer:</th>
					<td><input type="text" name="invnr" value="$invnr" /></td>
				</tr>
				<tr>
					<th>Hardware-ID:</th>
					<td><input type="text" name="hid" value="$hid" /></td>
				</tr>
			</table>
		</fieldset>
	</form>

EOF
}

sub action_search
{
	my @cust = param ('customer');
	my @rz   = param ('rz');
	my $name = param ('name') || '';
	my $fqdn = param ('fqdn') || '';
	my $invnr = param ('invnr') || '';
	my $hid = param ('hid') || '';
	my $show_all = param ('show_all') ? 1 : 0;

	my %search = ();
	
	$search{'kunde'} = \@cust if (@cust);
	$search{'rz'}    = \@rz   if (@rz);

	$search{'name'}  = "\%$name\%"    if ($name);
	$search{'fqdn'}  = "\%$fqdn\%"    if ($fqdn);
	$search{'ivnr'}  = "$invnr\%"     if ($invnr);
	$search{'hardware_id'} = "$hid\%" if ($hid);

	$search{'rack'}  = 0 if (!$show_all); # rack IS NOT NULL

	my @result = HousingDB::Hardware->search (%search);

	if (!@result)
	{
		print "\t<div>Leider wurden keine Ihren Kriterien entsprechenden Posten gefunden.</div>\n";
		return;
	}

	print <<EOF;
	<table class="result">
		<tr>
			<th>Hardware-ID</th>
			<th>Name</th>
			<th>Kunde</th>
			<th>RZ</th>
			<th>Rack</th>
		</tr>
EOF
	for (@result)
	{
		my $hw = $_;
		my $id      = $hw->id ();
		my $hid     = $hw->hardware_id ();
		my $name    = $hw->name ();
		my $kunde   = $hw->kunde ();

		my $rz_name   = '-';
		my $rack_name = '-';
		
		my $rack_id = $hw->rack ();
		if (!$rack_id)
		{
			$hid  = qq(<a href="$MySelf?action=add_hw&hw=$id">) . html_escape ($hid) . q(</a>);
			$name = qq(<a href="$MySelf?action=add_hw&hw=$id">) . html_escape ($name) . q(</a>);
		}
		else
		{
			my $rack    = HousingDB::Rack->retrieve ($rack_id);
			my $rz_id   = $rack->rz ();
			my $rz      = HousingDB::RZ->retrieve ($rz_id);

			$hid  = qq(<a href="$MySelf?action=edit_hw&hw=$id">) . html_escape ($hid) . q(</a>);
			$name = qq(<a href="$MySelf?action=edit_hw&hw=$id">) . html_escape ($name) . q(</a>);
			$rz_name   = qq(<a href="$MySelf?action=show_rz&rz=$rz_id">) . html_escape ($rz->name ()) . '</a>';
			$rack_name = qq(<a href="$MySelf?action=show_rack&rack=$rack_id">) . html_escape ($rack->name ()) . '</a>';
		}

		my $class   = $kunde == 1 ? 'noris' : 'customer';
		my $kunde_n = html_escape (name_kunde ($kunde));

		print <<EOF;
		<tr>
			<td class="$class">$hid</td>
			<td class="$class">$name</td>
			<td class="$class">$kunde_n</td>
			<td class="$class" style="text-align: right;">$rz_name</td>
			<td class="$class" style="text-align: right;">$rack_name</td>
		</tr>
EOF
	}
	print "\t</table>\n\n";
}

#
# INTERNAL
#

sub html_escape
{
	my @data = @_;

	for (@data)
	{
		$_ = encode_entities (defined ($_) ? $_ : '');
	}

	if (wantarray ())
	{
		return (@data);
	}
	else
	{
		return (join (', ', @data));
	}
}

=item B<_print_rz_map> (I<$rz_id>, I<$actions>)

Gibt eine Karte des RZ aus. Der Parameter I<$actions> muss so aufgebaut sein:

  $actions =
  {
    none     => 'add_rack',
    empty    => 'edit_rack',
    customer => 'edit_rack',
    noris    -> 'edit_rack'
  };

=cut

sub _print_rz_map
{
	my $rz_id = shift;
	my $actions = shift;
	
	my $rz = HousingDB::RZ->retrieve ($rz_id);
	my @racks = HousingDB::Rack->search (rz => $rz_id);
	my $pos = [];

	my $min_x = 0;
	my $max_x = 0;
	my $min_y = 0;
	my $max_y = 0;

	for (@racks)
	{
		my $rack = $_;
		my ($x, $y) = $rack->position ();

		$min_x = $x if ($min_x > $x);
		$max_x = $x if ($max_x < $x);
		$min_y = $y if ($min_y > $y);
		$max_y = $y if ($max_y < $y);
	}

	$min_x -= 2;
	$max_x += 3;
	$min_y -= 2;
	$max_y += 3;

	for (@racks)
	{
		my $rack = $_;
		my ($x, $y) = $rack->position ();

		$x -= $min_x;
		$y -= $min_y;

		$pos->[$y][$x] = [] unless (defined ($pos->[$y][$x]));
		push (@{$pos->[$y][$x]}, $rack);
	}

	print qq(\t\t<table class="rz">\n);
	for (my $y = $min_y; $y < $max_y; $y++)
	{
		print "\t\t\t<tr>\n";
		for (my $x = $min_x; $x < $max_x; $x++)
		{
			my $rel_x = $x - $min_x;
			my $rel_y = $y - $min_y;

			my $rack_id = 0;
			my $name;
			my $info;
			my $cust;

			my $type = 'none';
			
			if (defined ($pos->[$rel_y][$rel_x]))
			{
				for (@{$pos->[$rel_y][$rel_x]})
				{
					my $rack = $_;
					$rack_id = $rack->id ();
					$name = $rack->name ();
					$info = $rack->info ();
					$cust = $rack->kunde ();

					$cust ||= 1;
					$info ||= '';

					if ($cust != 1)
					{
						$type = 'customer';
						$info = ($info
							? "$info; " : '')
						. "Kunde #$cust:" . name_kunde($cust);
					}
					else
					{
						# ``hardware_id => 0'' == hardware_id IS NOT NULL
						my @hw = HousingDB::Hardware->search (rack => $rack_id, hardware_id => 0);
						my $num = 0;

						for (@hw)
						{
							my $h = $_;
							my $hwid = $h->hardware_id ();

							$num++ if ($hwid);
						}

						if ($num)
						{
							$type = 'noris';
						}
						else
						{
							$type = 'empty';
						}
					}
				}
			}

			print qq(\t\t\t\t<td class="rack $type">);
			if (defined ($actions->{$type}))
			{
				my $action = $actions->{$type};
				my $print = $name ? html_escape ($name) : '&nbsp;';
				my $title = $info ? qq( title=") . html_escape ($info) . '"' : '';

				print qq(<a href="$MySelf?action=$action&rz=$rz_id&rack=$rack_id&x=$x&y=$y"$title>$print</a>);
			}
			print "</td>\n";
		}
		print "\t\t\t</tr>\n";
	}
	print "\t\t</table>\n";
}

sub _print_rack
{
	my $rack_id = shift;
	my $actions = shift;
	
	my $rack = HousingDB::Rack->retrieve ($rack_id);
	my $rack_name = $rack->name ();
	my @hw = HousingDB::Hardware->search (rack => $rack_id, {order_by => 'unterste_he'});


	my @he = ();
	my $pos = [];
	my $width = 1;
	my $height = $rack->he ();

	for (@hw)
	{
		my $hw = $_;
		my ($he, $uhe) = $hw->he ();
		my $ohe = $uhe + $he - 1;

		my $done = 0;

		WIDTH: for (my $w = 0; $w < $width; $w++)
		{
			HEIGHT: for (my $h = $uhe; $h <= $ohe; $h++)
			{
				if (!defined ($pos->[$h]))
				{
					$pos->[$h] = [];
					next (HEIGHT);
				}

				if (defined ($pos->[$h][$w]))
				{
					next (WIDTH);
				}
			}

			# If we are here, we can use this width.
			for (my $h = $uhe; $h <= $ohe; $h++)
			{
				$pos->[$h][$w] = $hw;
			}
			$done = 1;
			last;
		}

		if (!$done)
		{
			for (my $h = $uhe; $h <= $ohe; $h++)
			{
				$pos->[$h][$width] = $hw;
			}
			$width++;
		}
	}

	print qq(\t\t<table class="rack">\n);
	for (my $h = $height; $h > 0; $h--)
	{
		print qq(\t\t\t<tr>\n);
		if (defined ($actions->{'empty'}))
		{
			my $act = $actions->{'empty'};
			print qq(\t\t\t\t<td class="he"><a href="$MySelf?action=$act&hw=0&rack=$rack_id&he=1&unterste_he=$h">$h</a></td>\n);
		}
		else
		{
			print qq(<\t\t\t\t<td class="he">$h</td>\n);
		}
		
		for (my $w = 0; $w < $width; $w++)
		{
			# The HE's are empty.
			if (!defined ($pos->[$h]) or scalar (@{$pos->[$h]}) == 0)
			{
				print qq(\t\t\t\t<td colspan="$width" class="empty">);
#				if (defined ($actions->{'empty'}))
#				{
#					my $act = $actions->{'empty'};
#					print qq(<a href="$MySelf?action=$act&hw=0&rack=$rack_id&he=1&unterste_he=$h">leer</a>)
#				}
#				else
#				{
#					print qq(leer);
#				}
				print qq(</td>\n);
				last;
			}

			if (!defined ($pos->[$h][$w]))
			{
				print qq(\t\t\t\t<td class="empty">);
#				if (defined ($actions->{'empty'}))
#				{
#					my $act = $actions->{'empty'};
#					print qq(<a href="$MySelf?action=$act&hw=0&rack=$rack_id&he=1&unterste_he=$h">leer</a>)
#				}
#				else
#				{
#					print qq(leer);
#				}
				print qq(</td>\n);
				next;
			}
			
			my $hw = $pos->[$h][$w];
			my ($he, $uhe) = $hw->he ();
			my $ohe = $uhe + $he - 1;
			my $name = html_escape ($hw->longname ());
			my $hw_id = $hw->id ();
			my $hardware_id = $hw->hardware_id ();
			my $cust = $hw->kunde ();
			my $many = $w;
			my $span = '';
			my $type = $cust == 1 ? 'noris' : 'customer';
			my $title = $hw->children ();

			if (ref ($title) eq 'HASH')
			{
				$title = 'Enth&auml;lt: ' . html_escape (keys %$title);
			}
			else
			{
				$title = '';
			}
			$title = html_escape ($hw->verylongname ()) . ($title ? "; " : "") . $title;
			$title = " title=\"$title\"";

			if (!$many)
			{
				for (my $i = $uhe; $i <= $ohe; $i++)
				{
					if (scalar (@{$pos->[$i]}) > 1)
					{
						$many++;
						last;
					}
				}
			}

			if ($h != $ohe)
			{
				next if ($many);
				last;
			}

			if ($many)
			{
				$span = qq(rowspan="$he");
			}
			else
			{
				$span = qq(colspan="$width" rowspan="$he");
			}

			print qq(\t\t\t\t<td $span class="$type">);
			if (defined ($actions->{$type}))
			{
				my $act = $actions->{$type};
				print qq(<a href="$MySelf?action=$act&hw=$hw_id"$title>$name</a>);
			}
			else
			{
				print $name;
			}
			print "</td>\n";

			last if (!$many);
		}
		print qq(\t\t\t</tr>\n);
	}
	print qq(\t\t</table>\n);
}

sub _print_hardware_children
{
	my $childs = shift;
	my $indent = @_ ? shift : '';

	return unless (ref ($childs) eq 'HASH');

	print "$indent<ul>\n";
	for (keys %$childs)
	{
		my $name = $_;
		my $next = $childs->{$name};

		if (ref ($next))
		{
			print "$indent\t<li>" . html_escape ($name) . "\n";
			_print_hardware_children ($next, $indent . "\t");
			print "$indent\t</li>\n";
		}
		else
		{
			print "$indent\t<li>" . html_escape ($name) . "</li>\n";
		}
	}
	print "$indent</ul>\n";
}

sub _get_all_customers
{
	my $retval = {};

	DoSelect (sub
		{
			my ($id, $name) = @_;
			$retval->{$name} = $id;
#		}, 'SELECT id, name FROM kunde WHERE ende IS NULL OR ende = 0 OR ende >= UNIX_TIMESTAMP(NOW())'
		}, 'SELECT DISTINCT h.kunde, k.name FROM hardware h LEFT JOIN kunde k ON h.kunde = k.id'
	);

	return ($retval);
}
