use strict;
use utf8;
use warnings;

package noris::Ticket::API::RT::SelectResult;

use Carp qw(croak);
use Dbase::Globals qw(find_descr);
use Dbase::Help qw(DoSel);
use Moose;
use noris::Ticket::API::RT::Field;

# Schöner wäre hier ein "isa => 'ArrayRef[noris::Ticket::API::RT::Field]'"
# mit coerce, aber das kann die Moose-Version unter Etch leider noch nicht.
has 'attributes' => ( is => 'ro', isa => 'ArrayRef', auto_deref => 1 );

# Das hier ist der Workaround für o.g. Unzulänglichkeit der alten Moose-
# Version. Es ist im Prinzip natürlich nur ein Cache, der aber sinnvoll
# erscheint, weil wir sonst in ->next() für jeden Datensatz neu in der
# %field_map nachschauen müssten.
has '_attributes' => (
    is         => 'ro',
    isa        => 'ArrayRef',
    auto_deref => 1,
    lazy       => 1,
    default    => sub {
        my $self = shift;
        [ map _field($_), $self->attributes ];
    }
);
has 'sth' => (
    is      => 'ro',
    isa     => 'Object',
    default => sub {
        my ($self) = @_;

        my @attr = $self->_attributes;

        # Tickets, die bereits "im_otrs" sind, sind so gut wie nicht existent.
        # (Muss als Sonderfall behandelt werden, weil er "merged" vorgeht, vgl.
        # #10005981):
        my @where = 'ticket.status != ' . find_descr( tickets => im_otrs => 1 );

		my %joins;
        {
            my $query = $self->query;

            while ( my ( $field, $filter ) = each %$query ) {
                $field = _field($field);

                my ( $operator, @values ) =
                  ref $filter ? @$filter : ( in => $filter );

                my $method = "where_$operator";
                push @where, $field->$method( map $field->api2rt($_), @values );

                ++$joins{$_} for $field->joins;
            }
        }

		++$joins{$_} for map $_->joins, @attr;

        my $sql = join ' ',
          SELECT => join( ',', map $_->select_name, @attr ) || 1,
          FROM => 'ticket', keys %joins,
          @where ? ( WHERE => join ' AND ', @where ) : ();

        noris::Ticket::API::_debug( sql => $sql );

        DoSel($sql);
    },
    lazy => 1,
);
has 'query' => ( is => 'ro', isa => 'HashRef', auto_deref => 1, required => 1 );
has 'connection' => ( is => 'ro', isa => 'noris::Ticket::API::RT::Connection' );

sub foreach_row {
    my ( $self, $fn ) = @_;
    my $row;
    while ( defined( $row = $self->next ) ) {
        $fn->(@$row);
    }
}

sub next {
    my $self   = shift;
    my @record = $self->sth->nextrow or return;
    my @attr   = $self->_attributes or return [];
    [ map $attr[$_]->rt2api( $record[$_] ), 0 .. $#record ];
}

sub count { shift->sth->rows }

{
    my %field_map = (
        _gen_field( created => Date => db_name => 'ticket.beginn' ),
        _gen_field( due => Date => db_name => 'ticket.endtermin' ),
        _gen_field( est_effort => SecsToHours => db_name => 'ticket.zeit' ),
        _gen_field(
            kunde     => String =>
              db_name => 'kunde.name',
            joins => ['JOIN kunde ON kunde.id = ticket.kunde'],
        ),
        _gen_field( info => String => db_name => 'ticket.infotext' ),
        _gen_field( last_action => Date => db_name => 'ticket.d_acted' ),
        _gen_field( merge_parent => Numeric => db_name => 'ticket.ticket' ),
        _gen_field( merge_root => Numeric => db_name => 'ticket.ticket' ),
        _gen_field( otrs_status => NotSupported => ),
        _gen_field(
            owner     => String =>
              db_name => 'person.user',
            joins => ['LEFT JOIN person ON ticket.bearbeiter = person.id'],
        ),
        _gen_field( pending_until => Date => db_name => 'ticket.termin' ),
        _gen_field(
            priority  => Numeric =>
              db_name => 'ticket.wichtig',
            format => '%02d',
        ),
        _gen_field(
            queue     => String =>
              db_name => 'queue.name',
            joins => ['JOIN queue ON queue.id = ticket.queue'],
        ),
        _gen_field( status => Status => ),
        _gen_field( ticket_number => Numeric => db_name => 'ticket.id' ),
        _gen_field( ticket_url => URL => db_name => 'ticket.id' ),
        _gen_field( title => String => db_name => 'ticket.subject' ),
        _gen_field(
            type      => String =>
              db_name => 'queue_areas.name',
            joins =>
              ['LEFT JOIN queue_areas ON queue_areas.id = ticket.queue_area'],
        ),
    );

    sub _field {
        my ($name) = @_;
        $field_map{$name} || croak("Unbekanntes Ticket-Attribut: $name");
    }
}

sub _gen_field {
    my ( $api_name, $type, @type_specific_args ) = @_;

    my $class = "noris::Ticket::API::RT::Field::$type";

    # Ohne eval würde $class als Dateiname interpretiert:
    eval "require $class";
    die $@ if length $@;

    $api_name => $class->new( api_name => $api_name, @type_specific_args );
}

1;
