package noris::REST::Backend;

=head1 NAME

noris::REST::Backend - Library für REST-basierte Backends

=head1 FUNKTIONEN

Alle Funktionen werden nur auf expliziten Wunsch exportiert.

=cut

use utf8;
use warnings;
use strict;

use base 'Exporter';
our @EXPORT_OK = qw(
  error
  error_missing_parameter
  get_param
  return_result
  success
);

use Data::DumpXML qw(dump_xml);

{
	my $cgi;

	sub cgi {
		require CGI;
		$cgi ||= new CGI;
	}
}

=head2 error($)

Erwartet als einziges Argument einen Fehlertext.
Gibt diesen Text als text/plain-Dokument mit HTTP-Status 400 aus
und beendet dann die Ausführung des Programms.

Beispiel:

	error('Das Ende der Welt ist nah.');

=cut

sub error($) {
	print cgi->header(
		-charset => 'utf-8',
		-status  => 400,
		-type    => 'text/plain'
	  ),
	  map /\n\z/ ? $_ : "$_\n", @_;
	exit;
}

=head2 error_missing_parameter($)

Soll verwendet werden, wenn ein erforderlicher CGI-Parameter fehlt.
Erwartet als einzigen Parameter den Name des fehlenden Parameters.
Ruft selbst L</error($)> auf, beendet also ebenfalls das Programm.

Beispiel:

	error_missing_parameter('user.name');

=cut

sub error_missing_parameter($) {
	my ($parameter) = @_;
	error qq(Parameter "$parameter" fehlt.);
}

=head2 get_param($$;%)

Erwartet als erstes Argument den Typ des erwarteten Parameters (d. h. den
entsprechenden L<CGI::Untaint>-Input-Handler) und als zweites seinen Namen.
Wird für den Typ L<C<undef>|perlfunc/undef> übergeben, so findet keine
Überprüfung des Wertes und auch kein Untaint statt.

Als weitere Argumente können Optionen übergeben werden, wobei jede Option
einen aus zwei Argumenten (Name und Wert) besteht.
Derzeit wird nur die Option I<optional> unterstützt; hat diese einen wahren
Wert, so wird kein Fehler geworfen, falls der gewünschte CGI-Parameter nicht
existiert, vielmehr gibt die Funktion in diesem Fall in skalarem Kontext
L<C<undef>|perlfunc/undef> und in einem Listenkontext eine leere Liste zurück.

Im skalaren Kontext liefert die Funktion den CGI-Paramter des angegebenen
Namens, bzw. wirft einen L<Fehler|/error_missing_parameter($)>, falls der
nicht gesetzt ist oder nicht dem gewünschten Format entspricht.

In einem Listenkontext wird eine Liste der CGI-Parameter mit dem
entsprechenden Namen zurückgegeben.
Hierbei wird kein Untaint unterstützt, d. h. der Typ-paramter muss
L<C<undef>|perlfunc/undef> sein.

Beispiele:

	my $username = get_param printable => 'user.name';

	my @dienste = get_param undef, 'dienst', optional => 1;

=cut

{
	my $handler;

	sub get_param($$;%) {
		my ( $as, $param, %option ) = @_;
		require CGI::Untaint;
		$handler ||= CGI::Untaint->new( cgi->Vars );
		if (wantarray) {
			die "CGI::Untaint wird nur im Skalar-Kontext unterstützt.\n"
			  if defined $as;
			my @param = cgi->param($param)
			  or $option{optional}
			  or error_missing_parameter $param;
			@param;
		}
		elsif (
			defined(
				my $value =
				  defined $as
				? $handler->extract( "-as_$as" => $param )
				: cgi->param($param)
			)
		  )
		{
			$value;
		}
		elsif ( @{ [ cgi->param($param) ] } ) {
			error qq(Parameter "$param" enthält einen ungültigen Wert.);
		}
		elsif ( $option{optional} ) { undef }
		else { error_missing_parameter $param }
	}
}

=head2 return_result($)

Gibt das Ergebnis des Backend-Aufrufs in dem (XML-)Format aus, in dem es das
L<noris::REST::Frontend> erwartet.
Als einziges Argument muss dazu ein Skalar (ggf. also eine Referenz) übergeben
werden.

Beispiel:

	return_result [ { foo => 'bar' }, 47, 11 ];

=cut

sub return_result($) {
	my ($result) = @_;
	print cgi->header( -charset => 'utf-8', -type => 'text/xml' ),
	  dump_xml($result);
}

=head2 success(;$)

Erwartet als einziges, optionales Argument eine Statusmeldung.
Gibt diese (oder als Default "OK") mit HTTP-Status 200 aus
und beendet dann die Ausführung des Programms.

Ist für (im Zweifel schreibende) Anfragen gedacht, die keine echte
Datenstruktur zurückliefern müssen.

Beispiel:

	success('Janz ruhig, et läuft!');

=cut

sub success(;$) {
	my ($status) = @_;
	unless ( defined $status ) { $status = "OK\n" }
	elsif ( $status !~ /\n\z/ ) { $status .= "\n" }

	print cgi->header( -charset => 'utf-8', -type => 'text/plain' ), $status;
	exit;
}

1;
