package noris::CreateTicket;

use utf8;
use strict;
use warnings;

=head1 NAME

noris::CreateTicket - Generieren neuer Tickets

=head1 SYNOPSIS

  use noris::CreateTicket;

  my $ticket = new noris::CreateTicket (%args);

  my $prio     = $ticket->get ('priority');
  my $old_info = $ticket->set (info => $new_info);

  my $ticket_id = $ticket->create () or die;

=head1 BESCHREIBUNG

Dieses Modul kann verwendet werden, um via TCP mit dem RT-Backend zu
kommunizieren, das über die Konfigurationsvariable RT_BACKEND festgelegt wird.
Im Gegensatz zum Mail-Backend kann wird so die Ticket-ID des erzeugten Tickets
zurückgeliefert.

=cut
  
use Carp (qw(carp cluck croak confess));
use Cf qw($RT_BACKEND $WDESCR);
use IO::Socket::INET (qw($CRLF));

our @ValidArgs = qw( area queue kunde extern info owner priority text );

return (1);

=head1 METHODEN

Das B<noris::CreateTicket>-Objekt kennt folgende Methoden:

=over 4

=item I<$obj> = B<noris::CreateTicket>-E<gt>B<new> ([I<$attr_name> =E<gt> I<$attr_value>, ...])

Erzeugt ein neues B<noris::CreateTicket>-Objekt, legt aber noch kein Ticket im
RT an. Fuer eine Liste gueltiger Argumente siehe L</"ATTRIBUTE">.

=cut

sub new
{
	my $pkg = shift;

	my %tmp = @_ ? @_ : ();
	my %obj = ();

	for (@ValidArgs)
	{
		my $arg = $_;
		$obj{$arg} = $tmp{$arg} if (defined ($tmp{$arg}));
	}

	return (bless \%obj, $pkg);
}

sub _do_create
{
	my $obj = shift;

	my $queue  = $obj->{'queue'} or confess ("Which queue?");
	my $extern = $obj->{'extern'} ? 'yes' : 'no';
	my $info   = $obj->{'info'} || '';
	my $kunde  = $obj->{'kunde'} || '';
	my $prio   = $obj->{'priority'} || 0;

	my $text   = $obj->{'text'} or confess ("No text given..");

	my $ret;
	my $ticket_id;
	#my $sequence_id;

	# Slurp mode
	local $/ = undef;

	$text =~ s/^(\.+)$/$1./gm;

	$queue =~ s/^\Q$WDESCR\E\.//;
	$info  =~ s/\s*\n[\s\n]*/ /g;
	$kunde =~ s/\s//g;
	$prio  =~ s/\D//g;

	my $socket = IO::Socket::INET->new(PeerAddr => $RT_BACKEND, Timeout => 20)
		or croak("Creating socket to $RT_BACKEND failed: $!");

	$socket->print ("X-RT-Area: $obj->{area}\n" ) if defined $obj->{area};
	$socket->print ("X-RT-Queue: $WDESCR.$queue\n");
	$socket->print ("X-RT-Extern: $extern\n");
	$socket->print ("X-RT-Info: $info\n") if ($info);
	$socket->print ("X-RT-Kunde: $kunde\n") if ($kunde);
	$socket->print ("X-RT-Priority: $prio\n") if ($prio);
	if ( my $owner = $obj->{owner} ) {
		$owner =~ s/\s+//g;
		$socket->print("X-RT-Owner: $owner\n");
	}

	$socket->print ($text);

	$socket->print (".$CRLF");

	defined( $ret = <$socket> ) and $ret =~ s/^\(#\d+\)\n//;
	  # Hinweis auf Zustellung in <E4>hnliches Ticket ignorieren

	if ($ret =~ m/^(\d+)/)
	{
		$ticket_id = $1;
		$obj->{'ticket_id'} = $ticket_id;
		return ($ticket_id);
	}

	if($ret =~ s/^ERROR:\s*//) {
		$ret =~ s/\n+\z//;
		confess $ret;
	}
	return;
}

=item I<$ticket_id> = I<$obj>-E<gt>B<create> ()

Sendet das Ticket an das RT und liefert die Ticket-ID zurueck. Zu diesem
Zeitpunkt muessen mindestens die Attribute B<queue> und B<text> gesetzt sein.

=cut

sub create
{
	my $obj = shift;

	my $ticket_id;
	my $sequence_id;

	if (defined ($obj->{'ticket_id'}) and defined ($obj->{'sequence_id'}))
	{
		$ticket_id   = $obj->{'ticket_id'};
		$sequence_id = $obj->{'sequence_id'};
	}
	else
	{
		($ticket_id, $sequence_id) = _do_create ($obj, @_);
		$obj->{'ticket_id'}   = $ticket_id;
		$obj->{'sequence_id'} = $sequence_id;
	}

	#return ($ticket_id, $sequence_id) if (wantarray ());
	#return ("$ticket_id-$sequence_id");
	return ($ticket_id) if (wantarray ());
	return ($ticket_id);
}

=item I<$attr_value> = I<$obj>-E<gt>B<get> (I<$attr_name>);

Liefert den aktuellen Wert des Attributs zurueck.

=cut

sub get
{
	my $obj = shift;
	my $key = shift;

	if (!grep { $_ eq $key } (@ValidArgs))
	{
		return;
	}

	return ($obj->{$key});
}

=item I<$old_value> = I<$obj>-E<gt>B<set> (I<$attr_name>, I<$new_value>);

Setzt das Attribut I<$attr_name> auf den Wert I<$new_value>. Liefert den
vorherigen Wert des Attributs zurueck.

=cut

sub set
{
	my $obj = shift;
	my $key = shift;
	my $val = shift;

	if ($obj->{'ticket_id'})
	{
		return;
	}

	if (!grep { $_ eq $key } (@ValidArgs))
	{
		return;
	}

	my $ret = $obj->{$key};
	$obj->{$key} = $val;

	return ($ret);
}

=back

=head1 ATTRIBUTE

Die folgenden Attribute existieren:

=over 4

=item B<queue> I<(zwingend notwendig)>

Queue in der das neue Ticket generiert werden soll. Optional mit einem
B<noris.>-Präfix (genauer gesagt dem, das in der Konfigurationsvariable
C<WDESCR> festgelegt ist.)

=item B<area>

Area, die gesetzt werden soll

=item B<kunde>

KundenB<name> des Kunden, der als B<Requestor> eingetragen werden wird.

=item B<extern>

Setzt das B<X-RT-Extern>-Feld - oder eben nicht..

=item B<info>

Wert fuer das B<Info>-Feld.

=item B<priority>

Wert fuer das B<Prority>-Feld.

=item B<text> I<(zwingend notwendig)>

Text fuer das Ticket.

=back

=head1 AUTHOR

Florian octo Forster E<lt>octo@noris.netE<gt>
fuer die noris network AG L<http://noris.net/>

=cut
