package HousingDB::Rack;

use utf8;
use strict;
use warnings;

BEGIN {
	unshift(@INC, ($ENV{'POPHOME'}||'@POPHOME@').'/lib')
		unless $ENV{'KUNDE_NO_PERLPATH'};
}

use Carp (qw(carp croak cluck confess));
use Dbase::Help;
use Dbase::Globals;
use Loader qw(log_update);

=head1 NAME

HousingDB::Rack

=head1 ATTRIBUTE

=over 4

=item B<id> I<(int2)>

=item B<rz> I<(int2)>

=item B<name> I<(char255)>

=item B<info> I<(char255)>

=item B<he> I<(int1)>

=item B<kunde> I<(int4)>

=item B<x> I<(int2)>

=item B<y> I<(int2)>

=back

=cut

our %ValidFields =
(
	id	=> 1,
	rz	=> 2,
	name	=> 3,
	info	=> 4,
	he	=> 5,
	kunde	=> 6,
	x	=> 7,
	y	=> 8
);

return (1);

=head1 METHODEN

=over 4

=item I<$obj> = HousingDB::Rack-E<gt>B<retrieve> (I<$rack_id>)

Loads the given rack from the database and returns the rack-object.

=cut

sub retrieve
{
	my $pkg = shift;
	my @id_list = @_;
	my @retval = ();

	confess ("No id given") unless (@id_list);

	for (@id_list)
	{
		s/\D//g;
		$_ = int ($_);
		die unless ($_);
	}

	DoSelect (sub
		{
			my $obj = {};
			@$obj{qw(id rz name info he kunde x y)} = @_;
			push (@retval, bless ($obj, $pkg));
		}, <<SQL);
	SELECT id, rz, name, info, he, kunde, x, y
	FROM rack
	WHERE id IN (${\join (', ', @id_list)})
SQL

	if (!@retval)
	{
		return;
	}
	elsif (wantarray ())
	{
		return (@retval);
	}
	elsif (@retval != 1)
	{
		cluck ("I was called in scalar context, but more than one object was returned by the database!");
	}
	return ($retval[0]);
}

sub create
{
	my $pkg = shift;
	my %data = @_;
	my $obj = \%data;

	for (qw(rz name info he kunde x y))
	{
		return (undef) unless (defined ($data{$_}));
	}

	$data{'name'} = qquote ($data{'name'});
	$data{'info'} = qquote ($data{'info'});

	$data{'id'} = Do ('INSERT INTO rack (rz, name, info, he, kunde, x, y) VALUES ('
		. join (', ', map { $data{$_} } (qw(rz name info he kunde x y))) . ')');
	log_update ('rack', 'id', $data{'id'}, undef, '*', undef);

	return (bless ($obj, $pkg));
}

sub extinguish
{
	my $obj = shift;
	die if (!$obj->{'id'});

	log_update ('rack', 'id', $obj->{'id'}, undef, '-', undef);
	Do ('DELETE FROM rack WHERE id = ' . $obj->{'id'});
}

=item HousingDB::Rack-E<gt>B<search> ([I<%where>], [I<$opts>])

Sucht Eintraege in der Datenbank. Die Suche kann durch I<%where> eingeschraenkt
werden. I<$opts> kann weitere Optionen angeben. Liefert eine Liste von
HousingDB::Rack-Objekten zurueck.

  %where =
  (
    field0 => $value0,
    field1 => $value1,
    ...
  );
  $opts =
  {
    order_by => 'field1'
  };

=cut

sub search
{
	my $pkg = shift;
	my $opts = pop if (scalar (@_) % 2);
	my %fields = @_;
	my $found_invalid = 0;

	for (keys %fields)
	{
		my $key = $_;
		my $val = $fields{$key};

		if (!defined ($ValidFields{$key}))
		{
			warn ("Not a valid field: $key");
			$found_invalid++;
			delete ($fields{$key});
			next;
		}

		if (!defined ($val))
		{
			$val = 'NONE';
		}
		elsif ($key eq 'name' or $key eq 'info')
		{
			$val = qquote ($val);
		}
		elsif ($key eq 'id' or $key eq 'rz' or $key eq 'he' or $key eq 'kunde' or $key eq 'x' or $key eq 'y')
		{
			$val =~ s/\D//g;
			$val ||= '0';
		}
		else
		{
			die ("You should not get here");
		}

		$fields{$key} = $val;
	}

	return (qw()) if (!%fields and $found_invalid);

	my $where = join (' AND ', map { $_ . ' = ' . $fields{$_} } (keys %fields));
	my $order = '';

	if (defined ($opts->{'order_by'}))
	{
		my $tmp = $opts->{'order_by'};
		if (defined ($ValidFields{$tmp}))
		{
			$order = " ORDER BY $tmp";
		}
		else
		{
			warn ("Not a valid field: $tmp");
		}
	}

	my @retval = ();

	my @id_list = ();
	DoSelect (sub
		{
			my $id = shift;
			push (@id_list, $id);
		}, "SELECT id FROM rack WHERE $where $order"
	);

	return if (!@id_list);

	@retval = retrieve ($pkg, @id_list);

	return (@retval);
}

=item $obj-E<gt>B<id> ()

Liefert die ID des Racks zurueck.

=cut

sub id
{
	my $obj = shift;
	return ($obj->{'id'});
}

=item $obj-E<gt>B<rz> ([I<$rz>])

Liefer die ID des RZ zurueck. Ggf. wird das Attribut gesetzt.

=cut

sub rz
{
	my $obj = shift;

	if (@_)
	{
		my $tmp = shift;
		_do_update ($obj, 'rz', $tmp);
	}
	
	return ($obj->{'rz'});
}

=item $obj-E<gt>B<name> ([I<$name>])

Liefer den Namen des Racks zurueck. Ggf. wird das Attribut gesetzt.

=cut

sub name
{
	my $obj = shift;

	if (@_)
	{
		my $tmp = shift;
		_do_update ($obj, 'name', $tmp, 'string');
	}
	
	return ($obj->{'name'});
}

=item $obj-E<gt>B<info> ([I<$info>])

Liefer den Infotext zurueck. Ggf. wird das Attribut gesetzt.

=cut

sub info
{
	my $obj = shift;

	if (@_)
	{
		my $tmp = shift;
		_do_update ($obj, 'info', $tmp, 'string');
	}
	
	return ($obj->{'info'});
}

=item $obj-E<gt>B<he> ([I<$he>])

Liefer die Anzahl der HEs zurueck. Ggf. wird das Attribut gesetzt.

=cut

sub he
{
	my $obj = shift;

	if (@_)
	{
		my $tmp = shift;
		_do_update ($obj, 'he', $tmp);
	}
	
	return ($obj->{'he'});
}

=item $obj-E<gt>B<kunde> ([I<$kunde>])

Liefer die ID des Kunden zurueck. Ggf. wird das Attribut gesetzt.

=cut

sub kunde
{
	my $obj = shift;

	if (@_)
	{
		my $tmp = shift;
		$tmp = 'NULL' unless (defined ($tmp));
		
		_do_update ($obj, 'kunde', $tmp);
	}
	
	return ($obj->{'kunde'});
}

=item $obj-E<gt>B<position> ([I<$x>, I<$y>])

Liefert die X- und Y-Position zurueck. Ggf. werden die Attribute gesetzt.

=cut

sub position
{
	my $obj = shift;

	if (scalar (@_) == 2)
	{
		my $x = shift;
		my $y = shift;

		_do_update ($obj, 'x', $x);
		_do_update ($obj, 'y', $y);
	}

	return ($obj->{'x'}, $obj->{'y'});
}

sub _do_update ($$$;$)
{
	my $obj = shift;
	my $field = shift;
	my $new_val = shift;
	my $old_val = $obj->{$field};
	my $id = $obj->{'id'};
	my $is_string = @_ ? shift : 0;

	if (($is_string and ($new_val eq $old_val))
			or (!$is_string and ($new_val == $old_val)))
	{
		return;
	}

	$obj->{$field} = $new_val;
	$new_val = qquote ($obj->{$field}) if ($is_string);

	$old_val ||= '-';
	
	log_update ('rack', 'id', $id, undef, $field, undef, $old_val);
	Do ("UPDATE rack SET $field = $new_val WHERE id = $id");
}

=back

=head1 SIEHE AUCH

L<HousingDB::Hardware>

=head1 AUTOR

Florian Forster E<lt>octo at noris.netE<gt> fuer die noris network AG
L<http://noris.net/>

=cut
