package Dbase::Password;

use strict;
use utf8;
use warnings;

use constant SALT_CHARS => ( '.', '/', 0 .. 9, 'A' .. 'Z', 'a' .. 'z' );

use Carp qw(croak);
use Dbase::Globals qw(content);
use UTFkram qw(safe_encode_utf8);

my %data;

sub new {
    my ( $package, $pass, $salt ) = @_;
    croak "Password must be defined.\n" unless defined $pass;
    $data{ my $self = bless \my ($dummy), $package } =
      { pass => $pass, salt => $salt };
    $self;
}

sub gen_salt {
    my $self   = shift;
    my $length = shift;
    $length = 2 unless defined $length;

    $self = $data{$self};

    my $salt = '';
    unless ( defined $self->{salt} ) {
        $salt .= +(SALT_CHARS)[ rand SALT_CHARS ] for 1 .. $length;
    }
    else {
        $salt = $self->{salt};
        $salt .= $salt while $length > length $salt;
        $salt = substr( $salt, 0, 1 ) . substr( $salt, 1 - $length );
    }

    $salt;
}

sub md5 {
    my $self = shift;
    crypt safe_encode_utf8( $data{$self}{pass} ), '$1$' . $self->gen_salt(8);
}

sub plain { $data{ +shift }{pass} }

sub DESTROY { delete $data{ +shift } }

1;

__END__

=encoding utf8

=head1 NAME

Dbase::Password - eingekapselte Kennwortobjekte

=head1 SYNOPSE

     use Dbase::Password ();

     my $password_object = Dbase::Password->new( $password, $salt );
     my $md5hash = $password_object->md5;

=head1 BESCHREIBUNG

Diese Klasse dient zum Verkapseln von Kennwörtern.
Durch die Verwendung von Inside-out-Objekten sind die Kennwörter insbesondere
nicht in Dumps sichtbar.

=head1 KONSTRUKTUR

=over 4

=item ->new( Kennwort [Salt] )

Die optionale Angabe eines Salts kann hilfreich sein, wenn sichergestellt
werden soll, dass sich für dasselbe Kennwort immer derselbe Hash ergibt,
um so z. B. changed_file glücklich zu machen.

=back

=head1 METHODEN

=over 4

=item ->md5

gibt einen MD5-Hash des Kennworts zurück

=item ->plain

gibt das Kennwort als Klartext zurück.
Wollen wir möglichst mal ausbauen, wird aktuell aber noch für die
WLAN-Radius-Konfiguration benötigt, vgl. RT#471216.

=back

