use strict;
use utf8;
use warnings;

package Dbase::Object;

use Moose;

use Carp qw(confess croak);
use Dbase::Help qw(qquote);

has _table => (
    is      => 'ro',
    isa     => 'Str',
    default => sub {
        my $self = shift;
        lc +( split /::/, ref($self) )[-1];
    },
    lazy => 1,
);

# Merkmale, die eindeutig und also als Schlüssel fürs Caching geeignet sind:
use constant _uniq_attributes => qw/id/;

# Workaround: undef-Werte aus den Argumenten entfernen, da (zumindest unter
# Etch) sonst für die fraglichen Attribute (aus welchen Gründen auch immer) das
# Default berechnet würde, obwohl sie als "lazy" gekennzeichnet sind.
# (Z. B. bei "Dbase::Hardware->new( id => ..., status => undef )".)
sub new {
    my ( $package, %attr ) = @_;
    while ( my ( $attr, $value ) = each %attr ) {
        delete $attr{$attr} unless defined $value;
    }
    $package->SUPER::new(%attr);
}

# Workaround^2: Unser Moose kennt noch keine BUILDARGS-Methode, sonst könnte
# man das mit einem "around BUILDARGS" erschlagen, ungefähr so (ungetestet):
# sub new {
#     my $orig  = shift;
#     my $class = shift;
#     my $args  = $class->$orig(@_);
#     while ( my ( $attr, $value ) = each %$args ) {
#         delete $args->{$attr} unless defined $value;
#     }
#     $args;
# };


sub BUILD {
    my ($self) = @_;
    for ( $self->_uniq_attributes ) {
        my $predicate_method = "has_$_";
        return if $self->$predicate_method;
    }
    require Dbase::Globals and Dbase::Globals->import('aufzaehlung')
      unless defined &aufzaehlung;
    croak 'Ein '
      . ref($self)
      . '-Objekt benötigt wenigstens eines der folgenden Attribute: '
      . aufzaehlung( { letzter_trenner => ' oder ' }, $self->_uniq_attributes );
}

# Lädt alle Attribute aus der Datenbank, sobald eines benötigt wird:
sub _load {
    my ( $self, $attribute ) = @_;
    require Dbase::Help and Dbase::Help->import('DoFn') unless defined &DoFn;
    my $attributes = $self->_attributes;
    my $where = join ' AND ', map "$_ = " . qquote( $self->$_ ),
      grep { my $predicate = "has_$_"; $self->$predicate }
      $self->_uniq_attributes
      or confess('Keine eindeutigen Attribute');
    @{$self}{ keys %$attributes } = DoFn(
        'SELECT '
          . join( ', ',
            map exists $attributes->{$_}{sql} ? $attributes->{$_}{sql} : $_,
            keys %$attributes )
          . ' FROM '
          . $self->_table
          . " WHERE $where"
    );
    $self->_after_load if $self->can('_after_load');
    $self->{$attribute} if defined $attribute;
}

has id => is => 'ro', predicate => 'has_id', required => 1;

{

    my %cache;    # für alle von dieser Klasse abgeleiteten Typen

    sub new_cached {
        my ( $package, %attr ) = @_;
        $package = ref $package if length ref $package;

        my $cache = $cache{$package};

        # Haben wir das schon im Cache?
        for ( $package->_uniq_attributes ) {
            defined( my $attr_value = $attr{$_} ) or next;
            if ( defined( my $cached_object = $cache->{$_}{$attr_value} ) ) {
                return $cached_object;
            }
        }

        # Falls nicht: neu bauen und cachen
        my $new_object = $package->new(%attr);
        for ( $package->_uniq_attributes ) {

            # Bewusst kein Zugriff über $self->$_,
            # weil das unnötige Datenbankabfragen provozieren könnte.
            defined( my $attr_value = $attr{$_} ) or next;
            $cache->{$_}{$attr_value} ||= $new_object;
        }

        $new_object;
    }
}

1;
