#!/usr/bin/perl -w
# nagios: -epn
##########################################################################
# Check delay of all configured snapmirrors of a netapp filer
# has nothing to do with http://netapp-monitoring.info/en/help/SnapMirror.html
# but follows the same purposes
# http://docs.oracle.com/cd/E11857_01/em.111/b28751/netapp_filer.htm
# Author: Katharina Drexel
# Date: 25.04.2013
# Reason: Ticket 14333864 
##########################################################################

use strict;
use Net::SNMP;
use Getopt::Std;

use lib "/usr/lib/nagios/plugins"  ;
use utils qw(%ERRORS &print_revision &support);

sub myusage {
	print "Usage: $0 -H <hostname> -C <community> -v <version> -w <warning> -c <critical>\n";
	exit $ERRORS{'UNKNOWN'};
}

my $PROGNAME = "check_netapp_snapmirror.pl";
my $help = $ARGV[0];
if ( $help eq "--help" ) {
	myusage();
}

my %args;
getopts('H:C:v:w:c:hdV', \%args);
my $debug;
#$args{d} ? $debug = 1 : $debug = 0;
if ( $args{d} ) {
	$debug = 1;
} else {
	$debug = 0;
}
print "DEBUGGING on\n" if ( $debug == 1 );
if ( $args{h} ) {
	print "help\n" if ( $debug == 1 );
        myusage();
}

if ( $args{V} ) {
	print_revision($PROGNAME,'$Revision: 0.2 $');
        	print "Copyright (c) 2013 Katharina Drexel 

This plugin reports the Snapmirror delay in seconds of a Netapp Storage device

";
	exit $ERRORS{'UNKNOWN'};
}
my $host;
if ( $args{H} ) {
	$host = $args{H};
} else {
	print "-H <hostname> missing\n";
	myusage();
}
my $community = $args{C};
$community = "public" unless $args{C};
print "COMM: $community\n" if ( $debug == 1 );

my $version = $args{v};
$version = "snmpv2" unless $args{v};
print "VER: $version\n" if ( $debug == 1 );

my $warn; my $crit;
if ( ! $args{w} ) {
	print "-w <warning in seconds delay> missing\n";
	myusage();
} else {
	$warn = $args{w};
}
if ( ! $args{c} ) {
	print "-c <critical in seconds delay missing\n";
	myusage();
} else {
	$crit = $args{c};
}
	
my ($session, $error) = Net::SNMP->session(
	-hostname => $host,
	-community => $community,
	-port => 161,
	-version => $version,
);

$session->max_msg_size(10000);

if (!defined $session) {
	printf "ERROR: %s.\n", $error;
	exit $ERRORS{'UNKNOWN'};
}

my $result = $session->get_table('1.3.6.1.4.1.789.1.9.20.1.2');
if (!defined $result) {
	printf "ERROR: %s.\n", $session->error();
	$session->close();
	exit $ERRORS{'UNKNOWN'};
}

my $snmpv;
if ( $version eq "snmpv1" ) {
	$snmpv = 1;
} elsif ( $version eq "snmpv3" ) {
	$snmpv = 3;
} else {
	$snmpv = "2c";
}
my $count = 0;
my $status = $ERRORS{'UNKNOWN'};
my $vol; my $lagsec;
my %warns; my %crits; my %oks;
for my $key ( keys %$result) {
	print "KEY: $key, RESULT: $$result{$key}\n" if ( $debug == 1);
	$vol = $$result{$key};
	my @oid=split (/\./,$key);
	my $index = $oid[-1];
	print "INDEX: $index\n" if ( $debug == 1 );
	# geht leider nicht mit netsnmp weil da komisch formatierte Sachen rauskommen
	my $delay = `/usr/bin/snmpget -v $snmpv -c $community $host 1.3.6.1.4.1.789.1.9.20.1.6.$index | /usr/bin/awk -F '[()]' '{print \$2}'`;
	print "DELAY: $delay\n" if ( $debug == 1 );
	$lagsec = $delay/100; 	#delay in seconds is  $lagsec
	if ( $lagsec > $crit ) {
		$crits{$vol} = $lagsec;
		$status = $ERRORS{'CRITICAL'} ;
	} elsif ( $lagsec > $warn ) {
		$warns{$vol} = $lagsec;
		$status = $ERRORS{'WARNING'} if $status != $ERRORS{'CRITICAL'};
	} else {
		$oks{$vol} = $lagsec;
		$status = $ERRORS{'OK'} if ( $status != $ERRORS{'WARNING'} and $status != $ERRORS{'CRITICAL'});
	}
	$count++;
}
$session->close();
#	my $res = $session->get_request('1.3.6.1.4.1.789.1.9.20.1.6.'.$index);
#	if (!defined $res) {
#		printf "ERROR: %s.\n", $session->error();
#		$session->close();
#		exit 3;
#   	}
#	$delay = $res->{'1.3.6.1.4.1.789.1.9.20.1.6.'.$index};
my $v;
if ( $status == $ERRORS{'CRITICAL'} ) {
	print "CRITICAL delay in snapmirrors\n";
	for $v (keys %crits ) {
		print "$v is $crits{$v} sec behind master\n";
	}
	for $v (keys %warns ) {
		print "$v is $warns{$v} sec behind master\n";
	}
	exit $status;
} elsif ( $status == $ERRORS{'WARNING'} ) {
	print "WARNING delay in snapmirrors\n";
	for $v (keys %warns ) {
		print "$v is $warns{$v} sec behind master\n";
	}
	exit $status;
} elsif ( $status == $ERRORS{'OK'} ) {
	print "OK: found $count snapmirrors\n"; 
	for $v (keys %oks ) {
		print "$v is $oks{$v} sec behind master\n";	
	}
} else {
	print "Plugin not working correctly";
	exit $status;
}
