package NNIS::MessageBus::Connection;

use 5.014;
use strict;
use warnings;
use utf8;

use EV;

use Time::HiRes qw(time);
use Params::Validate qw(validate);
use UUID::Tiny ':std';
use AnyEvent;
use AnyEvent::RabbitMQ;
use JSON::XS ();
use Data::Dumper;
use Scalar::Util qw(weaken reftype);

use NNIS::MessageBus::MultiCondVar qw(multi_cv);

use Moo;

has heartbeat => (
    is          => 'ro',
    default     => sub { 580 },
);

has timeout => (
    is      => 'rw',
    default => sub { 20 },
);

has source_service => (
    is          => 'rw',
    required    => 1,
);

has response_address => (
    is          => 'lazy',
);

has config      => (
    is          => 'rw',
    default     => sub { { } },
);

has debug       => (
    is          => 'rw',
    default     => sub { 0 },
);

has connection  => (
    is          => 'lazy',
);

has connection_async => (
    is          => 'lazy',
);

has channel_async => (
    is          => 'lazy',
);

has connector   => (
    is          => 'ro',
    required    => 1,
    weak_ref    => 1,
);

has channel     => (
    is          => 'lazy',
);

has last_contact  => (
    is          => 'ro',
    default     => sub { time },
);

sub is_alive {
    my $self = shift;

    return 0 if $self->{_is_dirty};

    # this is a nasty little trick to make the event loop work on
    # outstanding events, which might set $self->{_is_dirty} again
    if (AnyEvent::detect() eq 'AnyEvent::Impl::EV') {
        require EV;
        EV::run EV::RUN_NOWAIT;
    }
    else {
        my $cv = AnyEvent->condvar;
        my $timer = AnyEvent->timer(
            after       => 0.00001,
            cb          => sub { $cv->send },
        );
        $cv->recv;
    }

    return 0 if $self->{_is_dirty};
    return 1 if (time - $self->last_contact) < 1.5 * $self->heartbeat;
    return 0;
}

sub _check_alive {
    my $self = shift;
    die "This connection is dead\n" if $self->{_is_dirty};
    $self;
}

sub _tick {
    my $self = shift;
    $self->{last_contact} = time;
    $self;
}

sub background_connect {
    my $self = shift;
    $self->connection_async;
    return $self;
}

sub is_consuming {
    my $self = shift;
    $self->{_is_consuming} // 0;
}

sub set_consumers {
    my $self = shift;
    $self->{_consumers} = [@_];
    $self;
}

sub _set_up_consumers {
    my ($self) = @_;

    for my $consumer (@{ $self->{_consumers} }) {
        warn "Initiating consumption on " . $consumer->queue . "\n" if $self->debug;
        $self->channel->consume(
            queue      => $consumer->queue,
            on_success => sub {
                warn "Started consumption on " . $consumer->queue if $self->debug;
                $self->_tick;
            },
            on_failure => sub {
                $self->_fail;
                die sprintf "Consumption from queue %s failed: %s",
                    $consumer->queue, shift;
            },
            on_consume => sub {
                # incoming message!
                my $response = shift;
                $self->_tick;
                warn "Incoming message!" if $self->debug;
                my $header   = $response->{header}; # Net::AMQP::Protocol::Basic::ContentHeader
                my $body     = $response->{body};   # Net::AMQP::Frame::Body
                my $deliver  = $response->{deliver};# Net::AMQP::Frame::Method
                my $nmf_message = $self->_decode($body->payload);
                warn "Launching callback..." if $self->debug;
                my ($service_response, $service_status) = eval {
                    $consumer->callback->($nmf_message->{'nmf-body'});
                };
                warn "... callback terminated" if $self->debug;
                if ($@) {
                    warn "Consumer for $response->queue choked on a message: $@";
                    # TODO: send error message if desired
                }
                elsif ($consumer->has_response) {
                    warn "Preparing reply..." if $self->debug;
                    my ($message, $rk) = $self->_response_message($nmf_message, $service_response, $service_status);
                    warn "Sending reply..." if $self->debug;
                    $self->connector->send_async(
                        exchange        => $self->reply_exchange,
                        routing_key     => $rk,
                        body            => $self->_encode($message),
                    );
                }
                else {
                    warn "... no response necessary" if $self->debug;
                }
            },
        );
    }
}

sub start_consumption {
    my ($self, $error_cb) = shift;
    warn "Setting up consumers on $self...";
    $self->{_error_cb} = $error_cb;
    weaken $self->{_error_cb};
    $self->{_is_consuming} = 1;

    if ($self->channel_async->ready) {
        $self->_set_up_consumers;
    }
    else {
        $self->channel_async->cb(sub {
            $self->_set_up_consumers;
        });
    }

    $self;
}

sub ask {
    my $self = shift;
    return $self->ask_async(@_)->recv;
}

sub ask_async {
    my $self = shift;
    $self->_check_alive;
    validate(@_, {
        routing_key => 1,
        exchange    => 0,
        class       => 0,
        type        => 0,
        payload     => 1,
        timeout     => 0,
        version     => 0,
    });
    my %param = @_;
    my $query = $self->_message(%param, has_reply => 1);
    my $message_id = $query->{'nmf-header'}{'message-id'};
    my $body = $self->_encode($query);


    my $cv = multi_cv;
    $self->{_pending_responses}{ $message_id } = $cv;
    warn "Waiting for reply to message $message_id" if $self->debug;

    $self->send_async(
        routing_key     => $param{routing_key},
        exchange        => $param{exchange},
        body            => $body,
    );

    my $timeout = $param{timeout} || $self->timeout;
    if ($timeout) {
        # After lots of delays that aren't under the control of the event
        # loop, AnyEvent->now is out of sync with the wallclock time, and
        # can cause premature firing of timeouts.
        AnyEvent->now_update;
        $self->{_timers}{ $message_id } = AnyEvent->timer(
            after   => $timeout,
            cb      => sub {
                if ( my $cv = delete $self->{_pending_responses}{ $message_id } ) {
                    warn "Timeout for message-ID $message_id" if $self->debug;
                    $cv->croak("Timeout while waiting for reply");
                    $self->{timeouts}{ $message_id } = 1;
                    delete $self->{_timers}{ $message_id };
                }
            }
        );
    }

    return $cv;
}

sub send {
    my $self = shift;
    scalar $self->channel;
    $self->send_async(@_);
}

sub send_async {
    my $self = shift;
    $self->_check_alive;
    my %param = @_;
    my $exchange = $param{exchange};

    unless ($exchange) {
        my @parts = split /\./, $param{routing_key};
        if (@parts < 4) {
            die "Cannot infer exchange name from routing key '$param{routing_key}' (most have at least four dot-joined parts)\n";
        }
        $exchange = join '.', @parts[0..3];
    }
    $self->channel_async->cb(sub {
            $self->channel_async->recv->publish(
                routing_key => $param{routing_key},
                exchange    => $exchange,
                body        => $param{body},
                header => {
                    content_type   => 'application/json',
                    app_id         => $self->source_service,
                },
            );
        },
    );
}


sub _message_id {
    my $self = shift;
    # create_uuid() returns binary data, so turn into hex
    return unpack 'H*', create_uuid()
}

sub _message {
    my ($self, %param) = @_;
    my (undef, $service, $instance, $class, $type) = split /\./, $param{routing_key}, 5;
    $param{class} //= ucfirst $class;
    $param{type}  //= "$service.$type";
    my $msg = {
        'nmf-header' => {
            'nmf-version' => 1,
            'message-id' => $self->_message_id,
            'amqp-routing-key' => $param{routing_key},
            'source-service' => $self->source_service,
            'nmf-class' => {
                name                    => $param{class} // 'Query',
                type                    => $param{type},
                'type-version'          => $param{version} // '1.0',
            },
        },
        'nmf-body'  => $param{payload},
    };
    if ($param{has_reply}) {
        $msg->{'nmf-header'}{'nmf-class'}{'amqp-response-address'} = $self->response_address,
    }
    return $msg;
}

sub _response_message {
    my ($self, $orig, $body, $status) = @_;
    # TODO: send an error in status != error
    my $reply_address = $orig->{'nmf-header'}{'nmf-class'}{'amqp-response-address'};
    my $type_version  = $orig->{'nmf-header'}{'nmf-class'}{'type-version'};
    my $type_name     = $orig->{'nmf-header'}{'nmf-class'}{'type'};
    my $orig_msg_id   = $orig->{'nmf-header'}{'message-id'};
    my $nmf_message   = $self->_message(
        routing_key     => $reply_address,
        class           => 'Document',
        version         => $type_version,
        type            => $type_name,
        payload         => $body,
    );
    $nmf_message->{'nmf-header'}{'nmf-class'}{'in-response-to'} = $orig_msg_id;
    return ($nmf_message, $reply_address);
}

sub _build_connection {
    my ($self) = @_;
    $self->connection_async->recv;
}

sub _build_connection_async {
    my ($self) = @_;
    my $config = $self->config;

    warn "Connecting to $config->{host}..." if $self->debug();

    my $is_connected = 0;
    my $connected_cv = multi_cv;
    push @{ $self->{_tmp_connection} }, AnyEvent::RabbitMQ->new->load_xml_spec()->connect(
        host    => $config->{host}     // 'localhost',
        port    => $config->{port}     // 5672,
        user    => $config->{user}     // 'guest',
        pass    => $config->{password} // 'guest',
        vhost   => $config->{vhost}    // '/',
        tls     => $config->{tls}      // 1,
        tune    => {
            heartbeat => $self->heartbeat,
        },

        on_success => sub {
            my $connection = shift;
            warn "Successfully connected to $config->{host}" if $self->debug();
            $self->_tick;
            $connected_cv->send($connection);
        },
        on_failure => sub {
            $self->_fail('connection on_failure');
            $connected_cv->croak("Failed to connect to $config->{host}") unless $connected_cv->ready;
        },
        on_error => sub {
            $self->_fail('connection on_error');
        },
        on_read_failure => sub {
            $self->_fail('connection on_read_failure');
        },
        on_inactive => sub {
            $self->_fail('connection on_inactive');
        },
        on_return => sub {
            $self->_fail('connection on_return');
            die "Returned frame: ", Dumper shift;
        },
    );
    return $connected_cv;
}

sub reply_exchange {
    my $self = shift;
    return $self->config->{reply_exchange} // 'reply';
}

sub _build_response_address {
    my $self = shift;
    my $channel = $self->channel;
    my $cv = AnyEvent->condvar;
    my $response_address;
    my $queue = $channel->declare_queue(
        on_success => sub {
            warn "Queue declared!" if $self->debug;
            $self->_tick;
            my $method = shift;
            my $frame  = $method->method_frame;
            $response_address = $frame->queue;
            my $reply_exchange = $self->reply_exchange;
            $channel->bind_queue(
                queue           => $response_address,
                routing_key     => $response_address,
                exchange        => $reply_exchange,
                on_success      => sub {
                    warn "Bound Queue $response_address to $reply_exchange" if $self->debug;
                    $self->_tick;
                    $cv->send(1);
                },
                on_failure => sub {
                    $self->_fail('_build_response_address bind_queue on_failure');
                    $cv->send(0);
                },

            );
        },
        on_failure => sub {
            $self->_fail('_build_response_address on_failure');
            $cv->send(0)
        },
        queue      => '',
        auto_delete => 1,
        exclusive   => 1,
    );
    $cv->recv;
    unless ($response_address) {
        die "Error while declaring response queue";
    }
    $self->_consume($response_address);
    return $response_address;
}

sub _consume {
    my ($self, $response_address) = @_;
    my $cv = AnyEvent->condvar;
    say "Consuming from Queue $response_address" if $self->debug;
    $self->channel->consume(
        queue   => $response_address,
        on_consume => sub {
            my $response = shift;
            warn "Got response!" if $self->debug;
            my $header   = $response->{header}; # Net::AMQP::Protocol::Basic::ContentHeader
            my $body     = $response->{body};   # Net::AMQP::Frame::Body
            my $deliver  = $response->{deliver};# Net::AMQP::Frame::Method

            my $nmf_message = $self->_decode($body->payload);
            # TODO: validieren!
            my $reference = $nmf_message->{'nmf-header'}{'nmf-class'}{'in-response-to'};
            unless ($reference) {
                die "No 'in-response-to' field found in message body: ",
                    Dumper $nmf_message;
            }
            warn "In-Response-To is: $reference" if $self->debug;
            my $response_cv = delete $self->{_pending_responses}{$reference};
            if ($response_cv) {
                delete $self->{_timers}{ $reference };
                return $response_cv->send($nmf_message);
            }
            elsif (my $timer = delete $self->{timeouts}{ $reference }) {
                warn "Got reply after timeout for message with ID '$reference'" if $self->debug;
                return;
            }
            else {
                warn "Got an unexpected reply to message with ID '$reference'";
            }
        },
        on_success => sub {
            say "Consumation from Queue $response_address established" if $self->debug;
            $self->_tick;
            $cv->send(1);
        },
        on_failure => sub {
            $cv->send(0);
            die "Consumation from Queue $response_address failed!";
        },
    );
    $cv->recv;
}

sub _fail {
    my ($self, $cb_name) = @_;
    $self->_mark_dirty;
    if ($self->debug) {
        $cb_name //= 'an error callback';
        warn "$cb_name called\n";
    }
    if ($self->{_error_cb}) {
        $self->{_error_cb}->();
    }
    $self;
}

sub _build_channel {
    my $self = shift;
    $self->channel_async->recv;
}

sub _build_channel_async {
    my $self = shift;
    my $cv = multi_cv;
    $self->connection_async->cb(sub {
        $self->connection_async->recv->open_channel(
            on_success => sub {
                my $channel = shift;
                $self->_tick;
                $cv->send($channel);
            },
            on_failure  => sub {
                $self->_fail('channel on_failure');
                $cv->croak("Can't open channel");
            },
            on_close   => sub {
                $self->_fail('channel on_close');
            },
        )
    });
    return $cv;
}

sub _decode {
    my ($self, $buf) = @_;
    return JSON::XS::decode_json($buf);
}

sub _encode {
    my ($self, $structure) = @_;
    state $encoder = JSON::XS->new->utf8->convert_blessed;
    return $encoder->encode($structure) . "\n";
}

sub _mark_dirty {
    my $self = shift;
    $self->{_is_dirty} = 1;
    $self;
}

sub teardown {
    my $self = shift;

    return if $self->{_teardown_complete};

    say "teardown  " . ref($self) if $self->debug;
    delete $self->{_timers};
    if ( $self->{_pending_responses} ) {
        for my $cv (values %{ $self->{_pending_responses} }) {
            warn "Killing oustanding response in teardown" if $self->debug;
            $cv->croak('Lost connection to message bus');
        }
    }

    # Ticket 21298569: can cause trouble on teardown, so not
    # doing this in production
    if ($self->{channel} && $self->debug) {
        eval {
            $self->{channel}->delete_queue(
                queue       => $self->{response_address},
            );
        };
        warn "ERROR during $self teardown : $@" if $@ && $self->debug;
        delete $self->{channel};
    }
    $self->{_teardown_complete} = 1;
    say "Done with teardown " . ref($self) if $self->debug;
}

sub DESTROY {
    shift->teardown;
}

1;
__END__

# vim: tw=100
