#===============================================================#
#                                                               #
# ID                                                            #
#                                                               #
# resource_pool.py                                              #
#                                                               #
# Copyright (c) 2011 NetApp, Inc. All rights reserved.          #
# Specifications subject to change without notice.              #
#                                                               #
# Sample code to demonstrate how to:                            #
#        - list/create/delete a resource pool                   #
#        - list/add/delete members from a resource pool         #
#                                                               #
# This Sample code is supported from DataFabric Manager 3.8     #
# onwards.                                                      #
# However few of the functionalities of the sample code may     #
# work on older versions of DataFabric Manager.                 #
#===============================================================#

import sys
sys.path.append('../../../../../../../lib/python/NetApp')
from NaServer import *

# Prints the usage of this program
def print_usage() :
    print ("Usage:\n")
    print ("resource_pool.py <dfmserver> <user> <password> list [ResPoolName]\n")
    print ("resource_pool.py <dfmserver> <user> <password> create ResPoolName [ResourceTag]\n")
    print ("resource_pool.py <dfmserver> <user> <password> destroy ResPoolName\n")
    print ("resource_pool.py <dfmserver> <user> <password> member list ResPoolName\n")
    print ("resource_pool.py <dfmserver> <user> <password> member [add|del] ResPoolName MemberName\n")
    print ("<dfmserver>       -- Name/IP Address of the DFM server\n")
    print ("<user>            -- DFM server User name\n")
    print ("<password>        -- DFM server User Password\n")
    print ("<ResPoolName>     -- Resource pool name, mandatory for create & destroy options\n")
    print ("<MemberName>      -- Member to be added/removed from a resource pool, mandatory for add  & del options\n")
    sys.exit (1)

# Variables declaration
args      = len(sys.argv) - 1

if(args < 4):
    print_usage()

dfmserver = sys.argv[1]
dfmuser   = sys.argv[2]
dfmpw     = sys.argv[3]
command   = sys.argv[4]

# Setup DFM server connection
s = NaServer( dfmserver, 1, 0 )
s.set_style("LOGIN")
s.set_transport_type("HTTP")
s.set_server_type("DFM")
s.set_port(8088)
s.set_admin_user( dfmuser, dfmpw )

# List all resourcepools on the server
if ( command == "list" ) :

    if(args > 4):
        rname = sys.argv[5]
    else :
        rname = None
    
    if(rname):
        out   = s.invoke( "resourcepool-list-info-iter-start", "object-name-or-id", rname )
    
    else :
        out =  s.invoke( "resourcepool-list-info-iter-start")
    
    if ( out.results_status() == "failed" ) :
        print( "Error : " + out.results_reason() + "\n" )
        sys.exit(2)
    
    input = NaElement("resourcepool-list-info-iter-next")
    input.child_add_string( "maximum", out.child_get_int("records") )
    input.child_add_string( "tag",     out.child_get_string("tag") )
    out = s.invoke_elem(input)
    
    if ( out.results_status() == "failed" ) :
        print( "Error : " + out.results_reason() + "\n" )
        sys.exit(2)
    
    # Loop through the iteration records and print details
    print ("\nRESOURCEPOOLS:\n")
    print ("===================================================================\n")
    rps = out.child_get("resourcepools")
    
    if ( rps == None ) :
        print ("Error: No Resourcepools!\n")
        sys.exit(1)

    rpInfos = rps.children_get()
    
    if(rpInfos != None):
        for rpi in rpInfos :
            print ("Name\t\t:" + rpi.child_get_string("resourcepool-name"))
            print ("Status\t\t:" + rpi.child_get_string("resourcepool-status"))
            print ("# of Members\t:" + rpi.child_get_string("resourcepool-member-count"))
            print ("Tag\t\t:" + rpi.child_get_string("resource-tag"))
            print ("===================================================================\n")
    
elif ( command == "create" ): 
    # Below section creates a new resource pool
    if(args > 5) :
        rpName = sys.argv[5]
        rTag   = sys.argv[6]
    
    elif(args > 4):
        rpName = sys.argv[5]
    
    else :
        print_usage()

    rpCreate = NaElement("resourcepool-create")
    rp       = NaElement("resourcepool")
    rpInfo = NaElement("resourcepool-info")
    rpInfo.child_add_string( "resourcepool-name", rpName )
    if(args > 5) :
        rpInfo.child_add_string( "resource-tag",      rTag )
    rp.child_add(rpInfo)
    rpCreate.child_add(rp)
    out = s.invoke_elem(rpCreate)

    if ( out.results_status() == "failed" ) :
        print( "Error : " + out.results_reason() + "\n" )
        sys.exit(2)
    
    print ("Resourcepool " + rpName  + " created with ID : " + str(out.child_get_int("resourcepool-id")))
    
    # Iterate through the commandline args to see if you have mentioned
    # any members to be added, if yes add them iteratively.
    memNames = []
    if(args > 6):
        memNames = sys.argv[7:]
    
    for memName in memNames :
        out = s.invoke( "resourcepool-add-member", "resourcepool-name-or-id",
                rpName, "member-name-or-id", memName, "resource-tag", rTag )
                
        if ( out.results_status() == "failed" ) :
            print( "Error : " + out.results_reason() + "\n" )
            sys.exit(2)
        
        print ("\nAdded member " + memName + " to Resourcepool " + rpName + "\n")
    
elif ( command == "destroy" ) :
    # Destroy a resourcepool, but it needs to be empty.
    if(args > 4):
        rpName = sys.argv[5]
        
    out = s.invoke( "resourcepool-destroy", "resourcepool-name-or-id", rpName )
    
    if ( out.results_status() == "failed" ) :
        print( "Error : " + out.results_reason() + "\n" )
        sys.exit(2)
    
    print ("Resourcepool " + rpName + " destroyed!\n")
    
elif ( command == "member" ) :
    # Member operations  on a resource pool
    
    if(args > 7):
        subCommand = sys.argv[5]
        rpName     = sys.argv[6]
        memName    = sys.argv[7]
        rTag       = sys.argv[8]
    
    elif(args > 6):
        subCommand = sys.argv[5]
        rpName     = sys.argv[6]
        memName    = sys.argv[7]
        
    elif(args > 5):
        subCommand = sys.argv[5]
        rpName     = sys.argv[6]
    
    else:
        print_usage()
            
    if ( subCommand == "list" ) :
        # Begin the iterative API
        out = s.invoke( "resourcepool-member-list-info-iter-start", "resourcepool-name-or-id", rpName )
        
        if ( out.results_status() == "failed" ) :
            print( "Error : " + out.results_reason() + "\n" )
            sys.exit(2)
        
        # Use the tag and records returned by prev API call and retrive all records
        input = NaElement("resourcepool-member-list-info-iter-next")
        input.child_add_string( "maximum", out.child_get_int("records") )
        input.child_add_string( "tag",     out.child_get_string("tag") )
        out = s.invoke_elem(input)

        if ( out.results_status() == "failed" ) :
            print( "Error : " + out.results_reason() + "\n" )
            sys.exit(2)
        
        # Print necessary details from retrieved records
        print ("\nRESOURCEPOOL : " + rpName)
        print ("\n===================================================================\n")
        rpms = out.child_get("resourcepool-members")
        
        if ( rpms == None ) :
            print ("Error: No Members in Resourcepool!\n")
            sys.exit(2)
        
        rpmInfos = rpms.children_get()
        
        if(rpmInfos != None):        
            for rpmi in rpmInfos :
                print ("Name\t:" + rpmi.child_get_string("member-name"))
                print ("Status\t:" + rpmi.child_get_string("member-status"))
                print ("Type\t:" + rpmi.child_get_string("member-type"))
                print ("Tag\t:" + rpmi.child_get_string("resource-tag"))
                print ("===================================================================\n")
        
    elif ( subCommand == "add" ) :
        # Add a new member into an existing resource pool
        if ( args < 7 ) :
            print_usage()
        
        out = s.invoke( "resourcepool-add-member", "resourcepool-name-or-id",
            rpName, "member-name-or-id", memName, "resource-tag", rTag )
        
        if ( out.results_status() == "failed" ) :
            print( "Error : " + out.results_reason() + "\n" )
            sys.exit(2)
        
        print ("Added member " + memName + " to Resourcepool " + rpName + "\n")
    
    elif ( subCommand == "del" ) :
        # Remove a  member from an existing resource pool
        if ( args < 7 ) :
            print_usage()
        
        out = s.invoke( "resourcepool-remove-member", "resourcepool-name-or-id",
            rpName, "member-name-or-id", memName )
            
        if ( out.results_status() == "failed" ) :
            print( "Error : " + out.results_reason() + "\n" )
            sys.exit(2)
        
        print ("Removed member " + memName + " from Resourcepool " + rpName + "\n")
        
    else :
        print_usage()
    
else :
    print_usage()



