/**
*
* (c) Copyright 1993-2009 Hewlett-Packard Development Company, L.P.
*
* @ingroup   Data Compression (Dc)
* @file      da/dc/libdc.h
*
* @par Last Change:
* $Rev: 1981 $
* $Date: 2009-09-11 13:11:14 -0400 (Fri, 11 Sep 2009) $
*
* @brief     Compress/Decompress Library API
*
* @since     01.12.93   caka          Initial coding
*/

#if !defined(LIBDC_H)
#define LIBDC_H

#if !defined(UCHAR)
#define UCHAR   unsigned char
#endif

#if !defined(ULONG)
#define ULONG   unsigned long
#endif

/* ---------------------------------------------------------------------------
|    
|    Module Defines - Unique Data Compression Type Identifiers
|    
 -------------------------------------------------------------------------- */

#define    DC_NONE       0			/* No data compression	*/
#define    DC_LZ43       0x0b0b0b0b		/* Lempel-Ziv 4.3	*/
#define    DC_GZIP       1			/* GNU ZIP (not impl.)	*/


typedef int (*dcCallback)(UCHAR *, ULONG);
typedef int (*dcCallbackExtended)(UCHAR *, ULONG, void*);

/* ===========================================================================
| NetWare specific:
|  YES certification for NetWare requires that function names, that are
|  exported, start with company name prefix(ie. HPDcInit() instead of DcInit()). 
|
 ==============================================================================*/


/*========================================================================*//**
*
* @ingroup   Data Compression (Dc)
*
* @param     void
*
* @retval    DC_NONE	Initialization failed, compression disabled
* @retval    DC_LZ43	Initialization sucessful, Lempel-Ziv compression
|	  DC_...	Customer specific & implemented compression type
*
* @brief     This function is called by the Disk Agent's initialization
*            procedure at startup. It should initialize the DC module and
*            return an indication on what compression type will be used.
*
* @remarks   The current version od this API does not allow dynamic switching 
*            between different compression algorithms depending on the type of
*            input data. This might change in the future.
*
*//*=========================================================================*/
#if TARGET_WIN32
__declspec(dllexport)
#endif

#if TARGET_NETWARE
    extern ULONG
    HPDcInit (
        tchar *
    );
#else
    extern ULONG
    DcInit (
    #if PROTOTYPES
        tchar *
    #endif
    );
#endif

/*========================================================================*//**
*
* @ingroup   Data Compression (Dc)
*
* @param     readCallback		A pointer to the data input callback
* @param     writeCallback		A pointer to the data output callback
*
* @retval    0		Error
* @retval    1		OK
*
* @brief     This  is  the  actual  stream  compression   function.  It
*            receives its input data using the readCallback  data input
*            method  and  writes  its  output  using the  writeCallback
*            output method.  Here are a few guidelines the  compression
*            module developer should keep in mind:
*
*            1. The nature of the input and output  media is unknown to
*            the  compression  module.  Both input and output should
*            be  treated  as  read-only  and   write-only   streams,
*            respectively.
*
*            2. The DcCompress function is called once for each file to
*            be compressed.  It should read its input until EOF (the
*            input   handler   returns   zero)  and  make  sure  all
*            intermediate  output  buffers  are  flushed  before  it
*            returns.
*
*            3. Each  call to the  input  method  is  sooner  or  later
*            directly  translated  into a read() system call.  Thus,
*            for better  performance, the compression  engine should
*            try to read its input in large chunks and store them in
*            an internal input data buffer.
*
*            4. The  readCallback  function returns the number of bytes
*            read.  This number can, under certain circumstances, be
*            less than the number of bytes actually  requested.  The
*            compression  engine  should  not treat this as an error
*            but simply  issue a second read (this is similar to BSD
*            sockets,    reads   may    succeed    partially).   The
*            readCallback returns zero in case of error or at EOF.
*
*            5. The writeCallback function always succeedes completely.
*            If an error condition is encountered, the writeCallback
*            never returns.
*
*            readCallback returns:
*            >0	- Number of bytes read
*            0	- Error or EOF
*
*            writeCallback returns:
*            >0	- number of bytes written
*            0	- Error
*
* @remarks   Never call the readCallback to read zero bytes, as you
*            will be returned zero which means 'Error or EOF'. Also,
*            never attempt to write zero bytes for the same reason.
*
*//*=========================================================================*/
#if TARGET_WIN32 
__declspec(dllexport)
#endif
#if TARGET_NETWARE
    extern int
    HPDcCompress (
        dcCallback  readCallback,
        dcCallback  writeCallback
        );
#else
    extern int
    DcCompress (
    #if PROTOTYPES
        dcCallback  readCallback,
        dcCallback  writeCallback
    #endif
    );
#endif
/*========================================================================*//**
*
* @ingroup   Data Compression (Dc)
*
* @param     readCallback		A pointer to the data input callback
* @param     writeCallback		A pointer to the data output callback
* @param     externData		A pointer to the additional data.
*                              Compression library must pass this pointer
*                              to readCallback and writeCallback functions.
*
* @retval    0		Error
* @retval    1		OK
*
*//*=========================================================================*/

#if TARGET_WIN32 
__declspec(dllexport)
#endif
#if TARGET_NETWARE
    extern int
    HPDcCompress2 (
        dcCallbackExtended  readCallback,
        dcCallbackExtended  writeCallback,
        void                *externData
    );
#else
    extern int
    DcCompress2 (
    #if PROTOTYPES
        dcCallbackExtended  readCallback,
        dcCallbackExtended  writeCallback,
        void                *externData
    #endif
    );
#endif


/*========================================================================*//**
*
* @ingroup   Data Compression (Dc)
*
* @param     readCallback		A pointer to the data input callback
* @param     writeCallback		A pointer to the data output callback
*
* @retval    0		Error
* @retval    1		OK
*
* @brief     This  is  the  actual  stream decompression  function.  It
*            receives its input data using the readCallback  data input
*            method  and  writes  its  output  using the  writeCallback
*            output method.  Here are a few guidelines the  compression
*            module developer should keep in mind:
*
*            1. The nature of the input and output  media is unknown to
*            the decompression  module. Both input and output should
*            be  treated  as  read-only  and   write-only   streams,
*            respectively.
*
*            2. The DcDecompress  function is called once for each file
*            to be decompressed.  It should read its input until EOF
*            (the  input  handler  returns  zero)  and make sure all
*            intermediate  output  buffers  are  flushed  before  it
*            returns.
*
*            3. Both  input and  output  may be done in byte  chunks of
*            arbitrary length.   The readCallback  typically returns
*            data from an internal  OmniBack  data buffer.  However,
*            for  performance  reasons  it is a good  idea to output
*            larger  amounts  of data at  once,  since  calls to the
*            writeCallback  may be directly  translated into write()
*            system calls.
*
*            4. The readCallback may succeed partially.  A return value
*            of zero denotes an error or the end of the current file
*            to be decompressed.  The writeCallback  always succeeds
*            completely and never reports any errors back.
*
*            readCallback returns:
*            >0	- Number of bytes read
*            0	- Error or EOF
*
*            writeCallback returns:
*            >0	- Number of bytes written
*            0	- Error
*
* @remarks   Never call the readCallback to read zero bytes, as you
*            will be returned zero which means 'Error or EOF'. Also,
*            never attempt to write zero bytes for the same reason.
*
*//*=========================================================================*/
#if TARGET_WIN32 
__declspec(dllexport)
#endif
#if TARGET_NETWARE
    extern int
    HPDcDecompress (
        dcCallback  readCallback,
        dcCallback  writeCallback
    );
#else
    extern int
    DcDecompress (
    #if PROTOTYPES
        dcCallback  readCallback,
        dcCallback  writeCallback
    #endif
    );
#endif

/*========================================================================*//**
*
* @ingroup   Data Compression (Dc)
*
* @param     readCallback		A pointer to the data input callback
* @param     writeCallback		A pointer to the data output callback
* @param     externData		A pointer to the additional data.
*                              Compression library must pass this pointer
*                              to readCallback and writeCallback functions.
*
*//*=========================================================================*/

#if TARGET_WIN32 
__declspec(dllexport)
#endif
#if TARGET_NETWARE
    extern int
    HPDcDecompress2 (
        dcCallbackExtended  readCallback,
        dcCallbackExtended  writeCallback,
        void                *externData
    );
#else
    extern int
    DcDecompress2 (
    #if PROTOTYPES
        dcCallbackExtended  readCallback,
        dcCallbackExtended  writeCallback,
        void                *externData
    #endif
    );
#endif

/*========================================================================*//**
*
* @ingroup   Data Compression (Dc)
*
* @param     void
*
* @retval    0		Error
* @retval    1		OK
*
* @brief     This function is called to allow the DC module to perform
*            post-processing cleanup actions. Its return value is ignored.
*
*//*=========================================================================*/
#if TARGET_WIN32 
__declspec(dllexport)
#endif
#if TARGET_NETWARE
    extern int
    HPDcExit (
        void
    );
#else
    extern int
    DcExit (
    #if PROTOTYPES
        void
    #endif
    );
#endif

#endif /* LIBDC_H */
