//============================================================//
//                                                            //
//                                                            //
// apitest.cs                                                 //
//                                                            //
// Exploratory application for ONTAPI APIs                    //
// It lets you call any ONTAPI API with named arguments       //
//    (essentially a command-line version of the zexplore     //
//     utility)                                               //
//                                                            //
// Copyright 2002-2003 NetApp, Inc. All rights                //
// reserved. Specifications subject to change without notice. // 
//                                                            //
// This SDK sample code is provided AS IS, with no support or //
// warranties of any kind, including but not limited to       //
// warranties of merchantability or fitness of any kind,      //
// expressed or implied.  This code is subject to the license //
// agreement that accompanies the SDK.                        //
//                                                            //
// tab size = 4                                               //
//                                                            //
//============================================================//

using System;
using System.Collections.Generic;
using System.Text;
using NetApp.Manage;

namespace Netapp.Utility
{
    class ApiTest
    {
        private static void Usage()
        {
            Console.WriteLine("\nUsage:\n\t apitest [options] <host> <user> "
                + "<password> <ONTAPI-name> [<param-name> <arg> ...]\n");
            Console.WriteLine("Options:\n");
            Console.WriteLine("\t -t {type} \t Server type(type = filer, dfm, agent)\n");
            Console.WriteLine("\t -v {vfiler name}  Vfiler name, if the API has to be executed "
                + "in the context of a vfiler \n");
            Console.WriteLine("\t -r \t Use RPC transport\n");
            Console.WriteLine("\t -s \t Use SSL\n");
            Console.WriteLine("\t -p {port} \t Override port to use\n");
            Console.WriteLine("\t -c {timeout} \t Connection timeout value in seconds\n");
            Console.WriteLine("\t -i \t API specified as XML input, on the command line\n");
            Console.WriteLine("\t -I \t API specified as XML input, on standard input\n");
            Console.WriteLine("\t -x \t Show the XML input and output\n");
            Console.WriteLine("\t -X \t Show the raw XML input and output\n");
            Console.WriteLine("\t -h \t Use Host equiv authentication mechanism\n");
            Console.WriteLine("\n  Do not provide username and password for -h and -r options\n");
            Console.WriteLine("Examples:\n");
            Console.WriteLine("\t apitest sweetpea root tryme "
                + "system-get-version");
            Console.WriteLine("\n\t apitest amana root meat quota-report volume vol0\n");
            Environment.Exit(1);
        }

        private static void GetXMLInput(int inputXML,ref string[] args,ref int index)
        {
            String readXML = "";
            
            if (inputXML == 1)
            {
                if (args.Length == index + 1)
                {
                    Console.WriteLine("API not specified");
                    Usage();
                }
                for (int cnt = index + 1; cnt < args.Length; cnt++)
                {
                    readXML = readXML + args[cnt];
                }
            }
            else
            {
                String curLine = "";
                Console.WriteLine("Enter the input XML:\n");
                while ((curLine = Console.ReadLine()) != null)
                {
                    readXML += curLine;
                }
            }

            args = readXML.Split(new Char[] { ' ', '\n','\t','\r' });

            readXML = "";
            for (int cnt = 0; cnt < args.Length; cnt++)
            {
                if (!(args[cnt].Contains("\t") || args[cnt].Contains(" ")))
                {
                    readXML = readXML + args[cnt];
                }
            }
            args = readXML.Split(new Char[] { '\t', '\n' });
            index = -1;
        }

        static void Main(string[] args)
        {
            NaServer s;
            NaElement xi = null;
            NaElement xo;
            NaServer.TRANSPORT_TYPE transportType = NaServer.TRANSPORT_TYPE.HTTP;
            NaServer.SERVER_TYPE serverType = NaServer.SERVER_TYPE.FILER;
            NaServer.AUTH_STYLE authStyle = NaServer.AUTH_STYLE.LOGIN_PASSWORD;
            String vfiler = null;
            String type;
            int index = 0;
            int showXML = 0;
            int inputXML = 0;
            int port = -1;
            int timeOut = -1;
            Boolean status = true;
            Boolean useRPC = false;
            Boolean useHostsEquiv = false;

            if (args.Length < 3)
            {
                Usage();
            }
            index = 0;

            try
            {
                while (args[index].StartsWith("-"))
                {
                    switch (args[index][1])
                    {
                        case 't':
                            type = args[index + 1];
                            if (type.Equals("dfm"))
                            {
                                serverType = NaServer.SERVER_TYPE.DFM;
                            }
                            else if (type.Equals("agent"))
                            {
                                serverType = NaServer.SERVER_TYPE.AGENT;
                            }
                            else if (type.Equals("filer"))
                            {
                                serverType = NaServer.SERVER_TYPE.FILER;
                            }
                            else
                            {
                                Console.WriteLine("\nERROR: Invalid Option for Server type.");
                                Usage();
                            }
                            index = index + 2;
                            break;
                        case 'v':
                            vfiler = args[index + 1];
                            index = index + 2;
                            break;
                        case 'r':
                            authStyle = NaServer.AUTH_STYLE.RPC;
                            useRPC = true;
                            index++;
                            break;
                        case 's':
                            transportType = NaServer.TRANSPORT_TYPE.HTTPS;
                            index++;
                            break;
                        case 'p':
                            status = Int32.TryParse(args[index + 1], out port);
                            if (status == false)
                            {
                                Console.WriteLine("\nERROR: Invalid port no.");
                                Usage();
                            }
                            index = index + 2;
                            break;
                        case 'i':
                            inputXML = 1;
                            index++;
                            break;
                        case 'I':
                            inputXML = 2;
                            index++;
                            break;
                        case 'x':
                            showXML = 1;
                            index++;
                            break;
                        case 'X':
                            showXML = 2;
                            index++;
                            break;
                        case 'h':
                            authStyle = NaServer.AUTH_STYLE.HOSTSEQUIV;
                            useHostsEquiv = true;
                            index++;
                            break;
                        case 'c':
                            status = Int32.TryParse(args[index + 1], out timeOut);
                            if (status == false || timeOut <= 0)
                            {
                                Console.WriteLine("\nERROR: Invalid timeout value.");
                                Usage();
                            }
                            index = index + 2;
                            break;
                        default:
                            Console.WriteLine("\nERROR: Invalid Option.");
                            Usage();
                            break;
                    } //switch (args[index][1]) {
                } //while (args[index].StartsWith("-")){
            }
            catch (System.IndexOutOfRangeException)
            {
                Console.WriteLine("\nERROR: Invalid Arguments.");
                Usage();
            }
            if (authStyle == NaServer.AUTH_STYLE.LOGIN_PASSWORD &&
                            args.Length < 4)
            {
                Usage();
            }

            if (useHostsEquiv == true && useRPC == true)
            {
                Console.WriteLine("\nERROR: Invalid usage of authentication style. " +
                "Do not use -r option and -h option together.\n");
                System.Environment.Exit(1);
            }

            if (authStyle == NaServer.AUTH_STYLE.LOGIN_PASSWORD)
            {
                if (index == args.Length)
                {
                    Console.WriteLine("\nERROR: Host not specified.");
                    Usage();
                }
                if ((index + 1) == args.Length)
                {
                    Console.WriteLine("\nERROR: User not specified.");
                    Usage();
                }
                else if ((index + 2) == args.Length)
                {
                    Console.WriteLine("\nERROR: Password not specified.");
                    Usage();
                }
            }
        
            if (port == -1)
            {
                    switch (serverType)
                    {
                        default:
                        case NaServer.SERVER_TYPE.FILER:
                            port = (transportType == NaServer.TRANSPORT_TYPE.HTTP ? 80 : 443);
                            break;
                        case NaServer.SERVER_TYPE.DFM:
                            port = (transportType == NaServer.TRANSPORT_TYPE.HTTP ? 8088 : 8488);
                            break;
                        case NaServer.SERVER_TYPE.AGENT:
                            port = (transportType == NaServer.TRANSPORT_TYPE.HTTP ? 4092 : 4093);
                            break;
                    }
            }

            try
            {
                //1. Create an instance of NaServer object.
                //NaServer is used to connect to servers and invoke API's.
                if (vfiler != null)
                {
                    // Vfiler tunnelling requires ONTAPI version 7.1 to work 
                    s = new NaServer(args[index], 1, 7);
                }
                else
                {
                    s = new NaServer(args[index], 1, 0);
                }

                //2. Set the server type
                s.ServerType = serverType;

                //3. Set the transport type
                s.TransportType = transportType;

                //4. Set the authentication style for subsequent ONTAPI authentications.
                s.Style = authStyle;

                //5. Set the login and password used for authenticating when
                //an ONTAPI API is invoked.
                if (authStyle == NaServer.AUTH_STYLE.LOGIN_PASSWORD)
                {
                    s.SetAdminUser(args[++index], args[++index]);
                }

                //6. Set the port number
                s.Port = port;

                //7. Optional - set the vfiler name for vfiler tunneling
                if (vfiler != null)
                {
                    s.SetVfilerTunneling(vfiler);
                }

                //Set the request timeout.
                if (timeOut != -1)
                {
                    s.TimeOut = timeOut;
                }

                /* Invoke any  ONTAPI API with arguments
                    * in (key,value) pair 
                    * args[0]=filer,args[1]=user,args[2]=passwd
                    * args[3]=Ontapi API,args[4] onward arguments
                    * in (key,value) pair
                */
                try
                {
                    if (inputXML == 0)
                    {
                        if ((index+1) == args.Length)
                        {
                            Console.WriteLine("\nERROR: API not specified.");
                            Usage();
                        }
                        //8. Create an instance of NaElement which contains the ONTAPI API request
                        xi = new NaElement(args[++index]);
                    }
                    //Only use this for debugging
                    else
                    {
                        GetXMLInput(inputXML, ref args, ref index);
                        xi = s.ParseXMLInput(args[++index]);
                    }
                        
                    if (args.Length > index + 1)
                    {
                        for (int index2 = index + 1;index2 < args.Length;index2++)
                        {
                            //9. Optional - add the child elements to the parent 
                            xi.AddNewChild(args[index2], args[index2 + 1]);
                            index2++;
                        }
                    }

                    // Only use this for debugging purpose
                    if (showXML > 0)
                    {
                        if (showXML == 1)
                        {
                            Console.WriteLine("INPUT:\n" + xi.ToPrettyString(""));
                        }
                        else
                        {
                            s.DebugStyle = NaServer.DEBUG_STYLE.PRINT_PARSE;
                        }
                    }
                }
                catch (System.IndexOutOfRangeException)
                {
                    throw new NaApiFailedException("Mismatch in arguments passed "
                                                       + "(in (key,value) Pair) to "
                                                       + "Ontapi API", -1);
                }
                
                // 10. Invoke a single ONTAPI API to the server.
                // The response is stored in xo.
                xo = s.InvokeElem(xi);

                // Only use this for debugging purpose
               if (showXML > 0)
                {
                    if (showXML == 2)
                    {
                        //Simply return because the NaServer will print the raw XML OUTPUT.
                        return;
                    }
                    Console.WriteLine("OUTPUT:");
                }                    

                //11. Print the ONTAPI API response that is returned by the server
                Console.WriteLine(xo.ToPrettyString(""));
            }
            catch (NaApiFailedException e)
            {
                Console.Error.WriteLine(e.Message);
            }
            catch(Exception e)
            {
                Console.Error.WriteLine(e.Message);
            }
        }
    }
}
