#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "netapp_api.h"

#define shift() memmove(argv + 1, argv + 2, sizeof(*argv) * --argc)

void Usage(const char *name) {
	printf("Usage: %s -H <filer> -u <user> -p <password> [-P port] [-s] [-S] [-n min] [-l lagwarn] [-L lagcrit]\n", name);
	printf("       -H filer: Hostname or IP Address of the NetApp filer.\n");
	printf("       -u user: username for login. DO NOT USE \"root\" HERE!\n");
	printf("       -p password: Pasword for login.\n");
	printf("       -S: Use https (Default).\n");
	printf("       -s: Use http. Not recommended.\n");
	printf("       -P port: Port to connect to. Default: 443 for https, 80 for http.\n");
	printf("       -n min: At least <min> snapmirror must exist, otherwise the check will be critical.\n");
	printf("       -l lagwarn: Number of seconds for lag before the service state will change to warning.\n");
	printf("       -L lagcrit: Number of seconds for lag before the service state will change to critical.\n");
	exit(3);
}

na_server_t *init(na_server_transport_t prot, const char *host, int port, const char *user, const char *passwd) {
	char err[256];
	na_server_t *s;

	if (!na_startup(err, sizeof(err))) {
		printf("Critical: Error initializing NetApp API: %s\n", err);
		return NULL;
	}

	s = na_server_open(host, 1, 1); 

	na_server_set_transport_type(s, prot, 0);
	na_server_set_port(s, port);
	na_server_style(s, NA_STYLE_LOGIN_PASSWORD);
	na_server_adminuser(s, user, passwd);

	return s;
}

int main(int argc, char* argv[]) {
	na_server_transport_t prot = NA_SERVER_TRANSPORT_HTTPS;
	const char *host = NULL, *user = NULL, *passwd = NULL;
	int port = 0, min_num = 0, num = 0, num_ok = 0, num_warn = 0, num_crit = 0; 
	na_server_t *s;
	char long_output[10000] = {0};
	uint64_t lag = 0, lag_warn = UINT64_MAX, lag_crit = UINT64_MAX;
	
	while (argc > 1) {
		if (argc > 2 && !strcmp(argv[1], "-P")) {
			port = atoi(argv[2]);
			shift();
		} else if (argc > 2 && !strcmp(argv[1], "-H")) {
			host = argv[2];
			shift();
		} else if (argc > 2 && !strcmp(argv[1], "-s")) {
			prot = NA_SERVER_TRANSPORT_HTTP;
		} else if (argc > 2 && !strcmp(argv[1], "-S")) {
			prot = NA_SERVER_TRANSPORT_HTTPS;
		} else if (argc > 2 && !strcmp(argv[1], "-u")) {
			user = argv[2];
			shift();
		} else if (argc > 2 && !strcmp(argv[1], "-p")) {
			passwd = argv[2];
			shift();
		} else if (argc > 2 && !strcmp(argv[1], "-l")) {
			lag_warn = atoi(argv[2]);
			shift();
		} else if (argc > 2 && !strcmp(argv[1], "-L")) {
			lag_crit = atoi(argv[2]);
			shift();
		} else if (argc > 2 && !strcmp(argv[1], "-n")) {
			min_num = atoi(argv[2]);
			shift();
		} else {
			printf("Unknown option \"%s\".\n", argv[1]);
			Usage(argv[0]);
		}
		shift();
	}
	
	if (port < 1 || port > 65535) {
		if (prot == NA_SERVER_TRANSPORT_HTTP) port = 80;
		else port = 443;
	}
	if (!host || !user || !passwd) Usage(argv[0]);

	s = init(prot, host, port, user, passwd);
	if (!s) return 2;

	const char *available;
	na_elem_t *out, *e;
	na_elem_iter_t iter;
	
	out = na_server_invoke(s, "snapmirror-get-status", (void *) 0);
	if (na_results_status(out) != NA_OK) {
		printf("Crirical: Can't query netapp. Error %d: %s\n", na_results_errno(out), na_results_reason(out));
		return 2;
	}
	available = na_child_get_string(out, "is-available");
	if (strcmp(available, "true")) {
		printf("Critical: Snapmirror not available.\n");
		return 2;
	}
	iter = na_child_iterator(na_elem_child(out, "snapmirror-status"));
	for (e = na_iterator_next(&iter); e != NULL; e = na_iterator_next(&iter)) {
		int len = strlen(long_output);
		const char *name = na_child_get_string(e, "source-location");
		
		if (index(name, ':')) name = index(name, ':') + 1;
		lag = na_child_get_uint64(e, "lag-time", UINT64_MAX);
		if (lag > lag_crit) {
			if (len < 8191) snprintf(long_output + len, 8192 - len, "%s critical: %d seconds lag.\n", name, lag);
			++num_crit;
		} else if (lag > lag_warn) {
			if (len < 8191) snprintf(long_output + len, 8192 - len, "%s warning: %d seconds lag.\n", name, lag);
			++num_warn;
		} else {
			if (len < 8191) snprintf(long_output + len, 8192 - len, "%s ok: %d seconds lag.\n", name, lag);
			++num_ok;
		}
		++num;
	}

	if (num < min_num) {
		printf("Critical: Found only %d snapmirrors expected at least %d.\n%s", num, min_num, long_output);
		return 2;
	}
	if (num_crit) {
		printf("Critical: Found %d snapmirror, %d critical, %d warning, %d ok.\n%s", num, num_crit, num_warn, num_ok, long_output);
		return 2;
	}
	if (num_warn) {
		printf("Warning: Found %d snapmirror, %d warning, %d ok.\n%s", num, num_warn, num_ok, long_output);
		return 1;
	}
	printf("Ok: Found %d snapmirrors.\n%s", num, long_output);
	return 0;
}
