#!/usr/bin/perl

use strict;
use Getopt::Long qw(GetOptions :config no_ignore_case);
use LWP::Simple;
use lib "/usr/lib/nagios/plugins";


# per http auf einen collectd-Host connecten,
# check-rrd-fileage?$hostname aufrufen und die Ausgabe mit den Parametern 
# verwursten und einen gültigen Nagios-Status daraus erzeugen
# written by spfab@noris.net

my $collectd_server;
my $host;
my $warning;
my $critical;

my $cgi_script = 'check-rrd-fileage';


GetOptions('server=s'      => \$collectd_server,
	   'name=s'        => \$host,
           'warning=i'     => \$warning,
	   'critical=i'    => \$critical,
	   'help|?'        =>
	      sub { exec perldoc => -F => $0 or die "exec('perldoc -F $0'): $!\n" });


if (!$warning || $warning < 0 || !$critical || $critical < 0)
{
	die "wrong parameter";
}

if (!$collectd_server || $collectd_server eq '')
{
	die "no collectd-server provided";
}

if (!$host || $host eq '')
{
	die "no host to check";
}

my $url = "http://" . $collectd_server . "/cgi-bin/" . $cgi_script . "?" . $host;
my $oldest = get($url) || die "cannot get $url\n";

my $now = time();

my $exit_value = 0;
my $status_text = 'OK: rrd file age is ' . ($now - $oldest) . 'sec';

if (($now - $oldest) > $warning)
{
	$status_text = "WARNING: oldest rrd-file is " . ($now - $oldest) . "sec old";
        $exit_value = 1;
}

if ($now - $oldest > $critical)
{
	$status_text = "CRITICAL: oldest rrd-file is " . ($now - $oldest) . "sec old";
        $exit_value = 2;
}

if ($now - $oldest < 0)
{
        $status_text = "WARNING: no rrd-files found";
        $exit_value = 1;
}

print $status_text; 
exit $exit_value;

__END__

=head1 NAME

check_collectd_rrd_age.pl - ueberprueft das Alter von rrd-Dateien auf collectd-Servern

=head1 SYNOPSE

check_collectd_rrd_age.pl --server collect1.noris.net --name www1.noris.net --warning 600 --critical 1200

ueberprueft das Alter der aeltesten rrd-Datei fuer den Host www1.noris.net auf dem collectd-Server collect1.noris.net
und gibt bei einem Alter groesser 600 Sekunden den Status "WARNING" und bei einem Alter groesser 1200 Sekunden
den Status "CRITICAL" zurueck.

=head1 BESCHREIBUNG

Dieses Skript funktioniert nur in Verbindung mit dem cgi-Skript namens "check-rrd-fileage", welches 
derzeit auf collect1/2 laeuft.
Dieses macht auf folgende Art und Weise ein Stat auf alle rrd-Dateien eines Hosts:
 stat -c '%Y' $BASEDIR/*/*.rrd  2>/dev/null | awk 'BEGIN { foo=9999999999 } { if ($1 < foo) (foo = $1) } END { print foo }'
wobei $BASEDIR /var/lib/collectd/rrd/$hostname entspricht.
Es gibt die mtime als unix-timestamp der ältesten Datei zurueck.
=cut
