package noris::ManageUsers::Cyrus;

use strict;
use utf8;
use Cyrus::IMAP;           # depends: libcyrus-imap-perl22
use Cyrus::IMAP::Admin;    # depends: libcyrus-imap-perl22
use Data::Dump qw(pp);     # depends: libdata-dump-perl
use Log::Log4perl qw(get_logger :levels); # depends: liblog-log4perl-perl

use open IO => ':locale';

use Moose;                         # depends libmoose-perl
use Moose::Util::TypeConstraints;  # depends libmoose-perl

subtype 'SubfoldArr' => as 'ArrayRef[Str]';
coerce  'SubfoldArr' => from 'Str' => via { [ split /,/ ] };

has 'username'  => ( is => 'ro', isa => 'Str', required => 1 );
has 'password'  => ( is => 'ro', isa => 'Str', required => 1 );
has 'host'      => ( is => 'ro', isa => 'Str', required => 1 );
has 'separator' => ( is => 'ro', isa => 'Str', default  => '.' );
has 'subfolders' => (
    is         => 'ro',
    isa        => 'SubfoldArr',
    default    => sub { [] },
    auto_deref => 1,
    coerce     => 1,
);
has _handle   => ( is => 'rw', isa => 'Cyrus::IMAP::Admin' );

my $logger;

sub BUILD {
    my $self = shift;

    $logger = get_logger(__PACKAGE__);

    $self->_handle( $self->_get_handle() );

    unless ( $self->_handle ) {
        $logger->logdie(
            "Can't connect to IMAP Server Host " . $self->hostname );
    }

    $logger->debug( "Connected to Cyrus as " . $self->username );
}

sub _get_handle {
    my $self = shift;

    my $client = Cyrus::IMAP::Admin->new( $self->host )
      or $logger->logdie(
        "Can't establish IMAP connection to Server " . $self->host );

    $client->authenticate(
        -mechanism => 'login',
        -user      => $self->username,
        -password  => $self->password,
      )
      or $logger->logdie(
        'IMAP Authentification failed as User ' . $self->username );

    if ( defined( my $error = $client->error() ) ) {
        $logger->logdie("Unexpected IMAP error: $error");
    }

    return $client;
}

# Get User List from Cyrus
sub list_users {
    my $self = shift;

    $logger->debug("List Users called");

    my $handle    = $self->_handle;
    my $separator = $self->separator;

    my @user_mailboxes = $handle->list( '%', "user$separator" );
    if ( defined( my $error = $handle->error ) ) {
        $logger->logdie("Error getting IMAP userlist: $error");
    }

    my $re_user_prefix = qr/^user\Q$separator/;
    @user_mailboxes = map {
        ( my $username = $_->[0] ) =~ s/$re_user_prefix//;
        $username;
    } @user_mailboxes;

    $logger->debug("List Users: User Mailboxes: " . pp(@user_mailboxes) );
    return @user_mailboxes;
}

# Get Sub-Mailboxes for a specific User
sub _list_user_mailboxes {
    my ( $self, $user ) = @_;

    $logger->debug("List User Mailboxes called with user '$user'");

    my $handle    = $self->_handle;
    my $separator = $self->separator;

    my @user_mailboxes = $handle->list( '%', "user$separator$user$separator" );
    if ( defined( my $error = $handle->error ) ) {
        $logger->logdie("Error getting Mailbox List for User '$user': $error");
    }

    @user_mailboxes = map $_->[0], @user_mailboxes;

    $logger->debug( "List User Mailboxes: Subfolders of User '$user': "
          . pp(@user_mailboxes) );
    return @user_mailboxes;
}

# Create a new User
sub add_user {
    my ( $self, $user ) = @_;

    $logger->debug("Add User called with user '$user'");

    my $handle    = $self->_handle;
    my $separator = $self->separator;
    my $mailbox   = "user$separator$user";

    $logger->debug("Trying to create Mailbox '$mailbox'");
    $handle->create($mailbox)
      or $logger->error( "Error creating mailbox '$mailbox': " . $handle->error );

    $logger->info("Mailbox $mailbox created!");

    $logger->debug( 'Trying to set ACL for User '
          . $self->username
          . " for Mailbox: '$mailbox'" );
    $handle->setacl( $mailbox, $self->username => 'all' )
      or $logger->error( 'Add User: Error setting ACL for User '
          . $self->username
          . "for mailbox $mailbox: "
          . $handle->error );

    my $errors = 0;
    foreach my $folder ( $self->subfolders ) {
        unless ( $handle->create( $mailbox . $separator . $folder ) ) {
            $logger->error(
                "Add User: Error creating folder '$folder' for mailbox '$mailbox'");
            $errors++;
        }
        $logger->info("Folder '$mailbox$separator$folder' for Mailbox '$mailbox' created!");
    }

    return !$errors;
}

# Delete an existing User
sub delete_user {
    my ( $self, $user ) = @_;

    $logger->debug("Delete User called with user '$user'");

    my $errors  = 0;
    my $mailbox = 'user' . $self->separator . $user;
    $logger->debug("Trying to Delete Mailbox '$mailbox' for User '$user'");
    $self->_delete_mailbox( 'user' . $self->separator . $user ) or $errors++;

    # Are there any Subfolders to delete?
    # Check it now. Because (IMAP-Server dependendant config)
    # they get delete with the main Mailbox
    foreach my $dm ( $self->_list_user_mailboxes($user) ) {
        $logger->debug("Trying to Delete Mailbox '$dm' for User '$user'");
        $self->_delete_mailbox($dm) or $errors++;
    }

    $logger->info("User '$user' deleted!");
    return !$errors;
}

# Delete an existing Mailbox
sub _delete_mailbox {
    my ( $self, $mailbox ) = @_;

    $logger->debug("Delete Mailbox called with mailbox '$mailbox'");

    my $handle    = $self->_handle;
    my $separator = $self->separator;

    $logger->debug( 'Trying to set ACL for User '
          . $self->username
          . " form mailbox '$mailbox'" );
    $handle->setacl( $mailbox, $self->username, 'all' )
      or $logger->error( __LINE__
          . ": Error setting ACL for mailbox '$mailbox'"
          . $handle->error );

    $logger->debug("Trying to delete mailbox '$mailbox'");
    unless ( $handle->deletemailbox($mailbox) ) {
        $logger->error(
            "Cannot delete mailbox '$mailbox' : " . $handle->error );
        return 0;
    }
    $logger->info("Mailbox '$mailbox' deleted!");
    return 1;
}

no Moose;
__PACKAGE__->meta->make_immutable;

1;
__END__

=head1 NAME

noris::ManageUsers::Cyrus - Cyrus Modul für manage-users

=head1 BESCHREIBUNG

Dieses Modul ist Teil des manage-users Pakets und ist zuständig für die 
Administration der Benutzer in einem Cyrus-Server.

=head1 OPTIONEN

=over 4

=item username String

Der Benutzername mit dem sich das Modul am Cyrus Serven anmelden soll.

=item password String

Das Passwort mit dem sich das Modul am Cyrus Serven anmelden soll.

=item host String

Zu welchem Server soll sich das Skript connecten (z.B. localhost)

=item separator String

Über diesen Parameter kann das Trennzeichen zwischen Bentuzer, Mailboxnamen
und Unterverzeichnissen angegeben werden. 

    Siehe auch Parameter (unixhierarchysep: yes|no) in der Cyrus.conf 

Default: C<.>

=item subfolder String

Hier werden die Unterverzeichnisse angegeben die das Skript, beim erzeugen
eines neuen Postfachs, automatisch anlegen soll.

Beispiel: C<'subfolder =E<gt> 'Drafts,Sent,Trash''>

Default: C<''>

=back

=head1 PUBLIC METHODEN

=over 4

=item new()

Diese Funktion wird als erstes aufgerufen und der werden alle notwendigen
Parameter übergeben.

    use Noris::ManageUsers::Cyrus;

    my %BackendArd = ('username'   => 'cyradm',
                      'password'   => '123456',
                      'host'       => 'localhost',
                      'subfolders' => 'Drafts,Sent,Trash',
                     );
    my $BackendObject = Noris::ManageUsers::Cyrus->new(%BackendArg);

=item list_users()

list_users liefert eine Liste allen vorhandenen Usern im Cyrus Server zurück.

    my @user_list = $BackendObject->list_users()

=item add_user()

add_user wird der Username des zu generierenden Benutzers übergeben.

Das Modul generiert gleichzeitig alle Subfolder die im Parameter
subfolders übergeben wurden und gibt dem User mit dem man sich angemeldet
hat alle Rechte.

    $BackendObject->add_user('username');

=item delete_user()

delete_user wird der Username des zu löschenden Benutzers übergeben.
Das Modul löscht gleichzeitig auch alle Subfolder des jeweiligen
Benutzers.

    $BackendObject->delete_user('username');

=back

=head1 PRIVATE METHODEN

=over 4

=item BUILD

Mosse Framework Function.

=item _get_handle()

    my $Cyrus_Handle = $Self->_get_handle('username');

_get_handle verbindet sich zum Server und gibt als Rückgabewert das Handle
zum Server.

=item _list_user_mailboxes()

Listet alle Sub-Mailboxen (Verzeichnisse) eines Benutzers.

    my @sub_mailboxes = $Self->_list_user_mailboxes('username');

=item _delete_mailbox()

Löscht eine Mailbox oder eine Sub-Mailbox

    $Self->_delete_mailbox('user.foo');

oder

    $Self->_delete_mailbox('user/foo');

=back

=head1 AUTOR

 Stelios Gikas <stelios.gikas@noris.net>

