#!/usr/bin/perl
###############################################################################

#  ----------------------------------------------------------------------------
#
#  Copyright (C) 2002-2004, Eleven GmbH, Berlin
#
#  These coded instructions, statements, and computer programs contain
#  unpublished proprietary information of Eleven GmbH Berlin, and are copy 
#  protected by law. They may not be disclosed to third parties or copied 
#  or duplicated in any form, in whole or in part, without the specific, 
#  prior written permission of Eleven GmbH Berlin.
#
#  ----------------------------------------------------------------------------
#         \file: expurgateChecker.pl
#      $RCSfile: expurgateChecker.pl,v $ 
#
#     $Revision: 1.8 $
#
#        Author: Benjamin Pannier <bp@eleven.de>
#
#   Description: Script that searches for expurgate errors in a given logfile.
#
#      Commends: 
#  ----------------------------------------------------------------------------

###############################################################################

use strict;
use Socket;
use IO::Pipe;
use Net::SMTP;
use IO::Handle;
use IPC::Open3;
use Getopt::Std;
use Fcntl ':flock';
use POSIX ":sys_wait_h";

###############################################################################

my $installpath = "/usr/local/eleven/"; # where is expurgate installed to

my $expurgateLogFile = "/tmp/expurgate.log"; # syslog or expurgate file

my $interval = 2; # search every two seconds for changes in the logfile
my $analyseInterval = 60; # do statistic analyse every x seconds
my $errorInterval   = 120; # analyse errors no more than every x seconds

my $maxErrorAnalysesPerMail = 4; # how much reports should we include in a mail

my $sendErrorMailsEveryXSeconds = 2 * 60 * 60;   # every 2 hours send an error mail
my $sendStatisticMailsEveryXSeconds = 6 * 60 * 60; # every 6 hours send a statistic mail, not implemented yet

my $sendMaxPlainErrorsTotal = 40; # send max 40 complete error lines
my $sendMaxPlainErrorsType = 10; # send max 5 errors of a type

###############################################################################

# get some informations, configuration continues in next part

my $hostname = `hostname 2>&1`;
chomp $hostname;
if ( $? )
{
    die "ERROR: Can not get hostname: $hostname";
}
my $domain = `domainname 2>&1`;
chomp $domain;
if ( $? )
{
    die "ERROR: Can not get domainname: $domain";
}
if ( ( $domain =~ /[\(\[\]\)]+/ || $domain =~ /^\s*$/ ) && $hostname !~ /\./ )
{
    $domain = "domainname-is-empty.net";
}

my $whoami = `whoami 2>&1`;
chomp $whoami;
if ( $? )
{
    die "ERROR: Can not get username: $whoami";
}

if ( length( $domain ) )
{
    $hostname .= ".$domain";
}

###############################################################################

my $debug = 0;

my $sendSupportMailOnly = 0;
my $ticketNr;
my $commend;

my %options = ();
getopts("c:d:hi:l:nst:",\%options);

if ( defined $options{i} )
{
    $installpath = $options{i};
}

my $mailFrom = "$whoami\@$hostname"; # which mail from should we use to send emails

# where to send error reports
my %sendErrorMailsTo = (
    # Format:
    # "emailadr" => [ "receivingSMTPHost", "optional Publickeyfile" ],
    # If  a key file is given, the destination email address must be the same in the key file.
    
    "support\@eleven.de" => [ "mxa.expurgate.net", 
                              "$installpath/etc/ElevenSupport-publickey.asc" ],
    #"customer\@test.test" => [ "mailhostname.domain.test", "" ],
);

my %sendAnalyserMailsTo = (
    #"support-analyse\@eleven.de" => "mxa.expurgate.net",
    #"customer\@test.test" => "mailhostname.domain.test",
);

# better do not change, the code depends on this output
my $fullProcessListExe = "ps -aewwwo pid,pgid,ppid,euser,cputime,%cpu,stat,state,vsize,rss,pagein,%mem,tty,args";

# which commands will be executed if an error occures
my @errorAnalyseStatements = (
    $fullProcessListExe, # must be the first entry in this array
    "uname -a",
    "uptime",
    "$installpath/bin/expurgate -v",
    "grep -v '<!--' $installpath/etc/expurgate.xml | grep '<'",
    "cat /etc/init.d/eXpurgate",
    "cat /etc/init.d/expurgate",
    "ls -l $installpath/spool",
    "df -kal",
    "/usr/sbin/traceroute -n -w 2 exa.expurgate.net",
    "/sbin/traceroute -n -w 2 exa.expurgate.net",
    "ipcs",
    "/usr/sbin/ifconfig -a",
    "/sbin/ifconfig -a",
    "vmstat", 
    
    # solaris
    "vmstat -p",
    "vmstat -s",
    "netstat -nav",
    "ps -Aefo user,ruser,group,pid,ppid,pcpu,vsz,time,comm,args",
    "ps -auxelw",
    "top -n 50 -C -d 1 -I",
    
    # linux
    "free -bt",
    "netstat -nalep",
    "lsof | grep -i expurgate",
    "lsof | grep -i perl",
    # take your time
    "top -n 2 -C -d 1",
    
    # get the statistics again
    "vmstat -s",
    
);

# which commands will be executed once at startup
my @startupAnalyseStatements = (
    @errorAnalyseStatements, # must be the first entry in this array
    "sysctl -a",
);

# how should we check an expurgate server
my $expurgateAnalysisBase = "$installpath/bin/expurgate -f $installpath/etc/expurgate.xml -W ";
#my $expurgateAnalysisBase = "LD_ASSUME_KERNEL=2.4.19 $installpath/bin/expurgate -f $installpath/etc/expurgate.xml -W ";

# which server should we check
my @expurgateHosts = (
    &getIPAddresses( "exa.expurgate.net", ":55555", 1 ),
    &getIPAddresses( "exb.expurgate.net", ":55555", 1 ),
    &getIPAddresses( "exc.expurgate.net", ":55555", 0 ),
    &getIPAddresses( "exd.expurgate.net", ":55555", 0 ),
    &getIPAddresses( "exa-us.expurgate.net", ":55555", 0 ),
    &getIPAddresses( "exb-us.expurgate.net", ":55555", 0 ),
);

# which words should not occure in expurgate output
my $errorMatchRegex = "can not|timed out|error|lost|failed|warning";

###############################################################################

my $lockFile = "/tmp/expurgateTester.lock.$<";
my $statusDatabase = "/tmp/expurgateTester.db.$<";
my $errorMailTmpFile = "/tmp/expurgateTester.errmail.$<";
my $startupMailTmpFile = "/tmp/expurgateTester.startmail.$<";

###############################################################################

my $running = 1;
my $errorReportCounter = 1;

my $nextStatisticAnalyse = 0;
my $nextErrorAnalyse = 0;
my $errorAnalysesForNextMailLeft = $maxErrorAnalysesPerMail;
my $sendNextErrorMail = time + $sendErrorMailsEveryXSeconds;
my $sendNextStatisticMail = time + $sendStatisticMailsEveryXSeconds;

local *LOGFILE;

if ( defined $options{h} )
{
    print "Usage: $0 <options>\n";
    print "Options:\n";
    print "-h                    - this help text.\n";
    print "-c <commend>          - gives a short description why you using this script.\n";
    print "-d <debuglevel>       - sets the debug level of this script.\n";
    print "-i <installPath>      - the installpath of the eXpurgate installation.\n";
    print "-l <expurgatLogFile>  - the path to the eXpurgate logfile.\n";
    print "-n                    - do not encrypt any message.\n";
    print "-s                    - do not run this script in a loop, send exactly one report.\n";
    print "-t <ticket#>          - if used with '-s', it generates a report to an open support ticket.\n";
    exit 0;
}

if ( defined $options{d} )
{
    $debug = $options{d};
}

if ( defined $options{c} )
{
    $commend = $options{c};
}

if ( defined $options{l} )
{
    $expurgateLogFile = $options{l};
}

if ( defined $options{t} )
{
    $ticketNr = $options{t};
    
    if ( $ticketNr !~ /^[0-9]+$/ )
    {
        die "ERROR: invalid ticket nr!";
    }
}

if ( defined $options{s} )
{
    $sendSupportMailOnly = 1;
    
    while ( $commend =~ /^\s*$/ )
    {
        print "Please give a short reason for sending this report: ";
        $commend = <STDIN>;
        chomp $commend;
    }
    
    %sendErrorMailsTo = (
            "support\@eleven.de" => [ "mxa.expurgate.net", 
                                      "$installpath/etc/ElevenSupport-publickey.asc" ],
    );
}

if ( defined $options{n} )
{
    my $rcptTo;

    foreach $rcptTo ( keys %sendErrorMailsTo )
    {
         $sendErrorMailsTo{ $rcptTo }->[1] = "";
    }
}

my $gpgPath = "";

###############################################################################
###############################################################################
###############################################################################
###############################################################################
###############################################################################

{
    # verify files
    
    if ( ! open( LOCKFILE, ">>$lockFile" ) )
    {
        die "ERROR: Can not create or open lock file ($lockFile).";
    }
    flock( LOCKFILE, LOCK_EX ) || die "ERROR: Can not lock test file ($lockFile), is test script already running ?";
    
    # can we open the logfile
    my ( $ret, $errStr ) = &getLogfile( *LOGFILE, $expurgateLogFile );
    if ( ! $ret )
    {
        die "ERROR: $errStr";
    }
    close( LOGFILE );
    
    if ( ! open( TMPFILE, ">>$errorMailTmpFile" ) )
    {
        die "ERROR: Can not open tmp mail file ($errorMailTmpFile).";
    }
    close( TMPFILE );
    
    if ( ! -e "$installpath/bin/expurgate" )
    {
        die "ERROR: expurgate binary not found in $installpath/bin";
    }
}

{
    # verify encryption
    my %doEncryptingFor = ();
    
    foreach ( keys %sendErrorMailsTo )
    {
        if ( $sendErrorMailsTo{ $_ }->[1] ne "" )
        { 
            if ( ! -r $sendErrorMailsTo{ $_ }->[1] )
            {
                die "ERROR: Can not read public key file '" . $sendErrorMailsTo{ $_ }->[1]
                    . "' for email address: '$_'";
            }
            
            $doEncryptingFor{ $_ } = $sendErrorMailsTo{ $_ }->[1];
        }
    }
    
    if ( scalar keys %doEncryptingFor > 0 )
    {
        &verifyPKeys( \%doEncryptingFor );
    }
}

{
    # send startup mail
    my ( $ret, $errStr ) = &generateMailBody( $startupMailTmpFile, 1, \@startupAnalyseStatements );
    
    if ( ! $ret )
    {
        die "ERROR: Can not generate startup mail: $errStr";
    }
    
    my $mailContent;
    
    if ( $commend ne "" )
    {
        $mailContent = "Report[$whoami\@$hostname]\nCommend: $commend\n\n";
    }
    $mailContent .= (stat($startupMailTmpFile))[7] . "\n". `cat $startupMailTmpFile`;
    $mailContent .= "\nEnvironment:\n-------------------------------------------------------------------------------\n";
    
    foreach ( keys %ENV )
    {
        $mailContent .= "-$_ -> '$ENV{$_}'\n";
    }
    
    if ( ! $sendSupportMailOnly )
    {
        $mailContent .= "\nEarliest next mail: " . localtime( $sendNextErrorMail );
    }

    my $rcptTo;
    
    my $subject = "Startup Report";
    
    if ( $sendSupportMailOnly )
    {
        if ( $ticketNr ne "" )
        {
            $subject = "[Ticket#: $ticketNr] [$whoami] $commend";
        }
        else
        {
            $subject = "[$whoami]: $commend";
        }
    }
    
    foreach $rcptTo ( keys %sendErrorMailsTo )
    {
        if ( $sendErrorMailsTo{ $rcptTo }->[1] ne "" )
        {
            ( $ret, $errStr ) = &encryptContent( \$mailContent, $rcptTo );

            if ( ! $ret )
            {   
                die "ERROR: $errStr\n";
            }
        }
            
        ( $ret, $errStr ) = &sendMail( $rcptTo, 
                                       $sendErrorMailsTo{ $rcptTo }->[0], 
                                       $subject,
                                       \$mailContent );
        
        if ( ! $ret )
        {
            die "ERROR: $errStr\n";
        }
    }
    
    unlink $startupMailTmpFile;
}

if ( $sendSupportMailOnly )
{
    exit 0;
}

print "Start...\n";

do
{
    &debug( 8, "Start queue run." );
        
    my ( $ret, $errStr ) = &getLogfile( *LOGFILE, $expurgateLogFile );
    if ( ! $ret )
    {
        print "ERROR: $errStr\n";
    }
    else
    {
        close( LOGFILE );
        my $errCounter;
        my $errCounterTotal;
        
        ( $ret, $errStr, $errCounter, $errCounterTotal ) = 
                            &analyseLogfile( *LOGFILE, 
                                            $expurgateLogFile );
                                            
        if ( ! $ret )
        {
            print "ERROR: $errStr\n";
        }
        else
        {
            if ( $errCounter > 0 && 
                 $nextErrorAnalyse < time &&
                 $errorAnalysesForNextMailLeft > 0 )
            {
                # new errors detected
                ( $ret, $errStr ) = &generateMailBody( $errorMailTmpFile, 1, \@errorAnalyseStatements );
                
                $nextErrorAnalyse = time + $errorInterval;
                $errorAnalysesForNextMailLeft--;
                
                if ( ! $ret )
                {
                    print "ERROR: $errStr\n";
                }
            }
            else
            {
                &debug( 5, "NoErrorTest: $errCounter, $nextErrorAnalyse < ". time . ", $errorAnalysesForNextMailLeft" );
            }
        }
    }
    
    if ( $nextStatisticAnalyse < time )
    {
        # do analysing
        
        $nextStatisticAnalyse = time + $analyseInterval;
    }
    
    if ( $sendNextErrorMail < time && -r $errorMailTmpFile )
    {
        # send error mail now
        
        my %statusdb;
        
        if ( ! dbmopen( %statusdb, "$statusDatabase",0666) )
        {
            print "ERROR: Can not open db ($statusDatabase)\n";
        }
        
        my $errorReport = "\n-------------------------------------------------------------------------------\n";
        $errorReport .= "Count: " . $statusdb{"errorCount_${expurgateLogFile}"} . "\n\n";
        
        $statusdb{"errorCount_${expurgateLogFile}"} = 0;
        
        my $i = 1;
        foreach ( grep( /^errorstr_/, keys %statusdb ) )
        {
            s/^errorstr_//;
            $errorReport .= "$i: $_ -> " . $statusdb{"errorstr_$_"} . "\n";
            delete $statusdb{"errorstr_$_"};
            $i++;
        }
        
        $errorReport .= "\n----\n\n";
        
        for ( $i=0; $i < $statusdb{ "errorLinecounter" }; $i++ )
        {
            $errorReport .= $statusdb{ "errorLine_$i" } . "\n";
            delete $statusdb{ "errorLine_$i" };
        }
        
        $statusdb{ "errorLinecounter" } = 0;
        
        dbmclose( %statusdb );
        
        my $mailContent = (stat($errorMailTmpFile))[7] . "\n". `cat $errorMailTmpFile`;
        $mailContent .= $errorReport;
        
        $sendNextErrorMail = time + $sendErrorMailsEveryXSeconds;
        $mailContent .= "\nEarliest next mail: " . localtime( $sendNextErrorMail );

        my $rcptTo;
        
        foreach $rcptTo ( keys %sendErrorMailsTo )
        {
            if ( $sendErrorMailsTo{ $rcptTo }->[1] ne "" )
            {
                ( $ret, $errStr ) = &encryptContent( \$mailContent, $rcptTo );

                if ( ! $ret )
                {   
                    print "ERROR: $errStr\n";
                }
            }
            
            ( $ret, $errStr ) = &sendMail( $rcptTo, 
                                           $sendErrorMailsTo{ $rcptTo }->[0], 
                                           "Error Report $errorReportCounter",
                                           \$mailContent );
            
            if ( ! $ret )
            {
                print "ERROR: $errStr\n";
            }
        }
        
        $sendNextErrorMail = time + $sendErrorMailsEveryXSeconds;
        $errorAnalysesForNextMailLeft = $maxErrorAnalysesPerMail;
        $errorReportCounter++;
        
        unlink $errorMailTmpFile;
    }
    
    if ( $sendNextStatisticMail < time )
    {
        # send statistic mail now
        $sendNextStatisticMail = time + $sendStatisticMailsEveryXSeconds;
    }
    
    &debug( 8, "Stop queue run." );
    sleep $interval;
}
while ( $running );

print "Stop...\n";

exit 0;

###############################################################################
# -----------------------------------------------------------------------------

sub sendMail
{
    my ( $rcptTo, $mailHost, $addSubject, $refMailContent ) = @_;
    
    &debug( 3, "Sendmail: $rcptTo, $mailHost, $addSubject" );
    
    my $smtp = Net::SMTP->new("$mailHost");
    
    if ( ! $smtp )
    {
        $smtp = Net::SMTP->new("$mailHost");

        if ( ! $smtp )
        {
            return ( 0, "Can not get smtp object for host ($mailHost): $? $!" );
        }
    }
    
    if ( ! $smtp->mail( $mailFrom ) )
    {
        return( 0, "Can not set mail-from ($mailFrom): $? $!" );
    }
    
    if ( ! $smtp->to( $rcptTo ) )
    {
        return( 0, "Can not set rcpt-to ($rcptTo): $? $!" );
    }

    if ( ! $smtp->data() )
    {
        return( 0, "Can not say data: $? $!" );
    }

    if ( ! $smtp->datasend(
"To: $rcptTo
From: $mailFrom
Subject: eXchecker[$hostname], $addSubject\n\n") )
    {
        return( 0, "Can not send header: $? $!" );
    }
    
    if ( ! $smtp->datasend( $$refMailContent ) )
    {
        return( 0, "Can not send body: $? $!" );
    }

    if ( ! $smtp->dataend() )
    {
        return( 0, "Can not finish data: $? $!" );
    }

    if ( ! $smtp->quit() )
    {
        return( 0, "Can not send quit: $? $!" );
    }
    
    &debug( 2, "Mail sent: $rcptTo, $mailHost, $addSubject" );

    return ( 1, "" );
}

# -----------------------------------------------------------------------------

sub generateMailBody
{
    my ( $filename, $runExpurgateServerTest, $refCommands ) = @_;
    
    if ( ! open( TMPFILE, ">>$filename" ) )
    {
        return ( 0, "Can not open tmp mail file ($filename)." );
    }
    
    print TMPFILE "###############################################################################\n";
    print TMPFILE scalar( localtime() ) . "\n\n";
    
    my ( $ret, $errStr, $output );
    
    if ( $runExpurgateServerTest )
    {
        print TMPFILE "eXpurgate Server:\n";
    
        ( $ret, $errStr, $output ) = &testExurgateServer();
    
        print TMPFILE $$output;
        print TMPFILE "\n";
    }
    
    my $counter = 0;
    my $command;
    my @checkPids;
    
    foreach $command ( @$refCommands )
    {
        print TMPFILE "-------------------------------------------------------------------------------\n";
        print TMPFILE "Time: " . localtime() . "\n";
        print TMPFILE "Execute: $command\n";
        $output = `$command 2>&1`;

        my $status = $?;
        print TMPFILE "Status: $status\n";

        &debug( 6, "Executed: '$command' -> $status" );
                
        if ( $counter == 0 )
        {
            my $line;
            foreach $line ( split( /\r?\n/, $output ) )
            {
                &debug( 80, "Check Line: $line" );

                if ( $line =~ /^\s*(\S+)\s+\S+\s+(\S+)\s+\S+\s+([^:]+):([^:]+):([^\s]+)\s+\S+\s+\S+\s+\S+\s+(\S+)\s+.*(\S*bin.expurgate.*)$/i )
                {
                    my $pid = $1;
                    my $parentPid = $2;
                    my $cpuHour = $3;
                    my $cpuMin = $4;
                    my $cpuSec = $5;
                    my $memSize = $6;
                    my $cmdLine = $7;

                    &debug( 8, "Check eXpurgate: $pid $parentPid $cpuHour:$cpuMin:$cpuSec $memSize $cmdLine" );
                    
                    if ( $cpuHour > 0  ||
                         $cpuMin  > 0  ||
                         $cpuSec  > 10 ||
                         $memSize > 50000 )
                    {
                        &debug( 1, "Add berserker: $pid" );
                        push( @checkPids, $pid );
                        print TMPFILE "Berserk Process: $pid $parentPid $cpuHour:$cpuMin:$cpuSec $memSize $cmdLine\n";
                    }
                }
            }
        }
        
        if ( defined $output && $output ne "" )
        {
            $output =~ s/\x{1b}\[J/\n/sg;
            $output =~ s/\x{1b}(\[(K|\d*m|(\d+;\d+)?H))?/ /sg;
            print TMPFILE "Output:\n$output\n";
            
            &debug( 6, "Exe: $command -> $output" );
        }
        
        $counter ++;
    }
    
    my $pid;
    foreach $pid ( @checkPids )
    {
        my $exe = "strace -p $pid 2>&1";
        print TMPFILE "-------------------------------------------------------------------------------\n";
        print TMPFILE "Time: " . localtime() . "\n";
        print TMPFILE "Execute: $exe\n\n";

        &debug( 5, "Trace Berserker: $exe" );

        my $exePid = 0;
        
        if ( ( $exePid = open( OUT, "$exe|" )  ) )
        {
            my $counter = 0;

            my $rin = '';
            vec( $rin, fileno( OUT ), 1 ) = 1;

            my $start = time;

            while ( $counter < 50 && ( time - $start < 6 ) )
            {
                my $rout;
                my $nfound = select( $rout = $rin, undef, undef, 1 );

                if ( $nfound > 0 )
                {
                    my $line = scalar( <OUT> );
                    chomp $line;
               	    print TMPFILE $line . "\n";
                    $counter++;
                }
                elsif ( $nfound < 0 )
                {
                    last;
                }
            }
            kill( 15, $exePid );
            sleep 1;
            close( OUT );
            kill( 18, $pid ); # wake up traced processes again, strace stop them sometimes
        }
        else
        {
            print TMPFILE "ERROR: Can not execute: $?\n";
            &debug( 2, "ERROR: Could not trace berserker: $?" );
        }

        print TMPFILE "\n";
    }
    
    close( TMPFILE );
    
    return ( $ret, $errStr );
}

# -----------------------------------------------------------------------------

sub testExurgateServer
{
    my @pipes;
    
    for ( my $i = 0; $i < scalar( @expurgateHosts ); $i++ )
    {
        push( @pipes, new IO::Pipe );
    }
    
    my $pipe;
        
    for ( my $i = 0; $i < scalar( @expurgateHosts ); $i++ )
    {
        my $pid;
        $pipe = $pipes[ $i ];
        
        if ( ! ( $pid = fork() ) )
        {
            # child
            $pipe->writer();
            
            my $output = `$expurgateAnalysisBase $expurgateHosts[$i] 2>&1`;
            chomp $output;
            
            &debug( 5, "ChildExe: '$expurgateHosts[$i]' -> $output" );
            
            print $pipe  "$output";
            
            &debug( 8, "Child terminates now." );
            
            exit 0;
        }
        
        &debug( 6, "Fork: $i -> $pid" );
        
        $pipe->reader();
    }
    
    my $allOutput;
    
    my $i;
    for ( $i = 0; $i < scalar( @expurgateHosts ); $i++ )
    {
        my $output = "";
                
        &debug( 8, "Parent read from $i" );
        
        $pipe = $pipes[ $i ];
        
        while ( <$pipe> )
        {
            $output .= $_;
        }
        
        close $pipe;
                
        &debug( 7, "ParentOut: $i -> $output" );
        
        $allOutput .= "$expurgateHosts[$i]: $output\n";
    }
    
    my $child;
    while ( ( $child = waitpid( -1, WNOHANG ) ) > 0 ) 
    { 
        &debug( 8, "Child cleared: $child" );
    }
    
    return ( 1, "", \$allOutput );
}

# -----------------------------------------------------------------------------

sub analyseLogfile
{
    local ( *LOGFILE ) = shift( @_ );
    my ( $logfile ) = @_;
    
    my ( $ret, $errStr ) = &getLogfile( *LOGFILE, $logfile );
    
    if ( ! $ret )
    {
        return ( $ret, $errStr );
    }
    
    my %statusdb;
    
    if ( ! dbmopen( %statusdb, "$statusDatabase",0666) )
    {
        return ( 0, "Can not open database ($statusDatabase)");
    }
    
    my $errorCount = 0;
    
    my %errorLinesInThisRun;
    
    while ( <LOGFILE> )
    {
        chomp;
        
        my $errorStr = "";
        
        if ( /eXpurgate.[0-9]+.:?\s*(.*(${errorMatchRegex}).*)$/i )
        {
            $errorStr = $1;
        
            $errorCount++;
            
            if ( defined $statusdb{ "errorstr_" . $errorStr } )
            {
                $statusdb{ "errorstr_" . $errorStr } ++;
            }
            else
            {
                $statusdb{ "errorstr_" . $errorStr } = 1;
            }
            
            if ( ! defined $statusdb{ "errorLinecounter" } )
            {
                $statusdb{ "errorLinecounter" } = 0;
            }
            
            if ( $statusdb{ "errorstr_" . $errorStr } < ( $sendMaxPlainErrorsType + 1 )&&
                 $statusdb{ "errorLinecounter" } < $sendMaxPlainErrorsTotal &&
                 ! defined $errorLinesInThisRun{$errorStr} )
            {
                $statusdb{ "errorLine_" . $statusdb{ "errorLinecounter" } } = $_;
                $statusdb{ "errorLinecounter" }++;
                $errorLinesInThisRun{$errorStr} = 1;
            }
            
            &debug( 10, "Line match: '$_' -> '$errorStr' -> " . $statusdb{ "errorstr_" . $errorStr } );
        }
        else
        {
            &debug( 10, "Line do not match (${errorMatchRegex}): $_" );
        }
    }
    close( LOGFILE );
    
    &debug( 2, "Errors found: $errorCount" );
    
    $statusdb{"size_${logfile}"} = (stat($logfile))[7];
    
    if ( ! defined $statusdb{"errorCount_${logfile}"} )
    {
        $statusdb{"errorCount_${logfile}"} = 0;
    }
    
    $statusdb{"errorCount_${logfile}"} += $errorCount;
    
    my $counterTotal = $statusdb{"size_errorCount"};
    
    dbmclose( %statusdb );
    
    return ( 1, "", $errorCount, $counterTotal );
}

# -----------------------------------------------------------------------------

sub getLogfile
{
    local ( *LOGFILE ) = shift( @_ );
    my ( $logfile ) = @_;
    my %statusdb;
    
    if ( ! dbmopen( %statusdb, "$statusDatabase",0666) )
    {
        return ( 0, "Can not open database ($statusDatabase)");
    }

    my ($dev,$ino,$mode,$nlink,$uid,$gid,$rdev,$size,$atime,$mtime,$ctime,$blksize,$blocks);
    
    if ( ! ( ($dev,$ino,$mode,$nlink,$uid,$gid,$rdev,$size,$atime,$mtime,$ctime,$blksize,$blocks)
              = stat( $logfile ) ) )
    {
        if ( ! open( LOG, ">$logfile" ) )
        {
            dbmclose( %statusdb );
            return ( 0 , "Can not create logfile ($expurgateLogFile).");
        }
        close( LOG );
        
        if ( ! ( ($dev,$ino,$mode,$nlink,$uid,$gid,$rdev,$size,$atime,$mtime,$ctime,$blksize,$blocks)
              = stat( $logfile ) ) )
        {
            dbmclose( %statusdb );
            return ( 0 , "Can not stat logfile ($expurgateLogFile).");
        }
    }
        
    if ( ! open( LOGFILE, "$expurgateLogFile" ) )
    {
        dbmclose( %statusdb );
        return (0, "Can not open logfile ($expurgateLogFile).");
    }
    
    if ( ! defined $statusdb{"size_${logfile}"} )
    {
        $statusdb{"size_${logfile}"} = 0;
    }
    
    if ( $size >= $statusdb{"size_${logfile}"} ) 
    {
        &debug( 3, "Logfile opened, seek to " . $statusdb{"size_${logfile}"} );
        
	    if ( ! seek(LOGFILE, $statusdb{"size_${logfile}"}, 0) )
	    {
	        dbmclose( %statusdb );
	        return (0, "Can not seek in logfile ($expurgateLogFile).");
	    }
    }
    else
    {
        &debug( 9, "Logfile opened, $size - " . $statusdb{"size_${logfile}"} );
    }
    
    dbmclose( %statusdb );
    
    return ( 1, "" );
}

# -----------------------------------------------------------------------------

sub encryptContent
{
    my ( $refMailContent, $receiption ) = @_;

    if ( ! open( TMPFILE, ">$errorMailTmpFile" ) )
    {
        return ( 0, "Can not open tmp file: $errorMailTmpFile" );
    }

    print TMPFILE $$refMailContent;

    close TMPFILE;

    &debug( 4, "Crypt outputfile for '$receiption'" );

    my $out = `gpg --always-trust --no-secmem-warning --yes -ear $receiption $errorMailTmpFile 2>&1`;

    if ( $? )
    {
        return ( 0, "Can not crypt: $? $out" );
    }

    if ( ! open( TMPFILE, "${errorMailTmpFile}.asc" ) )
    {
        return ( 0, "Can not open crypted file: ${errorMailTmpFile}.asc $? $!" );
    }

    $$refMailContent = "";

    while ( <TMPFILE> )
    {
        $$refMailContent .= $_;
    }

    close TMPFILE;

    return ( 1, "" );
}

# -----------------------------------------------------------------------------

sub verifyPKeys
{
    my ( $refDoEncryptingFor ) = @_;
    
    if ( $gpgPath eq "" )
    {
        $gpgPath = `which gpg 2>&1`;
        chomp $gpgPath;
    
        if ( $? )
        {
            die "ERROR: Can not found encryption tool gpg in PATH: $gpgPath";
        }

        `$gpgPath -v`;
        if ( $? )
        {
            die "ERROR: Can not run encryption tool gpg: $gpgPath";
        }
    }
    
    foreach ( keys %$refDoEncryptingFor )
    {
        my $out = `$gpgPath --no-secmem-warning --import $refDoEncryptingFor->{$_} 2>&1`;
        
        if ( $? )
        {
            chomp $out;
            die "ERROR: Can not import key '$refDoEncryptingFor->{$_}' for '$_': $out";
        }
    
        &debug( 9, "PGP keyfile $refDoEncryptingFor->{$_} added: $out" );
    }
}

# -----------------------------------------------------------------------------

sub getIPAddresses
{
    my ( $hostname, $addon, $fatal ) = @_;
    
    my $host;
    my @result = ();

    
    if ( $hostname =~ /^[0-9]+\.[0-9]+\.[0-9]+\.[0-9]+$/ )
    {
        &debug( 6, "Add IP address for host '$hostname': $hostname" );
        return ( "$hostname$addon" );
    }
        

    my ( $name, $aliases, $addrtype, $length, @addrs);

    &debug( 8, "Resolve IP Addresses for: $hostname" );

    if ( ! ( ( $name, $aliases, $addrtype, $length, @addrs ) = 
                gethostbyname( $hostname ) ) )
    {
        if ( $fatal )
        {
            die "ERROR: Can not resolve hostname: $hostname $! $?";
        }
    }
    else
    {
        my $ip;

        foreach $ip ( @addrs )
        {
            &debug( 6, "Add IP address for host '$hostname': " . inet_ntoa( $ip ) );
            push( @result, inet_ntoa( $ip ) . "$addon" )
        }
    }

    if ( scalar @result == 0 && $fatal )
    {
        die "ERROR: Got no result ip addresses for: $hostname";
    }
    
    return @result;
}

# -----------------------------------------------------------------------------

sub debug
{
    my ( $debugLevel, $str ) = @_;
    
    if ( $debugLevel <= $debug )
    {
        print "DEBUG [$$] ($debugLevel/$debug): $str\n";
    }
}

# -----------------------------------------------------------------------------
