# (c) 1999 Matthias Urlichs, smurf@noris.de

package RT::database::manipulate;

use utf8;
use warnings; no warnings qw(once redefine uninitialized);
use strict;
use vars qw(@ISA @EXPORT_OK);
require Exporter;
@ISA = qw(Exporter);
@EXPORT_OK = qw(add_pointer change_priority
        stall steal notify reopen change_requestors merge kill give resolve
		change_subject change_info change_kunde change_area add_comment
		change_date_due add_new_request change_queue change_work remail
		add_correspondence docushare change_date_fdue take
		is_visible set_visibility change_status);

use Cf qw($RT_DOMAIN);
use RT::support::utils qw(
  can_create_request
  can_display_queue
  can_manipulate_queue
  can_manipulate_request
  depending_stalled
  get_effective_sn
  is_not_a_requestor
  is_rt_system
);
use RT::database qw(req_in add_request add_transaction update_request
		update_each_req update_seq);
use RT::support::mail qw(template_mail);
use RT::database::config qw(is_an_area is_a_queue);
use RT::AddOns;
use Dbase::Globals qw(content get_kunde get_person);
use Dbase::Help qw(Do DoFn DoSelect DoSeq isotime text_only);
use Docushare qw(link_ticket acltest);
use Fehler qw(probleme ffehler);

use vars qw(%req %queues %users $time);
BEGIN {
	*queues = \%RT::queues;
	*users = \%RT::users;
	*req = \%RT::req;
	*time = \$RT::time;
}

sub change_status($$$);

sub add_new_request($;$) {
	my($request,$content) = @_;
    my $queue_id = $request->param('queue');
    my $area = $request->param('area');
    # Nach Erkenntnissen in RT#288474 wird (nur) bei Followup-Tickets "to"
    # verwendet, aber im Requestor steht der User, der das Followup öffnet.
    my $requestors = $request->param('to');
    $requestors = $request->param('requestors') unless defined $requestors;
    my $mail2requestors = $request->param('mail2requestors');
    my $owner = $request->param('owner');
    my $subject  = $request->param('subject');
    my $info  = $request->param('info');
    my $priority  = $request->param('prio');
    my $status  = $request->param('status');
    my $date_created  = $request->param('date_created') || $time;
    my $date_told  = $request->param('date_told') || $time;
    my $id  = $request->param('new_ticket_id');
    my $date_due  = $request->param('date_due');
    my $date_fdue  = $request->param('date_fdue');
    # my $content = $request->param('content');
    my $user = $request->param('auth_user');
    my $from = $request->param('from') || $user;
    my $cc = $request->param('cc');
    my $bcc = $request->param('bcc');
    my $via_mail = $request->param('via_mail');
    my $kunde  = $request->param('kunde');
    my $extern = $request->param('extern');
	
    my $msg;
	$kunde=1 unless defined $kunde;
	$status= $date_due ? 'stalled' : 'open' unless defined $status;
	$subject="(kein Subject)" if not defined $subject or $subject eq "";
        
    my ($transaction, $serial);
    
	return (0,0,"Queue '$queue_id' gibt es nicht")
		unless defined $queues{$queue_id};

    unless (can_create_request($queue_id, $user)>0) {
        #use Carp 'cluck';
        #cluck("Create");

		return (0,0,"You don't have permission to create requests in this queue <$queue_id> <$user>");
    }
    
    #add the fact to each_req    
    $serial=add_request($id, $queue_id, $area, $requestors, $owner, $subject, $info, $priority, $status, $date_created, $date_told, $date_fdue, $date_due, $user);
    
    # note the creation in the transaction log
    $transaction=add_transaction($serial, $from, 'create','',$content,$user,undef,undef,$via_mail,$extern);
	return (0,0,$transaction) unless ref $transaction;

	if($kunde) {
    	$kunde = get_kunde($kunde);
    	Do("update ticket set kunde = $kunde where id = $serial") if $kunde;
	}

	$request->param("owner",$user)
		unless $request->param("owner") or can_manipulate_queue($queue_id,$user) > 0;

    if ($queues{$queue_id}{m_members_correspond} and ($queues{$queue_id}{m_mail} or not $via_mail)) {
		template_mail ('correspondence',$queue_id,$queues{$queue_id}{dist_list},"","", $serial ,$transaction, $subject, $user,'');
	}
	template_mail( correspondence => $queue_id, $mail2requestors && !$via_mail ? $requestors : undef, $cc, $bcc, $serial, $transaction, $subject, $user, '' )
		if $cc || $bcc || $requestors && $mail2requestors && !$via_mail;

	{
		my $autoreply = $request->param('autoreply');
		template_mail( autoreply => $queue_id, $requestors, '', '', $serial, $transaction, $subject, $user, '' )
		  if $autoreply || !defined $autoreply && $queues{$queue_id}{m_user_create} && $via_mail;
	}

	give($serial,$owner,$user) if $owner;

	RT::AddOns->dispatch( $queues{$queue_id}{name} => init_ticket => $serial, $date_created, $kunde, $subject );
    return ($serial,$transaction,"Request #$serial created.");
}

sub add_correspondence($;$$) {
	my($request,$content,$notify) = @_;
    my $serial = $request->param('serial_num');
    my $subject = $request->param('subject');
    my $user = $request->param('auth_user');
    my $from = $request->param('from') || $user;
    my $to = $request->param('to');
    my $cc = $request->param('cc');
    my $bcc = $request->param('bcc');
    my $status = $request->param('status');
    my $via_mail = $request->param('via_mail');
    my $extern = $request->param('extern');
    my $transaction;
    
    if (not $via_mail and
		not can_manipulate_request($serial, $user)>0) {
		return (0,"You don't have permission to modify request \#$serial");
    }

    req_in($serial, '_rt_system');
	$subject=$req{$serial}{subject} if not defined $subject or $subject eq "";
    my $queue_id=$req{$serial}{queue_id};
    
    $transaction=add_transaction($serial, $from, 'correspond','',$content,$user,undef,undef,$via_mail,$extern);
	return (0,$transaction) unless ref $transaction;
   
# read again as add_transaction overwrites it depending on user's privileges
    req_in($serial, '_rt_system');
    
#    if ($status and $req{$serial}{status} ne $status) {
#      	#print STDERR "Status of $serial becoming $status\n";
#      	my $opentrans = update_request($serial,'status',"$status", "_rt_system");
#      	#print "Reopening the request $opentrans\n$openmsg\n";
#		return (0,$opentrans) if $opentrans !~ /^\d+$/;
#    }
    
    if ((is_not_a_requestor($user,$serial))) {
	    my $res = update_each_req($serial, 'date_told', $time);
		return (0,$res) if $res;
	    template_mail('correspondence', $queue_id, $to, $cc, $bcc, $serial, $transaction, $subject, $user,'');
    } elsif($to ne "" or $cc ne "" or $bcc ne "") {
	    template_mail('correspondence', $queue_id, $to, $cc, $bcc, $serial, $transaction, $subject, $user,'');
	}
    
    if ($queues{$queue_id}{m_members_correspond} and ($queues{$queue_id}{m_mail} or not $via_mail)) {
      template_mail ('correspondence',$queue_id,$queues{$queue_id}{dist_list},"","", $serial ,$transaction, $subject, $user,'');
    }

	{
		my @res = change_status($serial,$via_mail ? '_rt_system' : $from, $status);
		return @res unless $res[0];
	}
  
    return ($transaction,"This correspondence has been recorded.");
}
  
sub add_comment($$) {
	my ($request, $content) = @_;

	my $serial = $request->param('serial_num');
	my $user = $request->param('auth_user');
    my $from = $request->param('from') || $user;
    my $via_mail = $request->param('via_mail');
    my $status = $request->param('status');
    my $extern = $request->param('extern');

    req_in($serial, '_rt_system');

	my $subject = $request->param('subject');

	my $to = $request->param('to');
	my $cc = $request->param('cc');
	my $bcc = $request->param('bcc');

    my $queue_id =$req{$serial}{queue_id}; 
	# print STDERR "Mail for $serial/$queue_id: ";

    if (not $via_mail and
		not can_manipulate_request($serial, $user)>0 and 
		not can_create_request($serial, $user)) {
		return (0,"You don't have permission to modify request \#$serial");
    }

    my $transaction = add_transaction($serial, $from, 'comments','',$content,$user,undef,undef,$via_mail,$extern);
	return (0,$transaction) unless ref $transaction;

    if ($queues{$queue_id}{m_members_comment} and ($queues{$queue_id}{m_mail} or not $via_mail)) {
		template_mail('comment', $queue_id,$queues{$queue_id}{dist_list},"","",$serial,$transaction,$subject,$user,$content);
    } elsif ($queues{$queue_id}{m_owner_comment} and $req{$serial}{owner} ne '' and ($queues{$queue_id}{m_mail} or not $via_mail)) {
		template_mail('comment', $queue_id, $req{$serial}{'owner'}, "","", $serial, $transaction, $subject, $user, $content);
	}

	if ($to or $cc or $bcc) {
		template_mail('correspondence', $queue_id, $to, $cc, $bcc, $serial, $transaction, $subject, $user, $content);
		## NEIN, das darf KEIN "comment" sein.
    }

	{
		my @res = change_status($serial, $via_mail ? '_rt_system' : $from, $status);
		return @res unless $res[0];
	}

    return ($transaction,"Your comments have been recorded.");
}

#
#change request's status to open
sub reopen($$) {
    my ($serial, $user) = @_;
    my $transaction;
 
	return (0,"Syntax error in reopen(); serial: $serial") unless $serial =~ /^\d+$/;

	return (0,"You don't have permission to modify request \#$serial")
    	unless can_manipulate_request($serial,$user)>0;
    
    $transaction = update_request($serial,'status','open', $user);
	return (0,$transaction) unless ref $transaction;
    return ($transaction,"Request #$serial has been reopened.");
}

sub resolve($$) {
    my ($serial, $user) = @_;
    my $transaction;

	return (0,"Syntax error in resolve(); serial: $serial") unless $serial =~ /^\d+$/;

	return (0,"You don't have permission to modify request \#$serial")
    	unless can_manipulate_request($serial,$user)>0;

	if ( length( my $error = join '; ', RT::AddOns->dispatch( $req{$serial}{queue}{name}, resolvable => $req{$serial} ) ) ) {
		return 0, $error;
	}
 
	unless ( $req{$serial}{owner} || is_rt_system($user) ) {
   		$transaction=update_request($serial,'owner',$user, $user);
		return (0,$transaction) unless ref $transaction;
	}

    $transaction=update_request($serial,'date_due', 0, $user);
	# return (0,$transaction) unless ref $transaction;
    $transaction=update_request($serial,'status', 'resolved', $user);
	return (0,$transaction) unless ref $transaction;
	foreach my $tick(depending_stalled($serial)) {
		reopen( $tick, "$serial\@$RT_DOMAIN" );
	}
    return ($transaction,"Request #$serial has been resolved.");
}

sub stall($$) {
    my ($serial, $user) = @_;
    my $transaction;

	return (0,"Syntax error in stall(); serial: $serial") unless $serial =~ /^\d+$/;

	return (0,"You don't have permission to modify request \#$serial")
    	unless can_manipulate_request($serial,$user)>0;
  
    $transaction = update_request($serial,'date_due',$time+3*60*60,$user)
		unless $req{$serial}{date_due};

    $transaction = update_request($serial,'status','stalled', $user);
	return (0,$transaction) unless ref $transaction;

    return ($transaction,"Request #$serial has been stalled.");
}

sub change_status($$$) {
    my ($serial, $user, $status) = @_;

	return (0,"Syntax error in change_status(); serial: $serial") unless $serial =~ /^\d+$/;
    req_in($serial,$user);
	return ([$serial,0], "No change")
		if $req{$serial}{status} eq $status;

	if($status eq "open" or $status eq "reopen") {
		return reopen($serial,$user);
	} elsif($status eq "stall" or $status eq "stalled") {
		return stall($serial,$user);
	} elsif($status eq "resolve" or $status eq "resolved") {
		return resolve($serial,$user);
	} elsif($status eq "kill" or $status eq "killed" or $status eq "dead") {
		return kill($serial,$user);
	} else {
		return (0,"State '$status' unknown in change_status()!");
	}
}

### TODO: Owner auf RT setzen,
###       in RT-JJJJ-Collection verlinken.

sub docushare($$$) {
    my ($serial, $id, $user) = @_;
  ffehler {
	$id =~ s/^File-//i;
	return (0,"Syntax error in docushare(); serial: $serial; id: $id") unless $serial =~ /^\d+$/ and $id =~ /^\d+$/;

	return (0,"You don't have permission to modify request \#$serial")
    	unless can_manipulate_request($serial,$user)>0;
  
	my $res = acltest($id,$users{$user}{name});
	return (0,"You cannot access this document.") unless $res;

	ffehler {
  		link_ticket($id,$serial) or goto problem;
	} sub {
		return (0,"Link-Fehler: ".$_[0]);
	};

    my $transaction = add_transaction($serial,$user,'docushare',"$id-$res","", $user);
	return (0,$transaction) unless ref $transaction;

    return ($transaction,"Link to Docushare #$id OK");

problem:
	my $prob;
	probleme { $prob .= "@_\n"; 1; };
	return(0,$prob);
  } sub {
	return (0,"@_");
  };
}

sub remail($$$$) {
    my ($serial, $trans, $addr, $user) = @_;
    my $transaction;

	return (0,"Syntax error in remail(); serial: $serial") unless $serial =~ /^\d+$/ and ref $trans;

	return (0,"You don't have permission to modify request \#$serial")
    	unless can_manipulate_request($serial,$user)>0;
 
 	$addr = text_only($addr);
    $transaction = add_transaction($serial,$user,'remail',$addr,$trans->[0]."-".$trans->[1], $user);

	return (0,$transaction) unless ref $transaction;
	my $StatusMsg=template_mail("remail", $req{$serial}{queue_id}, "","",$addr, $serial, $trans, "", $user,"");

    return ($transaction,$StatusMsg);
}

sub kill($$) {
    my ($serial, $user) = @_;

	return (0,"Syntax error in kill; serial: $serial") unless $serial =~ /^\d+$/;

	return (0,"You don't have permission to modify request \#$serial")
     	unless can_manipulate_request($serial,$user)>0;

	if ( my $error = RT::AddOns->dispatch( $req{$serial}{queue}{name} => kill => $serial ) ) {
		return 0, $error;
	}

    # Do("delete from ticketid where ticket = $serial");
    Do("update domainkunde set ticket=NULL where ticket = $serial");
    Do("update ipkunde set ticket=NULL where ticket = $serial");
    Do("update nic set ticket=NULL where ticket=$serial");
	
    my $transaction = update_request($serial,'status','dead', $user);    
	return (0,$transaction) unless ref $transaction;
    return ($transaction,"Request #$serial has been killed.");
}

sub merge($$$) {
    my ($serial, $merge_into, $user) = @_;
    my ($transaction);

	return (0,"Syntax error in merge(); serial: $serial; merge_into: $merge_into") unless $serial =~ /^\d+$/ and $merge_into =~ /^\d+$/;

	return (0,"You don't have permission to modify both requests you wish to merge")
    	unless can_manipulate_request($serial,$user)>0
		   and can_manipulate_request($merge_into,$user)>0;

	return (0, "$req{$serial}{owner} owns request \#$serial.  You can not merge it without 'stealing' it.")
    	if $req{$serial}{owner} and $req{$serial}{owner} ne $user;

	$serial = get_effective_sn($serial);
	$merge_into = get_effective_sn($merge_into);
    return (0,"Du mergst gerade #$serial auf #$merge_into -- SEHR witzig.")
		if $serial == $merge_into;

	DoSelect {
		my($p,$ts) = @_;
		Do("insert into ticketabo set ticket=$merge_into,person=$p,timestamp=$ts")
			unless DoFn("select count(*) from ticketabo where ticket=$merge_into and person=$p");
	} "select person,0+timestamp from ticketabo where ticket=$serial";

	if ( my $error = RT::AddOns->dispatch( $req{$serial}{queue}{name} => check_merge => $serial, $merge_into ) ) {
		return 0, $error;
	}
	if (
		$RT::req{$serial}{kunde_id} != $RT::req{$merge_into}{kunde_id}
		and my $error = RT::AddOns->dispatch(
			$req{$serial}{queue}{name} =>
			  change_kunde => $serial,
			$RT::req{$merge_into}{kunde_id}
		)
	  )
	{
		return 0, $error;
	}
	if ( my $error = RT::AddOns->dispatch( $req{$serial}{queue}{name} => do_merge => $serial, $merge_into ) ) {
		return 0, $error;
	}

	DoSeq("rtmupdate");
    $RT::reqcache{$serial} = undef;
    $RT::reqcache{$merge_into} = undef;
    Do("update ticket set d_acted = $time where id = $serial or id = $merge_into");
	Do("update ticket set ticket = $merge_into where ticket = $serial");

	update_each_req($merge_into,'date_created',$req{$serial}{'date_created'})
  		if $req{$merge_into}{'date_created'} > $req{$serial}{'date_created'};
    
	update_each_req($merge_into,'date_told',$req{$serial}{'date_told'})
		if $req{$merge_into}{'date_told'} < $req{$serial}{'date_told'} and $req{$serial}{'date_told'} > 0;

	update_each_req($merge_into,'date_due',$req{$serial}{'date_due'})
    		if defined $req{$serial}{date_due} and !defined $req{$merge_into}{date_due} || $req{$serial}{date_due} < $req{$merge_into}{date_due};

	update_each_req($merge_into,'date_fdue',$req{$serial}{'date_fdue'})
    	if $req{$merge_into}{'date_fdue'} < $req{$serial}{'date_fdue'} and $req{$serial}{'date_fdue'} > 0;

	{
    	req_in($serial,'_rt_system');
    	req_in($merge_into,'_rt_system');

		my $t_zeit = 0;
		defined $req{$_}{'t_zeit'} and $t_zeit += $req{$_}{'t_zeit'}
			for ($serial, $merge_into);
		update_each_req( $merge_into, work => $t_zeit/3600 )
			if $t_zeit and !defined $req{$merge_into}{t_zeit} || $req{$merge_into}{t_zeit} != $t_zeit;
	}

	$transaction = update_request($serial,'effective_sn',$merge_into, $user);    
	return (0,$transaction) unless ref $transaction;

    Do("delete from ticketlast where ( ticket = $serial or ticket = $merge_into ) and seq < 9999");

    update_request($merge_into,'status','open', $user);

	# Cache für Arbeitsstunden löschen
	undef $RT::stdcache{$merge_into};
	undef $RT::stdcache{$serial};
	DoSeq("stdupdate");

    return ($transaction,"Request #$serial has been merged into request #$merge_into.");

}


sub change_queue($$$;$) {
    my ($serial, $queue, $user, $area) = @_;
    my $transaction;
   
	return (0,"Syntax error in change_queue(); serial: $serial") unless $serial =~ /^\d+$/;

	return (0,"You don't have permission to modify request \#$serial")
    	unless can_manipulate_request($serial,$user)>0;
	return (0, "\'$queue\' is not a valid queue.")
    	unless is_a_queue($queue);
	return (0, "You may only requeue a request into a queue you have privileges to create requests in.")
    	unless can_create_request($queue, $user)>0;

	# Z. B. im Falle des RT::AddOn::Force_area würde es nicht helfen, hier
	# nur zu überprüfen, da das Ticket zwar momentan eine Area haben mag,
	# die durch den Queue-Wechsel aber eben verschwindet. Daher lieber
	# gleich einfach wieder öffnen, falls es ein Problem mit dem aktuellen
	# Status gibt.
	print STDERR "HERE: serial=$serial|user=$user|status=$req{$serial}{status}|queue=$queue\n";
	reopen( $serial, $user )
	  if $req{$serial}{status} eq 'resolved'
	  && RT::AddOns->dispatch( $queue => resolvable => $req{$serial} )
	  || $req{$serial}{status} eq 'dead'
	  && RT::AddOns->dispatch( $queue => kill => $serial );

	if ( my $error = RT::AddOns->dispatch( $req{$serial}{queue}{name} => change_queue => $serial ) ) {
		return 0, $error;
	}

	if ( !defined $area && defined $req{$serial}{area} ) {
		$area =
		  is_an_area( $queue, $req{$serial}{area} ) ? $req{$serial}{area} : '';
	}
	elsif ( $area =~ /^None$/i || $area eq '' ) {
		$area = "";
    } elsif (not is_an_area($queue, $area)) {
		return (0, "Queue \'$queue\' doesn't have an area called \'$area\'");
    }

	update_request($serial,'area',$area,$user);
   
	$transaction = update_request($serial,'queue_id',$queue, $user);
	return (0,$transaction) unless ref $transaction;

	# $transaction=update_request($serial,'owner','','_rt_system');
	## This doesn't make much sense

	return ($transaction,"Request #$serial moved to queue $queue.");
}

#change the ownership of a request
sub give($$$) {
    my ($serial, $owner, $user) = @_;
    my $transaction;
    
	return (0,"Syntax error in give(); serial: $serial") unless $serial =~ /^\d+$/;

	return (0,"You do not have access to modify request \#$serial")
    	unless can_manipulate_request($serial,$user)>0;

	return (0, "$req{$serial}{owner} owns request \#$serial.  You can not reassign it without 'stealing' it.")
    	if $req{$serial}{owner} and $req{$serial}{owner} ne $user;

	return(0,"$owner already owns request \#$serial.")
    	if $req{$serial}{owner} and $owner eq $req{$serial}{owner};
    
    my $qid = $req{$serial}{queue_id};
    my $subject = $req{$serial}{subject};

	$owner = ''
	  if !$owner || $owner eq '-' || $owner eq '--' || is_rt_system($owner);

    if ($owner eq '') {
	    return (0,"You must own request \#$serial before you can untake it.")
			unless $req{$serial}{'owner'} eq $user;
	
	    $transaction = update_request($serial,'owner','', $user);
		return (0,$transaction) unless ref $transaction;
    	return ($transaction, "Request #$serial untaken.");
    }
    
    if (not $req{$serial}{'owner'} and $owner eq $user) {	 
    	$transaction=update_request($serial,'owner',$user, $user);
		return (0,$transaction) unless ref $transaction;

    	return ($transaction, "Request #$serial taken.");
   	}
    
	return (0,"$owner does not have access to this queue")
    	if $owner ne "" and not can_display_queue($req{$serial}{queue_id},$owner)>0;
  
	return(0, "You can not change the ownership of a request owned by somebody else.")
    	if $req{$serial}{owner} and $req{$serial}{owner} ne $user;

	$transaction = update_request($serial,'owner',$owner, $user);
	return (0,$transaction) unless ref $transaction;

	template_mail('give',$qid,$users{$owner}{email},"","", $serial ,$transaction,$subject,$user,'');

    return ($transaction, "Request #$serial given to $owner.");
}

sub untake {
    my  ($serial, $user) = @_;
    give ($serial, "", $user);
}
sub take {
    my  ($serial, $user) = @_;
    give ($serial, $user, $user);
}

sub steal($$) {
    my ($serial, $user) = @_;
    my ($transaction);

	return (0,"Syntax error in steal; serial: $serial") unless $serial =~ /^\d+$/;

	return (0,"You don't have permission to modify request \#$serial")
    	unless can_manipulate_request($serial,$user)>0;
    
    #we don't need to read the req in because can_manipulate_req calls req_in
    #req_in($serial,$user);

    my $old_owner=$req{$serial}{'owner'};
    my $qid = $req{$serial}{'queue_id'};

	return (0,"You can only steal requests owned by someone else.")
    	if $old_owner and $old_owner eq $user;

	$transaction=update_request($serial,'owner',$user, $user);
	return (0,$transaction) unless ref $transaction;

	template_mail('steal',$qid,$users{$old_owner}{email},"","", $serial ,$transaction,$req{$serial}{subject}, $user,'');

	return ($transaction, "Request \#$serial stolen.");
}

sub change_requestors($$$) {
    my ($serial, $requestor, $user) = @_;
    my $transaction;
    my $msg;
    
	return (0,"Syntax error in change_requestors; serial: $serial") unless $serial =~ /^\d+$/;

	return (0,"You don't have permission to modify request \#$serial")
    	unless can_manipulate_request($serial,$user)>0;

    # ($user,$msg) = norm_requestors($user);
    # return (0,$msg) if $msg;

	$requestor = text_only($requestor);
    $transaction = update_request($serial,'requestors',$requestor,$user);
	return (0,$transaction) unless ref $transaction;
    return ($transaction, "Requestor changed to '$requestor'.");
}

sub change_area($$$) {
    my ($serial, $area, $user) = @_;
    my $transaction;

	return (0,"Syntax error in change_area(); serial: $serial") unless $serial =~ /^\d+$/;

	return (0,"You don't have permission to modify request \#$serial")
    	unless can_manipulate_request($serial,$user)>0;
    
    if($area =~ /^None$/i or $area eq "" or $area eq "--") {
		$area = "";
    } elsif (not is_an_area($req{$serial}{queue_id}, $area)) {
		return (0, "Queue \'$req{$serial}{queue_id}\' doesn't have an area called \'$area\'");
    }

    $transaction = update_request($serial,'area',$area,$user);
	return (0,$transaction) unless ref $transaction;
    return ($transaction, "Area of request #$serial changed to '$area'");
}

sub change_priority($$$) {
    my($serial, $priority, $user) = @_;
    my $transaction;

	return (0,"Syntax error in change_priority; serial: $serial") unless $serial =~ /^\d+$/;

	return (0,"You don't have permission to modify request \#$serial")
    	unless can_manipulate_request($serial,$user)>0;
	return "Priority nicht numerisch ('$priority')"
		if $priority !~ /^\d+$/;
    
    $transaction = update_request($serial,'priority',$priority, $user);
	return (0,$transaction) unless ref $transaction;

    return ($transaction, "Priority of request #$serial changed to $priority");
}

sub change_work($$$) {
    my($serial, $work, $user) = @_;
    my $transaction;

	return (0,"Syntax error in change_work(); serial: $serial") unless $serial =~ /^\d+$/;

	return (0,"You don't have permission to modify request \#$serial")
    	unless can_manipulate_request($serial,$user)>0;

	if ( $work =~ /^\s*(\d+):(\d\d)\s*\z/ ) { $work = $1 + $2/60 }
	elsif ( $work eq '-' ) {}
	elsif ( $work !~ /^\s*\d+?[.,]?\d*\s*\z/ ) {
		return "Arbeitszeit nicht numerisch ('$work')"
	}
	else {
		$work =~ y/,/./;
		$work ||= '-';
	}
    
    $transaction = update_request($serial,'work',$work, $user);
	return (0,$transaction) unless ref $transaction;

    $transaction,
    "Scheduled time for request #$serial "
      . ( $work eq '-' ? 'deleted' : "changed to $work hour" . ( $work != 1 && 's' ) );
}

sub add_pointer($$$$) {
	my($request,$serial,$new_queue,$content) = @_;
    my $user = $request->param('auth_user');
	my $pers = get_person($user);

	return (0,0,"Syntax error in add_pointer; serial: $serial") unless $serial =~ /^\d+$/;

	return (0,0,"You don't have permission to modify request \#$serial")
    	unless can_manipulate_request($serial,$user)>0;
	return (0,0,"You don't have permission to add to queue $new_queue")
    	unless can_manipulate_queue($new_queue,$user)>0 or
    	       can_create_request($new_queue, $user)>0;

    req_in($serial,$user);
	if($request->param('status') eq "open" and can_manipulate_queue($new_queue,$user)>0) {
		$request->delete("owner");
	} else {
		$request->param("owner",$user);
	}
	$request->param('queue',$new_queue);
	$request->param('status',"open") unless $request->param('status');
	$request->param(requestors=>$user);
	$request->param(mail2requestors=>1);

	my($new_serial,$transaction,$StatusMsg) = add_new_request($request,$content);
	return ($new_serial,$transaction,$StatusMsg) if $new_serial == 0;
	
    my $kunde = get_kunde($req{$serial}{kunde});
    Do("update ticket set kunde = $kunde where id = $new_serial") if $kunde > 0;
    # Do("update ticket set bearbeiter = $pers where id = $new_serial") if $pers > 0;

	RT::AddOns->dispatch( $new_queue => copy => $new_serial, $serial );

    my $res = add_transaction($serial,$user,'queue_ptr',$new_queue,$new_serial, $user);
	return (0,0,$res) unless ref $res;
    $res = add_transaction($new_serial,$user,'queue_ptr',$req{$serial}{queue_id},$serial, $user);

    return ($new_serial,$transaction,$StatusMsg);
}


sub change_date_due($$$) {
    my ($serial, $due_date, $user) = @_;
    my $transaction;

	return (0,"Syntax error in change_date_due(); serial: $serial") unless $serial =~ /^\d+$/;

	return (0,"You don't have permission to modify request \#$serial")
    	unless can_manipulate_request($serial,$user)>0;

    $transaction = update_request($serial,'date_due',$due_date,$user);
	return (0,$transaction) unless ref $transaction;

    return ($transaction, "Re-Open Date of request #$serial changed to ".isotime($due_date).".");
}

sub change_date_fdue($$$) {
    my ($serial, $fdue_date, $user) = @_;
    my $transaction;

	return (0,"Syntax error in change_date_fdue(); serial: $serial") unless $serial =~ /^\d+$/;

	return (0,"You don't have permission to modify request \#$serial")
    	unless can_manipulate_request($serial,$user)>0;

	$fdue_date ||= undef;

    $transaction = update_request($serial,'date_fdue',$fdue_date,$user);
	return (0,$transaction) unless ref $transaction;

	return $transaction,
	       defined $fdue_date ? "Due Date of request #$serial changed to ".isotime($fdue_date).'.'
	                          : "Due Date of request #$serial deleted."
	                        
}

sub change_subject($$$) {
    my ($serial, $subject, $user) = @_;
    my $transaction;

	return (0,"Syntax error in change_subject(); serial: $serial") unless $serial =~ /^\d+$/;

	return (0,"You don't have permission to modify request \#$serial")
    	unless can_manipulate_request($serial,$user)>0;
    
	$transaction =
	  update_request( $serial, subject => text_only($subject), $user );
	return (0,$transaction) unless ref $transaction;
    return ($transaction,"Request \#$serial\'s subject has been changed to \"$subject\".");
}

sub change_info($$$) {
    my ($serial, $info, $user) = @_;
    my $transaction;

	return (0,"Syntax error in change_info(); serial: $serial") unless $serial =~ /^\d+$/;

	return (0,"You don't have permission to modify request \#$serial")
    	unless can_manipulate_request($serial,$user)>0;
    
	$transaction = update_request( $serial, info => text_only($info), $user );
	return (0,$transaction) unless ref $transaction;
    return ($transaction,"Request \#$serial\'s state has been changed to \"$info\".");
}

sub change_kunde($$$) {
    my ($serial, $kunde, $user) = @_;
    my $transaction;

	return (0,"Syntax error in change_kunde(); serial: $serial") unless $serial =~ /^\d+$/;

	return (0,"You don't have permission to modify request \#$serial")
    	unless can_manipulate_request($serial,$user)>0;

	if ( my $error = RT::AddOns->dispatch( $req{$serial}{queue}{name} => change_kunde => $serial, $kunde ) ) {
		return 0, $error;
	}

    $transaction=update_request($serial,'kunde',$kunde, $user);
	return (0,$transaction) unless ref $transaction;
    return ($transaction,"Request \#$serial\'s kunde has been changed to \"$kunde\".");
}

sub notify($$$) {
    my ($serial, $notified, $user) = @_;
    my $transaction;

	return (0,"Syntax error in notify(); serial: $serial") unless $serial =~ /^\d+$/;

	return (0,"You don't have permission to modify request \#$serial")
    	unless can_manipulate_request($serial,$user)>0;
    
    $transaction = update_request($serial,'date_told',$notified, $user);
	return (0,$transaction) unless ref $transaction;
    return ($transaction, 'Notification Noted.');
}

sub is_visible($$) {
    my ($serial, $req) = @_;
    $req->{queue}{rtweb} and
    scalar DoFn <<_;
	SELECT COUNT(*)
	FROM   ticketid, ticket
	WHERE  ticketid.ticket = ticket.id
	   AND ticket.ticket = $serial
	   AND ticketid.extern='y'
	   AND ticketid.seq>0
_
}

# anders als die anderen sub-Routinen bezieht sich set_visibility()
# nicht auf ein Ticket, sondern einen einzelnen Eintrag, und wird auch
# in einem anderen Kontext aufgerufen
sub set_visibility($$$$) {
    my ($serial, $seq, $extern, $user) = @_;
    $extern = lc $extern;
    $extern = 'n' unless $extern eq 'y';
    return undef, "You don't have permission to modify request \#$serial"
        unless can_manipulate_request($serial,$user)>0;
    my $transaction = update_seq($serial,$seq,$extern,$user);
    return undef, $transaction unless ref $transaction;
    $transaction, "Made #$serial-$seq ".($extern eq 'n' && 'in').'visible.';
}

1;
