package noris::TroubleTicket;

=head1 NAME

noris::TroubleTicket - Interface fuer das Trouble-Ticket-System

=head1 SYNOPSIS

  use noris::TroubleTicket;

  my $tt = noris::TroubleTicket->new (%opts);

  $tt->set ($attribute, $value);
  $value = $tt->get ($attribute);

=cut

use utf8;
use strict;
use warnings;

BEGIN {
	unshift @INC, ( $ENV{'POPHOME'} || '@POPHOME@' ) . '/lib'
		unless $ENV{'KUNDE_NO_PERLPATH'};
}

use Carp qw(carp cluck croak confess);
use Exporter;

use POSIX qw(locale_h strftime);

use Email::Valid;
use noris::MIME::Words qw(encode_mimewords);
use Text::Wrap  qw(wrap);
use Time::Zone qw(tz_local_offset);

use Dbase ();
use Dbase::Help qw(DoN DoFn DoSelect DoTrans qquote);
use Dbase::Globals qw(bignum find_descr get_descr info_descr map_descr name_kunde);
use Fehler qw(ffehler probleme);

use Loader qw(log_update);

use noris::Kunde qw(kundeid_get_all kundeid_get_by_flags);
use noris::Person ();

use constant SUBJECT_PREFIX => 'Trouble Ticket';

@noris::TroubleTicket::EXPORT_OK = qw(tt_get_by_kunde tt_get_incomplete tt_get_descriptor_list);
@noris::TroubleTicket::ISA = ('Exporter');

=head1 ATTRIBUTE

=head2 Namen

=over 4

=item ticket I<(Ticket-ID)>

Ticket-ID des Tickets, dem dieses Trouble-Ticket zugeordnet ist.

=item subject I<(String)>

Betreff dieses Trouble-Tickets.

=item type I<(Descriptor)>

Typ des TTs (planmaessige Wartung, unerwarteter Ausfall, ...).

=item priority I<(Descriptor)>

Prioritaet (niedrig, mittel, hoch).

=item beginn I<(Zeit)>

Zeit, zu der das Problem beginnt. Dies kann durchaus in der Zukunft liegen.

=item ende I<(Zeit)>

Zeit zu der das Problem (voraussichtlich) behoben sein wird.

=item flag_kunde I<(Flag-Liste)>

Auswahl der Kunden/Empfaenger nach "Kunden-Flags".

=item flag_person I<(Flag-Liste)>

Auswahl der Kunden/Empfaenger nach "Personen-Flags".

=item additional_rcpt I<(String-Liste)>

Liste von E-Mail-Adressen, an die das Trouble-Ticket zusaetzlich geschickt
werden soll. Die Adressen werden mit C<Email::Valid> auf gueltigkeit
ueberprueft.

=item confirmer I<(Person-ID)>

Die bestaetigende Person. Das kann die gleiche Person sein, die das
Trouble-Ticket geschrieben hat. Muss aber nicht.. Dieser Wert kann nicht
gesetzt (also C<NULL> bzw. C<undef>) sein.

=item description I<(String)>

=item affected I<(String)>

=item comment I<(String)>

=item progess I<(String)>

=item resolve I<(String)>

Text fuer "Beschreibung", "Betroffen", "Anmerkungen", "Fortschritt" und
"Loesung"

=item select_kunden I<(enum)>

Spezifiziert wie die Kunden ausgewaehlt wurden. Gueltige Werte sind B<all>,
B<flags> und B<none>.

=back

=head2 Typen

Ja nach Typ des Attributs verhalten sich die Funktionen C<set> und C<get>
unterschiedlich:

=over 4

=item Descriptor

C<get> liefert im skalaren Kontext den Namen des gesetzten Descriptors zurueck.
Im Listenkontext wird der Namen und der Infotext zurueckgegeben.

C<set> erwartet den B<Namen> eines Descriptors. Mit der Methode
C<tt_get_descriptor_list> kann eine Liste aller moeglichen Werte ermittelt werden.

=item Flag-Liste

C<get> liefert eine Liste aller gesetzten Flags zurueck. Im skalaren Kontext
wird eine Array-Referenz zurueckgegeben.

C<set> erwartet eine Array-Referenz mit Descriptoren-Namen. Mit der Methode
C<tt_get_descriptor_list> kann eine Liste aller moeglichen Werte ermittelt werden.

=item Ticket-ID

Ticket-ID des zugeordneten RT-Tickets. Erwartet und zurueckgeliefert wird ein
Integer-Wert. IDs ge-merge-ter Tickets werden automatisch auf die Ticket-ID des
Mutter-Tickets gesetzt.

=item Person-ID

Erwartet bzw. liefert eine Personen-ID. C<get> liefert im Listen-Kontext eine
Liste mit den Eintraegen C<(I<$id>, I<$user>, I<$name>)>.

=item String

Keine Besonderheiten.

=item String-Liste

C<set> erwartet als Argument eine Array-Referenz. Der Rueckgabewert ist die
Anzahl ungueltiger E-Mail-Adressen. Ein von Null verschiedener Wert deutet also
auf einen Fehler hin.

C<get> liefert im skalaran Kontext eine Array-Referenz und im Listen-Kontext
eine Liste.

=item Zeit

Akzeptiert bzw. liefert einen Zeitpunkt in Epoch. Dabei muss B<beginn> vor
B<ende> liegen. B<ende> darf auch C<undef> sein, was als "Ende nicht bekannt"
interpretiert wird.

=back

=cut

our $ValidFields =
{
	id               => {get => \&_get_integer,    set => \&_set_id},
	ticket           => {get => \&_get_integer,    set => \&_set_ticket},
	subject          => {get => \&_get_string,     set => \&_set_string},
	type             => {get => \&_get_descriptor, set => \&_set_descriptor},
	timestamp        => {get => \&_get_time},
	beginn           => {get => \&_get_time,       set => \&_set_time},
	ende             => {get => \&_get_time,       set => \&_set_time},
	priority         => {get => \&_get_descriptor, set => \&_set_descriptor},
	flag_kunde       => {get => \&_get_flag,       set => \&_set_flag},
	flag_person      => {get => \&_get_flag,       set => \&_set_flag},
	additional_rcpt  => {get => \&_get_list,       set => \&_set_email},
	description      => {get => \&_get_string,     set => \&_set_string},
	affected         => {get => \&_get_string,     set => \&_set_string},
	comment          => {get => \&_get_string,     set => \&_set_string},
	progress         => {get => \&_get_string,     set => \&_set_string},
	resolve          => {get => \&_get_string,     set => \&_set_string},
	select_kunden    => {get => \&_get_select_kunden, set => \&_set_select_kunden},
	confirmer        => {get => \&_get_person}
};

our $DescriptorMap =
{
	type		=> 'tt_type',
	priority	=> 'tt_priority',
	flag_kunde	=> 'kunde',
	flag_person	=> 'pwdomain'
};

our $TextfieldMap =
{
	description	=> 'Problembeschreibung',
	affected	=> 'Betroffen',
	comment		=> 'Bemerkungen',
	progress	=> 'Fortschritt',
	resolve		=> 'Lösung'
};
our $TextfieldOrder = [qw(description affected comment progress resolve)];

$Text::Wrap::columns = 72;

# setlocale(LC_CTYPE, "fr_CA.ISO8859-1");
# setlocale (LC_TIME, 'de_DE.ISO8859-15');
# setlocale (LC_ALL, 'de_DE.iso88591');
setlocale (LC_TIME, 'de_DE');

return (1);

sub do_or_confess
{
	my $sql = join ('', @_);
	my $status = DoN ($sql); # has a scalar prototype!
	if ($status =~ s/^-//)
	{
		confess ("do_or_confess ($sql): $status");
	}

	return ($status);
}

# Vgl. RT#383375: Zu lange Aenderungen haben in der Vergangenheit dazu
# gefuehrt, dass es in `log_update' einen Fehler gab. Das soll nicht nochmal
# passieren.
sub log_update_or_confess
{
	my @argv = @_;
	my $status = undef;
	my $args = join (', ', map { defined ($_) ? $_ : '(undef)' } (@argv));

	ffehler (sub
		{
			$status = &log_update (@argv);
		},
		sub
		{
			my $errstr = join (', ', @_);
			$status = undef;
			confess ("log_update_or_confess ($args) failed: $errstr");
		});
	probleme (sub
		{
			my $errstr = join (', ', @_);
			$status = undef;
			confess ("log_update_or_confess ($args) had problems: $errstr");
		});

	return ($status);
}

=head1 METHODEN

=over 4

=item I<$obj> = noris::TroubleTicket-E<gt>B<new> (I<%args>)

Erzeugt ein neues Ticket-Objekt und setzt die entsprechenden Attribute.

=cut

sub new
{
	my $pkg = shift;
	my %opts = @_;

	my $obj = {};

	for (keys %opts)
	{
		my $key = $_;

		if (!defined ($ValidFields->{$key}))
		{
			$obj->{'_error'} = "No such field: $key";
			cluck ("No such field: $key")
				unless $ENV{TESTING3};
			next;
		}

		set ($obj, $key, $opts{$key});
	}

	return (bless $obj, $pkg);
}

sub _bitmask_to_descriptor_list
{
	my $attribute = shift;
	my $bitfeld   = bignum (shift);
	my $descr = $DescriptorMap->{$attribute} or confess ("Not a valid attribute: $attribute");
	my @list = ();

	my $i = 0;
	while ($bitfeld)
	{
		if ($bitfeld & 0x01)
		{
			my $name = get_descr ($descr, $i);
			push (@list, $name) if ($name);
		}
		$i++;
		$bitfeld >>= 1;
	}

	return (@list);
}

=item I<$obj> = noris::Troubleticket-E<gt>B<load> (I<$id>)

Laedt ein Trouble-Ticket aus der Datenbank.

=cut

sub load
{
	my $pkg = shift;
	my $id  = shift;
	my %obj;

	@obj{qw(ticket subject type priority timestamp beginn ende flag_kunde
			flag_person description affected comment progress
			resolve additional_rcpt confirmer)} = DoFn (<<SQL);
	SELECT rtticket, subject, type, priority, UNIX_TIMESTAMP(timestamp), beginn, ende, flag_kunde,
		flag_person, text_description, text_affected, text_comment,
		text_progress, text_resolve, additional_rcpt, confirmer
	FROM trouble_ticket
	WHERE id = $id
SQL

	$obj{'id'} = $id;

	#require Data::Dumper;
	#print STDERR Data::Dumper->Dump ([\%obj], ['obj']);

	if (!defined ($obj{'flag_kunde'}) and !defined ($obj{'flag_person'}))
	{
		$obj{'select_kunden'} = 'all';
	}
	else
	{
		$obj{'flag_kunde'}  = 0 if (!defined ($obj{'flag_kunde'}));
		$obj{'flag_person'} = 0 if (!defined ($obj{'flag_person'}));

		if (!$obj{'flag_kunde'} and !$obj{'flag_person'})
		{
			$obj{'flag_kunde'}  = [];
			$obj{'flag_person'} = [];
			$obj{'select_kunden'} = 'none';
		}
		else
		{
			$obj{'flag_kunde'}  = [_bitmask_to_descriptor_list ('flag_kunde',  $obj{'flag_kunde'})];
			$obj{'flag_person'} = [_bitmask_to_descriptor_list ('flag_person', $obj{'flag_person'})];
			$obj{'select_kunden'} = 'flags';
		}
	}
	#print STDERR "select_kunden = " . $obj{'select_kunden'}  . "\n";

	if (defined ($obj{'type'}))
	{
		my $id   = $obj{'type'};
		my $name = get_descr ($DescriptorMap->{'type'}, $id);
		$obj{'type'} = $name;
	}

	if (defined ($obj{'priority'}))
	{
		my $id   = $obj{'priority'};
		my $name = get_descr ($DescriptorMap->{'priority'}, $id);
		$obj{'priority'} = $name;
	}

	{
		my $tmp = $obj{'additional_rcpt'};
		$obj{'additional_rcpt'} = [split (' ', $tmp)];
	}

	$obj{'kunden'} = {};
	DoSelect (sub {
		my $kundeid   = shift;
		my $kundename = shift;
		$obj{'kunden'}->{$kundeid} = $kundename;
		#print STDERR "[tt #$id] kunde #$kundeid ($kundename)\n";
	}, <<SQL);
	SELECT t.kunde, k.name
	FROM trouble_ticket_kunde_assoc t
	LEFT JOIN kunde k ON t.kunde = k.id
	WHERE t.trouble_ticket = $id
SQL

	return bless(\%obj, $pkg);
}

sub _descriptor_list_to_bitmask
{
	my $attribute = shift;
	my @list = @_;
	my $descr = $DescriptorMap->{$attribute} or confess ("Not a valid attribute: $attribute");
	my $bitfield = bignum (0);

	for (@list)
	{
		my $name = $_;
		my $id   = find_descr ($descr, $name, 0x00);

		next if (!defined ($id));

		$bitfield |= bignum (1) << $id;
	}

	return ($bitfield);
}

sub _update_db_kunden
{
	my $obj = shift;
	my $id  = $obj->{'id'};

	my @create = ();
	my @remove = ();

	my %kunden = ();
	for ($obj->get_kunden ())
	{
		$kunden{$_} = 1;
	}

	DoSelect (sub {
		my $kundeid = shift;
		if (exists ($kunden{$kundeid}))
		{
			delete ($kunden{$kundeid});
		}
		else
		{
			push (@remove, $kundeid);
		}
	}, "SELECT kunde FROM trouble_ticket_kunde_assoc WHERE trouble_ticket = $id");

	@create = keys %kunden;

	#print STDERR "trouble_ticket_kunde_assoc create: (" . join (', ', @create) . ")\n";
	#print STDERR "trouble_ticket_kunde_assoc remove: (" . join (', ', @remove) . ")\n";

	if (@remove)
	{
		do_or_confess (<<SQL);
	DELETE FROM trouble_ticket_kunde_assoc
	WHERE trouble_ticket = $id AND kunde IN (${\join (', ', @remove)})
SQL
	}

	for (@create)
	{
		do_or_confess ("INSERT INTO trouble_ticket_kunde_assoc (trouble_ticket, kunde) VALUES ($id, $_)");
	}
}

sub _insert_db
{
	my $obj = shift;

	my $ticket = $obj->{'ticket'} || 'NULL';
	my $subject  = qquote ($obj->{'subject'});

	for (qw(description affected comment progress resolve))
	{
		if (!$obj->{$_}) { $obj->{$_} = undef; };
	}

	my $beginn   = time ();
	my $ende     = 'NULL';

	my $type     = 0;
	my $priority = 0;

	my $additional_rcpt = $obj->{'additional_rcpt'} || [];
	$additional_rcpt = qquote (join (' ', @$additional_rcpt));

	$beginn = $obj->{'beginn'} if (defined ($obj->{'beginn'}));
	$ende   = $obj->{'ende'} if (defined ($obj->{'ende'}) and $obj->{'ende'} >= $beginn);

	$obj->{'beginn'} = $beginn;
	$obj->{'ende'}   = $ende;

	if (defined ($obj->{'type'}))
	{
		my $name = $obj->{'type'};
		my $id   = find_descr ($DescriptorMap->{'type'}, $name, 0x01);
		$type = $id;
	}

	if (defined ($obj->{'priority'}))
	{
		my $name = $obj->{'priority'};
		my $id   = find_descr ($DescriptorMap->{'priority'}, $name, 0x01);
		$priority = $id;
	}

	my $flag_kunde  = undef;
	my $flag_person = undef;
	if ($obj->{'select_kunden'} eq 'flags')
	{
		$obj->{'flag_kunde'}  = [] unless ($obj->{'flag_kunde'});
		$obj->{'flag_person'} = [] unless ($obj->{'flag_person'});
		$flag_kunde  = _descriptor_list_to_bitmask ('flag_kunde',  @{$obj->{'flag_kunde'}});
		$flag_person = _descriptor_list_to_bitmask ('flag_person', @{$obj->{'flag_person'}});
	}
	elsif ($obj->{'select_kunden'} eq 'none')
	{
		$flag_kunde  = 0;
		$flag_person = 0;
	}

	my $id = DoTrans
	{
		my $tmp;
		do_or_confess (<<SQL);
		INSERT INTO trouble_ticket SET
			rtticket         = $ticket,
			subject          = $subject,
			type             = $type,
			beginn           = $beginn,
			ende             = $ende,
			priority         = $priority,
			flag_kunde       = ${\qquote($flag_kunde        )},
			flag_person      = ${\qquote($flag_person       )},
			text_description = ${\qquote($obj->{description})},
			text_affected    = ${\qquote($obj->{affected   })},
			text_comment     = ${\qquote($obj->{comment    })},
			text_progress    = ${\qquote($obj->{progress   })},
			text_resolve     = ${\qquote($obj->{resolve    })},
			additional_rcpt  = $additional_rcpt,
			confirmer        = ${\qquote($obj->{confirmer  })}
SQL
		($tmp) = DoFn ('SELECT MAX(id) FROM trouble_ticket');
		return ($tmp);
	};
	log_update_or_confess ('trouble_ticket', 'id', $id, undef,
			'rtticket', 'subject', '*', undef,
			$ticket, $subject);

	$obj->{'id'} = $id or die;

	_update_db_kunden ($obj);

	return ($id);
}

# 
# Aendert das unbestaetigte `offene' TT.
#
sub _update_db
{
	my $obj = shift;

	confess ("Object has no ID") if (!$obj->{'id'});

	my @fields = (qw(rtticket subject type beginn ende priority flag_kunde
				flag_person text_description text_affected
				text_comment text_progress text_resolve
				additional_rcpt confirmer));
	my @new_values = ();
	my @old_values = ();

	my @changes = ();

	@old_values = DoFn ('SELECT ' . join (', ', @fields) . ' FROM trouble_ticket WHERE id = ' . $obj->{'id'});

	push (@new_values, $obj->{'ticket'}, $obj->{'subject'});

	if (defined ($obj->{'type'}))
	{
		my $name = $obj->{'type'};
		my $id   = find_descr ($DescriptorMap->{'type'}, $name, 0x01);
		push (@new_values, $id);
	}
	else
	{
		push (@new_values, undef);
	}

	$obj->{'beginn'} = time () if (!defined ($obj->{'beginn'}));
	$obj->{'ende'} = undef if (defined ($obj->{'ende'}) and $obj->{'ende'} <= $obj->{'beginn'});
	push (@new_values, $obj->{'beginn'}, $obj->{'ende'});

	if (defined ($obj->{'priority'}))
	{
		my $name = $obj->{'priority'};
		my $id   = find_descr ($DescriptorMap->{'priority'}, $name, 0x01);
		push (@new_values, $id);
	}
	else
	{
		push (@new_values, undef);
	}

	if ($obj->{'select_kunden'} eq 'flags')
	{
		$obj->{'flag_kunde'}  = [] if (!defined ($obj->{'flag_kunde'} ));
		$obj->{'flag_person'} = [] if (!defined ($obj->{'flag_person'} ));
		push (@new_values, _descriptor_list_to_bitmask ('flag_kunde', @{$obj->{'flag_kunde'}}));
		push (@new_values, _descriptor_list_to_bitmask ('flag_person', @{$obj->{'flag_person'}}));
	}
	elsif ($obj->{'select_kunden'} eq 'none')
	{
		push (@new_values, 0, 0);
	}
	elsif ($obj->{'select_kunden'} eq 'all')
	{
		push (@new_values, undef, undef);
	}
	else { confess; }

	for (qw(description affected comment progress resolve))
	{
		$obj->{$_} = undef if (!$obj->{$_});
		push (@new_values, $obj->{$_});
	}

	my $additional_rcpt;
	$obj->{'additional_rcpt'} = [] if (!ref ($obj->{'additional_rcpt'}) or ref ($obj->{'additional_rcpt'}) ne 'ARRAY');
	$additional_rcpt = join (' ', @{$obj->{'additional_rcpt'}});
	push (@new_values, $additional_rcpt);

	# confirmer
	push (@new_values, undef);

	for (my $i = 0; $i < @fields; $i++)
	{
		my $field = $fields[$i];
		my $old_value = $old_values[$i];
		my $new_value = $new_values[$i];

		next if (!defined ($old_value) and !defined ($new_value));
		next if (defined ($old_value) and defined ($new_value) and $old_value eq $new_value);

		my $old_value_esc = qquote ($old_value);
		#print STDERR "log_update_or_confess ('trouble_ticket', 'id', $obj->{'id'}, undef, '$field', undef, $old_value_esc);\n";
		log_update_or_confess ('trouble_ticket', 'id', $obj->{'id'}, undef, $field, undef, $old_value);

		push (@changes, "$field = " . qquote ($new_value));
	}

	return unless @changes;

	my $sql = 'UPDATE trouble_ticket SET ' . join (', ', @changes) . ' WHERE id = ' . $obj->{'id'};
	#print STDERR "$sql\n";
	do_or_confess ($sql);

	_update_db_kunden ($obj);

	return ($obj->{'id'});
}

=item I<$id> = I$obj>-E<gt>B<save> ()

Speichert das Trouble-Ticket in der Datenbank. Liefert die ID zurueck. Ggf. ist
die ID schon vorher bekannt, in dem Fall wird der Eintrag in der Datenbank nur
aktualisiert.

=cut

sub save
{
	my $obj = shift;

	if ($obj->{'confirmer'})
	{
		$obj->{'_error'} = 'You cannot save an already confirmed Trouble-Ticket';
		return;
	}
	
	_save ($obj);
}

sub _save
{
	my $obj = shift;

	my $ret;

	my $confirmer = $obj->{'confirmer'} || 0;
	my $id = $obj->{'id'} || 0;
	my $ticket = $obj->{'ticket'} || 0;
	if ($confirmer and !$id)
	{
		$obj->{'_error'} = 'You cannot confirm a trouble-ticket that has not yet been saved!';
		return;
	}
	if ($confirmer and !$ticket)
	{
		$obj->{'_error'} = 'You need to create a Ticket before confirming a trouble-ticket!';
		return;
	}

	if ($confirmer)
	{
		$ret = _insert_db ($obj);
	}
	elsif ($id)
	{
		$ret = _update_db ($obj);
	}
	else
	{
		$ret = _insert_db ($obj);
	}

	return ($ret);
}

=item I<$success> = I<$obj>-E<gt>B<delete> ()

Loescht das Trouble-Ticket, falls es noch keine bestaetigte Version dieses
Trouble-Tickets gibt. Andernfalls wird das Trouble-Ticket nicht geloescht und
I<false> zurueckgegeben.

=cut

sub delete
{
	my $obj = shift;

	if (!is_deletable ($obj))
	{
		print STDERR "Cannot delete TT: " . $obj->{'_error'} . "\n";
		return;
	}

	my $id = $obj->{'id'} or die; # `is_deletable' should have caught this

	do_or_confess ("DELETE FROM trouble_ticket_kunde_assoc WHERE trouble_ticket = $id");
	do_or_confess ("DELETE FROM trouble_ticket WHERE id = $id");

	my $ticket  = $obj->{'ticket'}  || '-';
	my $subject = $obj->{'subject'} || '-';

	log_update_or_confess ('trouble_ticket', 'id', $id, undef,
			'rtticket', 'subject', '-', undef,
			$ticket, $subject);
	return (1);
}

=item I<$value> = I<$obj>-E<gt>B<get> (I<$attribut>)

Gibt den Wert des Attributs zurueck. Siehe auch oben unter L<ATTRIBUTE>.

=cut

sub get
{
	my $obj = shift;
	my $key = shift;

	if (!defined ($ValidFields->{$key}))
	{
		$obj->{'_error'} = "No such field: $key";
		cluck ("No such field: $key")
			unless $ENV{TESTING3};
		return;
	}

	if (!defined ($ValidFields->{$key}{'get'}))
	{
		$obj->{'_error'} = "Field `$key' not readable";
		return;
	}

	return ($ValidFields->{$key}{'get'}->($obj, $key, @_));
}

sub _get_descriptor
{
	my $obj = shift;
	my $key = shift;
	my $descr;

	my $id;
	my $name;
	my $info;

	if ($key ne 'type' and $key ne 'priority') { confess "Invalid key `$key'"; }
	$descr = $DescriptorMap->{$key};

	return if (!defined ($obj->{$key}));

	$name = $obj->{$key};
	return ($name) if (!wantarray ());

	$id   = find_descr ($descr, $name, 0x01);
	$info = info_descr ($descr, $id);
	
	return ($name, $info);
}

sub _get_flag
{
	my $obj = shift;
	my $key = shift;
	my $descr;
	my @flags = ();
	
	if ($key ne 'flag_kunde' and $key ne 'flag_person') { confess "Invalid key `$key'"; }
	$descr = $DescriptorMap->{$key};

	return if (!defined ($obj->{$key}) or !ref ($obj->{$key}));

	return (@{$obj->{$key}}) if (wantarray ());
	return ($obj->{$key});
}

sub _get_integer
{
	my $obj = shift;
	my $key = shift;

	return if (!defined ($obj->{$key}));
	return ($obj->{$key});
}

sub _get_list
{
	my $obj = shift;
	my $key = shift;

	$obj->{$key} = [] if (!ref ($obj->{$key}) or ref ($obj->{$key}) ne 'ARRAY');

	my @list = @{$obj->{$key}};

	return (@list) if (wantarray ());
	return (\@list);
}

sub _get_person
{
	my $obj = shift;
	my $key = shift;

	return if (!defined ($obj->{$key}));
	return ($obj->{$key}) if (!wantarray ());

	my $id = $obj->{$key};
	my $pers = noris::Person->load (id => $id);

	return if (!$pers);
	
	my $user = $pers->get ('user');
	my $name = $pers->get ('name');

	return ($id, $user, $name);
}

sub _get_select_kunden
{
	my $obj = shift;
	my $key = shift;

	if (!$obj->{$key})
	{
		$obj->{$key} = 'flags';
	}

	return ($obj->{$key});
}

sub _get_string
{
	my $obj = shift;
	my $key = shift;

	return if (!defined ($obj->{$key}));
	return ($obj->{$key});
}

sub _get_time
{
	my $obj = shift;
	my $key = shift;

	return if (!defined ($obj->{$key}));
	return ($obj->{$key});
}

=item I<$obj>-E<gt>B<set> (I<$attribut>, I<$wert>)

Setzt den Wert des Attributs. Siehe auch oben unter L<ATTRIBUTE>.

=cut

sub set
{
	my $obj = shift;
	my $key = shift;
	my $val = shift;
	my $old;

	if ($obj->{'confirmer'})
	{
		$obj->{'_error'} = "You cannot change a confirmed Trouble-Ticket!";
		return;
	}

	if (!defined ($ValidFields->{$key}))
	{
		$obj->{'_error'} = "No such field: $key";
		cluck ("No such field: $key")
			unless $ENV{TESTING3};
		return;
	}

	if (!defined ($ValidFields->{$key}{'set'}))
	{
		$obj->{'_error'} = "Field `$key' not writeable";
		return;
	}

	return ($ValidFields->{$key}{'set'}->($obj, $key, $val, @_));
}

sub _set_descriptor
{
	my $obj = shift;
	my $key = shift;
	my $val = shift;
	my $descr;

	my $id;
	my $name = $val;

	if ($key ne 'type' and $key ne 'priority') { confess "Invalid key `$key'"; }
	$descr = $DescriptorMap->{$key};

	$id = find_descr ($descr, $name, 0x00);
	
	if (!$id)
	{
		$obj->{'_error'} = "Not a valid $descr descriptor: $val";
		return;
	}

	return ($obj->{$key} = $name);
}

# `flags' sind ein Bitfeld von Descriptoren.
sub _set_flag
{
	my $obj = shift;
	my $key = shift;
	my $val = shift;
	my $descr;
	my @descriptor_id = ();

	if ($key ne 'flag_kunde' and $key ne 'flag_person') { confess "Invalid key `$key'"; }
	$descr = $DescriptorMap->{$key};

	if (!ref ($val))
	{
		$val = [$val];
	}

	if (ref ($val) ne 'ARRAY')
	{
		$obj->{'_error'} = "Flags need to be a sclar or array reference: $key";
		return;
	}

	for (@$val)
	{
		my $name = $_;
		my $id = find_descr ($descr, $name, 0x01);
	
		if (!defined ($id))
		{
			$obj->{'_error'} = "No such flag: $name";
			next;
		}

		next if (grep { defined ($_) and ($_ eq $name) } (@descriptor_id));

		push (@descriptor_id, $name);
	}

	$obj->{$key} = \@descriptor_id;

	_update_kunden ($obj);

	return ($obj->{$key});
}

sub _set_email
{
	my $obj = shift;
	my $key = shift;
	my $val = shift;

	my @invalid = ();

	$obj->{$key} = [];

	my $check = Email::Valid->new (-mxcheck => 1);
	for (@$val)
	{
		my $orig = $_ or next;
		my $mail = $check->address ($orig);

		if (!$mail)
		{
			push (@invalid, $orig);
			next;
		}

		push (@{$obj->{$key}}, $mail);
	}

	$obj->{'_error'} = "The following email-addresses are invalid: " . join (', ', @invalid) if (@invalid);

	return (scalar (@invalid));
}

sub _set_id
{
	my $obj = shift;
	my $key = shift;
	my $val = shift;

	$val ||= 0;
	$val =~ s/\D//g;

	if (!$val)
	{
		$obj->{'_error'} = "No valid ID given";
		return;
	}

	my $valid = DoFn ("SELECT COUNT(*) FROM trouble_ticket WHERE id = $val");

	if ($valid != 1)
	{
		$obj->{'_error'} = "No object with the ID $val exists!";
		return;
	}

	return ($obj->{$key} = $val);
}

sub _update_kunden
{
	my $obj = shift;

	$obj->{'flag_kunde'}  ||= [];
	$obj->{'flag_person'} ||= [];

	{
		my $kunde  = join (', ', @{$obj->{'flag_kunde'}});
		my $person = join (', ', @{$obj->{'flag_person'}});

		#print STDERR "[noris::TroubleTicket::_update_kunden] kundeid_get_by_flags (kunde => [$kunde], person => [$person]);\n";
	}

	$obj->{'kunden'} = kundeid_get_by_flags (kunde => $obj->{'flag_kunde'}, person => $obj->{'flag_person'});
}

sub _set_select_kunden
{
	my $obj = shift;
	my $key = shift;
	my $val = shift;

	if ($val =~ m/(all|none|flags)/)
	{
		$val = $1;
	}
	else
	{
		$val = 'flags';
	}

	if ($val eq 'all')
	{
		$obj->{'kunden'} = kundeid_get_all ();
	}
	elsif ($val eq 'flags')
	{
		$obj->{'flag_kunde'}  ||= [];
		$obj->{'flag_person'} ||= [];

		$obj->{'kunden'} = kundeid_get_by_flags (kunde => $obj->{'flag_kunde'}, person => $obj->{'flag_person'});
	}
	elsif ($val eq 'none')
	{
		$obj->{'kunden'} = {};
	}

	return ($obj->{$key} = $val);
}

sub _set_ticket
{
	my $obj = shift;
	my $key = shift;
	my $val = shift;
	my $tid;

	return if (!$val);
	
	# RT#325390: Mehrere, in ein gemeinsames Ticket zusammenge'merge'te
	# RT-Tickets sollen je ein Trouble-Ticket haben koennen. -octo
	($tid) = $val =~ m/(\d+)/;

	if (!$tid)
	{
		$obj->{'_error'} = "Invalid Ticket-ID `$val'";
		return;
	}

	if (!$obj->{'id'})
	{
		my $num = DoFn ("SELECT COUNT(*) FROM trouble_ticket WHERE rtticket = $tid");
		if ($num)
		{
			$obj->{'_error'} = "Another Trouble-Ticket is already associated with RT-Ticket #$tid";
			return;
		}
	}
	else
	{
		$obj->{$key} = DoFn ("SELECT rtticket FROM trouble_ticket WHERE id = " . $obj->{'id'});
		if (defined ($obj->{$key}) && ($obj->{$key}) && ($obj->{$key} != $val))
		{
			$obj->{'_error'} = "This Trouble-Ticket is already associated with RT-Ticket #" . $obj->{$key};
			return;
		}
	}

	$obj->{$key} = $tid;

	return ($tid);
}

sub _set_string
{
	my $obj = shift;
	my $key = shift;
	my $val = shift;

	return ($obj->{$key} = $val);
}

sub _set_time
{
	my $obj = shift;
	my $key = shift;
	my $val = shift;

	return ($obj->{$key} = undef)
		if ($key eq 'ende' and (!defined ($val) or ($val <= 0)));
	return ($obj->{$key} = undef)
		if ($key eq 'ende' and defined ($obj->{'beginn'}) and $val < $obj->{'beginn'});
	return if ($key eq 'beginn' and !defined ($val));
	return if ($key eq 'beginn' and defined ($obj->{'ende'}) and $val > $obj->{'ende'});

	return ($obj->{$key} = $val);
}

=item I<$fehlermeldung> = I<$obj>-E<gt>B<error> ()

Gibt die Fehlermeldung als String zurueck, falls eine aufgetreten ist. Dabei
wird der aktuelle Fehler geloescht, ein zweiter Aufruf wird daher keine
Probleme mehr feststellen. Die Vorzugsmethode um diese Methode zu benutzen ist
wie folgt:

  if (my $err = $tt->error ()) { die "Fehler im Trouble-Ticket-System: $err"; }

=cut

sub error
{
	my $obj = shift;
	my $str;

	return if (!$obj->{'_error'});

	$str = $obj->{'_error'};
	$obj->{'_error'} = '';

	return ($str);
}

=item I<$obj>-E<gt>B<add_kunde> (I<$kunde_id>)

Fuegt einen Kunden der Empfaengerliste hinzu, sofern er nicht sowieso schon
durch die Flags ausgewaehlt war. Die Methode erwartet eine Kunde-ID,
Kundennamen sind nicht zulaessig.

=cut

sub add_kunde
{
	my $obj = shift;
	my $kundeid = shift;
	my $kundename;

	if ($obj->{'confirmer'})
	{
		$obj->{'_error'} = "You cannot change a confirmed Trouble-Ticket!";
		return;
	}

	return if (defined ($obj->{'kunden'}{$kundeid}));

	$kundename = name_kunde ($kundeid);
	$obj->{'kunden'}{$kundeid} = $kundename if ($kundename);

	return;
}

=item I<$obj>-E<gt>B<remove_kunde> (I<$kunde_id>)

Entfernt den angegebenen Kunden aus der Empfaengerliste, falls er dort
aufgefuehrt ist.

=cut

sub remove_kunde
{
	my $obj = shift;
	my $kundeid = shift;

	if ($obj->{'confirmer'})
	{
		$obj->{'_error'} = "You cannot change a confirmed Trouble-Ticket!";
		return;
	}

	return if (!defined ($obj->{'kunden'}{$kundeid}));

	delete ($obj->{'kunden'}{$kundeid});

	return;
}

=item I<$obj>-E<gt>B<get_kunden> ()

Liefert eine Liste (oder eine Hash-Referenz in skalatem Kontext) der IDs aller
ausgewaehlten Kunden zurueck. Der Hash hat den Aufbau C<$kundeid =E<gt>
$kundename>.

=cut

sub get_kunden
{
	my $obj = shift;
	my @list;

	return if (!$obj->{'kunden'});

	return (keys %{$obj->{'kunden'}}) if (wantarray ());
	return ($obj->{'kunden'});
}

=item I<$str> = I<$obj>-E<gt>B<to_mail> ()

...

=cut

sub to_mail
{
	my $obj = shift;
	my $mail;

	my $to = 'Ausgewählte noris-network-Kunden:;';
	if ($obj->get ('select_kunden') eq 'all')
	{
		$to = 'Alle noris-network-Kunden:;';
	}

	my $subject = $obj->get ('subject');
	if (!$subject)
	{
		$obj->{_error} = 'Bei diesem Trouble Ticket fehlt ein Betreff.';
		return;
	}

	unless ( $obj->{description} ) {
		$obj->{_error} = 'Bei diesem Trouble Ticket fehlt die Problembeschreibung.';
		return;
	}

	$subject = ": $subject";
	if (defined (my $obj_ticket = $obj->get ('ticket'))) {
		$subject = " [noris #$obj_ticket]$subject";
	}
	$subject = SUBJECT_PREFIX . $subject;

	$mail = <<MAIL;
Content-Type: text/plain; charset=utf-8
Content-Transfer-Encoding: 8bit
From: support\@noris.net
To: ${\ encode_mimewords($to) }
Subject: ${\ encode_mimewords($subject) }
Precedence: bulk

MAIL

	my ($type_name, $type_info) = $obj->get ('type');
	if (!$type_name or !$type_info)
	{
		$obj->{_error} = 'Bei diesem Trouble Ticket fehlt der Typ.';
		return;
	}

	my ($prio_name, $prio_info) = $obj->get ('priority');
	if (!$prio_name or !$prio_info)
	{
		$obj->{_error} = 'Bei diesem Trouble Ticket fehlt die Priorität.';
		return;
	}

	# strftime(fmt, sec, min, hour, mday, mon, year, wday = -1, yday = -1, isdst = -1)
	# ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = localtime (time);

	my $beginn = $obj->get ('beginn');
	my $ende   = $obj->get ('ende');

	my $time_format = '%A, der %e. %B, %k:%M Uhr';
	my $offset = sprintf '+%02X00', ( tz_local_offset() / 3600 );

	$beginn = time () if (!$beginn);
	$beginn = strftime ($time_format, localtime ($beginn));
	if (defined ($ende))
	{
		$ende = strftime ($time_format, localtime ($ende)) . " $offset";
	}
	else
	{
		$ende = 'Ende nicht bekannt';
	}

	$mail .= <<MAIL;
========================================================================
Ticket-Typ:         $type_info
Störungsklasse:     $prio_info

Start des Problems: $beginn $offset
Ende  des Problems: $ende

MAIL

	for (@$TextfieldOrder)
	{
		my $field_name = $_;
		my $field_info = $TextfieldMap->{$field_name};
		my $field_text = $obj->get ($field_name);

		next if (!$field_text);

		{
			my @lines = split (m/(\r\n?|\n)/, $field_text);
			$field_text = join ("\n", wrap ('', '', @lines));
		}

		my $length_total = 72 - (2 + length ($field_info));
		my $length_right = int ($length_total / 2);
		my $length_left = $length_total - $length_right;

		my $padding_left  = '=' x $length_left;
		my $padding_right = '=' x $length_right;

		$mail .= <<MAIL;
$padding_left $field_info $padding_right
$field_text

MAIL
	}

	return ($mail);
}

=item I<$obj>-E<gt>B<confirm> (I<$person_id>)

Merkiert das Trouble-Ticket in seiner jetzigen Form als "bestaetigt".
Bestaetigte Trouble-Tickets sind zum Beispiel auf unserer Homepage fuer Kunden
sichtbar.

=cut

sub confirm
{
	my $obj = shift;
	my $id  = shift;

	if ($obj->{'confirmer'})
	{
		$obj->{'_error'} = "You cannot confirm a confirmed Trouble-Ticket!";
		return;
	}

	if (!$id)
	{
		$obj->{'_error'} = "`confirm' was called without a Person-ID";
		return;
	}

	my $num = DoFn ("SELECT COUNT(*) FROM person WHERE id = $id");

	if ($num != 1)
	{
		$obj->{'_error'} = "There is no person with the ID $id";
		return;
	}

	$obj->{'confirmer'} = $id;

	_save ($obj);
}

=item I<$obj>-E<gt>B<get_history> ()

Returns a list of changes. Each entry is of the form

  {time => $time, num => $num, user => $user}

where C<time> is the epoch when the change took place, C<num> is the number of
fields changed and C<user> is the username of the person doing the changes.

=cut

sub get_history
{
	my $obj = shift;
	
	return if (!$obj->{'id'});
	my $id = $obj->{'id'};

	my @changes = ();

	DoSelect (sub {
		my $time = shift;
		my $num  = shift;
		my $user = shift;
		push (@changes, {'time' => $time, num => $num, user => $user});
	}, <<SQL);
	SELECT UNIX_TIMESTAMP(ul.timestamp) AS epoch, COUNT(*) AS num_changes, p.user AS username
	 FROM ( updatelog ul, db_tabelle tbl, updatelog_spalten idx )
	 LEFT JOIN person p ON ul.person = p.id
	WHERE tbl.name = "trouble_ticket" AND idx.namen = "id"
	  AND ul.wert = ${\qquote ($id)}
	  AND ul.db_tabelle = tbl.id AND ul.indexspalten = idx.id
	GROUP BY ul.person, ul.timestamp, p.id
	ORDER BY ul.timestamp ASC
SQL
	return (@changes) if (wantarray ());
	return (\@changes);
}

=item I<$bool> = I<$obj>-E<gt>B<is_deletable> ()

Liefert I<true> wenn das Trouble-Ticket geloescht werden darf und I<false>
andernfalls. Ein Trouble-Ticket darf nur dann geloescht werden, wenn es
gespeichert wurde (also eine ID besitzt) und keine bestaetigte Version des
Trouble-Tickets existiert. Vgl. RT#257123

Wenn I<false> zurueckgegeben wird ist anschliessend eine Fehlermeldung gesetzt,
die den Grund angibt warum das TT nicht geloescht werden darf.

=cut

sub is_deletable
{
	my $obj = shift;

	my $id = $obj->{'id'};
	if (!$id)
	{
		$obj->{'_error'} = "Cannot delete Trouble-Ticket: It's not saved yet.";
		return;
	}

	if ($obj->{'confirmer'})
	{
		$obj->{'_error'} = "Cowardly refusing to delete a confirmed Trouble-Ticket.";
		return;
	}

	if ($obj->{'ticket'})
	{
		my $confirmed = DoFn <<SQL;
			SELECT count(*)
			FROM trouble_ticket
			WHERE rtticket = $obj->{'ticket'} AND confirmer IS NOT NULL
SQL
		if ($confirmed > 0)
		{
			$obj->{'_error'} = "Cowardly refusing to delete a confirmed Trouble-Ticket.";
			return;
		}
	}

	return (1);
}

=back

=head1 STATISCHE FUNKTIONEN

=over 4

=item I<@tt_ids> = B<tt_get_by_kunde> (I<$kunde_id>)

Liefert eine Liste (bzw. Array-Referenz im skalaren Kontext) der IDs aller
Trouble-Tickets die ein bestimmter Kunde sehen darf.

=cut

sub tt_get_by_kunde
{
	my $kundeid = shift;

	my @ret = ();
	DoSelect (sub {
		my $ttid = shift;
		push (@ret, $ttid);
	}, <<SQL);
	SELECT trouble_ticket FROM trouble_ticket_kunde_assoc WHERE kunde = $kundeid
SQL

	return (@ret) if (wantarray ());
	return (\@ret);
}

=item I<@tt_ids> = B<tt_get_incomplete> ()

Liefert alle Trouble-Ticket-IDs deren zugehoerige Trouble-Tickets noch nicht
vollstaendig sind.

=cut

sub tt_get_incomplete
{
	my @ret = ();

	my $tickets = {};
	my @tickets;

	DoSelect (sub {
		my $min_id = shift;
		my $max_id = shift;
		my $ticket = shift;

		#print STDERR "[RT#$ticket] min = $min_id; max = $max_id\n";

		$tickets->{$ticket} = {min => $min_id, max => $max_id};
	}, <<SQL);
	SELECT MIN(id), MAX(id), rtticket
	FROM trouble_ticket
	WHERE rtticket IS NOT NULL
	GROUP BY rtticket
SQL

	@tickets = map { $tickets->{$_}{'max'} } (keys %$tickets);

	DoSelect (sub {
		my $ticket  = shift;
		my $ende      = shift;
		my $resolve   = shift;

		#print STDERR "[RT#$ticket] ende = " . ($ende ? 'true' : 'false') . "; resolve = " . ($resolve ? 'true' : 'false') . ";\n";

		# min == max  =>  Das erste Ticket => noch nicht bestaetigt => noch nicht fertig. Vgl. RT#255874
		# -octo
		push (@ret, $tickets->{$ticket}{'min'}) if (!$ende or !$resolve
			or ($tickets->{$ticket}{'min'} == $tickets->{$ticket}{'max'}))
	}, <<SQL);
	SELECT rtticket, ende, text_resolve
	FROM trouble_ticket
	WHERE id IN (${\join (', ', @tickets)})
SQL

	DoSelect (sub {
		my $id = shift;
		push (@ret, $id);
	}, <<SQL);
	SELECT id
	FROM trouble_ticket
	WHERE rtticket IS NULL
SQL

	@ret = sort { $a <=> $b } (@ret);

	return (@ret) if (wantarray ());
	return (\@ret);
}

=item I<@descriptoren> = B<tt_get_descriptor_list> (I<$attribut>)

Liefert alle fuer ein Attribut definierten Descriptoren zurueck. Im skalaren
Kontext wird eine Hash-Referenz zurueck gegeben. Der Hash hat die Form:

  $id => {name => $name, info => $info}

Im Listen-Kontext wird eine Liste folgender Form zurueckgegeben. Die Liste ist
nach den IDs der Deskriptoren sortiert.

  $name0, $info0, $name1, $info1, ...

=cut

sub tt_get_descriptor_list
{
	my $key = shift;
	my $descr;
	my %tmp;

	my $ret = {};
	my @ret = ();

	$descr = $DescriptorMap->{$key};
	confess "Invalid key `$key'" if (!$descr);

	%tmp = map_descr( $descr, '!hide' );
	for (keys %tmp)
	{
		my $name = $_;
		my $id   = $tmp{$name};
		my $info = info_descr ($descr, $id);

		$ret->{$id} = {name => $name, info => $info};
	}

	return ($ret) if (!wantarray ());

	for (sort (keys %$ret))
	{
		my $id = $_;
		my $name = $ret->{$id}{'name'};
		my $info = $ret->{$id}{'info'};

		push (@ret, $name, $info);
	}

	return (@ret);
}

=back

=head1 DATENBANK-MADNESS

Nachdem das urspruengliche Datenbank-Konzept mit fuenf Tabellen und
Redundanz-Freiheit besser in das Konzept der Kunde-Datenbank eingegliedert
wurde, sieht es jetzt so aus:

Es gibt zwei Tabellen: Eine, die das Trouble-Ticket beinhaltet, und eine, die
die N:M-Zuordnung von Trouble-Tickets zu Kunden bewerkstelligt. Die zweite ist
ziemlich einfach und eingaenglich und wird daher nicht naeher beschrieben. Mit
der Tabelle fuer Trouble-Tickets schaut es ein wenig anders aus.

Jedes Trouble-Ticket besteht aus einem oder mehreren Eintraegen in der Tabelle.
Gibt es zu einem Trouble-Ticket mehrere Eintraege _muss_ das Feld C<rtticket>
gesetzt sein. Darueber findet die Zuordnung statt. Ausserdem muessen alls
Eintraege (bis auf einer) die Spalte C<confirmer> auf non-C<NULL> gesetzt
haben. Es gibt also n-1 bestaetigte "Versionen" eines Trouble-Tickets und
(immer genau) eine unbestaetigte Version des Trouble-Tickets.

Die Lebenszeit eines Trouble-Tickets laesst sich also wie folgt in Abschnitte
fassen:

  Erster Schritt: TT erstellen.
    id ! rtticket ! confirmer
    ---+----------+----------
     1 !     NULL !      NULL

  Zweiter Schritt: RT-Ticket erstellen.
    id ! rtticket ! confirmer
    ---+----------+----------
     1 !   123456 !      NULL

  Dritter Schritt: TT bestaetigen.
    id ! rtticket ! confirmer
    ---+----------+----------
     1 !   123456 !      NULL
     2 !   123456 !     54321

=head2 Logging

Aenderungen an der C<trouble_ticket>-Tabelle werden mit der kunde-Funktion
C<log_update> protokolliert. Nachdem Aenderungen B<immer nur> an der
unbestaetigten Fassung vorgenommen werden sind auch nur diese Aenderungen
interessant. Es gibt Eintraege im Update-Log fuer das Erstellen der
bestaetigten Eintraege, aber die sagen nur, dass die Eintraege angelegt wurden.

=head1 SIEHE AUCH

L<RT#218455>

=head1 AUTOR

Florian octo Forster E<lt>octo@noris.netE<gt> fuer die noris network AG
L<http://noris.net/>

=cut

# vim:background=light:hlsearch:incsearch
