use strict;
use utf8;
use warnings;

package noris::Ticket::API::RT::SelectResult;

use Carp qw(confess croak);
use Dbase::Globals qw(find_descr);
use Dbase::Help qw(DoSel DoSelect in_list);
use Moose;
use noris::Ticket::API::RT::Field;
use Tie::IxHash;
use Umlaut qw(binmodus);

# Schöner wäre hier ein "isa => 'ArrayRef[noris::Ticket::API::RT::Field]'"
# mit coerce, aber das kann die Moose-Version unter Etch leider noch nicht.
has 'attributes' => ( is => 'ro', isa => 'ArrayRef', auto_deref => 1, default => sub { [] } );

# Das hier ist der Workaround für o.g. Unzulänglichkeit der alten Moose-
# Version. Es ist im Prinzip natürlich nur ein Cache, der aber sinnvoll
# erscheint, weil wir sonst in ->next() für jeden Datensatz neu in der
# %field_map nachschauen müssten.
has '_attributes' => (
    is         => 'ro',
    isa        => 'ArrayRef',
    auto_deref => 1,
    lazy       => 1,
    default    => sub {
        my $self = shift;
        [ map _field($_), $self->attributes ];
    }
);

has 'origin' => ( is => 'ro', isa => 'HashRef' );

has 'sth' => (
    is      => 'ro',
    isa     => 'Object',
    default => sub {
        my ($self) = @_;

        my @attr = $self->_attributes;

        # Tickets, die bereits "im_otrs" sind, sind so gut wie nicht existent.
        # (Muss als Sonderfall behandelt werden, weil er "merged" vorgeht, vgl.
        # #10005981):
        my @where = 'ticket.status != ' . find_descr( tickets => im_otrs => 1 );

        tie my %joins, 'Tie::IxHash';
        {
            my $query = $self->query;

            while ( my ( $field, $filter ) = each %$query ) {
                $field = _field($field);

                for ( $field->is_listfield ? @$filter : $filter ) {

                    my ( $operator, @values ) = ref() ? @$_ : ( in => $_ );

                    my $method = "where_$operator";
                    push @where,
                      $field->$method( map $field->api2rt($_), @values );

                    ++$joins{$_} for $field->joins4query;
                }
            }
        }

        ++$joins{$_} for map $_->joins, @attr;

        my $sql = join ' ',
          SELECT => join( ',', map $_->select_name, @attr ) || 1,
          FROM => 'ticket',
          keys %joins, @where ? ( WHERE => join ' AND ', @where ) : (),

          # GROUP BY (nur) erforderlich für JOINS bei CIs und Leitungen
          'GROUP BY ticket.id';

        noris::Ticket::API::_debug( sql => "->sth:\n$sql" );

        DoSel($sql);
    },
    lazy => 1,
);
has 'query' => ( is => 'ro', isa => 'HashRef', required => 1 );
has 'connection' => ( is => 'ro', isa => 'noris::Ticket::API::RT::Connection' );

{
    my $queue_ptr;

    my $origin_debug = sub { '(' . join( '+', @{ ${ +shift } } ) . ')' };

    sub follow_tickets {
        my ( $self, %param ) = @_;

        my ( $follow_back, $follow_forth );
        {
            ref( my $follow_by = $param{follow_by} )
              or confess(
                'follow_tickets() braucht eine Parameterliste "follow_by"');

            for (@$follow_by) {
                if ( $_ eq 'main' || $_ eq 'split_source' ) {
                    $follow_back ||= 1;
                }
                elsif ( $_ eq 'sub' || $_ eq 'split_target' ) {
                    $follow_forth ||= 1;
                }
            }
        }

        my @ticket_numbers;
        {

            # Wir erzeugen erstmal ein neues SelectResult mit demselben Query,
            # damit sich bei "uns selbst" kein Cursor verschiebt und also bei
            # Bedarf auch noch ->next() funktioniert.
            # Außerdem brauchen wir hier die ticket_number, und zwar einerseits
            # nur die und sie andererseits auch, wenn sie in "unserer eigenen"
            # Attribut-Liste nicht angefragt wurde.
            my $base_result = $self->new(
                connection => $self->connection,
                attributes => ['ticket_number'],
                query      => $self->query,
            );
            while ( defined( my $row = $base_result->next ) ) {
                push @ticket_numbers, $row->[0];
            }
        }

        my %origin = map +( $_ => \[$_] ), @ticket_numbers;

        if ( $follow_back || $follow_forth ) {

            $queue_ptr ||= find_descr( tickett => queue_ptr => 1 );

            # Jetzt ausgehend von den Ticket-Nummern
            # die verbundenen Tickets ermitteln:
            my @sql_where = (
                'ticket.ticket = source_ticket.ticket',
                'source_ticket.id = ticketid.ticket',
                'ticketid.inhalt = dest_ticket.id',
                "ticketid.typ = $queue_ptr",
            );

            unless ( $follow_back && $follow_forth ) {
                push @sql_where, 'dest_ticket.id > source_ticket.id'
                  unless $follow_back;
                push @sql_where, 'dest_ticket.id < source_ticket.id'
                  unless $follow_forth;
            }

            my $graph;
            if ( defined $ENV{TICKET_API_RT_CONNECTION_GRAPH} ) {
                require GraphViz;
                $graph = noris::Ticket::API::RT::SelectResult::GraphViz->new(
                    defined $ENV{TICKET_API_RT_CONNECTION_GRAPH_OPTIONS}
                    ? split ',',
                    $ENV{TICKET_API_RT_CONNECTION_GRAPH_OPTIONS}
                    : ()
                );
                $graph->add_node(
                    $_,
                    fillcolor => '#CCCCFF',
                    style     => 'filled',
                ) for @ticket_numbers;
            }

            my @tickets = @ticket_numbers;

            while (@tickets) {
                my @new_tickets;
                my $sql = <<_
	SELECT ticket.id, dest_ticket.ticket
	FROM   ticket, ticket source_ticket, ticket dest_ticket, ticketid
_
                  . "\tWHERE "
                  . join( ' AND ',
                    @sql_where, in_list( 'ticket.id', '', @tickets ),
                  );
                noris::Ticket::API::_debug( sql => "->follow_tickets:\n$sql" );
                DoSelect {
                    my ( $source, $dest ) = @_;

                    if ($graph) {
                        $graph->add_node($dest);
                        $graph->add_edge( $source => $dest );
                    }

                    unless ( exists $origin{$dest} ) {

                        # neu entdecktes Ticket:
                        noris::Ticket::API::_debug(
                                follow => "Ticket $source vererbt origin "
                              . $origin_debug->( $origin{$source} )
                              . " an neu entdecktes Ticket $dest." );
                        $origin{$dest} = $origin{$source};

                        # außerdem müssen wir in diesem Fall weitersuchen:
                        push @new_tickets, $dest;
                    }
                    elsif ( ${ $origin{$source} } eq ${ $origin{$dest} } ) {

                        # Bereits bekanntes Ticket hat denselben Ursprung:
                        noris::Ticket::API::_debug(
                                follow => "Tickets $source und $dest "
                              . 'haben den gemeinsamen origin '
                              . $origin_debug->( $origin{$source} )
                              . '.' );
                    }
                    else {

                        # bereits bekanntes Ticket mit bislang anderem Ursprung
                        # => Zweige vereinigen:
                        noris::Ticket::API::_debug(
                            follow => 'Vereinige Origins: '
                              . join( ' + ',
                                map "Ticket $_ "
                                  . $origin_debug->( $origin{$_} ),
                                $source, $dest )
                              . '.'
                        );
                        my $origins = [
                            sort { $a <=> $b } map @{ ${ $origin{$_} } },
                            $source, $dest
                        ];
                        ${ $origin{$_} } = $origins for @$origins;
                    }

                    noris::Ticket::API::_debug(
                        follow_guts => sub {
                            require Data::Dump and Data::Dump->import('pp')
                              unless defined &pp;
                            '%origin = ' . pp( \%origin ) . "\n";
                        }
                    );
                }
                $sql;
                @tickets = @new_tickets;
            }

            if ($graph) {
                my $type = $ENV{TICKET_API_RT_CONNECTION_GRAPH} =~ /([^.]+)\z/
                  && $graph->can("as_$1") ? "as_$1" : 'as_png';
                open my $fh, '>', $ENV{TICKET_API_RT_CONNECTION_GRAPH}
                  or die
                  "open('>','$ENV{TICKET_API_RT_CONNECTION_GRAPH}'): $!\n";
                binmodus($fh);
                print $fh $graph->$type;
                close $fh;
            }
        }

        # %origin vereinfachen (Zwischen-Referenzen entfernen):
        $_ = $$_ for values %origin;

        noris::Ticket::API::_debug(
            origin => sub {

                # Tickets mit identischem "origin" zusammenfassen:
                my %origin_group;
                push @{ $origin_group{ join '+', @{ $origin{$_} } } }, $_
                  for sort { $a <=> $b } keys %origin;

                # um die Origin-Gruppen einfach numerisch nach der jeweils
                # ersten Ticket-Nummer sortieren zu können:
                no warnings 'numeric';

                join "\n", 'RT-Ticket ... folgte aus ...:',
                  map( "  $_ => " . join( '+', @{ $origin{$_} } ),
                    sort { $a <=> $b } keys %origin ),
                  '', 'Aus ... folgte(n) die RT-Tickets ...:',
                  map( "  $_ => " . join( '+', @{ $origin_group{$_} } ),
                    sort { $a <=> $b } keys %origin_group );
            }
        );

        return $self->new(
            connection => $self->connection,
            attributes => $param{attributes},
            query      => { ticket_number => [ in => keys %origin ] },
            origin     => \%origin,
        );
    }
}

sub foreach_row {
    my ( $self, $fn ) = @_;
    my $row;
    while ( defined( $row = $self->next ) ) {
        $fn->(@$row);
    }
}

sub next {
    my $self   = shift;
    my @record = $self->sth->nextrow or return;
    my @attr   = $self->_attributes or return [];
    [ map $attr[$_]->rt2api( $record[$_], $self ), 0 .. $#record ];
}

sub count { shift->sth->rows }

{
    my %field_map = (
        _gen_field(
            confitems => LinkObject =>
              db_name => 'confitem.name',
            db_table => 'confitem',
        ),
        _gen_field( changed => Date => db_name => 'ticket.d_acted' ),
        _gen_field( created => Date => db_name => 'ticket.beginn' ),
        _gen_field( customer_visibility => CustomerVisibility => ),
        _gen_field( due => Date => db_name => 'ticket.endtermin' ),
        _gen_field( est_effort => SecsToHours => db_name => 'ticket.zeit' ),
        _gen_field(
            kunde     => String =>
              db_name => 'kunde.name',
            joins => ['JOIN kunde ON kunde.id = ticket.kunde'],
        ),
        map( _gen_field(
                "incident_$_->[0]", $_->[1] =~ /^t_/ ? 'Date' : 'String',
                db_name => "rt_incidents.$_->[1]",
                joins =>
                  ['LEFT JOIN rt_incidents ON rt_incidents.ticket = ticket.id']
            ),
            [qw/ cause    ursache       /],
            [qw/ resolved t_entstoerung /],
            [qw/ response t_reaktion    /],
            [qw/ start    t_meldung     /],
            [qw/ type     art           /] ),
        _gen_field( info => String => db_name => 'ticket.infotext' ),
        _gen_field(
            leitungen => LinkObject =>
              db_name => 'leitung.name',
            db_table => 'leitung',
        ),
        _gen_field( locked => Locked => db_name => 'ticket.bearbeiter' ),
        _gen_field( merge_parent => Numeric => db_name => 'ticket.ticket' ),
        _gen_field( merge_root => Numeric => db_name => 'ticket.ticket' ),
        _gen_field( origin => Origin => db_name => 'ticket.id' ),
        _gen_field( otrs_status => NotSupported => ),
        _gen_field(
            owner     => String =>
              db_name => 'person.user',
            joins => ['LEFT JOIN person ON ticket.bearbeiter = person.id'],
        ),
        _gen_field( pending_until => Date => db_name => 'ticket.termin' ),
        _gen_field(
            priority  => Numeric =>
              db_name => 'ticket.wichtig',
            format => '%02d',
        ),
        _gen_field(
            queue     => String =>
              db_name => 'queue.name',
            joins => ['JOIN queue ON queue.id = ticket.queue'],
        ),
        _gen_field(
            sla_relevant => Flag =>
              db_name    => 'rt_incidents.flags',
            descr => 'rt_incidents_flags',
            name  => 'sla-relevant',
            joins =>
              ['LEFT JOIN rt_incidents ON rt_incidents.ticket = ticket.id']
        ),
        _gen_field( status => Status => ),
        _gen_field( ticket_email => EMail => db_name => 'ticket.id' ),
        _gen_field( ticket_number => Numeric => db_name => 'ticket.id' ),
        _gen_field( ticket_url => URL => db_name => 'ticket.id' ),
        _gen_field( title => String => db_name => 'ticket.subject' ),
        _gen_field(
            type      => String =>
              db_name => 'queue_areas.name',
            joins =>
              ['LEFT JOIN queue_areas ON queue_areas.id = ticket.queue_area'],
        ),
    );

    sub _field {
        my ($name) = @_;
        $field_map{$name} || croak("Unbekanntes Ticket-Attribut: $name");
    }
}

sub _gen_field {
    my ( $api_name, $type, @type_specific_args ) = @_;

    my $class = "noris::Ticket::API::RT::Field::$type";

    # Ohne eval würde $class als Dateiname interpretiert:
    eval "require $class";
    die $@ if length $@;

    $api_name => $class->new( api_name => $api_name, @type_specific_args );
}

{
    package noris::Ticket::API::RT::SelectResult::GraphViz;
    our @ISA = 'GraphViz';

    use Dbase::Help qw(DoFn qquote);

    sub add_node {
        my ( $self, $name, %option ) = @_;
        if (   $ENV{TICKET_API_RT_CONNECTION_GRAPH_VERBOSE}
            && !exists $option{label}
            && ( my ( $queue, $kunde, $subject ) = DoFn(<<_) ) ) {
	SELECT queue.name, kunde.name, ticket.subject
	FROM   ticket
	JOIN   kunde ON kunde.id = ticket.kunde
	JOIN   queue ON queue.id = ticket.queue
	WHERE  ticket.id = ${\ qquote($name) }
_
            $option{label} = "#$name in $queue ($kunde):\n$subject";
        }
        $self->SUPER::add_node( $name, %option );
    }
}

1;

__END__

=head1 NAME

noris::Ticket::API::RT::SelectResult -
zentraler Bestandteil des RT-Backends der Ticket-API

=head1 DEBUGGING

Über das Environment lassen sich für ->follow_tickets diverse Debugging-Routinen
aktivieren:

=over 4

=item TICKET_API_RT_CONNECTION_GRAPH=Dateiname

Zur Angabe einer Datei, in der ein Graph ausgegeben werden soll, der die
bei Bei ->follow_tickets() gefundenen Beziehungen zwischen den Tickets
darstellt.

Der Dateityp kann durch die Dateiendung festgelegt werden.
Falls der fragliche Typ von L<GraphViz> nicht unterstützt wird, wird ein PNG
erzeugt.

Beispiel:

    TICKET_API_RT_CONNECTION_GRAPH=graph.ps

Der Graph wird durch jeden neuen Aufruf von ->follow_tickets() überschrieben.

=item TICKET_API_RT_CONNECTION_GRAPH_OPTIONS=Name,Wert,Name,Wert,...

Zur Angabe von Optionen für den Graphen.
Beispiel:

    TICKET_API_RT_CONNECTION_GRAPH_OPTIONS=rankdir,1,pagewidth,5

Details s. L<GraphViz>.

=item TICKET_API_RT_CONNECTION_GRAPH_VERBOSE=1

Damit kann man veranlassen, dass die Graphen nicht nur die Nummern der
Tickets enthalten sondern zusätzliche Angaben (Queue, Kunde und Subject).

=back

->follow_tickets unterstützt außerdem folgende C<TICKET_API_DEBUG>-Optionen:

=over 4

=item follow

jede gefundene Ticket-Verknüpfung detailliert anzeigen

=item follow_guts

nach der Verarbeitung jeder Ticket-Vernüpfung den aktuellen Stand dumpen

=item origin

abschließend eine Übersicht ausgeben, welche Tickets mit welchen anderen
verknüpft sind

=item sql

SQL-Statements anzeigen

=back

