use strict;
use utf8;
use warnings;

package noris::Ticket::API::RT::Field::Date;

use Dbase::Help qw(in_range);
use Moose;
use Params::Validate qw(SCALAR UNDEF validate_pos);

extends 'noris::Ticket::API::RT::Field::String';

has select_name => (
    is      => 'ro',
    isa     => 'Str',
    lazy    => 1,
    default => sub {
        my $self = shift;
        'FROM_UNIXTIME(' . $self->db_name . ')';
    },
);

use constant RE_DATE => qr/^(\d+)-(\d\d)-(\d\d) (\d\d):(\d\d):(\d\d)\z/;

use constant DATE_OR_UNDEF => {
    type      => SCALAR | UNDEF,
    callbacks => {
        date_or_undef =>
          sub { my $value = shift; !defined $value || $value =~ RE_DATE }
    },
    default => undef,
};

sub _to_epoch($) {
    my $timestamp = shift;
    "UNIX_TIMESTAMP('$timestamp')";
}

sub where_range {
    my ( $self, $lower, $upper ) = validate_pos(
        @_,
        { isa => __PACKAGE__ },
        (DATE_OR_UNDEF) x 2
    );
    defined and $_ = _to_epoch($_) for $lower, $upper;
    in_range( $self->db_name, $lower, $upper );
}

for (qw(< <= = > >=)) {
    my $operator = $_;
    no strict 'refs';
    *{"where_$_"} = sub {
        my ( $self, $value ) = validate_pos(
            @_,
            { isa  => __PACKAGE__ },
            { type => SCALAR, regex => RE_DATE },
        );
        $self->db_name . " $operator " . _to_epoch($value);
    };
}

1;
