use strict;
use utf8;
use warnings;

package noris::Ticket::API::Connection;

use Moose;
use noris::Ticket::API::SelectResult;
use Params::Validate qw(ARRAYREF SCALAR validate_pos);

has 'backends' => (
    is         => 'ro',
    isa        => 'ArrayRef',
    auto_deref => 1,
    required   => 1,
);

has 'connections' => (
    is         => 'ro',
    isa        => 'ArrayRef',    # bei neuerem Moose evtl.: ArrayRef[Object]
    auto_deref => 1,
    lazy       => 1,
    default    => sub {
        my $self = shift;
        [
            map {
                $_ = "noris::Ticket::API::${_}::Connection" unless /::/;
                eval "require $_";
                confess("Error loading $_: $@") if length $@;
                $_->new;
              } $self->backends
        ];
    },
);

has '_user_locked' => (is => 'ro', isa => 'Bool', default => sub {''});

sub change_user {
    my ($self, $user_name) = @_;
    confess "Cannot change user, user is locked. It is probably a pooled connection." 
        if $self->_user_locked;
    $_->change_user($user_name) for $self->connections;
    return;  # die Rückgabewerte der Lowlevel-Connections sind irrelevant.
}

sub lock_user {
    my ($self) = @_;
    $self->{_user_locked} = 1;
}

sub is_open {
    my ($self) = @_;
    for my $conn ($self->connections) {
        return '' unless $conn->is_open();
    }
    return 1;
}

sub _report_call_to($@) {
    my $method = shift;
    require Data::Dump and Data::Dump->import('pp') unless defined &pp;
    my ( $package, $file, $line );
    {    # Ticket-API-interne Aufrufe überspringen:
        my $level = 0;
        do { ( $package, $file, $line ) = caller( ++$level ) }
          while __PACKAGE__ =~ /^\Q$package\E(?:\z|::)/;
    }
    "->$method" . pp(@_) . " called from $file line $line";
}

sub select_tickets {
    my ( $self, %param ) = @_;
    noris::Ticket::API::_debug(
        caller => sub {
            require Carp and Carp->import('longmess') unless defined &longmess;
            require Data::Dump and Data::Dump->import('pp') unless defined &pp;

            # Das würde indirekte Aufrufe via get_ticket() ausblenden, aber
            # letztlich ist das zu Debugging-Zwecken gar nicht so sinnig:
            # local %Carp::Internal = (
            #     %Carp::Internal, map +( $_ => 1 ),
            #     __PACKAGE__,     'noris::Ticket::API',
            # );
            '->select_tickets' . pp(%param) . ' called' . longmess();
        }
      )
      or noris::Ticket::API::_debug(
        query => sub { _report_call_to( 'select_tickets', %param ) . '.' } );
    return noris::Ticket::API::SelectResult->new(
        results => [ map $_->select_tickets(%param), $self->connections ] );
}

sub get_ticket {

    my ( $self, $ticket_number, $attributes ) = validate_pos(
        @_,
        { isa  => __PACKAGE__ },
        { type => SCALAR, regex => qr/^\d+\z/ },
        { type => ARRAYREF, optional => !wantarray },
    );

    if (wantarray) {
        {    # suche bei Bedarf rekursiv nach dem Haupt-Ticket:
            my $select_result = $self->select_tickets(
                attributes => [ 'merge_parent', @$attributes ],
                query      => { ticket_number   => $ticket_number },
            );
            my ( $merge_parent, @ticket_data );
            $select_result->foreach_row(
                sub {
                    confess("Es existieren mehrere Tickets #$ticket_number!?")
                      if defined $merge_parent;
                    ( $merge_parent, @ticket_data ) = @_;
                }
            );

            # Wir betrachten den (Sonder-)Fall, dass ein Merge-Parent nicht
            # ermittelt werden kann, weil die Tickets sich in unterschiedlichen
            # Backends befinden, mangels besserer Ideen erstmal so, als wär's
            # ein eigenständiges Ticket. Das ist auch der Grund, weshalb wir
            # hier merge_parent und nicht gleich merge_root verwenden, denn
            # (nur) so können wir bei mehrfach ineinander gemergten Tickets den
            # Baum zumindest ein Stück weit nach oben verfolgen.
            return @ticket_data
              if
              !$merge_parent # kein defined()-Test, weil JSON hier ein JSON::NotString-Objekt liefert
              || $merge_parent eq $ticket_number;

            $ticket_number = $merge_parent;
            redo;
        }
    }
    else {

        warn _report_call_to( 'get_ticket', @_ )
          . " with attributes, but in scalar context.\n"
          if ref $attributes && @$attributes;

        my $select_result = $self->select_tickets(
            attributes => [],
            query      => { ticket_number => $ticket_number },
        );
        my $no_tickets = $select_result->count;
        confess("Es existieren $no_tickets Tickets #$ticket_number!?")
          if $no_tickets > 1;
        $no_tickets || '';
    }
}

1;

__END__

=head1 NAME

noris::Ticket::API::Connection

=head1 WEITERE DOKUMENTATION

s. L<noris::Ticket::API>
