use strict;
use warnings;
use utf8;

package noris::Ticket::API;

use base 'Exporter';
our @EXPORT_OK = qw(
  clean_connection_pool
  get_connection
  get_pooled_connection
  in_out_list
  link_list
);

use Cf qw($TICKET_BACKENDS);
use noris::Ticket::API::Connection;
use Params::Validate qw(ARRAYREF CODEREF SCALAR validate_pos);

=begin testing

is_deeply [in_out_list(name => [1,3,5], [])], [name => ['in', 1, 3, 5]];
is_deeply [in_out_list(name => [], [1,2,3])], [name => [not_in => 1,2,3]];
is_deeply [in_out_list(name => [1,3,5], [1,2,3])], [name => [in => 5]];
is_deeply [in_out_list(name => [1,2,3], [1,2,3])], [name => ['in']];
is_deeply [in_out_list(name => [], [])], [];

=end testing

=cut

sub in_out_list {
    my ( $key, $positiv, $negativ ) =
      validate_pos( @_, { type => SCALAR }, ( { type => ARRAYREF } ) x 2 );

    return unless @$positiv || @$negativ;
    if (@$positiv) {
        my %negativ;
        @negativ{ map defined() && ".$_", @$negativ } = ();

        # Falls @$positiv == @$negativ, ist die Liste leer,
        # und das ist auch gut so.
        $key =>
          [ in => grep !exists $negativ{ defined() && ".$_" }, @$positiv ];
    }
    else {
        $key => [ not_in => @$negativ ];
    }
}

=begin testing

is_deeply [ link_list( name => [ [ 'foo', 'bar' ] ], [] ) ],
  [ name => [ [qw/has_any foo bar/] ] ];
is_deeply [ link_list( name => [ [] ], [] ) ], [];
is_deeply [ link_list( name => [ [qw/a b/], [], [qw/c/] ], [ [qw/d e/] ] ) ],
  [ name => [ [qw/has_any a b/], [qw/has_any c/], [qw/has_none d e/] ] ];

=end testing

=cut

sub link_list {
    my ( $key, $positiv, $negativ ) =
      validate_pos( @_, { type => SCALAR }, ( { type => ARRAYREF } ) x 2 );

    my @criteria;
    @$_ and push @criteria, [ has_any  => @$_ ] for @$positiv;
    @$_ and push @criteria, [ has_none => @$_ ] for @$negativ;
	return unless @criteria;
    $key => \@criteria;
}

sub _backends {
    return $ENV{TICKET_BACKENDS} if defined $ENV{TICKET_BACKENDS};
    $TICKET_BACKENDS;
}

sub get_connection {
    return noris::Ticket::API::Connection->new(
        backends => [ split ' ', _backends() ] );
}

my %ConnectionPool = ();

sub clean_connection_pool {
    if (@_) { delete @ConnectionPool{@_} }
	else { %ConnectionPool = () }
}

sub get_pooled_connection {
    my ($user) = @_;
    $user = defined $ENV{TICKET_API_USER} ? $ENV{TICKET_API_USER} : ''
      unless defined $user;
    my $backends = _backends();
    my $conn     = $ConnectionPool{$user}{$backends};
    return $conn if defined $conn && $conn->is_open();
    $conn = get_connection();
    $conn->change_user($user) if $user ne '';
    $conn->lock_user();
    $ConnectionPool{$user}{$backends} = $conn;
    return $conn;
}

sub _debug {
    my ( $types, $message ) = validate_pos(
        @_,
        { type => ARRAYREF | SCALAR },
        { type => CODEREF | SCALAR },
    );
    return unless _want_debug( ref $types ? @$types : $types );
	$message = $message->() if ref $message;
	$message .= "\n" if $message !~ /\n\z/;
	print $message;
}

sub _want_debug {
    return 1 if $ENV{DEBUG_TICKET_API}; # backwards compatibility
    return '' unless defined $ENV{TICKET_API_DEBUG};
    my %want_debug;
	@want_debug{ split /,/, $ENV{TICKET_API_DEBUG} } = ();
	return 1 if exists $want_debug{all};
	!exists $want_debug{$_} and return '' for @_;
	1;
}

1;
