package noris::Table;

use utf8;
use strict;
use warnings;

require Exporter;

=pod

Modul für Tabellen in kunde. Basiert auf Text::ASCIITable.

=head1 Allokation etc.

=head2 new(flag)

Generiert eine Standardtabelle.

=head2 skip(x)

Die Spalte mit der angegebenen Nummer wird unterdrückt.

Diese Funktion kann mehrfach aufgerufen werden (kumulativ).

Die erste Spalte hat die Nummer 1.

=head2 titel(bla, fasel, ...)

Liefert die Überschrift(en).

=head1 Daten füttern

=head2 daten()

Liefert die Daten ein.

=head2 ansi(BLA)

Liefert entweder BLA (ein ANSI-Code) oder einen Leerstring zurück,
je nachdem, ob die gewählte Ausgabe ANSI-Codes unterstützt oder nicht.

=head2 info(BLA)

Gibt Fortschrittsmeldungen aufs Terminal aus, wenn das Produzieren der Daten
jetzt länger dauern sollte (z. B. wegen DNS-Abfragen).

=head2 

=head2 drucken( { Option=>Wert }* )

Druckt die Tabelle aus.

Folgende Optionen können übergeben werden:

=over 4

=item trailer => Text

Der Trailer wird immer gedruckt, d. h. auch wenn die Tabelle leer ist.

=item pre => Text

=item post => Text

Präfix- und Postfixzeilen werden ausgegeben, wenn die Tabelle nicht leer
ist.

=item empty_is_ok => BOOL

Normalerweise wird an Stelle einer leeren Tabelle eine Warnung erzeugt.
Wenn man hier einen wahren Wert übergibt, wird stattdessen einfach die
Überschriftenzeile trotzdem ausgegeben.

=cut

use Cf qw($VRF_PREFIX);
use Dbase::Help;
use Fehler qw(fehler warnung);
use noris::ASCIITable;
use Loader qw(line_printer line_print_end show_progress);
use Term::ReadKey qw(GetTerminalSize);
use Term::ANSIColor qw(BOLD RESET UNDERLINE);
use IO::File;

our @ISA = qw(Exporter);
our @EXPORT = qw();
our @EXPORT_OK = qw();

my $TermHandle = IO::File->new("/dev/tty");

sub new
{
	my $cls = shift;
	my $t = noris::ASCIITable->new();
	$t->setOptions(allowANSI => 1);
	$t->setOptions(undef_as => "(?)");
	$t->setOptions(headingText => undef);
	$t->setOptions(headingStartChar => "");
	$t->setOptions(headingStopChar => "");
	$t->setOptions(hide_FirstLine => 1);
	$t->setOptions(hide_HeadLine => 1);
	$t->setOptions(hide_LastLine => 1);

	my($w,$h) = (0,0);
	($w,$h) = GetTerminalSize(undef,$TermHandle) if defined $TermHandle;

	no warnings 'once';
	my $obj = {table => $t, rows => -1, skip => [], 
		ansi => (($Db::pr_fh_modus||0)<=2 and not defined $Db::test_output), info => 0};
	$obj->{w} = $w;
	$obj->{h} = $h;

	bless $obj,$cls;
}

sub ansi {
	my $obj = shift;
	if($obj->{ansi}) { shift } else { "" }
}

sub info {
	my $obj = shift;
	return unless $obj->{w};

	my $i = join(" ",@_);
	$i = substr($i,0,$obj->{w}-6)."..." if length($i) > $obj->{w}-3;
	show_progress("  $i");
	show_progress( " " x ( $obj->{info} - length $i ) )
		if $obj->{info} > length($i);
	show_progress("\r");
	$obj->{info} = length($i);
}

sub skip {
	my $obj = shift;
	push(@{$obj->{skip}},@_);
}

sub titel {
	my ( $obj, @cols ) = @_;
	fehler "Titel bereits gesetzt!" if $obj->{rows} >= 0;
	$obj->{rows}=0;

	my $t = $obj->{table};
	my ( @columns, @colformat );

	for (@cols) {
		if ( ref $_ eq 'HASH' ) {
			push @columns, $_->{name};
			push @colformat, ( $_->{format} ? $_->{format} : "%s\t" );
		}
		else {
			push @columns, $_;
			push @colformat, "%s\t";
		}
	}
	$t->setOptions( colformat => \@colformat );

	$obj->{table}->setCols(@columns);

	$obj;
}

sub ausrichten {
	my $obj = shift;
	my $t = $obj->{table};
	my $dir = shift;
	foreach my $col(@_) {
		my $col = $t->{tbl_cols}[$col-1];
		$t->alignCol($col,$dir);
		$t->alignColName($col,$dir);
	}
}

sub daten {
	my $obj = shift;

	if ( $ENV{KUNDE_NO_TABLES} ) {
		my $t = $obj->{table};
		my %skipper;
		@skipper{ @{ $obj->{skip} } } = ();

		# if KUNDE_NO_TABLES print Titel
		# nicht in titel() rein (weil da skip noch nicht gesetzt ist)
		if ( $obj->{rows} == 0 ) {
			line_printer();
			my $colcount = 0;
			for ( @{ $t->{tbl_cols} } ) {
				printf $Db::pr_fh $t->{options}->{colformat}[$colcount], $_
				  if !exists $skipper{ $colcount + 1 };
				$colcount++;
			}
			print $Db::pr_fh "\n";
		}

		for ( my $colcount = 0 ; $colcount <= $#_ ; ++$colcount ) {
			next if exists $skipper{ $colcount + 1 };
			my $diff = length( $_[$colcount] ) - $t->count( $_[$colcount] );

			my @text_array  = $t->split_control_seqs_and_text( $_[$colcount] );
			my $beg_control = my $end_control = '';

			( undef, $beg_control ) = splice @text_array, 0, 2
			  if @text_array > 1 && $text_array[0] eq '';
			$end_control = pop @text_array unless @text_array & 1;
			my $col_text = join '', @text_array;

			printf $Db::pr_fh $beg_control if ($beg_control);
			printf $Db::pr_fh $t->{options}->{colformat}[$colcount], $col_text;
			printf $Db::pr_fh $end_control if ($end_control);
		}
		print $Db::pr_fh "\n";
	}
	else {
		if ( $obj->{rows} < 0 ) {
			$obj->titel( 1 .. 0 + @_ );
			$obj->{table}->setOptions( hide_HeadRow => 1 );
		}
	}
	$obj->{rows}++;
	$obj->{table}->addRow(@_);
	$obj;
}

sub drucken {
	my ( $obj, %option ) = @_;

	# if KUNDE_NO_TABLES wurde es schon gedruckt 
	if ( $ENV{KUNDE_NO_TABLES} ) {
		line_print_end();
		return;
	}

	my $t = $obj->{table};
	show_progress( ' ' x ( $obj->{info} + 2 ), "\r" ) if $obj->{info};

	if ( $obj->{rows} <= 0 && !$option{empty_is_ok} ) {
		if(defined $option{trailer} ) {
			if(ref $Db::test_output) {
				print $Db::test_output $option{trailer};
			} else {
				print $option{trailer};
			}
		}
		warnung "keine Daten";
		return;
	}
	my @skip = sort { $b <=> $a } @{$obj->{skip}};
	foreach my $i (@skip) {
		$i--; ## 0-based, please

		foreach (@{$t->{tbl_rows}}) {
			splice(@{$_},$i,1);
		}
		my $n;
		while(($a,$b)=each %{$t->{tbl_col}}) {
			$n = $a if $b == $i;
			$t->{tbl_col}{$a} = $b-1 if $b > $i;
		}
		delete $t->{tbl_col}{$n} if defined $n;
		splice(@{$t->{tbl_cols}},$i,1);
		foreach(@{$t->{tbl_multilinecols}}) {
			splice(@{$_},$i,1);
		}
	}

	my $da = ["+","+","-","+"];
	my $db = ["","",""];
	my $dc = [UNDERLINE,RESET,""];

	# PAGER wurde hier eher als störend empfunden, vgl. Ticket #10039117:
	#line_printer($obj->{rows}-2 > $obj->{h} or $t->getTableWidth() >= $obj->{w});
	line_printer();
	print $Db::pr_fh $option{pre    } if defined $option{pre    };
	print $Db::pr_fh $t->draw($da,$obj->{ansi} ? $dc : $db, $da,$db,$da,$da);
	print $Db::pr_fh $option{post   } if defined $option{post   };
	print $Db::pr_fh $option{trailer} if defined $option{trailer};
	line_print_end();
}
1;
