package noris::ServiceWeb::Globals;

use utf8;
use strict;
use warnings;
BEGIN {
	unshift @INC, ( $ENV{'POPHOME'} || '@POPHOME@' ) . '/lib'
		unless $ENV{'KUNDE_NO_PERLPATH'};
}

use Carp (qw(cluck confess));
use CGI (qw(param));
use Time::Local (qw(timelocal_nocheck));
use Exporter;
use Time::DaysInMonth ('days_in');

use noris::REST::Frontend (qw(query kunden));

@noris::ServiceWeb::Globals::EXPORT_OK = 
	(qw(get_selected_kunde print_select_kunde get_param_time
	get_selected_unterkunden print_select_unterkunden print_select
	print_time_select csv_escape html_escape

	get_acct_warnung
	get_kundeid_by_username get_unterkunden_by_kundeid
	print_kunde_select

	epoch_to_rfc1123));
@noris::ServiceWeb::Globals::ISA = ('Exporter');

our $Cache = {};

=head1 NAME

noris::ServiceWeb::Globals

=head1 SYNOPSIS

  use noris::ServiceWeb::Globals (qw(get_param_time get_selected_unterkunden
    print_select_unterkunden print_kunde_select print_select print_time_select
    csv_escape html_escape);

=cut

return (1);

=head1 EXPORTED FUNCTIONS

=over 4

=item I<$kundeid> = B<get_selected_kunde> ()

Returns the selected I<$kundeid> or I<false> if the param B<kunde> is not set
or set to an invalid value. The special value C<-1> means "all customers".

=cut

# $ak = { id => name };
sub get_selected_kunde
{
	my %opts = @_;

	my $sk = param ('kunde');
	my $ak;

	$ak = kunden (\%opts)->data ();

	if (scalar (keys %$ak) == 1)
	{
		my ($kid) = keys %$ak;
		return $kid;
	}

	# Check if there was any selection at all _after_ checking if there is
	# more than one potential customer. See RT#411651.
	return unless ($sk);
	return (-1) if ($sk == -1);
	return unless (defined ($ak->{$sk}));
	return ($sk);
}

=item B<print_select_kunde> ([I<\&hook>])

Prints a box to select the customer. Customers are automagically found using
the I<REMOTE_USER> environment variable.

If I<\&hook> is passed the referenced subroutine is calles with a hashref as
first argument. The hashref contains C<id =E<gt> name> mappings. The hash can
be modified in any way you want, but when changing the keys
B<get_selected_kunde> will stop working.

=cut

sub print_select_kunde
{
	my %opts = (@_);

	my $ak = kunden ( { Rolle => $opts{Rolle} } )->data ();
	my $sk = param ('kunde');

	return if (scalar (keys %$ak) < 2);

	$sk ||= 0;

	if (ref ($opts{'fixup_hook'}) eq 'CODE')
	{
		$opts{'fixup_hook'}->($ak);
	}

	print <<FORM;
		    <fieldset>
		      <legend>Kunde</legend>
		      <select name="kunde">
FORM
	if ($opts{'allow_select_all'})
	{
		my $selected = ($sk == -1 ? ' selected="selected"' : '');
		print qq(\t\t\t<option value="-1"$selected>Alle Kunden</option>\n);
	}
	for (sort { $ak->{$a} cmp $ak->{$b} } (keys %$ak))
	{
		my $kid = $_;
		my $kname = $ak->{$kid};
		my $selected = ($kid == $sk ? ' selected="selected"' : '');
		print qq(\t\t\t<option value="$kid"$selected>$kname</option>\n);
	}
	print <<FORM;
		      </select>
		    </fieldset>
FORM
}

=item B<print_select> (I<$name>, I<\@values>, I<\@names>, [I<$default>])

Prints a HTML B<select>-box names I<$name>. If given I<$default> is matched
against the I<\@values> list and the appropriate entry is preselected. The
result looks like this:

  <select name="$name">
    <option value="$value->[0]">$name->[0]</option>
    <option value="$value->[1]" selected="selected">$name->[1]</option>
    <option value="$value->[2]">$name->[2]</option>
    ...
    <option value="$value->[n-1]">$name->[n-1]</option>
    <option value="$value->[n]">$name->[n]</option>
  </select>

=cut

sub print_select
{
	confess ('Wrong number of arguments') unless ((scalar (@_) == 3) or (scalar (@_) == 4));

	my $name    = shift;
	my $values  = shift;
	my $names   = shift;
	my $default = @_ ? shift : '';

	if (scalar (@$values) != scalar (@$names))
	{
		confess ("Number of values and number of names must match!");
	}

	print qq(<select name="$name">\n);
	for (my $i = 0; $i < scalar (@$values); $i++)
	{
		my $val  = $values->[$i];
		my $name = $names->[$i];
		my $sel = '';

		if ((($val =~ m/^\d+$/) && ($default =~ m/^\d+$/) && ($val == $default))
				|| ($val eq $default))
		{
			$sel = ' selected="selected"';
		}

		print qq(  <option value="$val"$sel>$name</option>\n);
	}
	print "</select>\n";

	return;
}

=item B<print_time_select> ([I<$prefix>], [I<\%opts>])

Prints time selectors. The names are I<$prefix>B<year>, I<$prefix>B<month>,
I<$prefix>B<week>, I<$prefix>B<day>, I<$prefix>B<hour>, I<$prefix>B<minute>, I<$prefix>B<second>.

The options are:

=over 4

=item B<show_>I<{>B<year>I<,>B<month>I<,>B<day>I<,>B<hour>I<,>B<minute>I<,>B<second>I<}>

Wether or not to show year, month, ..., second. Default is to show all but seconds.

=item B<values_>I<{>B<year>I<,>B<month>I<,>B<day>I<,>B<hour>I<,>B<minute>I<,>B<second>I<}>

Values to show. Defaults to the this and the past two years for year, and all
values for the others: 1-12 for month, 1-31 for day, 0-23 for hour and 0-59 for
minute and second.

=item B<default>

Default time in epoch. Can be overwritten using the B<def_*> options (see below).

=item B<def_>I<{>B<year>I<,>B<month>I<,>B<day>I<,>B<hour>I<,>B<minute>I<,>B<second>I<}>

Default values. See B<default> for an alternative. If not given I<now> is used..

=back

=cut

sub print_time_select
{
	confess ('Wrong number of arguments') unless (scalar (@_) <= 2);

	my $prefix  = @_ ? shift : '';
	my $opts = @_ ? shift : {};

	$opts->{'show_year'}   = 1 unless (defined ($opts->{'show_year'}));
	$opts->{'show_month'}  = 1 unless (defined ($opts->{'show_month'}));
	$opts->{'show_day'}    = 1 unless (defined ($opts->{'show_day'}));
	$opts->{'show_hour'}   = 1 unless (defined ($opts->{'show_hour'}));
	$opts->{'show_minute'} = 1 unless (defined ($opts->{'show_minute'}));
	$opts->{'show_second'} = 0 unless (defined ($opts->{'show_second'}));

	$opts->{'values_year'}    = [(localtime)[5] + 1900, (localtime)[5] + 1899, (localtime)[5] + 1898] unless (defined ($opts->{'values_year'}));
	$opts->{'values_month'}   = [1 .. 12] unless (defined ($opts->{'values_month'}));
	$opts->{'values_day'}     = [1 .. 31] unless (defined ($opts->{'values_day'}));
	$opts->{'values_hour'}    = [0 .. 23] unless (defined ($opts->{'values_hour'}));
	$opts->{'values_minute'}  = [0 .. 59] unless (defined ($opts->{'values_minute'}));
	$opts->{'values_second'}  = [0 .. 59] unless (defined ($opts->{'values_second'}));

	$opts->{'names_month'} = [qw(Januar Februar M&auml;rz April Mai Juni Juli August September Oktober November Dezember)];
	$opts->{'names_day'}   = [map { "$_." } (1 .. 31)];

	$opts->{'default'}    = time unless (defined ($opts->{'default'}));
	$opts->{'def_year'}   = (localtime ($opts->{'default'}))[5] + 1900 unless (defined ($opts->{'def_year'}));
	$opts->{'def_month'}  = (localtime ($opts->{'default'}))[4] + 1    unless (defined ($opts->{'def_month'}));
	$opts->{'def_day'}    = (localtime ($opts->{'default'}))[3] unless (defined ($opts->{'def_day'}));
	$opts->{'def_hour'}   = (localtime ($opts->{'default'}))[2] unless (defined ($opts->{'def_hour'}));
	$opts->{'def_minute'} = (localtime ($opts->{'default'}))[1] unless (defined ($opts->{'def_minute'}));
	$opts->{'def_second'} = (localtime ($opts->{'default'}))[0] unless (defined ($opts->{'def_second'}));

	for (qw(day month year hour minute second))
	{
		my $span = $_;
		next unless ($opts->{"show_$span"});

		print_select ($prefix . $span,
			$opts->{"values_$span"},
			defined ($opts->{"names_$span"}) ? $opts->{"names_$span"} : $opts->{"values_$span"},
			$opts->{"def_$span"});
	}

	return;
}

=item I<$time> = B<get_param_time> ([I<$prefix>], [I<$default>])

Return the time-values with prefix I<$prefix> (as generated by
B<print_time_select>). If in doubt use I<$default> which itself defaults to
I<now>. Returns I<$time> which is in epoch.

=cut

sub get_param_time
{
	confess ('Wrong number of arguments') unless (scalar (@_) <= 2);

	my $prefix  = @_ ? shift : '';
	my $default = @_ ? shift : time;
	my @values = (localtime ($default))[0 .. 5];

	my $year   = param ($prefix . 'year');
	my $month  = param ($prefix . 'month');
	my $day    = param ($prefix . 'day');
	my $hour   = param ($prefix . 'hour');
	my $minute = param ($prefix . 'minute');
	my $second = param ($prefix . 'second');

	$year   = '' if (!defined ($year));
	$month  = '' if (!defined ($month));
	$day    = '' if (!defined ($day));
	$hour   = '' if (!defined ($hour));
	$minute = '' if (!defined ($minute));
	$second = '' if (!defined ($second));

	$year   =~ s/\D//g;
	$month  =~ s/\D//g;
	$day    =~ s/\D//g;
	$hour   =~ s/\D//g;
	$minute =~ s/\D//g;
	$second =~ s/\D//g;

	$year -= 1900 if ($year and $year > 1900);

	$values[5] = $year    if ($year);
	$values[4] = $month-1 if ($month);
	$values[3] = $day     if ($day);
	$values[2] = $hour    if (length ("$hour"));
	$values[1] = $minute  if (length ("$minute"));
	$values[0] = $second  if (length ("$second"));

	if ($values[5] < 70) { $values[5] = 70; }
	elsif ($values[5] > 138) { $values[5] = 138; }

	if    ( $values[4] < 0 )  { $values[4] = 0 }
	elsif ( $values[4] > 11 ) { $values[4] = 11 }

	# `days_in' want the month in the range 1..12
	if ($values[3] < 1) { $values[3] = 1; }
	elsif ($values[3] > days_in ($values[5], $values[4] + 1))
	{
		$values[3] = days_in ($values[5], $values[4] + 1);
	}

	if ($values[2] < 0) { $values[2] = 0; }
	elsif ($values[2] > 23) { $values[2] = 23; }

	if ($values[1] < 0) { $values[1] = 0; }
	elsif ($values[1] > 59) { $values[1] = 59; }

	if ($values[0] < 0) { $values[0] = 0; }
	elsif ($values[0] > 59) { $values[0] = 59; }

	return (timelocal_nocheck (@values));
}

=item I<@uk> = B<get_selected_unterkunden> (I<$kundeid>)

Get all "Unterkunden" that were selected. I<$kundeid> is used to do a sanity
check. Returns a list of customer-IDs. (see also B<print_select_unterkunden>)

=cut

sub get_selected_unterkunden
{
	confess ('Wrong number of arguments') unless (scalar (@_) == 1);

	my $kundeid = shift;

	my %uk = get_unterkunden_by_kundeid ($kundeid);
	my @sel = param ('unterkunden');
	my @ret = ();

	if (!@sel)
	{
		@ret = keys %uk;
	}
	else
	{
		for (@sel)
		{
			if (exists $uk{$_})
			{
				push (@ret, "$_");
			}
		}
	}

	return (@ret);
}

=item B<print_select_unterkunden> (I<$kundeid>)

Prints a selection box where one or more "Unterkunden" may be selected.

=cut

sub print_select_unterkunden
{
	confess ('Wrong number of arguments') unless (scalar (@_) == 1);

	my $kundeid = shift;

	my %unterkunden = get_unterkunden_by_kundeid ($kundeid);
	my @selected = get_selected_unterkunden ($kundeid);

	return unless (scalar (keys %unterkunden) > 1);

	my $num = scalar (keys %unterkunden);
	if ($num > 5) { $num = 5; }

	print <<FORM;
		    <fieldset>
		      <legend>Unterkunden</legend>
		      <select name="unterkunden" multiple="multiple" size="$num" style="float: left; margin-right: 1ex;">
FORM
	for (sort { $unterkunden{$a} cmp $unterkunden{$b} } (keys %unterkunden))
	{
		my $knr = $_;
		my $knm = $unterkunden{$knr};

		my $sel = (grep { $knr == $_ } (@selected)) ? ' selected="selected"' : '';

		print qq(\t\t\t<option value="$knr"$sel>$knm</option>\n);
	}
	print <<FORM;
		      </select>
		      Wenn Sie keine Unterkunden explizit ausw&auml;hlen werden
		      die Daten <strong>aller</strong> Unterkunden ausgegeben.
		    </fieldset>
FORM
}

=item I<$string> = B<csv_escape> (I<@strings>)

=item I<@array>  = B<csv_escape> (I<@strings>)

Escapes all strings in I<@strings> and returns them in a list or concatenated
with commatas.

=cut

sub csv_escape
{
	my @ret = @_;

	for (@ret)
	{
		$_ ||= '';
		if ($_ =~ m/\W/)
		{
			$_ =~ s/"/""/g;
			$_ = qq("$_");
		}
	}

	return (@ret) if (wantarray ());
	return (join (',', @ret));
}

=item I<$string> = B<html_escape> (I<@strings>)

=item I<@array>  = B<html_escape> (I<@strings>)


Escapes all strings in I<@strings> and returns them in a list or all strings
concatenated together.

=cut

sub html_escape
{
	my @ret = @_;

	for (@ret)
	{
		$_ ||= '';
		$_ =~ s/&/&amp;/g;
		$_ =~ s/</&lt;/g;
		$_ =~ s/>/&gt;/g;
		$_ =~ s/"/&quot;/g;
	}

	return (@ret) if (wantarray ());
	return (join ('', @ret));
}

=back

=cut

# XXX
# Deprecated functions..

=head1 DEPRECATED FUNCTIONS

=over 4

=item B<print_kunde_select> ()

Prints a box to select the customer. Customers are automagically found using
the I<REMOTE_USER> environment variable.

=cut

sub print_kunde_select # FIXME
{
	my $kunde = '';
	if (param ('kunde'))
	{
		$kunde = html_escape (param ('kunde'));
	}

	print <<FORM;
		    <fieldset style="border-color: red; border-width: 3px;">
		      <legend>Kunde</legend>
		      Username: <input type="text" name="kunde" value="$kunde" />
		    </fieldset>
FORM
}

=item I<$kundeid> = B<get_kundeid_by_username> (I<$username>)

Returns I<$kundeid>

=cut

sub get_kundeid_by_username
{
	confess ('Wrong number of arguments') unless (scalar (@_) == 1);

	my $username = shift;
	
	if (!exists ($Cache->{'get_kundeid_by_username'}))
	{
		my $q = query ('get_kunde.id_by_user.name', 'user.name' => $username);
		$Cache->{'get_kundeid_by_username'} = $q->data ();
	}

	if (!defined ($Cache->{'get_kundeid_by_username'}))
	{
		cluck ("No `kundeid' found for username ``$username''");
		return;
	}

	return ($Cache->{'get_kundeid_by_username'})
}


=item I<\%uk> = B<get_unterkunden_by_kundeid> (I<$kundeid>)

Returns a hashref of "Unterkunden".

=cut

sub get_unterkunden_by_kundeid
{
	confess ('Wrong number of arguments') unless (scalar (@_) == 1);

	my $kundeid = shift;

	if (!exists ($Cache->{'get_unterkunden_by_kundeid'}))
	{
		my $q = query ('get_unterkunden_by_kunde.id', 'kunde.id' => $kundeid);
		$Cache->{'get_unterkunden_by_kundeid'} = $q->data ();
	}

	if (!defined ($Cache->{'get_unterkunden_by_kundeid'}))
	{
		cluck ("No `unterkunde' found for KundeID $kundeid");
		return (qw());
	}

	return (%{$Cache->{'get_unterkunden_by_kundeid'}})
}

=item I<\%uk> = B<epoch_to_rfc1123> (I<$epoch>)

Returns the representation of I<$epoch> (or B<time()> if no argument is passed)
according to RFC1123. This string can be used for `Last-Modified' and/or
`Expires' HTTP-header.

=cut

sub epoch_to_rfc1123
{
	my @days = (qw(Sun Mon Tue Wed Thu Fri Sat));                                                     
	my @months = (qw(Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec));                               

	my $epoch = @_ ? shift : time ();                                                                 
	my ($sec, $min, $hour, $mday, $mon, $year, $wday, $yday) = gmtime($epoch);                        
	my $string = sprintf ('%s, %02d %s %4d %02d:%02d:%02d GMT', $days[$wday], $mday,                  
			$months[$mon], 1900 + $year, $hour ,$min, $sec);
	return ($string);                                                                                 
}

=back

=item I<Warnmeldung> = B<get_acct_warnung>

Returns I<Warnmeldung> oder I<leeren String>

=cut

sub get_acct_warnung {
    my $q = query('get_accounting_timestamp');

    return '' unless defined $q->data();

    if ( ( time() - 86400 ) > $q->data() ) {
        my @MoY = (qw(Jan Feb Mar Apr Mai Jun Jul Aug Sep Okt Nov Dez));
        my ( $sec, $min, $hour, $mday, $mon, $year, $wday, $yday, $isdst ) =
          localtime $q->data();

        my $last_acct_date = sprintf( "%d. %s. %d, %d:%02d",
            $mday, $MoY[$mon], $year + 1900, $hour, $min );

        return <<_
Bitte beachten Sie, dass unsere Accounting-Daten aktuell ausnahmsweise
erst bis $last_acct_date Uhr komplett verarbeitet sind.
Für spätere Zeitpunkte werden daher möglicherweise noch keine oder
unvollständige Daten angezeigt. Wir bitten um Ihr Verständnis.
_
    }

    return '';
}

=head1 AUTHOR

Florian octo Forster E<lt>octo@noris.netE<gt> for the noris network AG
Stelios Gikas E<lt>stelios.gikas@noris.netE<gt> for the noris network AG

=cut

# vim:background=light:noignorecase:hlsearch:incsearch
