package noris::REST::Frontend::QueryResult;

=head1 NAME

noris::REST::Frontend::QueryResult - Objekt für eine Antwort des REST-Backends

=head1 BESCHREIBUNG

Objekte dieser Klasse werden üblicherweise von der L<query()-Funktion aus
noris::REST::Frontend> zurückgeliefert.

Freundlicherweise ergeben sie in einem boolschen Kontext I<true>, wenn die
Abfrage erfolgreich war, und I<false> im Fehlerfall.

=cut

use utf8;
use warnings;
use strict;

use Encode qw(_utf8_on);

use overload bool => sub { !defined shift->errstr };

=head1 METHODEN / FUNKTIONEN

=head2 new

erwartet als (einziges) Argument ein L<HTTP::Response>-Objekt.
Man möchte den Konstruktor aber eigentlich nicht selbst aufrufen.

=cut

{
	my $parser;

	sub new {
		my $package = shift;
		$package = ref $package if length ref $package;

		my $self = bless {}, $package;
		$self->response = shift;
		$self->status   = $self->response->code;

		if ( $self->response->is_success ) {
			unless (
				defined(
					my $content_type = $self->response->header('Content-Type')
				)
			  )
			{
				$self->errstr = 'Antwort ohne Content-Type';
			}
			else {
				$content_type =~ s/;.*//;
				if ( $content_type eq 'text/plain' ) {
					$self->data = $self->response->content;
				}
				elsif ( $content_type eq 'text/xml' ) {
					require utf8
					  ; # Workaround für Fehler: Can't locate object method "SWASHNEW" via package "utf8" (RT#187900)
					unless ( defined $parser ) {
						require Data::DumpXML::Parser;
						$parser = Data::DumpXML::Parser->new;
					}
					_set_utf8_on( $self->data =
						  $parser->parse( $self->response->content )->[0] );
				}
				else {
					$self->errstr =
					  qq(Unerwarteter Content-Type "$content_type");
				}
			}
		}
		else {
			$self->errstr = $self->response->content;
		}
		$self;
	}
}

=head2 data

enthält die dekodierten Daten, die das Backend zurückgeliefert hat,
bzw. C<undef|L<perlfunc/undef>> im Fehlerfall

=cut

sub data : lvalue {
	shift->{data};
}

=head2 errstr

enthält im Fehlerfall einen Fehlertext und ist ansonsten
C<undef|L<perlfunc/undef>>.

=cut

sub errstr : lvalue {
	shift->{errstr};
}

=head2 response

enthält das L<HTTP::Response>-Objekt, mit dem das Objekt erzeugt wurde.
Sollte normalerweise nicht verwendet werden, weil sich die Implementation ändern
könnte.

=cut

sub response : lvalue {
	shift->{response};
}

=head2 status

enthält den numerischen HTTP-StatusCode, den das Backend geliefert hat

=cut

sub status : lvalue {
	shift->{status};
}

=head1 AUTOR

 Martin H. Sluka <fany@noris.net>
 für die noris network AG

=cut

# folgendes könnte man evtl. mal mit Data::Rmap eleganter lösen, vgl. RT#261799:
sub _set_utf8_on {
	for (@_) {
		if    ( ref eq '' )       { _utf8_on($_) }
		elsif ( ref eq 'ARRAY' )  { _set_utf8_on(@$_) }
		elsif ( ref eq 'HASH' )   { _set_utf8_on( values %$_ ) }
		elsif ( ref eq 'SCALAR' ) { _set_utf8_on($$_) }
		else { warn 'Unerwarteter Referenz-Typ: ' . ref() . "\n" }
	}
}

1;
