package noris::REST::Frontend;

=head1 NAME

noris::REST::Frontend - Kommunikationsmodul für externe Applikationen

=head1 SYNOPSE

  use noris::REST::Frontend qw(kunden query);

  my %kunden = %{kunden->data};

  if ( my $result = query { UseCache => 'rw' },
                          'get_kunde.id_by_user.name',
                          'user.name' => $username     )
  {
      print "User $username gehört zu Kunde Nr. " . $result->data;
  }
  else {
      die 'Fehler bei der Abfrage: ' . $result->errstr;
  }

=head1 BESCHREIBUNG

Aus Sicherheitsgründen soll von externen Servern nicht direkt auf die
Datenbank zugegriffen werden.
Vielmehr soll diese Kommunikation über REST-basierte Backend-Scripts
abgewickelt werden.
Dazu soll möglichst ausschließlich dieses Modul verwendet werden, so dass die
Details des REST-Protokolls bei Bedarf jederzeit geändert werden können.

Das Modul exportiert auf Anfrage die Funktion C<query()>, vgl. L<oben|/SYNOPSE>.
Alternativ kann es auch objekt-orientiert verwendet werden.

=cut

use utf8;
use warnings;
use strict;

use base 'Exporter';
our @EXPORT_OK = qw(kunden query);

use Carp qw(confess);
use Cf qw($REST_BACKEND $REST_CACHE_OPTIONS);
use Scalar::Util qw(blessed);
use noris::REST::Frontend::QueryResult ();

=head1 METHODEN / FUNKTIONEN

=head2 new

  my $frontend = new noris::REST::Frontend
                     Backend    => $url_path,
                     RaiseError => "Backend-Fehler:\n%s",
                     UseCache   => 'rw';

Für C<Backend> wird ein sinnvolles Default vorgegeben,
das man normalerweise nicht ändern möchte.

Wird unter C<RaiseError> ein definierter Wert angegeben, so wird im Falle eines
Fehlers bei der Backend-Abfrage L<Carp::confess()|Carp/confess> aufgerufen;
dabei wird der übergebene Optionswert als Format-String für
L<sprintf()|perlfunc/sprintf> verwendet, sprich da, wo die Fehlermeldung des
Backends eingesetzt werden soll, möge man C<%s> schreiben.

Mit C<UseCache> kann festgelegt werden, ob die Ergebnisse von Backend-Abfragen
zwischengespeichert werden sollen (dazu muss der Wert der Option ein "w"
enthalten) bzw. ein ggf. bereits zwischengespeichertes Ergebnis verwendet werden
darf (dazu muss der Wert der Option ein "r" enthalten), um so eine redundante
Backend-Abfrage zu vermeiden.
(Per Default erfolgt kein Caching.)

Alle o.g. Optionen lassen sich auch über gleichnamige Methoden abfragen und
setzen.

=cut

sub new {
    my $package = shift;
    $package = ref $package if length ref $package;
    bless { Backend => $REST_BACKEND, @_ }, $package;
}

sub Backend : lvalue {
    shift->{Backend};
}

sub RaiseError : lvalue {
    shift->{RaiseError};
}

sub UseCache : lvalue {
    shift->{UseCache};
}

=head2 query

Erwartet als erstes Argument den Namen einer Backend-Funktion (also den
Script-Namen) und als weitere Argumente eine Liste von Schlüsselworten und
zugehörigen Werten, vgl. L<Synopse|/SYNOPSE>.

Gibt ein L<noris::REST::Frontend::QueryResult>-Objekt zurück.

Optional kann vor dem ersten Argument noch eine Referenz auf einen Options-
Hash angegeben werden, also z. B. C<{UseCache =E<gt> 'rw'}>; die Optionen
gelten dann nur für diese Abfrage.

=cut

{
    my $cache;

    sub cache {
        require Cache::FileCache;
        $cache = Cache::FileCache->new( { split ' ', $REST_CACHE_OPTIONS } );
        {
            no warnings 'redefine';
            *cache = sub { $cache };
        }
        &cache;
    }
}

sub query {
    my $self =
      blessed $_[0] && $_[0]->isa(__PACKAGE__) ? shift: new __PACKAGE__;
    my %option = ( %$self, ref $_[0] ? %{ +shift } : () );
    $option{UseCache} = '' unless defined $option{UseCache};
    my ( $script, @query ) = @_;
    require URI;
    my $uri = URI->new_abs( $script, $option{Backend} );
    $uri->query_form(@query);

    if ( $option{UseCache} =~ /r/ && defined( my $value = cache()->get($uri) ) )
    {
        return $value;
    }

    require LWP::UserAgent;
    my $result =
      noris::REST::Frontend::QueryResult->new(
        ( $self->{ua} ||= new LWP::UserAgent )->get($uri) );

    cache()->set( $uri => $result ) if $option{UseCache} =~ /w/;

    confess sprintf $option{RaiseError}, $result->errstr
      if defined $option{RaiseError} && !$result;
    $result;
}

=head2 kunden

Gibt ein L<noris::REST::Frontend::QueryResult>-Objekt zurück, das (eine Referenz
auf) einen Hash enthält, dessen Schlüssel die IDs und dessen Werte die Namen
aller Kunden sind, auf deren Daten der aktuell angemeldete Benutzer zugreifen
können sollte.

Optional kann als Argument eine Referenz auf einen Options-Hash angegeben
werden, also z. B. C<{UseCache =E<gt> 'rw'}>; die Optionen gelten dann nur für
diese Abfrage.

C<{UseCache =E<gt> 'rw'}> ist für diese Funktion im Übrigen Default.

Im Options-Hash kann auch angegeben werden, in welcher Rolle der Zugriff
erfolgt, z. B. C<Rolle =E<gt> 'rz_access'>.
Default hierfür ist C<Rolle =E<gt> 'service'>.

=cut

sub kunden {
    my $self =
      blessed $_[0] && $_[0]->isa(__PACKAGE__) ? shift: new __PACKAGE__;
    my %option = ( %$self, ref $_[0] ? %{ +shift } : () );
    $option{UseCache} = 'rw' unless defined $option{UseCache};
    unless ( defined $ENV{REMOTE_USER} ) {
        return unless defined $option{RaiseError};
        confess sprintf $option{RaiseError},
          'Es scheint kein Benutzer angemeldet zu sein.';
    }
    $self->query(
        \%option, 'get_kunden_by_user.name',
        'user.name' => $ENV{REMOTE_USER},
        rolle       => defined $option{Rolle} ? $option{Rolle} : 'service',
    );
}

1;

