package noris::CreateTicket;

use utf8;
use strict;
use warnings;

=head1 NAME

noris::CreateTicket - Generieren neuer Tickets

=head1 SYNOPSIS

  use noris::CreateTicket;

  my $ticket = new noris::CreateTicket (%args);

  my $prio     = $ticket->get ('priority');
  my $old_info = $ticket->set (info => $new_info);

  my $ticket_id = $ticket->create () or die;

=head1 BESCHREIBUNG

Dieses Modul kann verwendet werden, um via TCP mit dem TicketServer zu
kommunizieren, das über die Konfigurationsvariable OTRS_TICKET_SERVER festgelegt
wird.
Im Gegensatz zum Mail-Backend kann wird so die Ticket-ID des erzeugten Tickets
zurückgeliefert.
Da der Ticketserver den Text in UTF-8 erwartet, sollten der Text entsprechend
kodiert sein

=cut
  
use Carp (qw(carp cluck croak confess));
use Cf qw($OTRS_TICKET_SERVER $WDESCR);
use IO::Socket::INET (qw($CRLF));

our @ValidArgs = qw(asuser queue kunde extern info owner priority text tickettype );

return (1);

=head1 METHODEN

Das B<noris::CreateTicket>-Objekt kennt folgende Methoden:

=over 4

=item I<$obj> = B<noris::CreateTicket>-E<gt>B<new> ([I<$attr_name> =E<gt> I<$attr_value>, ...])

Erzeugt ein neues B<noris::CreateTicket>-Objekt, legt aber noch kein Ticket im
TicketSystem an. Fuer eine Liste gueltiger Argumente siehe L</"ATTRIBUTE">.

=cut

sub new
{
	my $pkg = shift;

	my %tmp = @_ ? @_ : ();
	my %obj = ();

	for (@ValidArgs)
	{
		my $arg = $_;
		$obj{$arg} = $tmp{$arg} if (defined ($tmp{$arg}));
	}

	return (bless \%obj, $pkg);
}

sub _do_create
{
	my $obj = shift;

	my $queue  = $obj->{'queue'} or confess ("Which queue?");
	my $type   = $obj->{'extern'} ? 'external' : 'internal';
	my $info   = $obj->{'info'} || '';
	my $kunde  = $obj->{'kunde'} || '';
	my $prio   = $obj->{'priority'} || 0;
	my $ttype  = $obj->{'tickettype'} || '';

	my $text   = $obj->{'text'} or confess ("No text given..");

	my $ret;
	my $ticket_id;
	#my $sequence_id;

	# Slurp mode
	#local $/ = undef;

	$text =~ s/^(\.+)$/$1./gm;

	$queue =~ s/^\Q$WDESCR\E\.//;
	$info  =~ s/\s*\n[\s\n]*/ /g;
	$kunde =~ s/\s//g;
	$prio  =~ s/\D//g;

	my $socket;
	for ( split / /, $OTRS_TICKET_SERVER ) {
		$socket = IO::Socket::INET->new(
			PeerAddr => $_,
			Timeout  => 10,
		) and last;

		# Fehlermeldung lautet bei älteren IO::Socket-Versionen
		# "IO::Socket::INET: Timeout", bei neueren
		# "IO::Socket::INET: connect: timeout"
		warn "Error connect()ing to $_: $@" unless $@ =~ /: Timeout$/i;
	}
	croak("Creating socket to $OTRS_TICKET_SERVER failed: $!") unless $socket;

	if ($obj->{'asuser'}) {
		$socket->print("user $obj->{'asuser'}\n")
		  or warn "Error print()ing to $OTRS_TICKET_SERVER: $!\n";
	}

	$socket->print("ticket_correspondence monitoring $queue\n")
	  or warn "Error print()ing to $OTRS_TICKET_SERVER: $!\n";

	$socket->print ("X-noris-Ticket-Type: $obj->{tickettype}\n" ) if defined $obj->{tickettype};

	$socket->print ("X-noris-Ticket-Queue: $queue\n")
	  or warn "Error print()ing to $OTRS_TICKET_SERVER: $!\n";

	$socket->print ("X-noris-Ticket-ArticleType: $type\n")
	  or warn "Error print()ing to $OTRS_TICKET_SERVER: $!\n";

	if ($info) {
		$socket->print ("X-noris-Ticket-Info: $info\n")
		  or warn "Error print()ing to $OTRS_TICKET_SERVER: $!\n";
	}
	if ($kunde) {
		$socket->print ("X-noris-Ticket-Kunde: $kunde\n")
		  or warn "Error print()ing to $OTRS_TICKET_SERVER: $!\n";
	}
	if ($prio) {
		$socket->print ("X-noris-Ticket-Priority: $prio\n")
		  or warn "Error print()ing to $OTRS_TICKET_SERVER: $!\n";
	}
	if ( my $owner = $obj->{owner} ) {
		$owner =~ s/\s+//g;
		$socket->print("X-noris-Ticket-Owner: $owner\n")
		  or warn "Error print()ing to $OTRS_TICKET_SERVER: $!\n";
	}

	$socket->print ($text)
	  or warn "Error print()ing to $OTRS_TICKET_SERVER: $!\n";
	$socket->print (".$CRLF")
	  or warn "Error print()ing to $OTRS_TICKET_SERVER: $!\n";

	my $ticket_seq;
	{
	    defined( $ticket_seq = <$socket> ) or die "Stirb";

	    # Hinweis auf Zustellung an ähnliches Ticket ignorieren:
	    redo if $ticket_seq =~ /^OK/;
	    last if $ticket_seq =~ /^(\d+)-(\d+)\s*\z/;
	    redo if $ticket_seq =~ /^\(#\d+\)\n/;
	}
	$socket->close or die "Error closing connection to $OTRS_TICKET_SERVER: $!\n";

	unless ( defined $ticket_seq ) {
	    print STDERR "Das Ticketsystem lieferte keine Ticket-ID.\n";
	}
	elsif ( $ticket_seq !~ /^(\d+)-(\d+)\s*\z/ ) {
	    return $ticket_seq;
	}
	else {
	    return ($1, $2);
	}

}

=item I<$ticket_id> = I<$obj>-E<gt>B<create> ()

Sendet das Ticket an das RT und liefert die Ticket-ID zurueck. Zu diesem
Zeitpunkt muessen mindestens die Attribute B<queue> und B<text> gesetzt sein.

=cut

sub create
{
	my $obj = shift;

	my $ticket_id;
	my $sequence_id;

	if (defined ($obj->{'ticket_id'}) and defined ($obj->{'sequence_id'}))
	{
		$ticket_id   = $obj->{'ticket_id'};
		$sequence_id = $obj->{'sequence_id'};
	}
	else
	{
		($ticket_id, $sequence_id) = _do_create ($obj, @_);
		$obj->{'ticket_id'}   = $ticket_id;
		$obj->{'sequence_id'} = $sequence_id;
	}

	#return ($ticket_id, $sequence_id) if (wantarray ());
	#return ("$ticket_id-$sequence_id");
	return ($ticket_id) if (wantarray ());
	return ($ticket_id);
}

=item I<$attr_value> = I<$obj>-E<gt>B<get> (I<$attr_name>);

Liefert den aktuellen Wert des Attributs zurueck.

=cut

sub get
{
	my $obj = shift;
	my $key = shift;

	if (!grep { $_ eq $key } (@ValidArgs))
	{
		return;
	}

	return ($obj->{$key});
}

=item I<$old_value> = I<$obj>-E<gt>B<set> (I<$attr_name>, I<$new_value>);

Setzt das Attribut I<$attr_name> auf den Wert I<$new_value>. Liefert den
vorherigen Wert des Attributs zurueck.

=cut

sub set
{
	my $obj = shift;
	my $key = shift;
	my $val = shift;

	if ($obj->{'ticket_id'})
	{
		return;
	}

	if (!grep { $_ eq $key } (@ValidArgs))
	{
		return;
	}

	my $ret = $obj->{$key};
	$obj->{$key} = $val;

	return ($ret);
}

=back

=head1 ATTRIBUTE

Die folgenden Attribute existieren:

=over 4

=item B<queue> I<(zwingend notwendig)>

Queue in der das neue Ticket generiert werden soll.

=item B<area> Wird nicht mehr unterstützt

!!! Wird nicht mehr Untertstützt !!!
Area kann nicht mehr verwendet werden, da das noris::CreateTicket mittlerweile
über den Ticketserver läuft.

Stattdessen tickettype verwenden.

=item B<tickettype>

Ist das gleich wie vorher die Area. Die Syntax hat sich allerdings geändert.
Für gültige Tickettypen bitte ins entsprechende Ticketsystem nachschauen.

=item B<kunde>

KundenB<name> des Kunden, der als B<Requestor> eingetragen werden wird.

=item B<extern>

Setzt das B<X-noris-Tiket-ArticleType>-Feld - oder eben nicht..

=item B<info>

Wert fuer das B<Info>-Feld.

=item B<priority>

Wert fuer das B<Priority>-Feld.

=item B<text> I<(zwingend notwendig)>

Text fuer das Ticket.

=back

=head1 AUTHOR

Florian octo Forster E<lt>octo@noris.netE<gt>
fuer die noris network AG L<http://noris.net/>

Angepasst an den Ticket Server von
Stelios Gikas E<lt>sgikas@noris.netE<gt>

=cut
