package Net::DNS::Recurse;

use utf8;
use warnings;
use strict;

use Carp;
use Net::DNS;
use Net::DNS::Cache;
use Net::DNS::IN::RootServers;
use Socket ();

use constant PORT => getservbyname(qw(domain udp)) || 53;
use constant RCODE_OK => { map +( $_ => undef ), qw(NOERROR NXDOMAIN) };

sub new {
    my $package = shift;
    $package = ref $package if length ref $package;
    die 'USAGE: ' . __PACKAGE__ . '->new({<method> => <value>}*)' if @_ & 1;
    my $self = bless {}, $package;
    while (@_) {
        my ( $method, $value ) = splice @_, 0, 2;
        $self->$method($value);
    }
    $self;
}

sub cache {
    my ($self) = @_;
    croak 'USAGE: ' . __PACKAGE__ . '->cache()'
      unless length ref $self && $self->isa(__PACKAGE__)
      and @_ == 1;
    $self->{cache} ||= new Net::DNS::Cache;
}

sub debug {
    my ( $self, $value ) = @_;
    die 'USAGE: ' . __PACKAGE__ . '->debug([<value>])'
      unless length ref $self && $self->isa(__PACKAGE__)
      and @_ == 1 || @_ == 2 && defined $value;
    $self->{debug} = $value if @_ == 2;
    $self->{debug};
}

sub delay {
    my ( $self, $n ) = @_;
    die 'USAGE: ' . __PACKAGE__ . '->delay([<n>])'
      unless length ref $self && $self->isa(__PACKAGE__)
      and @_ == 2             && defined $n;
    $n > 5 ? 42 : 2**$n;
}

sub get_ns4 {
    my ( $self, $stack, $name ) = @_;
    die 'USAGE: ' . __PACKAGE__ . '->get_ns4(<stack>, <name>)'
      unless length ref $self && $self->isa(__PACKAGE__)
      and @_ == 3 && defined $name && $name !~ /^\./ && ref $stack;
    print "->get_ns4('$name') called from @{[caller]}\n" if $self->debug;
    $name = uc $name;
    my @ns;
    while ( length $name ) {
        if ( defined( my $ns = $self->cache->get( $name, 'NS' ) ) ) {
            return scalar @{ [ split /\./, $name ] }, map \$_->nsdname,
              $ns->data;
        }
        else { $name =~ s/^[^.]+\.?// }
    }
    0, map scalar Socket::sockaddr_in( PORT, Socket::inet_aton($_) ),
      values %Net::DNS::IN::RootServers;
}

sub query {
    my $self  = shift;
    my $stack = ref $_[0] ? shift: {};
    my $name  = shift;
    my $type  = shift;
    croak 'USAGE: ' . __PACKAGE__ . '->query([<stack>], <name>, [<type>])'
      unless length ref $self && $self->isa(__PACKAGE__)
      and !@_
      and defined $name && length $name;
    $type = 'A' unless defined $type;
    print "->query('$name','$type') called from @{[caller]}\n" if $self->debug;
    my $result;

    unless ( defined( $result = $self->cache->get( $name, $type ) )
        || defined( $result = $self->cache->get( $name, 'CNAME' ) ) )
    {
        my ( $level, @ns );
        if ( exists $stack->{ my $slot = "$name $type" } ) {
            return "loop resolving $name, type $type";
        }
        else { $stack->{$slot} = undef }
        ( $level, @ns ) = $self->get_ns4( $stack, $name );
        my $ns = 0;
      QUERY: while () {
            my $question = new Net::DNS::Packet $name, $type
              or return "Cannot create DNS question for '$name', type '$type'.";
            $question->header->rd(0);    # vgl. RT#86410
            my $question_data = $question->data;
            my $cycles        = 0;
            my $socket        = new IO::Socket::INET Proto => 'udp'
              or return "IO::Socket::INET->new(): $!\n";
            vec( my $vec = '', fileno $socket, 1 ) = 1;
            my %seen;

            my $_error
              ;   # zum Vormerken von Fehlern, bei denen der aktuelle NameServer
                  # aus der Liste geworfen werden soll; siehe continue-Block
          NS: while () {

                if ( ref $ns[$ns] ) {
                    if ( $seen{ lc ${ $ns[$ns] } }++ ) {
                        $_error = "loop for ${$ns[$ns]}; skipping";
                        next;
                    }
                    elsif (
                        ref(
                            my $answer = $self->query( $stack, ${ $ns[$ns] } )
                        )
                      )
                    {
                        for ( $answer->data ) {
                            if ( $_->type eq 'CNAME' ) {
                                $ns[$ns] = \$_->cname;
                                redo NS;
                            }
                            elsif ( $_->type eq 'A' ) {
                                $ns[$ns] =
                                  Socket::sockaddr_in( PORT,
                                    Socket::inet_aton( $_->address ) );
                            }
                            else { die 'Unexpected record type ' . $_->type }
                        }
                    }
                    elsif ( defined $answer ) {
                        $_error = "query for ${$ns[$ns]} returned: $answer";
                        next;
                    }
                    elsif ( $self->debug ) {
                        $_error = "could not resolve ${$ns[$ns]}";
                        next;
                    }
                }
                if ( $self->debug ) {
                    ( undef, my $addr ) = Socket::sockaddr_in( $ns[$ns] );
                    $addr = Socket::inet_ntoa($addr);
                    print "Asking $addr ("
                      . join ( ', ', map $_->string, $question->question )
                      . ')... ';
                }
                unless (
                    defined(
                        my $sent = $socket->send( $question_data, 0, $ns[$ns] )
                    )
                  )
                {
                    print "Error send()ing: $!\n" if $self->debug;
                    next;
                }
                elsif ( $sent != length $question_data ) {
                    print "Sent $sent of length $question_data bytes.\n"
                      if $self->debug;
                    next;
                }
                elsif ( $self->debug ) { print "Sent $sent bytes.\n" }
                if (
                    0 > (
                        my $selected = select my $rvec = $vec,
                        undef, undef, $self->delay($cycles)
                    )
                  )
                {
                    return "select() returned $selected: $!\n";
                }
                elsif ( $rvec eq $vec ) {
                    defined( my $peer = $socket->recv( my $answer_data, 4096 ) )
                      or return "recv(): $!\n";
                    my ( $peerport, $peeraddr ) = Socket::sockaddr_in($peer);
                    $peeraddr = Socket::inet_ntoa $peeraddr;
                    my ( $answer, $error ) = new Net::DNS::Packet \$answer_data;

                    # Workaround: Nicht auf !defined($error) abstellen:
                    # https://rt.cpan.org/Ticket/Display.html?id=53595
                    if ( defined $answer ) {
                        $answer->answerfrom($peeraddr);
                        $answer->answersize( length $answer_data );
                        if ( $self->debug ) {
                            ( my $answer_string = $answer->string ) =~
                              s/^/| /gm;
                            print $answer_string;
                        }
                        unless (
                            exists RCODE_OK->{ my $rcode =
                                  $answer->header->rcode } )
                        {
                            $_error = "answer had rcode $rcode";
                        }
                        elsif ( $answer->header->tc ) {
                            print "Answer is truncated.\n" if $self->debug;
                        }
                        else {
                            $self->cache->learn($answer);
                            last QUERY
                              if defined( $result =
                                  $self->cache->get( $name, $type ) )
                              || defined( $result =
                                  $self->cache->get( $name, 'CNAME' ) )
                              || $answer->header->aa
                              || $rcode eq 'NXDOMAIN';
                            my ( $_level, @_ns ) =
                              $self->get_ns4( $stack, $name );
                            if ( $_level > $level ) {
                                print
"Answer led from level $level to level $_level.\n"
                                  if $self->debug;
                                $level = $_level;
                                @ns    = @_ns;
                                $ns    = 0;
                                next QUERY;
                            }
                            else {
                                $_error =
"answer was not helpful (still at level $_level)";
                            }
                        }
                    }
                    elsif ( $self->debug ) {
                        print
"Error decoding DNS answer from $peeraddr:$peerport: $error\n";
                    }
                }
            }
            continue {
                if ( defined $_error ) {
                    my $discarded = splice @ns, $ns--, 1;
                    print "\u$_error.\n" if $self->debug;
                    $discarded =
                      ref $discarded
                      ? $$discarded
                      : Socket::inet_ntoa +
                      ( Socket::sockaddr_in $discarded)[1];
                    return
"no valid nservers left after discarding $discarded, because $_error"
                      unless @ns;
                    undef $_error;
                }
                unless (@ns)            { return 'no valid nservers left' }
                elsif  ( ++$ns > $#ns ) {
                    $ns = 0;
                    ++$cycles;
                }
                return "Giving up after $cycles cycles at level $level ("
                  . (
                    join ',',
                    map ref()
                    ? $$_
                    : Socket::inet_ntoa( ( Socket::sockaddr_in $_)[1] ),
                    @ns
                  )
                  . ').'
                  if $cycles >= 3;
            }
        }
    }
    $result;
}

1

__END__

Änderungen:

2002-10-07 ->query() konnte bei unerwarteten DNS-Antworten in Endlosschleife
           geraten, weil $level immer undef blieb; gefixt

2002-10-08 s/send/query/ in USAGE-Meldung von ->query() 

2002-10-11 Net::DNS::Resolver-Aufrufe durch eigenen Code ersetzt, weil wir
           sinnvollerweise auch bei unvollständigen Antworten weiterfragen
           müssen.

2003-04-29 Wir müssen _erst_ die Antwort analysieren (bzw. in $result kopieren)
           und _dann_ schauen, ob sie autoritativ war.

2004-02-05 get_ns4() hatte eine Endlosschleife für (kaputte) Namen, die mit
           einem Punkt beginnen, z. b. ".staedtler.dk". Stirbt jetzt ggf.
