package Net::DNS::Cache;

use utf8;
use warnings;
use strict;

use Carp;
use Net::DNS;
use Net::DNS::Cache::Slot;

sub new { my $self = bless {}, __PACKAGE__ }

sub cleanup {
    my $self = shift;
    croak 'USAGE: ' . __PACKAGE__ . '->cleanup()'
      unless length ref $self && $self->isa(__PACKAGE__)
      and @_ == 1;
    my $deleted = 0;
    $_->ttl < 0 and delete $self->{$_} and ++$deleted for keys %$self;
    $deleted;
}

sub get {
    my ( $self, $name, $type, $class ) = @_;
    croak 'USAGE: ' . __PACKAGE__ . '->get(<name>, [<type>, [<class>]])'
      unless length ref $self && $self->isa(__PACKAGE__)
      and @_ >= 2             && @_ <= 4
      and defined $name;
    my $slot = $self->_slotkey( $name, $class, $type );
    delete $self->{$slot} if exists $self->{$slot} && $self->{$slot}->ttl < 0;
    $self->{$slot};
}

sub learn {
    my $self = shift;
    croak 'USAGE: ' . __PACKAGE__ . '->learn(<Net::DNS::Packet>+)'
      unless length ref $self
      && $self->isa(__PACKAGE__)
      && !grep !$_->isa('Net::DNS::Packet'), @_;
    for my $packet (@_) {
        $self->replace($packet->answerfrom, $packet->$_) for qw(additional authority answer);
    }
}

sub replace {
    my $self       = shift;
    my $answerfrom = shift;
    croak 'USAGE: ' . __PACKAGE__ . '->replace(<answerfrom>, <Net::DNS::RR>+)'
      unless length ref $self && $self->isa(__PACKAGE__)
      and !grep !$_->isa('Net::DNS::RR'), @_;
    for my $r (@_) {
        delete $self->{ $self->_slotkey( $r->name, $r->class, $_ ) }
          for $r->type, 'ANY';
    }
    for my $r (@_) {
        for ( $r->type, 'ANY' ) {
            $self->{ my $slot = $self->_slotkey( $r->name, $r->class, $_ ) } ||=
              new Net::DNS::Cache::Slot
              ttl        => $r->ttl,
              answerfrom => $answerfrom;
            $self->{$slot}->data($r);
        }
    }
}

sub _slotkey {
    my ( $self, $name, $class, $type ) = @_;
    croak 'USAGE: ' . __PACKAGE__ . '->slot(<name>, [<type>, [<class>]])'
      unless length ref $self && $self->isa(__PACKAGE__)
      and @_ >= 2             && @_ <= 4
      and defined $name;
    $type  = 'A'  unless defined $type;
    $class = 'IN' unless defined $class;
    "\U$name $class $type";
}

1
