package HousingDB::Hardware;

use utf8;
use strict;
use warnings;

BEGIN {
	unshift(@INC, ($ENV{'POPHOME'}||'@POPHOME@').'/lib')
		unless $ENV{'KUNDE_NO_PERLPATH'};
}

use Carp (qw(cluck confess));

use Dbase::Help;
use Dbase::Globals;
use Dbase::IP;
use Loader qw(log_update);

=head1 NAME

HousingDB::Hardware

=head1 ATTRIBUTE

=over 4

=item B<id> I<(int4)>

=item B<hardware_id> I<(char255)>

=item B<name> I<(char250)>

=item B<seriennr> I<(char250)>

=item B<ivnr> I<(char255)>

=item B<kunde> I<(int4)>

=item B<info> I<(char255)>

=item B<beginn> I<(int4)>

=item B<ende> I<(int4)>

=item B<rack> I<(int2)>

=item B<unterste_he> I<(int1)>

=item B<he> I<(int1)>

=back

=cut

our %ValidFields =
(
	id		=> 1,
	hardware_id	=> 2,
	name		=> 3,
	seriennr	=> 4,
	ivnr		=> 5,
	kunde		=> 6,
	info		=> 7,
	beginn		=> 8,
	ende		=> 9,
	rack		=> 10,
	unterste_he	=> 11,
	he		=> 12,
	eigentuemer	=> 13,
	status		=> 14
);

return (1);

sub _retrieve
{
	my $pkg = shift;
	my %args = @_;
	my @retval = ();
	my @id_list = ();
	my $order_by = '';

	if (!defined ($args{'id_list'}))
	{
		confess ("Missing argument 'id_list'");
	}
	@id_list = @{$args{'id_list'}};

	if (defined ($args{'order_by'}) && (exists ($ValidFields{$args{'order_by'}})))
	{
		$order_by = 'ORDER BY ' . $args{'order_by'};
	}

	for (@id_list)
	{
		s/\D//g;
		$_ = int ($_);
		confess unless ($_);
	}

	DoSelect (sub
		{
			my $obj = {};
			@$obj{qw(id hardware_id name seriennr ivnr kunde info beginn ende rack unterste_he he eigentuemer status ip_id)} = @_;
			push (@retval, bless ($obj, $pkg));
		}, <<SQL);
	SELECT id, hardware_id, name, seriennr, ivnr, kunde, info, beginn, ende, rack, unterste_he, he, eigentuemer, status, ip
	FROM hardware
	WHERE id IN (${\join (', ', @id_list)})
	$order_by
SQL

	if (!@retval)
	{
		return;
	}
	elsif (wantarray ())
	{
		return (@retval);
	}
	elsif (@retval != 1)
	{
		cluck ("I was called in scalar context, but more than one object was returned by the database!");
	}
	return ($retval[0]);
} # _retrieve

=head1 METHODEN

=over 4

=item HousingDB::Hardware-E<gt>B<retrieve> (I<$id>)

Laed das Hardware-Objekt mit der entsprechenden ID.

=cut

sub retrieve
{
	my $pkg = shift;
	my $id = shift;

	return (_retrieve ($pkg, id_list => [$id]));
} # retrieve

sub create
{
	my $pkg = shift;
	my %data = @_;
	my $obj = \%data;

	for (keys %$obj)
	{
		delete ($obj->{$_}) unless (defined ($ValidFields{$_}));
	}

	if (defined ($obj->{'hardware_id'}) or !defined ($obj->{'name'}))
	{
		return (undef);
	}

	my @keys = sort { $ValidFields{$a} <=> $ValidFields{$b} } (keys (%$obj));
	my @vals = map { qquote ($obj->{$_}) } (@keys);

	return (undef) unless (scalar (@keys) > 1);

	my $sql = 'INSERT INTO hardware (' . join (', ', @keys) . ') VALUES (' . join (', ', @vals) . ')';
	print STDERR "Creating hardware: $sql\n";

	$obj->{'id'} = Do ($sql);
	log_update ('hardware', 'id', $obj->{'id'}, undef, '*', undef);

	return (bless ($obj, $pkg));
}

=item HousingDB::Hardware-E<gt>B<search> ([I<%where>], [I<$opts>])

Sucht Eintraege in der Datenbank. Die Suche kann durch I<%where> eingeschraenkt
werden. I<$opts> kann weitere Optionen angeben. Liefert eine Liste von
HousingDB::Hardware-Objekten.

  %where =
  (
    field0 => $value0,
    field1 => $value1,
    ...
  );
  $opts =
  {
    order_by => 'field1'
  };

=cut

sub search
{
	my $pkg = shift;
	my $opts = pop if (scalar (@_) % 2);
	my %fields = @_;
	my @fields = ();
	my $found_invalid = 0;

	my %field2table =
	(
		name		=> 'hardware',
		hardware_id	=> 'hardware',
		seriennr	=> 'hardware',
		ivnr		=> 'hardware',
		info		=> 'hardware',
		id		=> 'hardware',
		beginn		=> 'hardware',
		ende		=> 'hardware',
		rack		=> 'hardware',
		unterste_he	=> 'hardware',
		he		=> 'hardware',
		kunde		=> 'hardware',
		rz		=> 'rack',
		fqdn		=> 'ipkunde'
	);

	for (keys %fields)
	{
		my $key = $_;
		my $val = $fields{$key};
		my $table;

		if (!defined ($field2table{$key}))
		{
			warn ("Not a valid field: $key");
			$found_invalid++;
			delete ($fields{$key});
			next;
		}

		$table = $field2table{$key};

		if (!defined ($val))
		{
			push (@fields, "$table.$key IS NULL");
		}
		elsif ($key eq 'name' or $key eq 'hardware_id' or $key eq 'seriennr' or $key eq 'ivnr' or $key eq 'info')
		{
			$val =~ s/"/""/g;

			if ($val)
			{
				push (@fields, qq($table.$key LIKE "$val"));
			}
			else
			{
				push (@fields, qq($table.$key IS NOT NULL));
			}
		}
		elsif ($key eq 'id' or $key eq 'beginn' or $key eq 'ende' or $key eq 'rack'
				or $key eq 'unterste_he' or $key eq 'he')
		{
			$val =~ s/\D//g;

			if ($val)
			{
				push (@fields, qq($table.$key = $val));
			}
			else
			{
				push (@fields, qq($table.$key IS NOT NULL));
			}
		}
		elsif ($key eq 'kunde' or $key eq 'rz')
		{
			my @tmp;

			if (!ref ($val))
			{
				@tmp = ($val);
			}
			else
			{
				@tmp = @$val;
			}

			$val = join (', ', grep { $_ != 0 } (map { int ($_) } (@tmp)));

			if ($val)
			{
				push (@fields, "$table.$key IN ($val)");
			}
			else
			{
				print STDERR "Invalid $key: $val\n";
				$found_invalid++;
			}
		}
		elsif ($key eq 'fqdn')
		{
			$val =~ s/"/""/g;
			if ($val)
			{
				push (@fields, qq(ipkunde.name LIKE "$val"));
			}
			else
			{
				push (@fields, q(ipkunde.name IS NOT NULL));
			}
		}
		else
		{
			die ("You should not get here");
		}
	}

	#print STDERR "Fields:", @fields, "; Invalid: ", $found_invalid, ";\n";

	if (!@fields and $found_invalid)
	{
		warn ("No where clauses but $found_invalid invalid definitions. Returning empty list.");
		return (qw());
	}

	push (@fields, 'hardware.enthalten_in IS NULL');
	push (@fields, '(hardware.ende IS NULL OR hardware.ende > UNIX_TIMESTAMP(NOW()))');

	my $where = '';
	$where = 'WHERE ' . join (' AND ', @fields);

	my @retval = ();
	my $sql = <<SQL;
	SELECT hardware.id
	FROM hardware
	LEFT JOIN rack ON hardware.rack = rack.id
	LEFT JOIN ipkunde ON hardware.ip = ipkunde.id
	$where
SQL

	#print STDOUT qq(<div style="position: absolute; bottom: 1ex; left: 1ex; background-color: yellow; color: red; border: 2px inset red; font-weight: bold;">$sql</div>\n);
	my $id_list = [];
	DoSelect (sub
		{
			my $id = shift;
			push (@$id_list, $id);
		}, $sql
	);

	return if (!@$id_list);

	@retval = _retrieve ($pkg, id_list => $id_list,
		(defined ($opts->{'order_by'}) ? (order_by => $opts->{'order_by'}) : ()));

	return (@retval);
} # search

=item $obj-E<gt>B<he> ([I<$hoehe>, [I<$unterste_he>]])

Gibt die Hoehe des Geraets (in HE) und die Position im Rack (die unterste HE)
zurueck. Falls gegeben wird die Hoehle und evtl. die Position gesetzt.

=cut

sub he
{
	my $obj = shift;

	if (@_)
	{
		my $old = $obj->{'he'} || '-';
		my $new;
		$obj->{'he'} = shift;

		$new = qquote ($obj->{'he'});

		log_update ('hardware', 'id', $obj->{'id'}, undef, 'he', undef, $old);
		Do ("UPDATE hardware SET he = $new WHERE id = " . $obj->{'id'});
	}
	if (@_)
	{
		my $old = $obj->{'unterste_he'} || '-';
		my $new;
		$obj->{'unterste_he'} = shift;

		$new = qquote ($obj->{'unterste_he'});

		log_update ('hardware', 'id', $obj->{'id'}, undef, 'unterste_he', undef, $old);
		Do ("UPDATE hardware SET unterste_he = $new WHERE id = " . $obj->{'id'});
	}
	
	return ($obj->{'he'}, $obj->{'unterste_he'});
}

=item $obj-E<gt>B<rack> ([I<$rack_id>])

Gibt die rack-ID zurueck, dem dieses Geraet zugeordnet ist oder I<NULL>, wenn
dieses Geraet keinem Rack zuzuordnen ist. Wird eine Rack-ID uebergeben wird das
Feld gesetzt.

=cut

sub rack
{
	my $obj = shift;

	if (@_)
	{
		my $new_val = shift;
		my $old_val = $obj->{'rack'} || 0;

		if (!$new_val or ($new_val != $old_val))
		{
			my $id = $obj->{'id'};
			my $standort = DoFn ("SELECT standort FROM hardware WHERE id = $id");

			$obj->{'rack'} = $new_val;
			$new_val = $obj->{'rack'} ? qquote ($obj->{'rack'}) : 'NULL';

			log_update ('hardware', 'id', $id, undef, 'rack', undef, $old_val ? $old_val : '-');

			# Hier den aktuellen Benutzer als neuen Standort angeben.
			log_update ('hardware', 'id', $id, undef, 'standort', undef, $standort) if (defined ($standort));
			Do ("UPDATE hardware SET rack = $new_val, standort = NULL WHERE id = $id");
		}
	}

	return ($obj->{'rack'});
}

=item $obj-E<gt>B<id> ()

Gibt die ID der Hardware zurueck. Dies ist die ID in der B<hardware>-Tabelle,
und B<NICHT> die sog. "B<Hardware-ID>".

=cut

sub id
{
	my $obj = shift;
	return ($obj->{'id'});
}

=item $obj-E<gt>B<hardware_id> ()

Gibt die Hardware-ID zurueck. Siehe vorhergehende Methode.

=cut

sub hardware_id
{
	my $obj = shift;
	return ($obj->{'hardware_id'});
}

=item $obj-E<gt>B<name> ([I<$name>])

Liefert pseudo-intelligent einen Namen der gut klingt: Falls der Hardware ein
IP-Objekt zugeordnet ist, wird entweder der zugehoerige (IP-)Name oder die
IP-Adresse zurueckgegeben. Falls kein IP-Objekt zugeordnet ist wird das
I<name>-Feld zurueckgegeben, also idR. der Geraetetyp.

Wird I<$name> uebergeben wird das I<name>-Feld der Hardware gesetzt, falls es
sich um Hardware B<ohne> Hardware-ID handelt.

=item $obj-E<gt>B<longname> ()

Liefert ausführliche Information zum Objekt, analog zu C<list_hardwares()>.

=item $obj-E<gt>B<verylongname> ()

Liefert noch ausführlichere Information, für den Tooltip in der Rackliste.

=cut

sub name
{
	my $obj = shift;

	my $retval = '';
	
	if (@_)
	{
		if (defined ($obj->{'hardware_id'}) and $obj->{'hardware_id'})
		{
			warn ("Will not change name for hardware with hardware_id!");
		}
		else
		{
			my $old = $obj->{'name'};
			my $new = shift (@_);

			if ($old ne $new)
			{
				log_update ('hardware', 'id', $obj->{'id'}, undef, 'name', undef, $old);
				Do ('UPDATE hardware SET name = ' . qquote ($new) . ' WHERE id = ' . $obj->{'id'});
				$obj->{'name'} = $new;
			}
		}
	}

	if (!$obj->{'ip'})
	{
		ip ($obj);
	}

	return ($obj->{'ip_name'}) if ($obj->{'ip_name'});
	return ($obj->{'ip'}) if ($obj->{'ip'});
	return ($obj->{'name'});
}

sub longname
{
	my $obj = shift;
	#id hardware_id name seriennr ivnr kunde info beginn ende rack unterste_he he eigentuemer status

	my $retval = '';

	$retval = $obj->{'name'};
	if($obj->{'ip_id'}) {
		ip ($obj) unless $obj->{'ip'};

		if($obj->{'ip_name'}) {
			$retval .= ' (' . $obj->{'ip_name'} . ')';
		} else {
			$retval .= ' [' . $obj->{'ip'} . ']';
		}
	}
	if($obj->{'kunde'} > 1) {
		$retval .= " (".name_kunde($obj->{'kunde'}).")";
	}
	if($obj->{'hardware_id'}) {
		$retval .= ' [' . $obj->{'hardware_id'} . ']';
	}

	return $retval;
}

sub verylongname
{
	my $obj = shift;
	#id hardware_id name seriennr ivnr kunde info beginn ende rack unterste_he he eigentuemer status

	my $retval = '';

	if($obj->{'hardware_id'}) {
		$retval = '#' . $obj->{'hardware_id'} . ': ';
	}
	$retval .= $obj->{'name'};
	if($obj->{'ip_id'}) {
		ip ($obj) unless $obj->{'ip'};

		if($obj->{'ip_name'}) {
			$retval .= ' (' . $obj->{'ip_name'} . ')';
		} else {
			$retval .= ' ';
		}
		$retval .= '[' . $obj->{'ip'} . ']';
	}
	if($obj->{'kunde'} > 1) {
		$retval .= " (".name_kunde($obj->{'kunde'}).")";
	}

	return $retval;
}

=item $obj-E<gt>B<realname> ()

Dumme Version der I<name>-Methode: Liefert in jedem Fall den Wert des
I<name>-Feldes.

=cut

sub realname
{
	my $obj = shift;
	return ($obj->{'name'});
}

=item $obj-E<gt>B<ip> ()

Gibt Informationen ueber das zugehoerige IP-Objekt zurueck: I<Name> und
I<IP-Adresse> jeweils falls vorhanden, falls keines von beiden gesetzt ist
liefert die Methode einen leeren String.

=cut

sub ip
{
	my $obj = shift;

	if (!$obj->{'ip'})
	{
		my $id = $obj->{'id'} or die;
		my ($name, $ip,$bits) = DoFn (<<SQL);
			SELECT ip.name, ip.ip6,ip.bits FROM hardware h
			LEFT JOIN ipkunde ip ON h.ip = ip.id
			WHERE h.id = $id
SQL
		$name ||= '';

		$obj->{'ip'} = (defined $bits) ? Dbase::IP->new_db($ip,$bits)->str : '';
		$obj->{'ip_name'} = $name ? $name : '';
	}

	return ($obj->{'ip_name'} . ' (' . $obj->{'ip'} . ')') if ($obj->{'ip_name'} and $obj->{'ip'});
	return ($obj->{'ip_name'}) if ($obj->{'ip_name'});
	return ($obj->{'ip'});
}

=item $obj-E<gt>B<seriennr> ()

Gibt die Seriennummer zurueck.

=cut

sub seriennr
{
	my $obj = shift;
	return ($obj->{'seriennr'});
}

=item $obj-E<gt>B<ivnr> ()

Gibt die Inventarnummer zurueck.

=cut

sub ivnr
{
	my $obj = shift;
	return ($obj->{'ivnr'});
}

=item $obj-E<gt>B<kunde> ()

Gibt die Kundennummer zurueck.

=cut

sub kunde
{
	my $obj = shift;
	return ($obj->{'kunde'});
}

=item $obj-E<gt>B<info> ()

Gibt den Infotext zurueck.

=cut

sub info
{
	my $obj = shift;
	return ($obj->{'info'});
}

=item $obj-E<gt>B<beginn> ()

Gibt den Beginn des Gueltigkeitsbereich zurueck.

=cut

sub beginn
{
	my $obj = shift;
	return ($obj->{'beginn'});
}

=item $obj-E<gt>B<ende> ([I<$time>])

Gibt das Ende des Gueltigkeitsbereich zurueck. Wenn I<$time> uebergeben wird,
wird das Feld gesetzt und der alte Wert zurueckgegeben. I<$time> ist in Epoch,
also in standard UNIX time (Sekunden seit dem 1. Januar 1970, 0:00 Uhr).

=cut

sub ende
{
	my $obj = shift;
	my $ret = $obj->{'ende'};

	if (@_)
	{
		my $nv = shift;
		my $id = $obj->{'id'};
		my $log_ende = isotime ($ret);

		log_update ('hardware', 'id', $id, undef,
			'ende', undef,
			$log_ende);
		Do (<<SQL);
			UPDATE hardware
			SET ende = ${\qquote ($nv)}
			WHERE id = $id
SQL
		$obj->{'ende'} = $nv;
	}

	return ($ret);
}

=item I<$eigentuemer> = $obj-E<gt>B<eigentuemer> ()

=item (I<$et_id>, I<$et_info>) = $obj-E<gt>B<eigentuemer> ()

Liefert den Eigentuemer als String oder Eigentuemer-ID und -Info als Array
zurueck.

=cut

sub eigentuemer
{
	my $obj = shift;
	my $et = $obj->{'eigentuemer'};

	return unless ($et);
	
	my $et_info = mpersinfo ($et);

	return ($et, $et_info) if (wantarray ());
	return ("#$et: $et_info");
}

# fany wollte das nicht:
sub eigentuermer_info
{
	my $obj = shift;
	my $et = $obj->{'eigentuemer'};

	return unless ($et);

	my @ret = ();
	my %val;
	@val{qw(Telefon Fax Handy ISDN E-Mail Adresse)} =
		DoFn ("SELECT fon, fax, pager, isdn, email, adr FROM person WHERE id = $et");

	if ($val{'Adresse'})
	{
		push (@ret, ['Adresse', $val{'Adresse'}]);
	}

	if ($val{'E-Mail'})
	{
		push (@ret, ['E-Mail-Adresse', puny_decode($val{'E-Mail'}, 4)]);
	}

	for (qw(Telefon Fax Handy ISDN))
	{
		push (@ret, [$_, $val{$_}]) if ($val{$_});
	}

	if (wantarray ())
	{
		return (@ret);
	}
	else
	{
		return join ('; ', map { $_->[0] . ': ' . $_->[1] } (@ret));
	}
}

=item $obj-E<gt>B<status> ()

Liefert I<Info> oder I<Name> des zugeordneten I<hardware_status> Deskriptors.

=cut

sub status
{
	my $obj = shift;

	if (defined ($obj->{'status'}))
	{
		my $ret = info_descr ('hardware_status', $obj->{'status'});
		$ret = get_descr ('hardware_status', $obj->{'status'}) unless ($ret);

		return ($ret);
	}

	return;
}

=item $obj-E<gt>B<children> ()

Liefert eine (ggf. mehrdimensionale) Hashref der Kinder einer Hardware zurueck.
Falls eine Hardware keine Kinder besitzt wird nur die ID und der Name der
Hardware zurueck gegeben. Der Hash schaut ungefaehr so aus:

  $children = {
    "$id" => {
        name     => "[$hardware_id] $name",
        children => {
            "$id" => { name => "[$hardware_id] $name", },
            "$id" => { name => "[$hardware_id] $name", },
            .
            :
        },
      },
    "$id" => { name => "$name [$hardware_id]", },
    .
    :
  };

=cut

sub children
{
	my $obj = shift;
	my $id = $obj->{'id'};
	return (_get_chrildren ($id));
}

sub _get_chrildren
{
	my $id = shift;
	my $childs;
	my $ret = {};

	DoSelect (sub
		{
			my $cid = shift;
			my $hid = shift;
			my $name = shift;

			$name .= " [$hid]" if ($hid);
			$childs->{$cid} = $name;
		}, qq(SELECT id, hardware_id, name FROM hardware WHERE enthalten_in = "$id")
	);

	return if !$childs;

	for ( keys(%$childs) ) {
		$ret->{$_}->{name} = $childs->{$_};
		my $child = _get_chrildren($_);
		$ret->{$_}->{children} = $child if defined $child;
	}
	return ($ret);
}

=item $obj-E<gt>B<remove_from_rack> ()

Traegt die Hardware aus dem Rack aus. Setzt

  hardware.rack = NULL
  hardware.unterste_he = NULL
  hardware.standort = $REMOTE_USER

Vgl. auch C<RT#387074>

=cut

sub remove_from_rack
{
	my $obj = shift;
	my $id = $obj->{'id'};

	# Save the old values
	my $ov_rack = $obj->{'rack'} || '-';
	my $ov_unterste_he = $obj->{'unterste_he'} || '-';
	my $ov_standort = DoFn ("SELECT standort FROM hardware WHERE id = $id");

	my $nv_standort = get_person ($ENV{'REMOTE_USER'});

	if (!defined ($ov_standort))
	{
		$ov_standort = '-';
	}

	log_update ('hardware', 'id', $id, undef,
		'rack', 'unterste_he', 'standort', undef,
		$ov_rack, $ov_unterste_he, $ov_standort);
	Do (<<SQL);
		UPDATE hardware
		SET rack = NULL,
		    unterste_he = NULL,
		    standort = $nv_standort
		WHERE id = $id
SQL
	$obj->{'rack'} = undef;
	$obj->{'unterste_he'} = undef;

	return (1);
}

=item I<%graph_links> = $obj-E<gt>B<ms_power_graph> ()

Generiert die URL zu den Stromverbrauchsgraphen der Messstelle

=cut

sub ms_power_graphs {
    my ( $self, $id) = @_;

    my @pg;
    my ( $msid, $sname, $tcollect, $ip_name ) = DoFn <<SQL;
SELECT mess_stelle.id, mess_stelle.name, mess_typ.collectd_typ, ipkunde.name
  FROM mess_stelle JOIN mess_typ ON mess_stelle.typ = mess_typ.id
    JOIN ipkunde ON mess_stelle.snmp_host = ipkunde.id
 WHERE mess_stelle.id = $id
   AND ( ipkunde.ende IS NULL OR ipkunde.ende >= UNIX_TIMESTAMP(NOW()) )
   AND ( mess_stelle.ende IS NULL OR mess_stelle.ende >= UNIX_TIMESTAMP(NOW()) )
SQL

    return () unless $msid;

    my $cold =
        'https://collect.noris.net/cgi-bin/collection3/bin/graph.cgi?'
      . 'plugin=snmp;'
      . 'end=' . time() . ';'
      . "hostname=$ip_name;"
      . "type=$tcollect;"
      . "type_instance=$sname;";

    my $day   = 'begin=' . ( time() - 86400 *   1 ) . ';';
    my $week  = 'begin=' . ( time() - 86400 *   7 ) . ';';
    my $month = 'begin=' . ( time() - 86400 *  31 ) . ';';
    my $year  = 'begin=' . ( time() - 86400 * 365 ) . ';';

    my %links = (
        ms_id                => $msid,
        ms_name              => $sname,
        collectd_graph_day   => $cold . $day,
        collectd_graph_week  => $cold . $week,
        collectd_graph_month => $cold . $month,
        collectd_graph_year  => $cold . $year,
    );
    return %links;
}

=item I<@collectd_links> = $obj-E<gt>B<power_graph> ()

Generiert die URL zu den Stromverbrauchsgraphen eines Racks

=cut

sub power_graphs {
    my $obj = shift;

    my $cold =
        'https://collect.noris.net/cgi-bin/collection3/bin/graph.cgi?'
      . 'plugin=snmp;'
      . 'begin=' . ( time() - 86400 * 31 ) . ';'
      . 'end=' . time() . ';';

    my $collect_link = "http://collect.noris.net/cgi-bin/collection3/bin/index.cgi?";

    my @pg;
    DoSelect(
        sub {
            my ( $vid, $msid, $sname, $tcollect, $ip_name ) = @_;
            my $extra =
                "hostname=$ip_name;"
              . "type=$tcollect;"
              . "type_instance=$sname;";

            my %links = (
                messstelle_id  => $msid,
                collectd_graph => $cold . $extra,
            );
            push( @pg, \%links );
        },
        <<SQL);
SELECT mess_verbraucher.id, mess_stelle.id, mess_stelle.name, mess_typ.collectd_typ, ipkunde.name
  FROM mess_verbraucher JOIN mess_stelle ON mess_verbraucher.mess_stelle = mess_stelle.id
    JOIN mess_typ ON mess_stelle.typ = mess_typ.id
    JOIN ipkunde ON mess_stelle.snmp_host = ipkunde.id
 WHERE mess_verbraucher.hardware = $obj->{'id'}
   AND ( ipkunde.ende IS NULL OR ipkunde.ende >= UNIX_TIMESTAMP(NOW()) )
   AND ( mess_stelle.ende IS NULL OR mess_stelle.ende >= UNIX_TIMESTAMP(NOW()) )
SQL

    return @pg;
}

=back

=head1 SIEHE AUCH

L<HousingDB::Rack>

=head1 AUTOR

Florian Forster E<lt>octo at noris.netE<gt> fuer die noris network AG
L<http://noris.net/>

=cut
