package Fehler;
use utf8;
use warnings;
use strict; use warnings;
use Carp qw();
use POSIX qw(_exit);
use Term::ANSIColor qw(CYAN RED RESET);
use Error qw(:try with);

$Error::Debug=1;
# use Dbase::Help qw(DoTransExit); # würde zu Import-Schleife führen (RT#272498)

# kein cutoff bittedanke
$Carp::MaxArgLen = 1000;
$Carp::MaxArgNums = 0;

require Exporter;
our @ISA = qw(Exporter);
our $VERSION = "0.1";
our @fehler; ## TEMP
our @EXPORT_OK = qw(warnung problem fehler nfehler
				    ohne_warnung hat_warnung hat_problem
				    warnungen probleme ffehler re_fehler
					report_status report_fehler add_to_fehler
					@fehler
					);
## @fehler ## TEMP

our $ohne_warnungen;
our $last_bug;
our $in_ffehler;

#BEGIN {
#	eval "use Devel::DollarAt;";
#	$Error::Debug = 1;
#}

my @warnungen;
my @probleme;

sub report_fehler(;$$);

sub import {
	my $pkg = shift;
	my($callpkg) = caller;
	my @arg;
	foreach my $arg(@_) {
		if($arg eq ":die") {
			no strict 'refs';
			${"${callpkg}::_fehler_die"} = 1;
		} else {
			push(@arg,$arg);
		}
	}
	Fehler->export_to_level(1, $pkg, @arg);
}

=head1 Fehler -- Fehler/Problem/Warnmeldungen abfangen

=head2 Verwendung

	use Fehler;

	... code ...
	ohne_warnung {    ## ignoriert Warnungen 
		if(warnung("willst du dies auf die Nase fallen lassen?")) {
			# der Code wird nicht weiter ausgeführt
			return;
		}
	};
	problem("das würde auf die Nase fallen");
	fehler("RUMMS -- das ist ein Problem");

	probleme {
		# es geht nicht weiter
		warnungen {1;} # löscht alle Warnungen
		1;
	};
	warnungen {
		# frage ob OK, 
		1;
	} or # ... alles OK ...


	my $was = datensatz_anlegen();
	
	ffehler { # bearbeiten; das kann fehlschlagen
		datensatz_bearbeiten($was);
	} sub { # wieder weg damit
		datensatz_löschen($was);
	}

=cut

=head1 ohne_warnung(Code)

Läßt eine Prozedur ohne Warnungen durchlaufen.

=head2 Verwendung

	ohne_warnung {
		code_der-warnungen_produziert();
	};

=head2 Auswirkung

Im aufgerufenen Code auftretende Aufrufe von warnung() werden ignoriert;
diese Aufrufe liefern 0 zurück.

=cut

sub ohne_warnung(&) {
	{
		package Fehler::warnkill;
		sub DESTROY {
			my $self = shift;
			$Fehler::ohne_warnungen = $$self;
		}
	}
	my($code) = @_;
	my $ow = $ohne_warnungen;
	my $wk = bless \$ow,"Fehler::warnkill";
	$ohne_warnungen = 1;

	&$code();
}

=head1 hat_warnung()

Liefert die Anzahl anstehender Warnungen zurück.

Diese Funktion sollte nur in Ausnahmefällen verwendet werden, da
Unterfunktionen ihren Erfolgsstatus immer via Rückgabewert zurückliefern
sollen.

=head2 Verwendung

	if(hat_warnung()) {
		return "Das machen wir jetzt vorsichtshalber nicht.\n";
	}

=cut

sub hat_warnung {
	scalar @warnungen;
}

=head1 hat_problem()

Liefert die Anzahl anstehender Probleme zurück.

Diese Funktion sollte nur in Ausnahmefällen verwendet werden, da
Unterfunktionen ihren Erfolgsstatus immer via Rückgabewert zurückliefern
sollen.

=head2 Verwendung

	return if hat_problem();

=cut

sub hat_problem {
	scalar @probleme;
}

=head1 warnung(Meldung...)

Wenn ein Problem ansteht: Ignoriere die Warnparameter und iefere 2 zurück.

Wenn Warnungen aktiv sind: Speichere die Warnparameter, liefere 1 zurück.

Andernfalls, d.h. wenn der Code unter C<ohne_warnungen{}> läuft:
Ignoriere die Warnparameter und liefere 0 zurück.

=head2 Verwendung

	my $was = welche_aktion();
	ist_gefaehrlich($was)
		and warnung("bla","fasel","Jemand schwingt die rote Fahne. Anhalten?")
		and return wantarray ? () : undef;
	
	aktion($was);

=head2 Auswirkung

Die Parameter werden gespeichert und können via C<warnungen{}>
abgerufen werden.

=head2 Sonderlocke

Wenn das erste Element ein Hash ist und ein C<tstatus>-Element enthält,
so schreibe die Meldung dort hinein.

=cut

sub warnung(@) {
	# return 2 if hat_problem();
	return 0 if $ohne_warnungen;

	if(ref $_[0] eq "HASH" and exists $_[0]{'tstatus'}) {
		my $r = shift;
		push(@{$r->{'tstatus'}},["warning",@_]);
		return wantarray ? () : undef;
	}

	my $arg = [ @_ ];
	$arg->[0] =~ s/^### /* /;
	push(@warnungen, $arg);
	1;
}

=head1 problem(Meldung...)

Speichert die Problemparameter.

=head2 Verwendung

	my $input = eingabe();
	eingabe_konsistent($input)
		or return problem("tabelle","spalte","Kann so nicht gespeichert werden");

	eingabe_verarbeiten($input);

=head2 Auswirkung

Die Parameter werden gespeichert und können via C<probleme{}>
abgerufen werden.

C<problem()> liefert immer C<undef> zurück.

=head2 Sonderlocke

Wenn das erste Element ein Hash ist und ein C<tstatus>-Element enthält,
so schreibe die Meldung dort hinein.

=cut

sub problem(@) {
	if(ref $_[0] eq "HASH" and exists $_[0]{'tstatus'}) {
		my $r = shift @_;
		if($ENV{'TESTING2'} and (
				"@_" =~ /keine vern..?nftige NS-Antwort/i or
				"@_" =~ /^keine NS-Records$/i or
				0)) {
			push(@{$r->{'tstatus'}},["info",@_]);
		} else {
			push(@{$r->{'tstatus'}},["error",@_]);
		}
		return wantarray ? () : undef;
	} elsif(not defined $_[0]) {
		shift @_;
	}
	my $arg = [ @_ ];
	$arg->[0] =~ s/^### /* / if defined $arg->[0];
#	push(@$arg,(caller(1))[3]);
#	push(@$arg,(caller(2))[3]);
	push(@probleme, $arg);
	wantarray ? () : undef;
}

=head1 fehler(Meldung...)

Meldet einen (normalerweise fatalen) internen Fehler.

=head2 Verwendung

	ffehler {
		fehler("Grummel","Brummel","Hier ist was schiefgelaufen!");
	} sub {
		print STDERR "Fehler! ",join(" ",@_),"\n";
	};

=head2 Auswirkung

Die Prozedur verlässt die erste Prozedur eines ffehler()-Aufrufs und führt
den zweiten Teil aus.

=head1 nfehler(Meldung...)

Wie C<fehler()>, aber es wird kein Stackdump oder ähnliche Metadaten
generiert, angehängt, was-auch-immer.

=cut

{
package Fehler::Daten;
}
{
package Fehler::Bug;
use Term::ANSIColor qw(RED RESET);
use Carp qw(longmess);

our @ISA = qw(Error);

sub new {
    my $self  = shift;
    my(@data) = @_;

    local $Error::Depth = $Error::Depth + 1;
	my($package, $file, $line) = caller(1);

    $Fehler::last_bug = $self->SUPER::new(-file => $file, -line => $line, -text => $data[0], -object => bless(\@data,"Fehler::Daten"));

	return $Fehler::last_bug;
}

sub stringify {
    my $self = shift;
    my $text = $self->report_fehler(1);
    $text;
}

sub report_fehler($;$$) {
	my($self,$flag,$kn) = @_;
	$flag=0 unless defined $flag;
	Dbase::_Help::DoTransExit(2) if $flag&4;
	if(defined $kn) {
		$kn = " *** ".$kn;
	} else {
		$kn = "";
	}
	my @fehler = @{$self->object};

	my $res = "\n*** Fehler$kn:\n";
	if($flag & 2) { # nur die erste Zeile
		$res .= $fehler[0];
	} else {
		$res .= join("\n",map {defined $_ ? $_ : "<undef>"} @fehler);
	}
	$res = "(Die Datenbankverbindung wurde neu aufgebaut)"
		if $flag&4 and $res =~ /server has gone away/;
	$res .= "\n";
	print STDERR RED, $res, RESET unless $flag&1;
	$res;
}

} # end package Fehler::Bug

sub fehler(@) {
	my @fehler;
	@fehler = @_ if @_;

	@fehler = @{Error::prior()->object} 
		if not @fehler and ref Error::prior() and ref Error::prior()->object;
	@fehler = @Fehler::fehler unless @fehler; ## TEMP
	@fehler = ("(kein Fehlertext angegeben)") unless @fehler;

	my $ldump;
	my @msg;

	eval {
		if(Dbase::_Help::in_test()) {
			$Carp::MaxArgLen = 10000;
			$Carp::MaxArgNums = 0;
		}
	};

	no strict qw(subs);
	if(defined $Apache::{request} or %{*Apache2::RequestRec::}) {
		# Dieses Konstrukt fällt (manchmal) wegen eines obskuren Perl-Bugs
		# ganz gewaltig auf die Nase
		@msg = split("\n",Carp::longmess("...Stack-Dump...\n"));
	} else {
		# Unter Apache geht dieses Konstrukt nicht
		my $pid=open(FEHLER,"-|");
		unless($pid) {
			eval {
				print Carp::longmess("\n...Stack-Dump...\n");
				_exit(0);
			};
			print "EVAL: $@\n";
			_exit(0);
		}
		@msg = <FEHLER>;
		close(FEHLER);
	}

	push(@fehler, 
		grep { 
			s#^\s+##; s#\s*\n?\z##;
			s#\s?called.at.#\n @ #;
			s#(?:\s|\\x\{[a9]\})+# #;
			s#::(crypter)\(.*\)#::$1(*GEHEIM*)#;
			s#/usr/(?:pop/lib|src/pop/datenbank)/kunde/##;
			if(m#^Fehler::f+ehler# or m#^at .* line \d+$# ) { 0; }
			elsif($ldump) { $ldump=0; 1; }
			elsif(m#\.\.\.Stack-Dump\.\.\.#) { $ldump=1; 1; }
			else { m#/usr/pop/lib/rt/#i or not m#/usr/pop/lib# } } 
				@msg)
		unless $ENV{'KUNDE_DEBUG'} or grep { m#\.\.\.Stack-Dump\.\.\.# } @fehler;
	push(@fehler, @msg) if $ENV{'KUNDE_DEBUG'};

	@Fehler::fehler = @fehler; ## TEMP
	eval { goto testfehler; };
	Fehler::Bug->new(@fehler);
	eval { goto fehler; } unless $in_ffehler;
	die $Fehler::last_bug;
}


sub nfehler(@) {
	my @fehler;
	@fehler = @_ if @_;
	@fehler = ("(kein Fehlertext angegeben)") unless @fehler;

	eval { goto testfehler; };
	die with Fehler::Bug(@fehler);
}


=head1 ffehler { Code } sub { Fehlerbehebung }

Fangprozedur, um im Fehlerfall Seiteneffekte aufheben zu können.

=head2 Verwendung

	ffehler {
		$a++;
		fehler "Blubb";
		$a--;
		return "OK";

	} sub {
		$a--;
		goto fehler; # um das Problem zu propagieren
	}

=cut

###
# Dieser Code ist deswegen so unheimlich nervig, weil das Error-Modul
# davon ausgeht, dass bei jedem die()-Aufruf innerhalb dieses Codes, dem
# ein Objekt übergeben wird, dieses Objekt eine Unterklasse von Error ist.
#
# Hilfreicherweise ist Apache (siehe APR::Error) anderer Meinung.
#
sub throwit($) {
	my $err = shift;
	throw Error::Simple($err) unless ref $err;
	throw Error::Simple("".$err, $err->{rc}) if ref $err eq 'APR::Error';
	throw $err if UNIVERSAL::isa($err, 'Error');
	throw Fehler::Bug($err);
}
sub ffehler(&&) {
	my $what = shift;
	my $err = shift;
	local $in_ffehler = 1;
	my $whath = sub {
		my $ok = 0;
		my @res = ();
		my $wa = wantarray;
		eval {
			if($wa) { @res = &$what(); }
			elsif(defined $wa) { $res[0] = &$what(); }
			else { &$what(); }
			$ok = 1;
		};
		throwit((%{*Devel::DollarAt::} and ref $@) ? $@."\n".$@->backtrace : $@) unless $ok;
		wantarray ? @res : $res[0];
	};
	my $errh = sub {
		$in_ffehler = 0;
		my $ok = 0;
		my @res = ();
		my $wa = wantarray;
		eval {
			if($wa) { @res = &$err(@{$_[0]->object}); }
			elsif(defined $wa) { $res[0] = &$err(@{$_[0]->object}); }
			else { &$err(@{$_[0]->object}); }
			$ok = 1;
		};
		throwit($@) unless $ok;
		wantarray ? @res : $res[0];
	};
	my $errs = sub {
		$in_ffehler = 0;
		my $ok = 0;
		my @res = ();
		my $wa = wantarray;
		eval {
			no warnings "uninitialized";
			$_[0]="0 <undef>" if not defined $_[0];
			if($wa) { @res = &$err(0+$_[0], "".$_[0]); }
			elsif(defined $wa) { $res[0] = &$err(0+$_[0], "".$_[0]); }
			else { &$err(0+$_[0], "".$_[0]); }
			$ok = 1;
		};
		throwit($@) unless $ok;
		wantarray ? @res : $res[0];
	};

	my($pkg) = caller;
	no strict 'refs';
	if (${"${pkg}::_fehler_die"}) {
		try { &$whath(); }
		catch Fehler::Bug with { &$errh($_[0]); }
		catch Error::Simple with { &$errs($_[0]); }
		otherwise { $in_ffehler = 0; &$err(shift); };
	} else {
		try { &$whath(); }
		catch Fehler::Bug with { &$errh($_[0]); }
		catch Error::Simple with { &$errs($_[0]); }
		otherwise {
			my $e = shift;
			print STDERR "AUA 2: Unbekannter Fehlertyp: ".ref($e)." $e\n";
		}
	}
}


=head1 warnungen { Code }

Liste alle anstehenden Warnungen auf.

=head2 Verwendung

	# melde alle Bla-Fehler
	warnungen {
		if($_[0] eq "bla") {
			print STDERR "Eine Bla-Warnung ist aufgetreten: @_\n";
			return 1;
		} elsif($_[0] eq "blubb") {
			# speichere für den späteren blubb-Fehlerhandler, daß das
			# Problem hier auftrat
			## push(@_,"(via BlaFasel)"); ## geht aktuell nicht, nie verwendet
			return 0;
		}
		0;
	};

=head2 Auswirkungen

Dem Code werden alle anstehenden Warnungen übergeben.

Liefert der Code C<true> zurück, gilt die Warnung als bearbeitet und 
wird gelöscht. Ansonsten wird sie gespeichert und beim nächsten Aufruf
von C<warnungen> wieder übergeben.

Die Warnmeldung wird als Parameter übergeben.

Die Funktion liefert die Anzahl der erledigten Warnungen zurück.

=head1 probleme

siehe B<warnungen()>, entsprechend.

=cut

sub probcall($@) {
	my($handler,@in) = @_;
	my @out;
	my $nr = 0;

	foreach my $prob(@in) {
		if(&$handler(@$prob)) {
			$nr++;
		} else {
			push(@out, $prob);
		}
	}
	($nr,@out);
}
sub warnungen(&) { my $nr; ($nr,@warnungen) = probcall($_[0],@warnungen); $nr;}
sub probleme(&)  { my $nr; ($nr,@probleme)  = probcall($_[0],@probleme);  $nr;}

sub report_status(;$$) {
	my $num;
	my($flag,$kn) = @_;
	$flag=0 unless defined $flag;
	if(defined $kn) {
		$kn = " *** ".$kn;
	} else {
		$kn = "";
	}
	
	$num = hat_warnung;
	if($num) {
		print STDERR CYAN, "\n*** ".(($num>1)?"$num Warnungen":"1 Warnung")."$kn:\n";
		warnungen { print STDERR join("\n",@_)."\n"; };
		print STDERR "\n", RESET;
	}

	$num = hat_problem;
	if($num) {
		Dbase::_Help::DoTransExit(2) if $flag&4;
		print STDERR RED, "\n*** ".(($num>1)?"$num Probleme":"1 Problem")."$kn:\n";
		probleme { print STDERR join("\n",map { defined $_ ? $_ : "<NULL>" } @_),"\n"; };
		print STDERR "\n", RESET;
	} else {
		Dbase::_Help::DoTransExit(1) if $flag&8;
	}
}

=head2 report_fehler ( [ Flag [ Zusatz ]] )

Ausgabe der gespeicherten Fehlermeldung, evtl. mit Zusatztext.

Flag-Bits:

=over 4

=item 1

Nicht auf stderr ausgeben, nur zurückliefern.

=item 2

Nur die erste Zeile ausgeben, insbes. also kein Traceback.

=item 4

Aktuelle Transaktion abbrechen und neu starten.

=back


=head2 report_status ( [ Flag [ Zusatz ]] )

Ausgabe der gespeicherten Problem-Meldungen, evtl. mit Zusatztext.

Flag-Bits:

=over 4

=item 4

Aktuelle Transaktion abbrechen und neu starten, falls ein Problem
gemeldet wurde.

=item 8

Aktuelle Transaktion bestätigen und neu starten, falls B<kein> Problem
gemeldet wurde.

=back


=cut

sub report_fehler(;$$) {
	my($flag,$kn) = @_;
	my $e = Error::prior();
	$e = $last_bug if not defined $e;
	if(defined $e) {
		if (UNIVERSAL::isa($e,"Error::Simple")) {
			return $e->stringify();
		} else {
			return $e->report_fehler($flag,$kn);
		}
	}
	print STDERR <<_;

*** Ein Fehler ist aufgetreten, wurde aber nicht korrekt geloggt ?!?
*** Bitte oeffne ein Entwicklungsticket und kopiere deine letzten
    Aktionen in selbiges.
*** Danke.
_
}
sub add_to_fehler(@) {
	if(ref Error::prior() and ref Error::prior()->object) {
		unshift @{Error::prior()->object},@_;
	} else {
		Carp::confess("kaputter Fehleraufruf: @_");
	}
};
sub re_fehler() {
	Error::prior()->throw();
};

1;

=head2 Altlasten

Fehler führen zum Programmende.

Warnungen und Probleme werden bei Programmende auf C<stderr> ausgegeben.

=cut

END {
	report_status;
}

1; # Ende
