use strict;
use utf8;
use warnings;

package Dbase::Ticket;

{
    my $ticket_api;

    sub _ticket_api() {
        return $ticket_api if defined $ticket_api;
        require noris::Ticket::API;
        $ticket_api = noris::Ticket::API::get_pooled_connection();
    }
}

sub new {
    my ( $package, %self ) = @_;
    $package = ref $package if length ref $package;
    bless \%self, $package;
}

use constant {
    ATTR2OBJECT => {
        kunde => [ 'Dbase::Object::Kunde',  'name' ],
        owner => [ 'Dbase::Object::Person', 'user' ],
    },
};

{

    # Wir merken uns, welche Ticket-Attribute schon mal gebraucht wurden,
    # und laden die bei weiteren Tickets dann gleich alle auf einmal.
    my %attributes;

    sub _load {
        my ( $self, $attribute ) = @_;

        ++$attributes{$attribute};

        # Dass das auch ohne TicketNummer "funktioniert",
        # brauchen wir fürs Tool get_tickets, aus dem dieser Code
        # auch ursprünglich stammt. => Evtl. mal überarbeiten...
        if ( defined $self->{ticket_number}
            && $self->{ticket_number} =~ /^\d+\z/ )
        {
            my @missing = grep !exists $self->{$_}, keys %attributes;
            @{$self}{@missing} =
              _ticket_api()->get_ticket( $self->{ticket_number}, \@missing );
        }
        $self->{$attribute};
    }
}

sub AUTOLOAD {
    my $self = shift;
    my ($attribute) = our $AUTOLOAD =~ /([^:]+)\z/ or die;
    $self->_load($attribute) unless exists $self->{$attribute};

    # Attribut ggf. in ein Objekt verwandeln.
    # Beachte: Es kann passieren, dass das Attribut schon anderweitig "auf
    # Vorrat" geladen (oder beim Konstruktur übergeben), aber noch nicht in
    # ein Objekt verwandelt wurde.
    if ( my $object_spec = ATTR2OBJECT->{$attribute}
        and defined $self->{$attribute} && !ref $self->{$attribute} )
    {
        my ( $package, @args ) = @$object_spec;
        eval "require $package";
        die $@ if length $@;
        $self->{$attribute} =
          $package->new_cached( @args => $self->{$attribute} );
    }

    $self->{$attribute};
}

sub DESTROY { }    # damit nicht AUTOLOAD aufgerufen wird

sub arbeitszeit {
    my $self = shift;
    return $self->{arbeitszeit} if exists $self->{arbeitszeit};
    require Dbase::Globals;
    my %ticket_zeit;
    @ticket_zeit{qw(ohne_faktoren mit_faktoren)} =
      Dbase::Globals::time4ticket( $self->ticket_number );
    $self->{arbeitszeit} = { ist => \%ticket_zeit };

    my $est_effort = $self->est_effort;
    $self->{arbeitszeit}{soll} = $est_effort * 3600
      if defined $est_effort
      && $est_effort ne '';    # OTRS liefert bei nicht gesetzter Sollzeit "".

    $self->{arbeitszeit};
}

{
    my %cache;

    sub new_cached {
        my ( $package, %attr ) = @_;
        $cache{ $attr{ticket_number} } ||= $package->new(%attr);
    }
}

1;

__END__

=encoding utf8

=head1 NAME

Dbase::Ticket - Objekt, das ein Ticket repräsentiert

=head1 SYNOPSE

	use Dbase::Ticket;
	my $ticket = Dbase::Ticket->new_cached( ticket_number => 123456 );
	my $subject = $ticket->title;

=head2 KONSTRUKTOREN

=over 4

=item ->new( ticket_number => $ticketnummer )

Anlegen eines neuen Objekts

=item ->new_cached( ticket_number => $ticketnummer )

legt nur dann ein neues Objekt an, wenn nicht zuvor über diesen Konstruktur
bereits eines für dieselbe Kundennummer angelegt wurde.

=back

=head2 METHODEN

Es können grundsätzlich beliebige Attribute, die von der Ticket-API unterstützt
werden, über die jeweils gleichnamige Methode abgefragt werden.

Ansonsten:

=over 4

=item ->arbeitszeit

... ist eine Referenz auf einen Hash mit folgendem Aufbau:

    {
        ist => {
            mit_faktoren  => $sekunden,
            ohne_faktoren => $sekunden,
        },
        soll => $est_effort * 3600,
    }
=item ->kunde

Hierbei handelt es sich um ein L<Dbase::Object::Kunde>-Objekt.

=item ->owner

Hierbei handelt es sich um ein L<Dbase::Object::Person>-Objekt.

=back

