use strict;
use utf8;
use warnings;

package Dbase::Object::Person;

use Moose;
extends 'Dbase::Object';

use overload '""' => sub { shift->name },

  # zur Vermeidung von Endlos-Rekursion in Class::MOP::Attribute:
  bool => sub { 1 },
  ;

use constant _attributes => {
    abteilung => { sql   => 'abt' },
    kunde     => { bless => [qw/Dbase::Object::Kunde id/] },
    pass      => {
        bless          => [qw/Dbase::Password/],
        add_args4bless => sub { shift->user },
    },
    peer_ip => { sql => 'uremip', bless => [qw/Dbase::Object::IP id/] },
    map +( $_ => {} ),
    qw/email homedir id name pager uid user/
};

use constant _uniq_attributes => qw/id user/;

# TODO: Das hier gehört in Dbase::Object; ich weiß nur noch nicht, wie:
while ( my $object_attr = each %{ __PACKAGE__->_attributes } ) {
    has $object_attr => (
        is   => 'rw',    # w für Attribute, die in Objekte umgewandelt werden
        lazy      => 1,
        predicate => "has_$object_attr",
        default   => sub { shift->_load($object_attr) },
    );
}
while ( my ( $attr, $def ) = each %{ __PACKAGE__->_attributes } ) {
    defined( my $bless = $def->{bless} ) or next;
    my ( $package, @args ) = @$bless;
    around $attr => sub {
        my ( $orig, $self ) = @_;
        my $value = $self->$orig;
        return $value if !defined $value || ref $value;
        eval "require $package";
        die $@ if length $@;
        $self->$orig(
            $package->new(
                @args, $value,
                $def->{add_args4bless} ? $def->{add_args4bless}->($self) : ()
            )
        );
    };
}

has anrede => (
    is      => 'ro',
    lazy    => 1,
    default => sub {
        my $name = shift->name;

        my $anrede;
        if ( defined $name ) {
            my ( $herrfrau, $vorname ) = split ' ', $name;
            $anrede = { Frau => 'Liebe', Herr => 'Lieber' }->{$herrfrau}
              and defined $vorname
              and $anrede .= " $vorname";
        }
        $anrede = 'Guten Tag' unless defined $anrede;

        $anrede;
    },
);

has as_string => (
    isa     => 'Str',
    is      => 'ro',
    lazy    => 1,
    default => sub {
        my $self = shift;
        require Dbase::Globals;
        Dbase::Globals::mpersinfo( $self->id );
    },
);

sub _after_load {
    my $self = shift;
    return unless defined $self->{pager};
    require Dbase::Globals;
    $self->{pager} = Dbase::Globals::rufnummernliste( $self->{pager} );
}

has vorgesetzter => (
    is      => 'ro',
    lazy    => 1,
    default => sub {
        my $self = shift;
        require Dbase::Help and Dbase::Help->import('DoFn')
          unless defined &DoFn;
        my $id = $self->id;
        unless ( defined $id ) {
            require Carp;
            require Data::Dump;
            Carp::confess(
                ref($self) . '-Objekt ohne ID: ' . Data::Dump::pp($self) );
        }
        my $vorgesetzter =
          DoFn("SELECT vorgesetzter FROM perso WHERE person = $id");
        $vorgesetzter = $self->new_cached( id => $vorgesetzter )
          if defined $vorgesetzter;
        $vorgesetzter;
    },
);

__PACKAGE__->meta->make_immutable;

no Moose;

1;

__END__

=encoding utf8

=head1 NAME

Dbase::Object::Person - Objekt, das eine Person repräsentiert

=head1 SYNOPSE

  use Dbase::Object::Person;
  my $person = Dbase::Object::Person->new( id => 42 );
  my $name = $person->name;

=head2 KONSTRUKTOREN

=over 4

=item -E<gt>new( id => $person_id, ... )

Anlegen eines neuen Personen-Objekts

=item -E<gt>new_cached

Falls für diese Person mit dieser Methode bereits ein Objekt angelegt wurde,
wird das bestehende Objekt zurückgeliefert, ansonsten ein neues erzeugt und
gecacht.

=back

=head2 METHODEN

Folgende Attribute können abgefragt werden:

=over 4

=item -E<gt>id

Datenbank-ID der Person

=item -E<gt>abteilung

Abteilung der Person

=item -E<gt>anrede

Anrede für die Person

=item -E<gt>email

primäre E-Mail-Adresse der Person

=item -E<gt>homedir

das Home-Verzeichnis der Person

=item -E<gt>kunde

der primäre Kunde der Person als L<Dbase::Object::Kunde>

=item -E<gt>name

Name der Person

=item -E<gt>pass

Kennwort der Person als L<Dbase::Password>

=item -E<gt>pager

L<Liste der zugeordneten Mobilrufnummern|Dbase::Globals/rufnummernliste>

=item -E<gt>uid

Benutzer-ID der Person

=item -E<gt>user

Benutzername der Person

=item -E<gt>vorgesetzter

Vorgesetzter als L<Dbase::Object::Person>-Objekt (nur für Mitarbeiter)

=back

