package Dbase::IP;

use utf8;
use strict;
use warnings;

require Exporter;

=pod

Modul für IP-Adresse und -Adressbereiche.

=head1 Allokation etc.

=head2 new(txt)

Generiert aus dem angegebenen String ein IP-Objekt.

=head2 new_db(ip6,bits)

Generiert aus den angegebenen Datenbankfeldern ein IP-Objekt.

=head2 bitmask(NUM,$flag)

Liefert eine neue Adresse mit entsprechend gesetzter Bitmaske.
Wirft einen C<fehler>, wenn in der Adresse Bits gesetzt sind, die von der
neuen Maske ausgeblendet werden.

Wenn Flag&1, so wird kein Fehler geworfen, sondern die entsprechenden Bits
werden abgeschaltet.

Wenn Flag&2, wird ein problem() zurückgeliefert anstelle eines Fehlers.

=head2 mask4()

Liefert eine neue Adresse, in der nur die untersten 32 Bit gesetzt sind,
als IPv4-Adresse. Dient primär zur Anzeige von VRFs.


=head1 Anzeige

=head2 str( [ Flag ] )

Liefert die Adresse in textueller Form zurück.

Flags:

=over 4

=item 1 

Wert in der Form "Anfang ... Ende (/Bits)".

=item 2

Wert in der Form "Anfang - Ende".

=item 4

Adressen aus VRFs werden im VRF-Format "Name!V4adr" geliefert.

=item 8

Adressen aus VRFs werden nur als "V4adr" geliefert.

=back

=head2 str4()

Liefert die Adresse in IPv4-Schreibweise zurück. Falls es eine v6-Adresse
ist, wird ein Fehler geworfen. VRF-Adressen werden wie normale
IPv4-Adressen behandelt.

=head2 str6()

Gibt die Adresse in IPv6-Schreibweise aus, auch wenn es eine IPv4-Adresse
ist.

=head1 Arithmetik

Alle Operationen (+ -) beziehen sich auf die angegebene Netzadresse.

	10.2.3.4 + 1 == 10.2.3.5
	10.2.3.4/30 + 1 == 10.2.3.8/30
	10.2.3.8/29 - 1 == 10.2.3.0/29
	10.2.3.16/29 - 10.2.3.0/29 = 2
	10.2.3.16/28 - 10.2.3.0/28 = 1

=head1 Abfragen

=head2 old_ip4()

Gibt die Adresse aus (altes internes IPv4-Format)

=head2 db_ip4()

Gibt die Adresse aus (32-Bit-Zahl als Datenbankeintrag).

C<undef>: Die Adresse ist als IPv6-Datum nicht darstellbar.

=head2 db_ip6()

Gibt die Adresse aus (32-Byte-String für die Datenbank)

=head2 db_bits()

Gibt die Anzahl der Bits aus (Links-Shift für ~0-Maske, für die Datenbank)

=head2 dbs( [ Tabellenname ] )

Gibt die Inhalte der Datenbankfelder (ip6,bits) als String aus, für
SELECT-Befehle.

=head2 dbi( [ Tabellenname ] )

Wie vor, jedoch ist das Füllzeichen ein Komma, für C<INSERT>- und
C<UPDATE>-Befehle.

=head2 dbsub( [ Tabellenname ] [, Start ] )

Wie dbs(), jedoch als SELECT-Befehl für Subnetze. C<Start> gibt eine
optionale Anfangsposition für die Suche an. (Die Netzmaske der
Startadresse wird ignoriert.)

=head2 is_v4()

Liefert ein Flag, ob die Adresse eine IPv4-Adresse ist.

=head2 is_v4rf()

liefert 2, wenn die Adresse im VRF-Bereich liegt.
Ansonsten wie is_v4().

=head2 vrf_id()

liefert die VRF-Nummer, so das Netz ein VRF ist, ansonsten einen Fehler.

=head2 min_mask()

Liefert die Bitzahl, die die Netmask zu dieser Adresse mindestens haben muss.

=head1 Unterschiede zu L<NetAddr::IP>

Der Hauptunterschied ist, dass C<Dbase::IP> Netze abbildet, aber
B<nicht> Adressen innerhalb von Netzen.

Daraus folgt, dass

	10.1.2.4 +1 == 10.1.2.5
	10.1.2.4/30 +1 == 10.1.2.8/30

statt

	10.1.2.4 +1 == 10.1.2.4
	10.1.2.4/30 +1 == 10.1.2.5/30

Letztere Adresse wäre im Kontext von C<Dbase::IP> ungültig.

=cut

use Cf qw($VRF_PREFIX);
use Dbase::Help;
use Net::IP qw(ip_get_version);
use NetAddr::IP qw(V4mask);
use NetAddr::IP::Util qw(hasbits add128 sub128 shiftleft notcontiguous addconst);
use NetAddr::IP::Lite qw(Zero Ones);    # _nicht_ :aton, vgl. RT#427581

use Fehler qw(fehler problem);

our @ISA = qw(Exporter NetAddr::IP);
our @EXPORT = qw();
our @EXPORT_OK = qw();

use overload '""' => sub { $_[0]->str(); },
             '++' => sub { $_[0]->add1(); },
             '--' => sub { $_[0]->sbt1(); },
             '+' => sub { $_[0]->addc($_[1]); },
             '-' => sub { $_[0]->sbtc($_[1]); },
             '+=' => sub { $_[0]->add($_[1]); },
             '-=' => sub { $_[0]->sbt($_[1]); };

my $vrf_prefix;

my @_bitcache;
# Liefert einen 128-bit-Vektor, in dem nur das Bit $b gesetzt ist.
# (gezählt von rechts)
sub _onebit($)
{
	my($b) = @_;
	fehler "OneBit $b" if $b<0 or $b>127;

	return $_bitcache[$b] if defined $_bitcache[$b];

	my $r = $b ? shiftleft(Ones,$b) : Ones;
	$r &= ~ shiftleft(Ones,$b+1) if $b < 127;

	$_bitcache[$b] = $r;
	$r;
}

# Multipliziert einen 128bit-Vektor $v mit einer Zahl $m.
sub _mult($$)
{
	my($m,$v) = @_;
	fehler "Multiplikation mit negativer Zahl ($m * $v)!" if $m < 0;
	return Zero if $m == 0;
	$v = _onebit($v);
	return $v if $m == 1;

	my $res = ($m & 1) ? $v : Zero;
	while(1) {
		$m >>= 1;
		last unless $m;
		$v = shiftleft($v,1);
		if($m & 1) {
			my $carry;
			($carry,$res) = add128($res,$v);
			fehler "Overflow mult $m $res $v" if $carry;
		}
	}
	$res;
}

# Dividiert einen 128bit-Vektor $v durch einen 1-Bit-Vektor.
# _div(n,_mult(n,b)) == b.
sub _div($$)
{
	my($m,$v) = @_;
	return 0 if $v eq Zero;
	$m = _onebit($m);
	return 1 if $v eq $m;

	my $res = 0;
	my $s = 1;
	while(1) {
		if(hasbits($m & $v)) {
			$res += $s;
			$v &= ~$m;
		}
		last if $v eq Zero;
		$s <<= 1;
		$m = shiftleft($m,1);
	}
	$res;
}

# Liefert die N-nächste IP-Adresse (bzw. das entsprechende Netz) zurück.
sub addc
{
	my ($ip,$num) = @_;
	fehler "Cannot add two IP addresses" if ref $num;
	return $ip->sbtc(-$num) if $num < 0;

	$ip = $ip->copy;
	my $carry;
	($carry,$ip->{addr}) = add128($ip->{addr}, _mult($num, $ip->db_bits));
	fehler "Overflow $ip" if $carry;
	$ip;
}

# Zahl: Liefert die N-vorherige IP-Adresse (bzw. das entsprechende Netz) zurück.
# Adresse: Liefert die Differenz.
# $n1+($n2-$n1) == $n1
sub sbtc
{
	my ($ip,$num) = @_;

	if(ref $num) {
		my $bits = $ip->db_bits;
		fehler "Adressen sind verschieden breit: $ip $num"
			if $bits != $num->db_bits;
		return -sbtc($num,$ip) if $ip < $num;
		my($carry,$res) = sub128($ip->{addr}, $num->{addr});
		fehler "Overflow $ip" unless $carry;
		return _div($ip->db_bits,$res);

	} else {
		return $ip->addc(-$num) if $num < 0;
		$ip = $ip->copy;
		my $carry;
		($carry,$ip->{addr}) = sub128($ip->{addr}, _mult($num,$ip->db_bits));
		fehler "Overflow $ip" unless $carry;
		return $ip;
	}
}

# Addiert N zu der gegebenen Adresse (bzw. dem Netz).
sub add
{
	my ($ip,$num) = @_;
	fehler "Cannot add two IP addresses" if ref $num;
	return $ip->sbt(-$num) if $num < 0;

	my $carry;
	($carry,$ip->{addr}) = add128($ip->{addr}, _mult($num, $ip->db_bits));
	fehler "Overflow $ip" if $carry;
	$ip;
}

# Zieht von der gegebenen Adresse (bzw. dem Netz) N ab.
sub sbt
{
	my ($ip,$num) = @_;
	fehler "Cannot inline-subtract two IP addresses" if ref $num;
	return $ip->add(-$num) if $num < 0;

	my $carry;
	($carry,$ip->{addr}) = sub128($ip->{addr}, _mult($num, $ip->db_bits));
	fehler "Overflow $ip" unless $carry;
	$ip;
}

# Liefert die darauffolgende Adresse / Netz.
sub add1
{
	my $ip = shift;

	my $carry;
	($carry,$ip->{addr}) = add128($ip->{addr}, _onebit($ip->db_bits));
	fehler "Overflow $ip" if $carry;
	$ip;
}

# Liefert die vorherige Adresse / Netz.
sub sbt1
{
	my $ip = shift;

	my $carry;
	($carry,$ip->{addr}) = sub128($ip->{addr}, _onebit($ip->db_bits));
	fehler "Overflow $ip" unless $carry;
	$ip;
}

sub _check_bits
{
	my $ip = shift;
	fehler ("Falsche Bitmaske", $ip)
		if $ip->db_ip6 ne $ip->networkaddr->db_ip6;
}

sub min_mask
{
	my $ip = shift;
	my $nc;
	(undef,$nc) = notcontiguous($ip->{addr});
	128-$nc;
}


our $RE_VRFname = qr/\w[-\w]*\w/;

#Generiert aus dem angegebenen String ein IP-Objekt.
sub new
{
	my $what = shift;
	my($str,$bits) = @_;
	my $vrf;
	$vrf = $1 if $str =~ s/^($RE_VRFname)!//;    # vgl. RT441266
	$bits=0 unless defined $bits;

	# NetAddr::IP::Lite ist viel zu permissiv
	return undef unless ip_get_version( ($str =~ m#(.*)/\d+$#) ? $1 : $str );

	# ... oder auch wieder nicht: 10.020.003.4 => 10.20.3.4
	$str =~ s/(^|[.:])0+(?=[1-9])/$1/g;

	my $ip = NetAddr::IP::Lite->new($str);
	return undef unless defined $ip;
	if(defined $vrf) {
		require Dbase::Help;
		require Dbase::Globals;
		$vrf = Dbase::Globals::get_vrf($vrf);
		return undef unless defined $vrf;
		$vrf = Dbase::IP->new($VRF_PREFIX)->bitmask(32) + $vrf;
		substr($ip->{addr},0,12)=substr($vrf->{addr},0,12);
		$ip->{isv6} = 1;
	} else {
		$ip->{isv6} = 0
			if $ip->{isv6} and not hasbits($ip->{addr} & V4mask);
	}
	bless $ip,$what;
	$ip = $ip->bitmask($bits) if $bits;

	$ip->_check_bits();
	$ip;
}


#Generiert aus den angegebenen Datenbankfeldern ein IP-Objekt.
#(ip6,bits)
sub new_db
{
	my $what = shift;
	my($ip,$bits) = @_;

	return undef if not defined $bits;
	fehler "Die Adresse ist nicht definiert" unless defined $ip;

	return undef unless defined $ip;
	$ip =~ s/(....)/$1:/g; $ip =~ s/:$//;
	$ip = NetAddr::IP::Lite->new($ip,128-$bits);

	$ip->{isv6} = 0
		if $ip->{isv6} and not hasbits($ip->{addr} & V4mask);
	bless $ip,$what;
	$ip->_check_bits();
	$ip;
}


#Setzt die Bitmaske der Adresse. Meldet ein C<problem>, wenn in der Adresse
#Bits gesetzt sind, die von der neuen Maske ausgeblendet werden.
#(NUM)
sub bitmask
{
	my $ip = shift;
	my($num,$flag) = @_;
	$flag=0 unless defined $flag;

	$ip = $ip->copy;

  return problem "Zu hoher Wert bei Bitmaske eingestellt: $ip $num"
    if ($num < 0);

  if ($ip->{isv6}) {
    return problem "Falsche Bitmaske: $ip $num bei IPv6-Adressen"
      if ($num > 128);
  } else {
    return problem "Falsche Bitmaske: $ip $num bei IPv4-Adressen"
      if ($num > 32);
  }

	$ip->{mask} = NetAddr::IP::Lite->new("::/".(128-$num))->{mask};
	if($flag & 2) {
		return problem "Falsche Bitmaske: $ip $num"
			if $ip->min_mask < $ip->db_bits;
	} elsif($flag & 1) {
		$ip->{addr} &= $ip->{mask};
	} elsif($num) {
		$ip->_check_bits();
	}
	$ip;
}

sub networkaddr
{
	my($adr) = @_;
	$adr = NetAddr::IP::Lite::network($adr);
	$adr->bitmask(0);
}

sub broadcastaddr
{
	my($adr) = @_;
	$adr = NetAddr::IP::Lite::broadcast($adr);
	$adr->bitmask(0);
}

sub mask4
{
	my $ip = shift;

	$ip = $ip->copy;

	substr($ip->{addr},0,12) = "\0"x12;
	$ip->{isv6} = 0;
	$ip;
}

#Gibt die Adresse in textueller Form aus.
sub str
{
	my $ip = shift;
	my($flag) = @_;
	$flag=0 unless defined $flag;

	my $res = "";
	if($ip->version == 6) {
        $vrf_prefix = Dbase::IP->new($VRF_PREFIX) 
            if $flag & (8|4) and not defined $vrf_prefix;
		my $masklen;
        if($flag & (8|4) and $ip->is_v4rf > 1) {
			require Dbase::Help;
			require Dbase::Globals;
			if($flag & 8) {
            	$res = $ip->mask4->addr;
			} else {
            	my $vip = $ip->bitmask(32,1);
            	$ip = $ip->mask4;
            	$res = DoFn "select vrf from ipkunde where ${\$vip->dbs}";
            	$res = ($vip - $vrf_prefix->bitmask(32)) if not defined $res;
				$res .= "!" . $ip->addr;
			}
			$masklen = 32-$ip->db_bits;
		} else {
			$res .= $ip->short;
			$masklen = $ip->masklen;
		}

		if($ip->db_bits) {
			if($flag & 2) {
				$res .= " - ".$ip->broadcastaddr;
			} elsif($flag & 1) {
				$res .= " ... ".$ip->broadcastaddr->short." (/".$ip->masklen4.")";
			} else {
				$res .= "/$masklen";
			}
		}
	} else {
		$res = $ip->addr;
		if($ip->db_bits) {
			if($flag & 2) {
				$res .= " - ".$ip->broadcastaddr->addr;
			} elsif($flag & 1) {
				$res .= " ... ".$ip->broadcastaddr->addr." (/".$ip->masklen.")";
			} else {
				$res .= "/".$ip->masklen;
			}
		}
	}
	$res;
}

sub masklen4()
{
	my $ip = shift;
	return ($ip->is_v4rf ? 32 : 128) - $ip->db_bits;
}

sub vrf_id()
{
	my $ip = shift;
	fehler "Dies ist kein VRF" unless $ip->is_v4rf > 1;

	$vrf_prefix = Dbase::IP->new($VRF_PREFIX) 
		unless defined $vrf_prefix;

	return $ip->bitmask(32,1) - $vrf_prefix->bitmask(32);
}

sub revstr {
	# 0x01020304 -> 4.3.2.1.in-addr.arpa
	my $ip = shift;

	my $str;
	my @a;
	if($ip->is_v4) {
		$str = "in-addr";
		@a = split(/\./,$ip->addr);
	} else {
		$str = "ip6";
		@a = split(//,$ip->db_ip6);
	}
	$str = join(".",reverse @a).".$str.arpa";
	return $str;
}

#Gibt die Adresse in IPv4-Schreibweise aus.
sub str4
{
	my $ip = shift;
	fehler "keine IPv4-Adresse: $ip"
		unless $ip->is_v4rf;
	$ip = $ip->mask4; # für VRFs
	return $ip->str;
}


#Gibt die Adresse in IPv6-Schreibweise aus, auch wenn es eine IPv4-Adresse ist.
sub str6
{
	my $ip = shift;
	my $ip2 = $ip->copy;
	$ip2->{isv6} = 1;
	return $ip2->str;
}


#Gibt die Adresse aus (32-Bit-Zahl in historischer Form).
#undef: Die Adresse ist als IPv6-Datum nicht darstellbar.
sub old_ip4
{
	my $ip = shift;
	return undef if $ip->{isv6};
	return unpack("N",substr($ip->{addr},-4));
}

#Gibt die Adresse aus (32-Bit-Zahl als Datenbankeintrag).
#undef: Die Adresse ist als IPv6-Datum nicht darstellbar.
sub db_ip4
{
	my $ip = shift;
	return undef if $ip->{isv6};
	my $res = unpack("N",substr($ip->{addr},-4));
	$res -= 2**32 if $res >= 2**31;
	$res;
}

#Gibt die Adresse aus (32-Byte-String für die Datenbank)
sub db_ip6
{
	my $ip = shift;
	return unpack("H*",$ip->{addr});
}

#Gibt die Anzahl der DB-Bits aus (Links-Shift für ~0-Maske)
sub db_bits
{
	my $ip = shift;
	if($ip->{isv6}) {
		return 128-$ip->masklen;
	} else {
		return 32-$ip->masklen;
	}
}


#Gibt die Inhalte der Datenbankfelder (ip6,bits) als String aus, für
#SELECT-Befehle.
sub dbs
{
	my $ip = shift;
	my($tname) = @_;
	if(defined $tname) {
		$tname .= ".";
	} else {
		$tname = "";
	}
	my $res;
	$res = "${tname}ip6='".$ip->db_ip6."'";
	$res .= " and ${tname}bits=".$ip->db_bits;

	"( $res )";
}

#Wie vor, jedoch für die Suche nach untergeordneten Netzen.
sub dbsub
{
	my($ip,$tname,$start) = @_;
	$start=$ip unless defined $start;

	return "1=0" unless $ip->db_bits; # unterhalb einer Adresse ist nichts!
	my $nip = $ip+1;

	if(defined $tname) {
		$tname .= ".";
	} else {
		$tname = "";
	}
	my $res;
	$res = "${tname}ip6>='".$start->db_ip6."' and ${tname}ip6<'".$nip->db_ip6."'";
	$res .= " and ${tname}bits<".$ip->db_bits;

	"( $res )";
}

#Wie vor, jedoch für INSERT und UPDATE-Befehle.
sub dbi
{
	my $ip = shift;
	my($tname) = @_;

	if(defined $tname) {
		$tname .= ".";
	} else {
		$tname = "";
	}
	my $res;
	$res = "${tname}ip6='".$ip->db_ip6."'";
	$res .= ", ${tname}bits=".$ip->db_bits;

	$res;
}


#Liefert ein Flag, ob die Adresse eine IPv4-Adresse ist.
sub is_v4
{
	my $ip = shift;
	return $ip->version == 4;
}

#Liefert ein Flag, ob die Adresse eine IPv4-Adresse ist oder, weil in
#einem VRF, als solche behandelt werden soll.
sub is_v4rf
{
	my $ip = shift;
	return 1 if $ip->version == 4;
	$vrf_prefix = Dbase::IP->new($VRF_PREFIX) 
		if not defined $vrf_prefix;
	return 2 if $vrf_prefix->contains($ip) and $ip->db_bits <= 32;
	return 0;
}

1;

