package Dbase::Globals;

use utf8;
use strict;
use warnings;

require Exporter;

#our %EG;  # zwei Zeilen wegen stupider Perl-Warnung
#    %EG = ( NL=>1, IT=>1, FR=>1, AT=>1, BE=>1, CH=>2, DE=>0, GB=>1 );
#our %LAND = ( I=>"IT", F=>"FR", B=>"BE", I=>"IT", A=>"AT", D=>"DE",
#	"UK"=>"GB",	USA=>"US");

=pod

Unterstützende Funktionen für den Datenbankzugriff

=head1 Suche nach Personen, Kunden, etc.

=head2 get_person( STRING [Restriktion] [Flags] [Kunde] )

Sucht einen Personeneintrag und liefert dessen ID zurück bzw. ein
L<Problem|Fehler/problem>, falls das Suchkriterium nicht eindeutig ist
bzw. L<undef|perlfunc/undef>, falls gar keine entsprechende Person gefunden
wird.

STRING kann sein:

=over 4

=item die ID des Eintrags

=item das Suchkürzel

=item ein NIC-Handle

=item der Username

=item der Realname

=item die User-ID (in der Form C<uid=NUMMER>)

=item die Mailadresse

=item die Telefon-, Fax- oder Handynummer (aber nur exakt wie in der DB)

=back

Die Restriktion kann sein:

=over 4

=item ein C<pwdomain>-Flag

Datensätze, in denen das Flag nicht gesetzt ist, werden ignoriert.

=item C<'?'> und ein (numerisches) C<person>-Datenfeld

Datensätze, in denen das Feld NULL oder 0 ist, werden ignoriert.

=back

Folgende Flags sind definiert:

=over 4

=item 1

Es werden nur Mailadresse, Usernamen oder NIC-Handles berücksichtigt.

=item 2

Es werden nur Usernamen berücksichtigt.

=item 8

Allen SELECTs wird ein FOR UPDATE hintangestellt, um die Datensätze zu
sperren.

=back

Das optionale Argument Kunde, in dem eine Kundennummer erwartet wird, wird
ggf. verwendet, wenn eine Suche nicht eindeutig ist, um die Suche hilfsweise
auf Personen einzuschränken, die diesen primären Kunden haben oder bei diesem
assoziiert sind.


Ausgabefunktionen siehe unten unter L<Personeneinträge>.


=head2 get_kunde( STRING [Flag] )

Sucht einen Kunden und liefert dessen ID zurück.

I<STRING> kann sein:

=over 4

=item die Kundennummer

=item der Name des Kunden

=item ein Aliasname des Kunden (C<kunde I<Name> u>)

=item eine Domain des Kunden

=item eine IP-Adresse des Kunden

=item eine dem Kunden zugeordnete Mailadresse (C<kunde I<Name> M>)

=item der Username von einer dem Kunden zugeordneten Person

=item eine Mailadresse von einer dem Kunden zugeordneten Person

=back

Weitere Funktionen siehe unten unter L<Kunden>.

Falls I<Flag> &1 gesetzt ist, wird eine Fehlermeldung ausgegeben, wenn der
Kunde nicht gefunden wurde.
Bei I<Flag> & 2 ruft die Funktion stattdessen die() auf.

Bei I<Flag> & 4 wird der angegebene Name aus dem internen Cache gelöscht.
Die Funktion liefert in diesem Fall die gecachte ID zurück oder, falls
keine im Cache war, C<undef>.
Falls C<undef> als Name übergeben wird, wird der gesamte Cache gelöscht
und die Anzahl der gelöschten Einträge zurückgegeben.

I<Flag&8> sperrt den Kunden, mittels eines FOR UPDATE hinter dem SELECT.


=head2 find_kunde_by_mail( MailAdr )

Diese Funktion liefert die ID des Kunden zurück,
zu dem eine Mailadresse gehört,
hilfsweise wird versucht, über die Domain oder eine übergeordnete Domain
einen Kunden zuzuordnen.
Sofern auch dabei keiner gefunden wird, gibt die Funktion 0 zurück.


=head2 get_kunden(Kunden)

Wrapper um get_kunde() zur Auswahl von Kundengruppen.
Liefer eine (ggf. leere) Liste von Kunden zurück.

Versteht zusätzlich zu get_kunde() folgende Syntax:

=over 4

=item tp=Person

=item vp=Person

um alle Kunden auszuwählen, bei denen die C<Person> als technischer
bzw. vertrieblicher Ansprechpartner eingetragen ist.
Person wird dabei von get_person() ausgewertet.
Wird die C<Person> nicht gefunden, ist das ein Fehler.

=item tp!Person

=item vp!Person

wie oben, jedoch um Kunden, bei denen die fragliche C<Person> als technischer
bzw. vertrieblicher Ansprechpartner eingetragen ist, auszuschließen, sprich alle
übrigen Kunden auszuwählen (auch solche ohne entsprechenden Ansprechpartner).

=back

Sonstige Angaben werden einfach an get_kunde() durchgereicht; falls get_kunde()
keinen entsprechenden Kunden findet, wird eine leere Liste zurückgegeben.


=head2 get_reseller(Kundennr)

Liefert den Reseller-Typ eines Kunden, oder C<undef>.

Im Listenkontext: Liefert Tupel C<(Typ, Name)>.


=head2 find_reseller(Kundennr)

Liefert den zu einem Kunden gehörenden Reseller.

=over 4

=item a

Der Kunde ist selber Reseller

=item b

Im Kunden ist ein Reseller eingetragen

=item c

Für einen übergeordneten Kunde trifft (a) oder (b) zu.

=back

=head2 get_ipkunde_byhost( STRING )

Liefert die ID des IP-Eintrags in der I<ipkunde>-Tabelle. Der STRING darf ein
Hostname oder eine IP-Addresse sein.
Der gefundene Eintrag darf nur eine IP-Adresse sein und kein Netzbereich.

=head2 get_ipkunde( STRING [Flag] )

Liefert die ID des IP-Eintrags in der I<ipkunde>-Tabelle, die zur IP-Adresse
I<STRING> gehört. Dabei wird nach übergeordneten Netzen gesucht.

I<STRING> kann eine IP-Adresse ohne (C<10.1.1.1>) oder mit (C<10.1.1.0/25>)
Netzangabe sein.

I<Flag&1> bewirkt, dass die letzte beendete Adresse zurückgeliefert wird.

I<Flag&2> schaltet die Suche in übergeordneten Bereichen ab.

I<Flag&8> sperrt den Eintrag, mittels eines FOR UPDATE hinter dem SELECT.


=head2 get_ipnum( ipaddr [bits] [Flag] )

Liefert die ID des IP-Eintrags in der I<ipkunde>-Tabelle, die zum IP-Netz
I<ipaddr> gehört. Dabei wird normalerweise nach übergeordneten Netzen gesucht.

Die Angabe der Bits ist im DB-Format, d.h. 0==Hostadresse.

Weitere Hilfsfunktionen siehe unten unter L<IP-Adressen>.

Flag-Bedeutung siehe L<get_ipkunde>.


=head2 get_vrf( str )

Liefert die VRF-Nummer des durch 'str' bezeichneten IPv4-VRFs zurück.


=head2 hostgroupinfo( ID [ Name Beschreibung ] )

Liefert eine Textdarstellung des angegebenen Hostgroup-Objekts.
Zur Optimierung können Name und Beschreibung optional gleich mitgegeben werden.


=head2 ipinfo( IP-ID [ IP Bits Name Ende ] )

Liefert eine Textdarstellung des angegebenen IP-Objekts.
Zur Optimierung können IP, Bits, Name und Ende optional gleich mitgegeben
werden.


=head1 Ident, Descriptoren, etc.pp.

Datenbanken mögen Nummern. Menschen nicht, und Programme auch nicht.

Deshalb gibt es in der Datenbank eine Liste, in der verzeichnet ist

=over 4

=item eine Nummer

=item ein Name (ein Wort)

=item eine einzeilige Beschreibung

=item eine Bitmask mit Typklasse

=back

In der Datenbank ist grundsätzlich die Nummer verzeichnet, gelegentlilch
auch ein Buchstabe (historisch gewachsen...). In den Perlskripten wird
entweder der Buchstabe direkt verwendet, oder die Namen werden via
C<find_descr()> zu Zahlen umgewandelt.

Da es von diesen Tabellen mehrere gibt (Dienst, Ziel, Quelle, ...), wird
der I<Ident>-Parameter zur Unterscheidung verwendet.

Zusätzlich gibt es Fälle, in denen nicht alle I<Ident>-Namen angezeigt
werden sollen. Beispielsweise sollen beim Domainantrag nur die für diesen
relevanten Dienste angezeigt werden. Deshalb kann man die Listen filtern.

Wegen der Dualität Zeichen vs. Ziffer sollten die ID-Nummern, die beides
darstellen können, d.h. 48 bis 57 (die ASCII-Zeichen 0 bis 9), nicht
verwendet werden.


=head2 get_descr( Ident ID [Flag] )

Liefert zu einer I<Ident>-ID den zugehörigen Namen.

Wenn I<Flag>&1, ist eine undef-ID OK.

Wenn I<Flag>&2, liefern unbekannte Werte C<?> statt C<undef>.


=head2 info_descr( Ident ID )

Liefert zu einer I<Ident>-ID den zugehörigen Text.


=head2 test_descr_flag( Ident ID Flag )

Meldet, ob für diesen Deskriptor das betreffende _ident-Flag gesetzt ist.


=head2 find_descr( Ident Name [ Flag ] )

Liefert die zum betreffenden I<Ident>-Namen gehörende Zahl zurück.

	$desc = find_descr('dienst','contact');
	die "Bäh" unless defined $desc;
	$pers = DoFn("select id from kundemail
	               where kunde = 1 and dienst = $desc");

Ist das Flag gesetzt, wird ein Fehler geworfen, wenn der bezeichnete
Deskriptor nicht existiert.


=head2 list_descr( Ident [numerisch] [Liste] [Sub])

Liste eine zahlenbasierte Deskriptorliste auf, z.B. "dienst".

I<numerisch> zeigt an, daß numerisch anstatt nach Deskriptorname
sortiert werden soll.

I<Liste> kann die Kriterien enthalten, nach denen die Liste gefiltert
werden soll.

Ist I<Sub> angegeben, wird der Code mit jedem Eintrag mit den Argumenten
C<(Ident, numID, Tag, Infotext, Gruppen)> aufgerufen.


=head2 map_descr( Ident [Liste])

Liefert eine Liste "Name => ID" des Deskriptors.

I<Liste> kann die Kriterien enthalten, nach denen die Liste gefiltert
werden soll.


=head2 enum_descr( Ident [numerisch] [Liste] )

Liste eine zahlenbasierte Deskriptorliste, z.B. "dienst", in Kurzform auf.

I<numerisch> zeigt an, daß numerisch anstatt nach Deskriptorname
sortiert werden soll.

I<Liste> kann die Kriterien enthalten, nach denen die Liste gefiltert
werden soll.


=head2 cenum_descr( Ident [numerisch] [Liste] )

Liste eine zeichenbasierte Deskriptorliste, z.B. "ziel", in Kurzform auf.

I<numerisch> zeigt an, daß nach Zeichen anstatt nach Deskriptorname
sortiert werden soll.

I<Liste> kann die Kriterien enthalten, nach denen die Liste gefiltert
werden soll.


=head2 clist_descr( Ident [numerisch] [Liste] [Sub])

Liste eine zeichenbasierte Deskriptorliste auf, z.B. "ziel".

I<numerisch> zeigt an, daß nach Zeichen anstatt nach Deskriptorname
sortiert werden soll.

I<Liste> kann die Kriterien enthalten, nach denen die Liste gefiltert
werden soll.

Ist I<Sub> angegeben, wird der Code mit jedem Eintrag mit den Argumenten
C<(Ident, ID, Tag, Infotext, Gruppen)> aufgerufen. I<ID> ist entweder
numerisch oer in der Form C<'X'>.


=head2 gen_descr( Ident ID Name [ Infotext [ Gruppe... ] ] )

Assoziiert eine ID mit einem I<Ident>-Namen und optional einem Infotext
und/oder diversen Gruppen. Ist die ID C<undef>, wird die nächste freie
ID ausgewählt.

Diese Funktion sollte normalerweise nicht verwendet werden; Deskriptoren
werden via Kundeprogramm gepflegt. Sie ist aber für Tests sehr
hilfreich.

Ein Fehler wird gemeldet, wenn C<gen_descr> irgendwas außer dem Infotext
updaten würde.

Rückgabewert ist die ID.


=head2 a_descr( Ident [Filter] )

Listet alle I<Ident>-Namen auf.

In Stringkontext wird die Liste als Text mit Leerstellen getrennt ausgegeben.


=head2 i_descr( Ident [Filter] )

Listet alle I<Ident>-Nummern auf.

In Stringkontext wird die Liste als Text mit Leerstellen getrennt ausgegeben.


=head2 flag_names( Bits [Ident [alt]] )

Generiert eine Liste von Namen, die mit den Bits korrespondieren.

	$pwuse = DoFn("select pwuse from person where id = 1");
	print "Flags       : ".flag_names($pwuse,"pwdomain")."\n";

Ist für I<alt> etwas angegeben, so werden die Unterschiede zwischen den
alten und neuen Flags zurückgeliefert.

Im Arraykontext werden die Daten als Liste zurückgeliefert.


=head2 DoSelect_flaglist( Ident sub SelectString [selectSub printSub ] )

Wie DoSelect, jedoch wird als zusätzlicher Parameter ein Deskriptorname
übergeben. Als erstes Datenbankfeld muss ein Flagbits-Feld des
entsprechenden Deskriptors SELECTiert sein.

Die Prozedur wird nicht mit dem Bitfeld aufgerufen, sondern mit einem
String, in dem die ausgewählten Deskriptoren durch ihre Display-Zeichen
repräsentiert sind.

Rückgabewert ist eine Liste der verwendeten Zeichen und ihrer
Deskriptornamen, damit unter der Tabelle eine Erklärung dieser Zeichen 
ausgegeben werden kann.

Optional können zwei weitere Prozeduren mitgegeben werden. Die erste dient
dazu, die Ausgabe vorzufiltern; ein Datensatz wird nicht ausgegeben, wenn 
diese Prozedur einen false-Wert liefert. Parameter ist der
Original-SQL-Datensatz.

Die zweite wird vor dem ersten Datensatz aufgerufen und dient dazu, diese
im Kopf auszugeben. Parameter ist das Array der Zeichen plus Doku, wie es
auch von DoSelect_flaglist zurückgeliefert wird.


=head2 print_flaglist( outFile head liste... )

Die von DoSelect_flaglist() generierte Liste von Buchstaben und ihre
Bedeutung wird in outFile ausgegeben. "head" bedeutet, dass die Liste am
Anfang der Tabelle gedruckt wird.


=head2 get_gruppen( Ident Liste [strict] [Feld] )

In der Datenbank können Deskriptoren (beispielsweise ein Dienst)
genauer unterschieden werden (beispielsweise, ob er in Personenrecords
oder Tarifen verwendet wird).

Diese Funktion generiert aus den Kriterien in der Liste Flags zur
Kennzeichnung der einzelnen Deskriptoren.

Wenn C<strict> gesetzt ist, wird via C<problem()> ein Fehler gemeldet,
wenn eine (nicht negierte Gruppe) nicht gefunden wurde. Ansonsten wird
der unbekannte Eintrag übersprungen.

Wird ein C<Feld>-Name übergeben, so ist der Rückgabewert ein SQL-Schnipsel
für ein WHERE-Statement.


=head2 get_bitmap( Ident Liste [strict] )

Diese Funktion generiert aus den Kriterien in der Liste Flags zur
Kennzeichnung der einzelnen Deskriptoren.

Der Unterschied zu get_gruppen() ist, dass hier nicht eine Bitmap zu den
Kriterien selber erstellt wird, sondern eine Bitmap der
Deskriptor-Werte, die zu diesen Kriterien passen.

Wenn C<strict> gesetzt ist, wird via C<problem()> ein Fehler gemeldet,
wenn eine Gruppe nicht gefunden wurde. Ansonsten wird der unbekannte
Eintrag übersprungen.

Nicht gefundene negierte Gruppen werden in jedem Fall übersprungen.


=head2 test_gruppe( Ident Deskr $grs $grc )

Liefert I<true>, wenn der angegebene I<Ident>-Deskriptor
durch die Variablen C<$grs $grc> nicht ausgefiltert wird.


=head2 test_gruppen( Bits $grs $grc )

Liefert I<true>, wenn die Variablen C<$grs $grc> das gegebene Bitfeld
nicht ausfiltern. Es muß ein I<grs>-Bit und darf kein C<grc>-Bit gesetzt
sein.


=head2 test_alle_gruppen( Bits $grs $grc )

Liefert I<true>, wenn die Variablen C<$grs $grc> das gegebene Bitfeld
nicht ausfiltern.  Es müssen alle I<grs>-Bits und es darf kein
C<grc>-Bit gesetzt sein.


=head2 Beispiel für die C<*_gruppe>-Funktionen

	my ($grs,$grc) = get_gruppen("dienst_ident","domain,!hide");

	# für jeden evtl. interessanten Dienst
		return unless test_gruppe("dienst",$dienst,$grs,$grc);


=head2 test_flag (Ident Descr Flags)

Prüft, ob der Deskriptor in den Flags gesetzt ist.


=head2 is_dienst (Dienst_X Dienst_Y)

Prüft, ob sich ein Dienst X (via tarifeq-Tabelle) zu einem anderen
Dienst Y reduzieren läßt.


=head1 Dienste-Tabelle

Diese Hilfsfunktionen sind langweilig. ;-)

=head2 name_dienst ( Nummer ) => Name

Liefert den Namen eines Diensts.

=head2 find_dienst ( Name ) => Nummer

Findet einen Dienst.


=head1 Personeneinträge

Die Datenbank referenziert alle Personen als Nummer.

Je nach Anforderung des Programms gibt es ein paar Hilfsfunktionen, diese
Nummern lesbar zu machen.


=head2 get_handle ( Person Registrare $res SetHandle SetRegistrar )

Liefert den ersten bei einem der Registrare bekannten Handle für die
Person. Wenn die Person keinen betreffenden Handle besitzt und der
SetHandle angegeben ist, wird der Handle für SetRegistrar oder im ersten
der Registrare gesetzt und zurückgeliefert.

Die Person ist eine Zahl, die Handles Text.

'$res' ist der evtl. Status eines Automaten, der bei Problemen
modifiziert wird.

	get_handle(get_person("smurf"),"denic,ripe");


=head2 persinfo( PersonID )

Liefert eine einzeilige Kurzinfo zur Person (Name, Username, Mailadresse,
Zusatztext; jeweils soweit vorhanden).


=head2 mpersinfo( PersonID [ Flag ] )

Liefert Name mit Mailadresse für die Person.

Gibt es weder Name noch Mailadresse, wird Username oder Suchkürzel oder 
Zusatzinfo verwendet.

Flag & 1: Liefert nur den Namen.


=head2 mmpersinfo( PersonID )

Liefert Mailadresse für die Person, oder C<undef>.


=head2 kpersinfo( PersonID, ResellerID )

Liefert einen Kurznamen für die Person: Username, Suchkürzel,
Mailadresse, Realname, oder Zusatzinfo; je nachdem was als erstes
gefunden wurde.

Wird eine Reseller-ID angegeben, so wird der vom Reseller vergebene
Personenhandle zurückgeliefert.


=head2 kkpersinfo( PersonID Flag )

wie C<kpersinfo>, aber gecacht.

Flag=1: ein alter Cacheinhalt ignoriert.
Flag=2: undef ist OK.

=head2 persfirma( PersonID )

Liefert den Firmennamen zu einer Person.

=head1 IP-Adressen

Siehe L<Dbase::IP>.


=head1 Datumsmanipulation

In der Datenbank werden alle Datums- und Zeitangaben im Unixformat
gespeichert. Das hat zum einen die üblichen historischen Gründe (die
zuerst verwendete Datenbank hatte einfach kein Datum), zum anderen
brauchen weitere Berechnungen die Zeit sowieso als Zahl.


=head2 this_date( Sekunden )

Liefert den Tagesanteil der angegebenen Zeit, in der lokalen Zeitzone.


=head2 this_time( Sekunden )

Liefert den Tageszeitanteil der angegebenen Zeit, in der lokalen Zeitzone.


=head2 iso_intervall( Sekunden [Flag])

Liefert eine für Menschen lesbare Version des übergebenen Zeitraums.

	iso_intervall(3610) =E<gT> "01:00:10"

Die folgenden Flags sind definiert:

=over 4

=item 1

Sekunden werden immer angegeben.

=item 2

Wenn I<Tag> gesetzt ist, wird der Wert als Tageszeit in der lokalen
Zeitzone angesehen.

=item 4

Tage werden nicht ausgewiesen.

=item 8

undef-Eingaben resultieren in '-'.

=item 16

Sekunden werden nie angegeben.

=back

=head2 nice_intervall( Sekunden [Flag] )

Liefert eine für Menschen lesbare Version des übergebenen Zeitraums.

	nice_intervall(3610+24*3600) =E<gT> "1 Tag 1 Std 10 sec"

Die folgenden Flags sind definiert:

=over 4

=item 1

Garnix wird als 00:00 angezeigt, nicht als "-"

=item 2

Tage werden nicht extra angezeigt:

	nice_intervall(3610+24*3600,1) =E<gT> "25 Std 10 sec"

=back

=head2 svdaterange( Beginn Ende Flag )

Gibt den Zeitraum zwischen den beiden Datumsangaben (die Null sein
können) in lesbarer Form aus. Zeitangaben werden nicht ausgegeben;
Tagesangaben werden unterdrückt

	svdaterange(unixtime("1999-01-01"),0) =E<gt> "ab 1999-01"
	svdaterange(unixtime("1999-01-01"),unixtime("1999-02-01"))
		=E<gt> "Monat 1999-01"

Flags:

=over 4

=item 1

Auch bei nur einem Monat wird C<von ... bis ...> statt
nur C<Monat ...> ausgegeben.

=item 2

Statt "immer" wird nichts ausgegeben, wenn sowohl C<von> als auch C<bis> leer
sind.

=item 4

Die Tages-Unterdrückung wird unterdrückt.

=back


=head2 sdaterange( Beginn Ende [ Flag ] )

Gibt den Zeitraum zwischen den beiden Datumsangaben (die Null sein
können) in lesbarer Form aus. Zeitangaben werden nicht ausgegeben;
Tagesangaben werden unterdrückt

	sdaterange(unixtime("1999-01-01"),0) =E<gt> "1999-01--XXX"

Flags:

=over 4

=item 1

Kurzform: das "XXX" fällt weg; gibt es weder Beginn noch Ende, wird der
leere String zurückgeliefert.

=back


=head2 daterange( Beginn Ende )

Gibt den Zeitraum zwischen den beiden Datumsangaben (die Null sein
können) in lesbarer Form aus. Zeitangaben werden nicht ausgegeben.

	daterange(unixtime("1999-01-02 12:34:56"),0) =E<gt> "ab 1999-01-02"


=head2 timerange( beginn ende )

Gibt den Zeitraum zwischen den beiden Datumsangaben (die Null sein
können) in lesbarer Form aus. Die Zeitangabe wird um Mitternacht
unterdrückt.

	timerange(unixtime("1999-01-01 00:00:00"),0) =E<gt> "ab 1999-01-01"


=head2 enddate( ende )

Gibt eine Unixzeitangabe zurück, die das Ende des angegebenen Zeitraums
(z.B. eines Monats) repräsentiert.

	enddate("1999-01");
	enddate("2000");


=head2 rangedate( beginn [ende] )

Gibt zwei Unix-Zeitangaben zurück, die den Bereich zwischen den angegebenen
Datumsangaben repräsentieren.

	rangedate("1999-01-01-2000-01-01");
	rangedate("1999-01-01 - 2000-01-01");
	rangedate("1999-01-01 - 2000-02");


=head2 zeit_klartext( Stunde Minute )

Gibt eine Zeitangabe als Text aus, zwecks die Sprachausgabe.

	zeit_klartext(11,22) => "elf Uhr zweiundzwanzig";


=head2 in_euro( datum )

Liefert C<true>, wenn Geldbeträge für dieses Datum in Euro vorliegen.


=head1 Accounting


=head2 add_acct( Kunde Dienst Ziel JJJJMM TT Quelle Bytes Pakete Text Flag )

Fügt einen Accountingdatensatz ein, oder updatet den bestehenden Datensatz.

	add_acct("POP","ip","das_ziel", 199901,23, "die_quelle",1234,123,"Test");

Der Schlüssel für Accounting sind B<alle> obigen Datenfelder außer
I<Bytes> und I<Pakete>. Intern werden diese Daten auf einen Hashschlüssel
und eine Sequenznummer abgebildet.

Die Flags bedeuten:

=over 4

=item 1

einen existierenden Datensatz mit diesem Text ersetzen

=item 2

I<Immer> einen zusätzlichen, neuen Datensatz generieren

=back


=head1 Kunden


=head2 name_kunde( Kundennr [Flag] )

Diese Funktion liefert den Namen eines Kunden zurück.

Flags:

=over 4

=item 1

Lösche den internen Hash (beim Umbenennen)

=item 2

Hole den langen Namen (kunde.hauptperson=>name), wenn vorhanden.

=back


=head2 unterkunden( KundeNr [Var] [Flag] )

Liefert im Arraykontext eine Liste aller Unterkunden von I<Kunde>, inkl.
dem Kunden selbst.

Im Stringkontext wird ein SQL-C<select>-Fragment der Form

	( I<Var> = I<Nummer1> or I<Var> = I<Nummer2> ... )

zurückgeliefert. Der Defaultwert für I<Var> ist "kunde".

Flags:

=over 4

=item 1 [Flag]

Falls Flag auf 1 gesetzt wird werden nur die aktiven Unterkunden
beruecksichtigt. Der Defaultwer für I<Flag> ist 0 (auch nicht aktive
anzeigen).

=back


=head2 oberkunden( KundeNr [Var] )

Liefert im Arraykontext eine Liste aller übergeordneten Kunden von
I<Kunde>, inkl. dem Kunden selbst.

Im Stringkontext wird ein SQL-C<select>-Fragment der Form

	( I<Var> = I<Nummer1> or I<Var> = I<Nummer2> ... )

zurückgeliefert. Der Defaultwert für I<Var> ist "kunde".


=head2 oberkunde( KundeNr, Flag )

Liefert den übergeordneten Kunden zu I<Kunde>, oder den Kunden selbst
wenn er kein Unterkunde ist.

An diesen "Oberkunden" wird z.B. die Rechnung gestellt.

=over 4

Flag & 1: bei gesetztem eigene_re-Flag eines Unterkunden wird dieser
zurückgeliefert.

=back

=head2 ist_unterkunde( KundeNr [BasisNr] )

Liefert I<true>, wenn der Kunde ein (direkter oder indirekter)
Unterkunde des Kunden I<Basis> ist.

Default für I<Basis> ist der POP.


=head2 resellerkunden( KundeNr )

Liefert eine Liste von Kunden, deren Reseller der mit I<KundeNr>
bezeichnete Kunde ist. Der Kunde selbst ist in der Liste verzeichnet,
ebenso wie alle Unterkunden.


=head2 ist_intern( MailAdr [BasisNr] )

Liefert I<true>, wenn die angegebene Mailadresse zum Kunden I<Basis> oder
einem seiner Unterkunden gehört.

Default für I<Basis> ist der POP, daher auch der Funktionsname.


=head1 Sonstiges


=head2 beendet( Epoch )

gibt eine leere Zeichenkette zurück, falls I<Epoch> C<undef> ist,
C< (beendet ab ...)>, falls I<Epoch> in der Zukunft liegt und
C< (beendet seit ...)>, falls I<Epoch> in der Vergangenheit liegt


=head2 name_nic( NIC-Nr )

Diese Funktion liefert den Namen eines NICs zurück.



=head2 aufzaehlung( [{ trenner => STRING, letzter_trenner => STRING }] Liste )

Liefert eine Zeichenkette zurück, die die Elemente der Liste mit dem angegebenen
Trenner (Default: C<, >) verbindet, mit Ausnahme des letzten Elements, das mit
C<letzter_trenner> (Default: C< und >) angehängt wird; Beispiele:

  'eins, zwei, drei und vier' eq aufzaehlung( qw/eins zwei drei vier/ )

  'eins, zwei oder drei' eq
  aufzaehlung( { letzter_trenner => ' oder ' }, qw/eins zwei drei/ )


=head2 is_hotline( [Stunden-Art-ID] )

Ermittelt, ob die fragliche Stunden-Art zur Gruppe Hotline gehört,
sprich liefert einen entsprechenden ja/nein-Wert.
Beim Aufruf ohne Argument wird eine Liste aller Stunden-Art-IDs,
die zur Gruppe Hotline gehören, zurückgeliefert


=head2 stunden_bereich( Person Kunde Beginn Ende Flags sub )

Diese Funktion ruft für jeden Zeiteintrag im angegebenen Bereich die
übergebene Funktion auf, ggf. eingeschränkt auf eine Person und/oder einen
Kunden. Überschneidungen werden herausgeschnitten und führen dazu, dass
ein Eintrag mehr als einmal zurückgeliefert wird, mit unterschiedlichen
Zeiten.

Sortiert ist das Ganze nach Person und Arbeitszeit.

Der Prozeduraufruf geschieht mit folgenden Parametern:

=over 4

=item ID des Stundeneintrags

=item Beginn

=item Dauer

=item Person

=item Kunde

=item Faktor

=item Infotext

=item Ticket-ID

=item Art des Stundeneintrags

=item Flags (aus der der Art des Stundeneintrags)

=item Name der Stundeneintragsart ("Normal", "Hotline", ...)

=item Inhalt-Flags

=item Zähler

Der Zähler ist B<immer> das letzte Element in dieser Liste.

=back

Der Aufruf dieser Prozedur ist relativ datenbankintensiv. Wenn nur die
Zeit summiert werden soll, ist ein SELECT der effizientere Weg.

Folgende Inhalt-Flags sind definiert:

=over 4

=item 1

Das zurückgelieferte Ticket ist die ID bzw. die Nummer eines OTRS-Tickets.

B<obsolet>.

=back

Folgende Flags sind definiert:

=over 4

=item 1

Hotline-Einsätze werden kumuliert: liegen weniger als 15 Minuten
zwischen Einsätzen, gelten sie als ein Einsatz.

=item 2

Die ersten 30 Minuten eines (kumulierten) Hotline-Einsatzes werden
weggelassen. Ohne Flag 1 ist Flag 2 nicht sinnvoll!

=item 4

Einträge, die keine Hotline-Einsätze sind, werden ignoriert und gar nicht
zurückgeliefert. Ist nur Flag 4, jedoch nicht 1+2 angegeben, werden für
<15min-Pausen zwischen Hotline-Einsätzen Pseudo-Einträge zurückgeliefert.

=item 16

Liefert die OTRS-Ticket-ID, wenn eingetragen.

B<obsolet>.

=item 32

statt der (internen) OTRS-Ticket-ID wird die
(sichtbare) Nummer zurückgeliefert.

B<obsolet>.

=back

Ist von den Flags 1+2+4 nur Flag 4 gesetzt, werden Fülleinträge
ausgegeben, wenn zwischen zwei Einträgen nicht mehr als 15 Minuten liegen.
ID, Zähler und kunde sind C<undef>. Für den Faktor wird der Wert des
nachfolgenden Eintrags verwendet. Die darauf folgenden Felder fehlen.


=head2 flush_std_cache( person start dauer )

Die effektiven Zeit eines Stundeneintrags (inkl. Faktor und
exkl. darin verschachtelte Einträge) wird im Stundendatensatz
(Feld stunden.cache) gecacht.

Wird ein Eintrag verändert, so müssen die Datensätze, die davon betroffen
sind (mithin: die ihn enthalten), geupdatet werden. Diese Funktion löscht
das Cache-Feld der betreffenden Einträge.

Diese Funktion ruft C<update_stunden_cache()> auf, die den Cache neu
berechnet.


=head2 update_stunden_cache( [Beginn] [Ende] [Person] )

Die effektive Zeit eines Stundeneintrags (inkl. Faktor und
exkl. darin verschachtelte Einträge) wird im Stundendatensatz
(Feld stunden.zeit) gecacht.

Dieses Feld wird hier für diejenigen Datensätze neu berechnet, bei denen
sein Wert NULL ist.

Die Transaktion, in der dieser Aufruf stattfindet, sollte wiederholbar
sein (d.h. "DoTrans {...} 2"), da konkurrierende Änderungen ein Rollback
auslösen können.


=head2 arbeitszeit_heute( Person )

Diese Funktion liefert die Zeit (in Sekunden) zurück, die der Betreffende
am aktuellen Tag arbeiten müsste, damit er am Monatsende genau sein 
Überstunden-Soll erreicht.


=head2 soll_bis_heute( Person )

Diese Funktion liefert die Zeit (in Sekunden) zurück, die der Betreffende
diesen Monat (bis inkl. des aktuellen Tags) arbeiten müsste, um seine
Sollzeit zu erreichen. Überstunden des Vormonats werden nicht
berücksichtigt.


=head2 time4ticket( TicketID [Beginn] [Ende] [Kunden] [ohne_Kunden] )

Diese Funktion liefert die Summe der auf das angegebene Ticket (sowie ggf.
gemergte) gebuchten Arbeitszeiten in Sekunden.

I<Neu seit kunde 1.2009.4>: Bei nicht existenten oder nicht (mehr) selbständigen
(weil mit anderen zusammengefassten) Tickets wird C<undef> zurückgeliefert.
(In Listenkontext zweimal.) C<0> bzw. C<0,0> bedeutet hingegen, dass es das
Ticket zwar gibt, aber keine Arbeitszeit darauf gebucht wurde.

In skalarem Kontext erhält man die Arbeitszeit unter Berücksichtigung
eingetragener Faktoren, in Listenkontext als ersten Wert die Arbeitszeit ohne
Faktoren und erst als zweiten Wert die Arbeitszeit mit Faktoren.

Optional kann mit C<Beginn> ein Start- und/oder mit C<Ende> ein
Endzeitpunkt angegeben werden. Es werden dann nur entsprechende
Arbeitszeiteintragungen berücksichtigt; maßgeblich ist der Startzeitpunkt
der Zeiteintragung.

C<Kunden> und C<ohne_Kunden> enthalten optional jeweils eine Referenz auf eine
Liste von Kunden; es werden dann nur Zeiteintragungen berücksichtigt, die
bzw. die nicht auf einen der angegebenen Kunden gebucht wurden.


=head2 time4kunde( Kunde [Beginn] [Ende] )

Diese Funktion liefert die Summe der für einen bestimmten Kunden
gebuchten Arbeitszeiten in Sekunden. 
In skalarem Kontext erhält man die Arbeitszeit unter Berücksichtigung
eingetragener Faktoren, in Listenkontext als ersten Wert die Arbeitszeit ohne
Faktoren und erst als zweiten Wert die Arbeitszeit mit Faktoren.

Optional kann mit C<Beginn> ein Start- und/oder mit C<Ende> ein
Endzeitpunkt angegeben werden. Es werden dann nur entsprechende
Arbeitszeiteintragungen berücksichtigt; maßgeblich ist der Startzeitpunkt
der Zeiteintragung.


=head2 time4kunden( [Beginn] [Ende] [Limit] )

Diese Funktion liefert die Summe der für alle Kunden
gebuchten Arbeitszeiten in Sekunden, als Liste.

Jedes Listenelement ist eine Referenz auf eine weitere Liste mit
 * Kundennummer,
 * Arbeitszeit ohne Faktor,
 * Arbeitszeit mit Faktor.

Optional kann mit C<Beginn> ein Start- und/oder mit C<Ende> ein
Endzeitpunkt angegeben werden. Es werden dann nur entsprechende
Arbeitszeiteintragungen berücksichtigt; maßgeblich ist der Startzeitpunkt
der Zeiteintragung.

Das optionale Limit gibt an, wieviele Kunden maximal zurückgeliefert
werden. Die Liste ist nach absteigender Arbeitszeit sortiert.


=head2 mail2mime( Mail )

Konvertiert eine Mail nach MIME.

Die Mail kann als Text, Dateideskriptor, oder MIME::Entity vorliegen. Im letzteren Fall wird gar nichts getan.


=head2 mime2mail( Mail [from to...] )

Konvertiert eine Mail in Text. Adressat und Empfänger werden entweder 1:1
übernommen oder aus der Mail extrahiert.

Die Mail kann als Text, MIME::Entity, Code-Referenz, Dateideskriptor, oder
Array-Referenz (C<[Mail from to...]>) vorliegen.

Empfängeradressen im x-noris-to:-Header ersetzen die entsprechenden
Adressen im to:-Header, um Mails zu Reviewzwecken o.ä. umzuleiten.
Dito für die anderen Header mit Empfängeradressen.


=head2 rund(Zahl [ Stellen ])

liefert einen sauber gerundeten Integer zurück. Die Zahl wird dabei vorher
durch 10^(Stellen) geteilt; Default ist 0. Negative Werte sowie
Math::BigInt werden korrekt behandelt.

Ist die Stellenzahl negativ, so wird nicht geteilt, nur gerundet.

	rund(123.4, 0)  => 123
	rund(123.4, 1)  => 12
	rund(123.4, -1) => 120


=head2 preis (Zahl [Komma])

Akzeptiert eine Angabe mit Nachkommastellen.
Liefert diese Zahl als Integer mit der angegebenen Stellenzahl hinter
dem Komma. Der Default sind 3 Nachkommastellen.


=head2 print_komma( Wert minKomma maxKomma gesamt )

Spucke einen Wert mit Nachkommstellen aus.


=head2 sendmail( $text $from @to )

Schickt eine Mail ab. C<$text> muß alle notwendigen Header enthalten.
(Verwendet zur Konvertierung in Text mime2mail().)


=head2 charset( $utf8flag )

Liefert den für das aktuelle Locale gültigen charset-Wert zurück.

Wenn das utf8flag übergeben wird, wird utf8 (Flag gesetzt) bzw. Latin9
(Flag nicht gesetzt) erzwungen.


=head2 mimeheader( $fh $utf8flag $contenttype )

Schreibt die für die aktuelle Locale gültigen MIME-Header in den Filehandle
und setzt das Encoding korrekt.

Wenn das utf8flag übergeben wird, wird utf8 (Flag gesetzt) bzw. Latin9
(Flag nicht gesetzt) erzwungen.

Default für den Content-Type ist text/plain.

Ist der Filehandle C<undef>, werden die Headerzeilen als Text
zurückgeliefert. Der korrekte Modus des Filehandles, an den die Ausgabe
letztlich geht, ist dann manuell zu setzen:

=over 4

=item UTF8 an

utf8modus()

=item UTF8 aus

latinmodus()

=item Default

textmodus()

=back


=head2 message_id()

Generiert eine neue Message-ID.

<b>Achtung:</b> Die spitzen Klammern drumherum sind nicht Bestandteil
der ID, weil die betreffende Datenbanktabelle I<ticketid> die ID ohne 
Klammern speichert. Beim Einfügen in Mailheader muss der Aufrufende 
diese selber dazusetzen:

	my $msgid = message_id();
	$msg->head->replace("message-id", "<$msgid>");


=head2 rwdiff()

Liefert die Differenz zwischen den Datenbank-Zeitstempeln in der Lese-
und Schreibdatenbank.


=head2 gen_passwd( Länge )

Generiert einen Passwortstring.


=head2 stufung( Bytes )

Versucht die Bytes einigermaßen sinnvoll auszugeben (xx MB),

	stufung(1024*1024*123) => "123 MB";

Verwendung: in Übersichtslisten des Kundeprogramms.

=head2 stufung_in( Bytes )

Die inverse Funktion zu C<stufung>.

	stufung_in("123 MB") => 123*1024*1024


=head2 flatten( proc data... )

Die Funktion wird mit Textzeilen (inkl. LF am Ende) aufgerufen.
Referenzen und Dateien in den Argumenten werden aufgelöst, d.g. rekursiv
ausgegeben bzw. gelesen.

=head2 if_defined( Name Wert [Sub-Routine] )

Gibt eine leere Liste zurück, falls C<Wert> L<undef> ist.  Andernfalls wird eine
Liste bestehend aus dem C<Name>n und den Rückgabewerten der C<Sub-Routine>
zurückgegeben oder falls keine L<Sub-Routine> übergeben wurde, eine
zweielementige Liste bestehend aus C<Name> und C<Wert>.

Beispiele:

	if_defined( foo => undef );                   # ()
	if_defined( foo => 'bar' );                   # qw(foo bar)
	if_defined( foo => 'bar', sub { uc shift } ); # qw(foo BAR)

Die Funktion darf nur in einem Listenkontext verwendet werden.

=head2 rufnummernliste( Rufnummern )

bastelt aus einer Liste von Rufnummern im datenbanküblichen Format eine Liste
von Hash-Referenzen, die man z. B. an ein Template übergeben kann, und gibt im
Listenkontext diese und in skalarem Kontext eine Referenz darauf zurück.

=head2 chomped( string )

gibt das zurück, was vom C<string> nach einem L<perlfunc/chomp()> übrig bleibt

=head2 puny_encode( string [flag] )

Kodiert den String zu punycode.


=head2 puny_decode( string [flag] )

Dekodiert den String zu UTF-8 oder Latin1.

	Flag	4	Übersetzung einer Mail-Adresse bzw. eines
			Umleitungs-Ziels (= automatisches Abschneiden
			des User-Teils bzw. des Channel-Typs)
		2	obsolet (analysiere $LANG auf UTF8)
		1	obsolet (war: verwende Latin1)

=head2 bignum( [ Zahl ] )

Diese Funktion meldet 1 zurück, wenn 64-bit-Zahlen mit bignum-Arithmetik
nachgebildet werden sollen / müssen.

Wird eine Zahl übergeben, so wird diese ggf. als bignum zurückgeliefert,
ansonsten "normal".

=head2 is_holiday( [Zeitpunkt] [Schema] )

Gibt 1 zurück, falls der als Unix-Timestamp angegebene Zeitpunkt
(Default: jetzt) ein Feiertag ist und 2, falls es sich um einen halben
Feiertag handelt.
Dabei kann optional ein Feiertagsschema angegeben werden, das in
L<noris::Date::Calendar::Profiles> definiert ist (andernfalls gibt die
Funktion -1 zurück); per Default wird sonst C<DE-BY-noris> (also bayerische
Feiertage mit den noris internen Spezifikationen) verwendet.

=head2 in_period( Zeitraum-Definition [Zeitpunkt] )

Gibt 1 zurück, falls der angegebene Zeitpunkt (Default: jetzt) innerhalb der
L<Zeitraum-Definition|Time::Period> liegt.
Als Erweiterung der von L<Time::Period|Time::Period> unterstützten Syntax
können Feiertage dabei mittels C<-SCHEMA> explizit ausgenommen bzw. mit
C<+SCHEMA> explizit eingeschlossen werden.

=head2 umlaut_e( Text )

Verwandelt Ä -> AE, ..., ß -> ss

=head2 get_rz_from_ipkunde( IPKunde )

Liefert die RZ ID für den jeweiligen IP-Kunde Eintrag.
Dazu schaut er zuerst nach ob Standort gesetzt ist und dieser einem RZ 
entsrpicht. Sollte da keins gefunden werden, wird nachgeschaut ob ein
Rack gesetzt ist. Wird keins gefunden, wird C<undef> zurückgegeben.

=cut

use strict;
use warnings;
no warnings qw(uninitialized);
use Cf qw($WDESCR $DEFAULTKLASSE $MAILDOM $BIGNUM $VRF_PREFIX);
use Dbase::Help;
use Dbase::Help qw(
  date_add_ymd
  date_start
  Do
  DoFn
  DoSelect
  DoTrans
  DoTime
  in_list
  isotime
);
use Math::BigInt;
use Time::Local;
use Config;
use Fehler qw(fehler problem warnung ffehler);
use Net::IDN::Punycode qw(decode_punycode encode_punycode);
use noris::MIME::Words qw(encode_mimewords);
use UTFkram qw(decode_anything safe_encode_utf8);
use Dbase::IP;
use Date::Calc qw(Days_in_Month);
use Umlaut qw(binmodus latinmodus utf8modus);

our @ISA = qw(Exporter);
our @EXPORT = qw(
	get_person get_kunde get_ipnum get_ipkunde get_tarif 
	get_descr gen_descr a_descr i_descr list_descr info_descr
	find_descr enum_descr mpersinfo persinfo kkpersinfo kpersinfo 
	svdaterange sdaterange daterange add_acct timerange map_descr
	cenum_descr clist_descr
	nice_intervall unterkunden oberkunden oberkunde
	ist_unterkunde ist_intern flag_names get_gruppen test_alle_gruppen
	test_gruppen test_gruppe name_kunde zeit_klartext bignum
	iso_intervall this_date this_time enddate rangedate get_handle
	sendmail rwdiff gen_passwd mmpersinfo stufung stufung_in
	print_komma flatten test_descr_flag get_reseller mail2mime
	mime2mail in_euro preis test_flag no_crlf find_reseller
	resellerkunden message_id is_dienst umlaut_e
	);
our @EXPORT_OK = qw(
  $RE_valid_username
  %descr_id
  %descr_name
  addr_from_block
  aufzaehlung
  beendet
  chomped
  content
  def_or_minus
  DoSelect_flaglist
  explain_child_error
  find_dienst
  find_kunde_by_mail
  flush_std_cache
  gen_flaglist
  get_bitmap
  get_ipkunde_byhost
  get_kunden
  get_vrf
  hostgroupinfo
  if_defined
  in_period
  ipinfo
  is_holiday
  is_hotline
  is_valid_username
  mimeheader
  name_dienst
  name_nic
  persfirma
  print_flaglist
  puny_decode
  puny_encode
  rufnummernliste
  rund
  get_rz_from_ipkunde
  soll_bis_heute
  stunden_bereich
  time4kunde
  time4kunden
  time4ticket
  update_stunden_cache
);

my @ghash;
my %scache2;# Nr => String
my %scache; # String => Nr
sub bignum(;$);
sub content($);
sub rund($;$);

sub hostgroupinfo {
	my ( $id, $name, $beschreibung ) = @_;
	$id =~ /^\d+\z/ or die "Ungültige Hostgroup-ID: $id\n";
	( $name, $beschreibung ) =
	  DoFn("SELECT name, beschreibung FROM hostgroup WHERE id = $id")
	  or return
	  if @_ < 3;
	sprintf( '#%3d: ', $id ) . $name
	  . ( defined $beschreibung && " - $beschreibung" );
}

sub beendet($) {
    my ($ende) = @_;
    my $time = DoTime();
    defined $ende
      && (' (beendet '
        . ( $ende < DoTime() ? 'seit' : 'ab' ) . ' '
        . isotime($ende)
        . ')' );
}

sub ipinfo {
	my ( $id, $ip6, $bits, $name, $ende ) = @_;
	$id =~ /^\d+\z/ or die "Ungültige IP-ID: $id\n";
	( $ip6, $bits, $name, $ende ) =
	  DoFn("SELECT ip6, bits, name, ende FROM ipkunde WHERE id = $id")
	  or return
	  if @_ < 5;

	my $ip_o = Dbase::IP->new_db( $ip6, $bits );
	sprintf( '#%5d: ', $id )
	  . ( defined $name && "$name " ) . '['
	  . $ip_o->str(4) . ']'
	  . beendet($ende);
}

sub persinfo ($) {
	my($persid) = @_;
	return undef unless $persid;
	my($ret) = DoT("select name,user,email,zusatzinfo from person where id = $persid");
	if(ref $ret) {
		my($name,$uname,$mail,$zusa) = @{$ret->[0]};
		my($res);
		$name = "" unless content $name;
		if (content $uname) { $uname = lc $uname; } else { $uname = ""; }
		if (content $mail) { $mail = puny_decode($mail, 4); } else { $mail = ""; }
		$zusa = "" unless content $zusa;

		$ret  = $name;
		if($uname ne "") {
			if($name eq "") {
				$ret = $uname;
			} else {
				$ret .= " ($uname)";
			}
		}
		$ret .= " <" if ($name ne "" or $uname ne "") and $mail ne "";
		$ret .= $mail;
		$ret .= ">" if ($name ne "" or $uname ne "") and $mail ne "";
		$ret .= ": " if ($name ne "" or $uname ne "" or $mail ne "") and $zusa ne "";
		$ret .= $zusa;
		$ret = sprintf ("#%5d %s",$persid,$ret) if $ret ne "";
		$ret = "?_$persid" if $ret eq "";
		$ret;
	} else {
		"??_$persid";
	}
}


sub kpersinfo ($;$) {
	my($persid,$reseller) = @_;
	return undef unless $persid;
	if($reseller) {
		my $id = DoFn("select handle from resellernic where person=$persid and reseller=$reseller");
		return $id if defined $id;
		return "$WDESCR-$persid";
	}
	my($ret) = DoT("select user,name,email,zusatzinfo,suchbegriff from person where id = $persid");
	if(ref $ret) {
		my($user,$name,$mail,$zusa,$such) = @{$ret->[0]};
		my($res);
		return lc $user if content $user;
		return $such if content $such;
		return puny_decode($mail, 4) if content $mail;
		return $name if content $name;
		return $zusa if content $zusa;
		"?_$persid";
	} else {
		"??_$persid";
	}
}

my %kkpers;
sub kkpersinfo ($;$) {
	my($persid,$flag) = @_;
	$flag=0 if not defined $flag;
	return ($flag&2)?"-":undef unless $persid;
	delete $kkpers{$persid} if $flag&1;
	return $kkpers{$persid} if exists $kkpers{$persid};
	$kkpers{$persid} = kpersinfo($persid);
}

sub persfirma ($) {
    my ($persid) = @_;

    my $firmname;
    my ( $kid, $mpid, $pname ) = DoFn("SELECT kunde, mperson, name FROM person WHERE id = $persid");
    if ($mpid) {
        $firmname = DoFn("SELECT name FROM person WHERE id = $mpid");
        return $firmname if defined $firmname;
    }

    $firmname = DoFn("SELECT person.name
                        FROM person JOIN kunde ON person.kunde = kunde.id
                       WHERE kunde.id = $kid
                         AND person.id = $persid");
    return $firmname;
}

my %mpers;
sub mpersinfo ($;$) {
	my($persid,$flag) = @_;
	$flag |= 0;
	return undef unless $persid;
	return $mpers{$persid} if not $flag and exists $mpers{$persid};

	my($ret) = DoT("select name,email,zusatzinfo,suchbegriff,user from person where id = $persid");
	if(ref $ret) {
		my($name,$mail,$zusa,$such,$user) = @{$ret->[0]};
		my($res);
		if(content $name) {
			$name =~ s/^(Herr|Frau)\s+//;
			return $name if $flag & 1;
			if(content $mail) {
				$name .= " <".($mail).">";
			}
			return $mpers{$persid} = $name;
		}
		return puny_decode($mail, 4) if content $mail;
		return $such if content $such;
		return $zusa if content $zusa;
		return lc $user if content $user;
		$mpers{$persid} = "?_$persid";
	} else {
		$mpers{$persid} = "??_$persid";
	}
}


sub mmpersinfo ($) {
	my($persid) = @_;
	return undef unless $persid;

	my $res = DoFn("select email from person where id = $persid");
	return puny_decode($res, 4) if $res;
	return undef;
}

my %kid;
my $knflush = time();
sub get_kunde ($;$) {
	my($name,$flag) = @_;
	$flag=0 unless defined $flag;

    if ( $flag & 4 ) {    # Cache löschen
        if ( defined $name ) { return delete $kid{$name} }
        else {
            my $cachesize = keys %kid;
            undef %kid;
            return $cachesize;
        }
    }

	return undef unless content($name);
	return 0 if $name eq "-";

	my $upd = ($flag&8) ? "for update" : "";

	if($knflush+600 < time()) {
		$knflush = time();
		%kid = ();
	}
	my $id = $kid{$name};
	return $id if $id;

	if ( $name =~ /^\d+\z/ and $id = DoFn("SELECT id FROM kunde WHERE id = $name $upd") ) {
		$kid{$name} = $id;
		return $id;
	}

	# wird z. B. für test/56_domainmanuell benötigt:
	if ( $name =~ /^domain#(\d+)\z/ and $id = DoFn("SELECT kunde FROM domainkunde WHERE id = $1 $upd") ) {
		$kid{$name} = $id;
		return $id;
	}

	$id = DoFn("select kunde from rechnungen where rnr = $1 $upd")
		if $name =~ /^R(\d+)$/;
	return $id if $id;

	$id = DoFn("select id from kunde where name = '${\quote $name}' $upd")
		if $name !~ /^\d+$/;
	if($id) {
		$kid{$name} = $id;
		return $id;
	}

	return undef if $name =~ /^\d+$/;

	$id = DoFn("select kunde from uucpkunde where uucpkunde.name = '${\quote $name}' $upd") 
		if $name !~ /^\d*$/;
	if($id) {
		$kid{$name} = $id;
		return $id;
	}

	if ($name !~ /^\d*$/ and length($name) <= 100) {
		( my $nname = $name ) =~ s'.*@'';
		$nname = puny_encode(lc $nname);
		do {
			$id = DoFn("select kunde from domainkunde where domain = '${\quote $nname}' and ( ende is null or ende > UNIX_TIMESTAMP(NOW()) ) $upd");
			return $id if $id;
		} while ($nname =~ s/.*?\.(.+\..)/$1/);
	}
	if (length($name) <= 100) {
		$id = get_ipkunde($name, $flag&8);
		$id = DoFn("select kunde from ipkunde where id = $id $upd") if $id;
		return $id if $id;
	}
	if($name !~ /^\d+$/) {
		$id = DoFn("select kunde from person where user = ${\qquote $name} $upd");
		return $id if $id;
	}
	if($name !~ /^\d+$/) {
		$id = DoFn("select distinct kunde from person where email = ${\qquote $name} $upd", undef, 1);
		return $id if $id;
		if ( ( my $puny = puny_encode(lc $name) ) ne lc $name ) {
			$id = DoFn("select distinct kunde from person where email = ${\qquote $puny} $upd", undef, 1) and return $id;
		}
	}
	if ( my ($name) = $name =~ /^\!(.+)/ ) {
		# neueste beendete Domain
		my $qpuny = quote puny_encode(lc $name);
		$id = DoFn("select distinct kunde from domainkunde where domain = '$qpuny' and ende > 0 and ende < UNIX_TIMESTAMP(NOW()) order by beginn desc limit 1 $upd")
			if $name !~ /^\d*$/ and length($name) <= 100;
		return $id if $id;

		if (length($name) <= 100) {
			$id = get_ipkunde($name,1) and
			$id = DoFn("select kunde from ipkunde where id = $id $upd") and
			return $id;
		}
	}

	$id = DoFn <<_ and return $kid{$name} = $id;
	SELECT   kunde
	FROM     hardware
	WHERE    hardware_id = ${\qquote($name)}
	ORDER BY beginn DESC
	LIMIT    1
	$upd
_
	if($name !~ /\s/) {
		my $nname = quote($name);
		$id = DoFn("select person.kunde,nic.person from person,nic where person.id = nic.person and nic.handle = '$nname' $upd");
		return $id if $id;
		# Da die Namen der Handles, nach denen hier gesucht wird, von
		# den Domain-Resellern vergeben werden, sollte der folgende
		# Suchschritt sicherheitshalber der letzte in der Kette bleiben:
		$id = DoFn("SELECT reseller FROM resellernic WHERE handle='$nname' $upd") and return $id;
	}

	die     "Kunde '$name' unbekannt!\n" if $flag&2;
	problem "Kunde '$name' unbekannt!\n" if $flag&1;
	undef;
}

sub get_kunden($) {
	my ($kunden) = @_;
	if ( my ( $ap_typ, $kriterium, $person ) = $kunden =~ /^([tv]p)([!=])(\S+)$/i ) {
		defined( my $ansprechpartner = get_person($person) ) or return;
		my @kunden;
		$ap_typ = 'ap_' . { tp => 'technik', vp => 'vertrieb' }->{$ap_typ};
		$kriterium = '!=' if $kriterium eq '!';

		DoSelect { push @kunden, @_ } <<_;
	SELECT   id
	FROM     kunde
	WHERE    $ap_typ $kriterium $ansprechpartner${\( $kriterium eq '!=' && " OR $ap_typ IS NULL" )}
	ORDER BY id
_
		@kunden;
	}
	elsif ( defined( my $kunde = get_kunde($kunden) ) ) { $kunde }
	else { () }
}

sub get_reseller($;$) {
	my($kunde,$whatever) = @_;
	my $rtyp = DoFn("select art from reseller where kunde = $kunde");
	return wantarray ? () : undef unless defined $rtyp;
	my $rn = get_descr("reseller",$rtyp);
	return problem "Reseller-Eintrag #$rtyp für Kunde #$kunde kaputt"
		unless defined $rn;
	return wantarray ? ($rtyp,$rn) : $rn;
}

sub find_reseller($) {
	my($kunde) = @_;
	return undef unless defined $kunde;

	do {
		my $reseller;

		return $kunde if DoFn("select count(*) from reseller where kunde=$kunde");
		($kunde,$reseller) = DoFn("select kunde,reseller from kunde where id=$kunde");
		return $reseller if $reseller;
	} while($kunde);
	return undef;
}

sub get_ipkunde_byhost($) {
	my ($name) = @_;
	my ( $ipk, $ip, $bits );

	DoSelect {
		die "Es wurden mehrere Hosts mit dem angegeben Namen gefunden!\n"
		  if defined $ipk;
		( $ipk, $ip, $bits ) = @_;
	} "SELECT id, ip6, bits FROM ipkunde WHERE name = ${\ qquote $name} AND ( ende IS NULL OR ende >= UNIX_TIMESTAMP(NOW()))";
	die "Der angegebene Hostname darf kein Netzbereich sein.\n"
	  if $bits;
	return $ipk if defined $ipk;

	( $ipk, my $ip_o ) = get_ipkunde( $name, 2 );
	die "Es wurde keine aktiver Eintrag in der DB gefunden.\n"
	  unless defined $ipk;
	die "Die angegebene IP darf kein Netzbereich sein.\n"
	  if $ip_o->db_bits;

	return $ipk;
}

sub get_ipkunde($;$) {
	my($name,$flags) = @_;
	return $1 if $name =~ /^I?(\d+)$/;

	my $ip;
	eval { $ip = Dbase::IP->new($name); };
	return undef unless defined $ip;
	return get_ipnum($ip,$flags);
}

sub get_vrf($) {
	my($str) = @_;
	my $sel;

	if($str =~ /^\d+$/) {
		my $ip = Dbase::IP->new($VRF_PREFIX)->bitmask(32) + $str;
		$sel = $ip->dbs . " and vrf is not null";
	} else {
		$sel = "vrf=${\qquote $str}";
	}
	my($ip,$bits) = DoFn "select ip6,bits from ipkunde where $sel and (ende is null or ende >= UNIX_TIMESTAMP(NOW()))";
	return undef unless defined $bits;
	return Dbase::IP->new_db($ip,$bits)
		- Dbase::IP->new($VRF_PREFIX)->bitmask(32)
}

sub get_ipnum($;$) {
	my($ip,$flag) = @_;
	my $bits=$ip->db_bits;

	$flag=0 unless defined $flag;
	my $upd = ($flag&8) ? "for update" : "";
	my $alt;
	if($flag&1) {
		# neueste beendete IP-Adresse
		$alt = "ende > 0 and ende < UNIX_TIMESTAMP(NOW()) order by beginn desc limit 1";
	} else {
		$alt = "(ende is null or ende >= UNIX_TIMESTAMP(NOW()))";
	}

	while (1) {
		my($id,$adr,$bit) = DoFn("select id,ip6,bits from ipkunde where ${\ $ip->dbs } and $alt $upd");
		return wantarray ? ($id,Dbase::IP->new_db($adr,$bit)) : $id if defined $bit;

		last if $flag & 2;
		last if ++$bits > $ip->bits;
		$ip = $ip->bitmask($bits,1);
	}
	wantarray ? () : undef;
}

our $RE_valid_username = qr/[a-z][-._a-z0-9]{1,31}/;
# RT#408318: Wir akzeptieren hier bewusst nur Kleinbuchstaben,
# weil wir nicht wollen, dass Usernamen mit Großbuchstaben in der
# Datenbank landen. Wer Großbuchstaben erlauben will, sollte die
# vor dem Aufruf dieser Funktion umwandeln.
sub is_valid_username($) { shift =~ /^$RE_valid_username$/ }

sub get_person ($;$$$) {
	my($name,$restrict,$flag,$kunde) = @_;
	return undef unless $name;
	my($id);
	my $where = "where";
	$flag = 0 unless $flag;
	my $upd = ($flag & 8) ? "for update" : "";

	if($restrict) {
		foreach my $res(split(/,/,$restrict)) {
			if($res =~ s/^\?//) {
				$where .= " person.$res is not null and" if $res;
			} else {
				$res = find_descr("pwdomain",$res);
				$where .= " (pwuse & ".(bignum(1)<<$res).") and" if $res;
			}
		}
	}
	$id = DoFn("select id from person $where id = $1 $upd") if ($name =~ /^(?:${WDESCR}-|#)?(\d+)$/i);
	return $id if $id;

	$id = DoFn("select id from person $where user = ${\qquote $name} $upd")
				if is_valid_username( lc $name );
	return $id if $id or $flag & 2;

	$id = DoFn("select id from person $where suchbegriff = ${\qquote $name} $upd")
				if not $flag & 1 and is_valid_username( lc $name );
	return $id if $id;

	{
		my( @ids, $suche );
		my $qname = qquote($name);

		Suche: {

			$suche = 'mit diesem Handle';
			DoSelect { push @ids, @_ } <<_ and last Suche;
	SELECT   DISTINCT person
	FROM     nic, person
	$where   nic.person = person.id AND nic.handle = $qname
	ORDER BY person.id $upd
_

			$id = DoFn("select id from person $where uid = $1 $upd") and return $id
			  if $name =~ /^uid=(\d+)$/i;

			unless ( $flag & 1 ) {
				$suche = 'dieses Namens';
				DoSelect { push @ids, @_ } <<_ and last Suche;
	SELECT id FROM person $where name = $qname ORDER BY id $upd
_
			}

			$suche = 'mit dieser E-Mail-Adresse';
			DoSelect { push @ids, @_ } <<_ and last Suche;
	SELECT id FROM person $where email = $qname ORDER BY id $upd
_
			if ( ( my $puny = puny_encode(lc $name, 4) ) ne lc $name ) {
				DoSelect { push @ids, @_ } <<_ and last Suche;
	SELECT id FROM person $where email = ${\ qquote($puny) } ORDER BY id $upd
_
			}

			# known bug: findet die Person nicht, wenn irgendwo mehrere Rufnummern eingetragen sind:
			if ( not $flag & 1 and $name =~ /^\+[0-9 ]+$/ && length $name > 2 ) {
				$suche = 'mit dieser Rufnummer';
				DoSelect { push @ids, @_ } <<_ and last Suche;
	SELECT id FROM person $where $qname IN (fon,fax,pager) ORDER BY id $upd
_
			}

			undef $suche;
		}

		if (@ids) {
			return "@ids" if @ids == 1;

			my $hinweis = '';
			if ( defined $kunde ) {
				my %person;
				DoSelect { ++$person{+shift} } <<_;
	SELECT id FROM person WHERE kunde = $kunde AND ${\ in_list( id => '', @ids ) }
_
				DoSelect { ++$person{+shift} } <<_;
	SELECT person FROM kundemail WHERE kunde = $kunde AND ${\ in_list( person => '', @ids ) }
_
				if ( keys %person == 1 ) {
					my ($person) = keys %person;
					return $person;
				}
				elsif ( keys %person > 1 ) {
					@ids = sort keys %person;
					$hinweis = ' bei diesem Kunden';
				}
			}

			return problem( "Es gibt$hinweis mehrere Personen $suche:\n"
			                . join "\n", map sprintf("#%5d: %s", $_, mpersinfo($_)), @ids )
		}
	}

	if(not $flag & 1) {
		$id = DoFn("SELECT person FROM mailassoc WHERE email = ${\qquote $name} $upd") and return $id;
	}

	# Da die Namen der Handles, nach denen hier gesucht wird, von
	# den Domain-Resellern vergeben werden, sollte der folgende
	# Suchschritt sicherheitshalber der letzte in der Kette bleiben:
	unless ($flag & 1) {
		my $wh = '';
		if ($name =~ s/^([^:]+)://) {
			my $kd = get_kunde($1) or undef $wh;
			$wh = " AND reseller = $kd" if defined $wh;
		}
		$id = DoFn("select person from resellernic where handle = '${\quote $name}'$wh $upd")
			and return $id
			if defined $wh;
	}
	$id;
}

sub get_tarif ($) {
	my($name) = @_;
	return undef if $name eq "";
	return DoFn("select id from tarif where name = '${\quote $name}'");
}

sub test_gruppen($$$) {
	my($grp,$set,$clr) = @_;
	return 0 if $grp & $clr;
	return 1 if $grp & $set;
	return 1 if $set == 0;
	return 0;
}

sub test_alle_gruppen($$$) {
	my($grp,$set,$clr) = @_;
	return 0 if $grp & $clr;
	return 0 if ~$grp & $set; # Bit in $grp aus, aber in $set ein
	return 1;
}

sub test_gruppe($$$$) {
	my($desc,$grp,$set,$clr) = @_;
	if(defined $ghash[$grp]) {
		$grp = $ghash[$grp];
	} else {
		$grp = $ghash[$grp] = DoFn("select descr.gruppe from descr_typ,descr where descr.descr = $grp and descr_typ.id = descr.typ and descr_typ.name = '${\quote $desc}'");
	}
	return test_gruppen($grp,$set,$clr);
}

sub get_gruppen($$;$$) {
	my($name,$gruppen,$strict,$field) = @_;
	my $setg = 0;
	my $clrg = 0;
	return wantarray ? (0,0) : 0 
		if not defined $gruppen or $gruppen eq "" or $gruppen eq "-";
	foreach my $gruppe(split(/,/,$gruppen)) {
		my $not = ($gruppe =~ s/^\!//);
		my $grp = find_descr($name,$gruppe);
		unless(defined $grp) {
			next if $not or not $strict;
			return problem("Deskriptor $name/$gruppe unbekannt");
		}
		if($not) {
			$clrg |= (bignum(1)<<$grp);
		} else {
			$setg |= (bignum(1)<<$grp);
		}
	}

	if ( defined $field ) {
		join ' AND ',
		     $setg ? "$field & $setg = $setg" : (),
		     $clrg ? "$field & $clrg = 0"     : ()
	}
	elsif (wantarray) { $setg, $clrg }
	else              { $setg        }
}

sub list_descr ($;$$$) {
	my($name,$numer,$gruppe,$sub) = @_;
	my $text = '';
	my($setg,$unsetg) = get_gruppen($name."_ident",$gruppe);
	my $descr = DoFn "select id from descr_typ where name=${\qquote $name}";
	return unless defined $descr;

	my $disp = DoFn "select count(*) from descr where typ=$descr and idchar is not null";
	my($err) = DoSelect {
		my($id,$bla,$grp,$inf,$dis) = @_;
		return unless test_gruppen($grp, $setg, $unsetg);
		if(ref $sub) {
			&$sub( $name, $id, $bla, $inf, $grp );
		} else {
			$text .= sprintf("%3d ",$id);
			if($disp) {
				if(defined $dis) {
					$text .= "$dis ";
				} else {
					$text .= "  ";
				}
			}
			$text .= sprintf("%-20s %-5s %s\n",$bla,$grp?scalar flag_names($grp,$name."_ident"):"",$inf);
		}
	} "select descr,bla,gruppe,infotext,idchar from descr where typ=$descr order by ".($numer?"descr":"bla");

	$text;
}

sub map_descr ($;$) {
	my($name,$gruppe) = @_;
	my($setg,$unsetg) = get_gruppen($name."_ident",$gruppe);
	my @res;
	DoSelect {
		my($id,$bla,$grp) = @_;
		return unless test_gruppen($grp, $setg, $unsetg);
		push(@res,$bla,$id);
	} "select descr.descr,descr.bla,descr.gruppe from descr_typ,descr where descr_typ.name = '${\quote $name}' and descr_typ.id = descr.typ";

	@res;
}

sub clist_descr(;$$$$) {
	my($name,$numer,$gruppe,$sub) = @_;
	my($text);
	my($setg,$unsetg) = get_gruppen($name."_ident",$gruppe);
	my($err) = DoSelect {
		my($nam,$id,$bla,$grp,$inf) = @_;
		return unless test_gruppen($grp, $setg, $unsetg);
		if(ref $sub) {
			&$sub( $nam, $id>32 ? sprintf("'%c'",$id) : $id, $bla, $inf, $grp );
		} else {
			$text .= sprintf("%-10s  ",$nam) if $name eq "";
			if($id > 32) {
				$text .= sprintf("'%c' %-20s %-5s %s\n",$id,$bla,$grp?scalar flag_names($grp,$nam."_ident"):"",$inf);
			} else {
				$text .= sprintf("%3d %-20s %-5s %s\n",$id,$bla,$grp?scalar flag_names($grp,$nam."_ident"):"",$inf);
			}
		}
	} "select descr_typ.name,descr.descr,descr.bla,descr.gruppe,descr.infotext from descr_typ,descr where ".(($name ne "") ? "descr_typ.name = '${\quote $name}' and " : "").
		"descr_typ.id = descr.typ order by descr_typ.name,".($numer?"descr.descr":"descr.bla");

	$text;
}


sub a_descr {
	my($name,$gruppe) = @_;
	my(@data);
	my($setg,$unsetg) = get_gruppen($name."_ident",$gruppe);
	my($err) = DoSelect {
		my($name,$grp) = @_;
		return unless test_gruppen($grp, $setg, $unsetg);
		push(@data,$name);
	} "select descr.bla,descr.gruppe from descr_typ,descr where descr_typ.name = '${\quote $name}' and descr_typ.id = descr.typ order by descr.descr";

	wantarray ? @data : join(" ",@data);
}


sub i_descr {
	my($name,$gruppe) = @_;
	my(@data);
	my($setg,$unsetg) = get_gruppen($name."_ident",$gruppe);
	my($err) = DoSelect {
		my($name,$grp) = @_;
		return unless test_gruppen($grp, $setg, $unsetg);
		push(@data,$name);
	} "select descr.descr,descr.gruppe from descr_typ,descr where descr_typ.name = '${\quote $name}' and descr_typ.id = descr.typ order by descr.descr";

	wantarray ? @data : join(" ",@data);
}


sub enum_descr ($;$$) {
	my($name,$numer,$gruppe) = @_;
	my $text;
	my $len;
	my($setg,$unsetg) = get_gruppen($name."_ident",$gruppe);
	DoSelect {
		my($name,$id,$grp) = @_;
		return unless test_gruppen($grp, $setg, $unsetg);
		if($text ne "") {
			if($len+length($id)+length($name) < 75) {
				$text .= " ";
				$len++;
			} else {
				$text .= "\n";
				$len = 0;
			}
		}
		$text .= "${id}_$name";
		$len += length($id)+length($name)+1;
	} "select descr.bla,descr.descr,descr.gruppe from descr_typ,descr where descr_typ.name = '${\quote $name}' and descr_typ.id = descr.typ order by ".($numer?"descr.descr":"descr.bla");

	$text;
}


sub get_bitmap ($$;$) {
	my($name,$gruppe,$strict) = @_;
	my $res = 0;
	my($setg,$unsetg) = get_gruppen($name."_ident",$gruppe,$strict);
	return 0 if not $setg and not $unsetg and $gruppe =~ /(?:^|,)\w/;

	DoSelect {
		my($id,$grp) = @_;
		return unless test_gruppen($grp, $setg, $unsetg);
		$res |= bignum(1) << $id;
	} "select descr.descr,descr.gruppe from descr_typ,descr where descr_typ.name = ${\qquote $name} and descr_typ.id = descr.typ";

	$res;
}


sub cenum_descr ($;$$) {
	my($name,$numer,$gruppe) = @_;
	my($text,$len);
	my($setg,$unsetg) = get_gruppen($name."_ident",$gruppe);
	my $err = DoSelect {
		my($name,$id,$grp) = @_;
		return unless test_gruppen($grp, $setg, $unsetg);
		$id = pack("C",$id) if $id > 32;
		if($len+length($id)+length($name) < 75) {
			$text .= " ";
			$len++;
		} else {
			$text .= "\n";
			$len=0;
		}
		$text .= "${id}_$name";
		$len += length($id)+length($name);
	} "select descr.bla,descr.descr,descr.gruppe from descr_typ,descr where descr_typ.name = '${\quote $name}' and descr_typ.id = descr.typ order by ".($numer?"descr.descr":"descr.bla");

	$text;
}

our %descr_id;
our %descr_name;

sub get_descr ($$;$) {
	# Klasse, Zahl -> Text 
	my($name,$id,$flag) = @_;
	$flag=0 if not defined $flag;

	my($res);
	return ($flag&1)?"-":undef unless defined $id;
	return $res if defined($res = $descr_id{$name}{$id});
	my($sid) = $id;
	$sid = ord($sid) if ($sid =~ s/^\'(.)\'$/$1/ || $sid =~ /^\D$/);
	$res = DoFn("select descr.bla from descr,descr_typ where descr_typ.name = '${\quote $name}' and descr_typ.id = descr.typ and descr.descr = $sid");
	if(defined $res) {
		$res = $sid if $res =~ /^\s*$/;
		$descr_id{$name}{$id} = $res;
	}
	$res = '?' if not defined $res and $flag&2;
	$res;
}

sub info_descr ($$) {
	# Klasse, Zahl -> Text 
	my($name,$id) = @_;
	my($res);
	my($sid) = $id;
	$sid = ord($sid) if ($sid =~ s/^\'(.)\'$/$1/ || $sid =~ /^\D$/);
	DoFn("select descr.infotext from descr,descr_typ where descr_typ.name = '${\quote $name}' and descr_typ.id = descr.typ and descr.descr = $sid");
}

sub find_descr ($$;$) {
	# Klasse, irgendwas -> Zahl
	# ist das optionale dritte Argument wahr, wird ein fehler() geworfen,
	# falls der fragliche Deskriptor nicht gefunden wurde.
	my($name,$id,$fehler) = @_;
	my $res = $descr_name{$name}{$id};
	return $res if defined $res;

	$res = DoFn("select descr.descr from descr,descr_typ where descr_typ.name = '${\quote $name}' and descr_typ.id = descr.typ and descr.bla = '${\quote $id}'");
	if(not defined $res and $id =~ /^\d+$/) {
		$res = DoFn("select descr.descr from descr,descr_typ where descr_typ.name = '${\quote $name}' and descr_typ.id = descr.typ and descr.descr = $id");
	}
	if(not defined $res and (length($id) == 1 or (length($id) == 3 and $id =~ s/^'(.)'$/$1/))) {
		$res = DoFn("select descr.descr from descr,descr_typ where descr_typ.name = '${\quote $name}' and descr_typ.id = descr.typ and descr.descr = ".ord($id));
	}
	if ( defined $res ) { $descr_name{$name}{$id} = $res }
	elsif ( $fehler ) { fehler qq(Es gibt keinen $name-Deskriptor "$id".) }
	else { undef }
}

sub test_flag($$$) {
	# Klasse, irgendwas, Flagbits => bool
	my($name,$id,$bits) = @_;
	my $d = find_descr($name,$id);
	return undef unless defined $d; # gibt es nicht
	if($bits & (bignum(1)<<$d)) {
		return 1;
	} else {
		return 0;
	}
}

sub test_descr_flag ($$$) {
	# Klasse, irgendwas, Name => Bool
	my($name,$id, $txt) = @_;
	my $did = find_descr($name,$id);
	return problem "Deskriptor '$name/$id' unbekannt" unless defined $did;
	my $dfl = DoFn("select descr.gruppe from descr,descr_typ where descr.typ = descr_typ.id and descr.descr = $did and descr_typ.name = '${\quote $name}'");
	return 0 unless defined $dfl;
	my $dif = find_descr($name."_ident",$txt);
	return problem "Deskriptortyp '$name/$txt' nicht gefunden" unless defined $dif;
	$dfl & (bignum(1)<<$dif);
}

sub gen_descr ($$$;$@);
sub gen_descr ($$$;$@) {
	# Klasse, Zahl, Text
	my($name,$id,$bla,$info,@gruppen) = @_;
	@ghash = ();

	$bla = quote($bla);
	$id = ord($id) if defined $id and ($id =~ s/^'(.)'$/$1/ or $id =~ /^\D$/);
	my($dtyp) = DoFn("select id from descr_typ where name=${\qquote $name}");
	$dtyp = Do("insert into descr_typ set name=${\qquote $name}")
		unless $dtyp;
	
	my($nid) = DoFn("select descr from descr where typ=$dtyp and bla=${\qquote $bla}");
	if(defined $nid) {
		if(defined $id) {
			fehler "Descr '$name//$bla' hat bereits ID '$nid'" if $nid != $id;
		} else {
			$id = $nid;
		}
	} else {
		if(defined $id) {
			$nid = DoFn("select bla from descr where typ=$dtyp and descr=$id");
			if(defined $nid) {
				fehler "Descr '$name/$id' hat bereits Name '$bla'" if $nid ne $bla;
			} else {
				Do("insert into descr set typ=$dtyp, descr=$id, bla=${\qquote $bla}");
			}
		} else {
			$id = DoFn("select max(descr) from descr where typ=$dtyp");
			$id++; # OK auch wenn $id==undef
			Do("insert into descr set typ=$dtyp, descr=$id, bla=${\qquote $bla}");
		}
	}
	Do("update descr set infotext=${\qquote $info} where typ=$dtyp and descr=$id") if defined $info and $info ne "";

	if(@gruppen == 1) {
		if($gruppen[0] eq "-") {
			@gruppen = ();
		} else {
			@gruppen = split(/\,/,$gruppen[0]);
		}
	}
	if(@gruppen) {
		my $akl = DoFn("select gruppe from descr where typ=$dtyp and descr=$id");
		my $nkl=0;
		if(@gruppen == 1 and not defined $gruppen[0]) {
			$nkl=0;
		} else {
			foreach my $kl(@gruppen) {
				$nkl |= bignum(1)<<gen_descr("${name}_ident",undef,$kl);
			}
		}
		fehler "Gruppenliste geändert: $nkl, war $akl" if $akl and $nkl != $akl;
		Do("update descr set gruppe=$nkl where typ=$dtyp and descr=$id");
	}
	$id;
}


sub daterange ($$) {
	my($beginn,$ende) = @_;
	my($res);

	if($beginn and $ende) {
		"von ".(isodate $beginn)." bis ".(isodate $ende);
	} elsif($beginn > time) {
		"ab ".(isodate $beginn);
	} elsif($beginn) {
		"seit ".(isodate $beginn);
	} elsif($ende) {
		"bis ".(isodate $ende);
	} else {
		"";
	}
}

sub sdaterange ($$;$) {
	my($beginn,$ende,$flag) = @_;
	$flag ||= 0;
	my $res = "";

	if($beginn and $ende) {
		my($sbeg,$send);
		$sbeg = sisodate($beginn,1);
		$ende -= 2*3600+3; # zwei Stunden Zeitzone plus Schaltsekunde plus Fuzz
		$ende = $beginn if $ende < $beginn; # grins
		$send = sisodate($ende,2);

		# Start mit Tagesangabe => Ende auch mit Tag, und umgekehrt
		if($send =~ /\-\d+\-/) {
			$sbeg = sisodate($beginn,0);
		} elsif($sbeg =~ /\-\d+\-/) {
			$send = sisodate($ende,0);
		} 

		if($sbeg eq $send) {
			$res = $sbeg;
		} else {
			$res = "$sbeg--$send";
		}
	} elsif($beginn) {
		$res = sisodate($beginn,1)."--";
		$res .= "XXX" if $flag&1;
	} elsif($ende) {
		$res = "XXX" if $flag&1;
		$res .= "--".(sisodate($ende,2));
	} else {
		$res = "?immer?" unless $flag&1;
	}
	$res;
}

sub svdaterange ($$;$) {
	my($beginn,$ende,$flag) = @_;
	$flag=0 unless $flag;
	my($res);

	if($beginn and $ende) {
		$ende -= 2*3600+3; # zwei Stunden Zeitzone plus Schaltsekunde plus Fuzz
		$ende = $beginn if $ende < $beginn; # grins

		my($sbeg,$send);
		if($flag & 4) {
			$sbeg = isodate($beginn);
			$send = isodate($ende);
		} else {
			$sbeg = sisodate($beginn,1);
			$send = sisodate($ende,2);

			# Start mit Tagesangabe => Ende auch mit Tag, und umgekehrt
			if($send =~ /\-\d+\-/) {
				$sbeg = sisodate($beginn,0);
				$flag |= 4;
			} elsif($sbeg =~ /\-\d+\-/) {
				$send = sisodate($ende,0);
				$flag |= 4;
			} 
		}

		if($sbeg eq $send and not $flag&1) {
			if($flag & 4) {
				"am $sbeg";
			} else {
				"Monat $sbeg";
			}
		} else {
			if ($ende <= $beginn and not $flag&1) {
				"Monat $sbeg";
			} elsif($sbeg =~ /\-\d+\-/) {
				"von $sbeg bis $send";
			} else {
				"Monate $sbeg bis $send";
			}
		}
	} elsif($beginn > time) {
		"ab ".(($flag&4)?isodate($beginn):sisodate($beginn,1));
	} elsif($beginn) {
		"seit ".(($flag&4)?isodate($beginn):sisodate($beginn,1));
	} elsif($ende) {
		"bis ".(($flag&4)?isodate($ende):sisodate($ende,2));
	} elsif($flag & 2) {
		"";
	} else {
		"immer";
	}
}

sub timerange ($$;$) {
	my($beginn,$ende, $ohne_text) = @_;
	my($res);

	if($beginn and $ende) {
		(($ohne_text)?'':'von ').(isotime $beginn).(($ohne_text)?'-':' bis ').(isotime $ende);
	} elsif($beginn > time) {
		(($ohne_text)?'':'ab ').(isotime $beginn);
	} elsif($beginn) {
		(($ohne_text)?'':'seit ').(isotime $beginn);
	} elsif($ende) {
		(($ohne_text)?'':'bis ').(isotime $ende);
	} else {
		'';
	}
}


my %nicname;
sub name_nic($) { # Nummer => Name
	my $id = shift;
	fehler "name_nic mit nichtnumerischem Argument",$id
		unless $id =~ /\A\d+\z/;
	return get_descr("nic",$id);
}


sub find_dienst($) { # Name => Nummer
	my $name = shift;
	return scalar DoFn "select id from dienst where name=${\qquote $name}";
}

my %dname;
sub name_dienst($) { # Nummer => Name
	my $id = shift;
	fehler "name_dienst mit nichtnumerischem Argument",$id
		unless $id =~ /\A\d+\z/;
	return $dname{$id} if defined  $dname{$id};
	my $name = DoFn "select name from dienst where id=$id";
	$dname{$id} = $name;
	return $name;
}


# Flags:
# 1: Acct-Eintrag mit diesem Text ersetzen
# 2: IMMER einen neuen Acct-Eintrag generieren
# 
sub add_acct {
	my($cust,$dien, $key,$jjmm,$tt,$quelle, $bytes,$pkt,$txt, $flag) = @_;

	$flag=0 if not defined $flag;

	$txt = undef if not content $txt or $txt eq "-";

	if($jjmm == 0) {
        my($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = localtime(DoTime);
        $tt = $mday;

        my $jj = $year; my $mm = $mon + 1;
	    $jj += 1900;
	    $jjmm = $jj*100 + $mm;
	}
	unless($cust =~ /^\d+$/) {
		my $d = get_kunde($cust);
		fehler "Kunde '$cust' unbekannt" unless $d;
		$cust = $d;
	} elsif($cust == 0) {
		fehler "kein Kunde angegeben";
	}
	unless($quelle =~ /^\d+$/) {
		my $d = find_descr("quelle",$quelle);
		fehler "Quelle '$quelle' unbekannt" unless defined $d;
		$quelle = $d;
	} elsif($quelle == 0) {
		$quelle = find_descr("quelle",$WDESCR);
	}
	unless($dien =~ /^\d+$/) {
		my $d = find_dienst $dien;
		fehler "Dienst '$dien' unbekannt" unless defined $d;
		$dien = $d;
	}
	if($key) {
		my $d = find_descr( name_dienst ( $dien ) => $key ) ||
		        find_descr( ziel                  => $key ) ||
		        find_descr( acctinfo              => $key );
		fehler "Ziel '$key' für Dienst '".name_dienst($dien)."' unbekannt" unless defined $d;
		$key = $d;
	} else {
		$key = find_descr( name_dienst( $dien ), $DEFAULTKLASSE    ) ||
		       find_descr( ziel               => $DEFAULTKLASSE, 1 );
	}

	my $hash = hash($cust,$dien,$key,$jjmm,$tt,$quelle);
	my $seq;

	my $where = (defined $txt) ? "acctassoc.info=${\qquote $txt}" : 'acctassoc.info IS NULL';

	DoTrans {
		# Fälle:
		# - $flag&2: neuen Datensatz anlegen
		# - $flag&1: ersetzen (d.h. den alten löschen, sofern vorhanden)
		# - sonst: existierenden Datensatz updaten, sonst neu anlegen
		#
		# Wenn $flag&2, dann brauchen wir nach dem alten gar nicht erst
		# zu suchen.

		unless($flag & 2) {
			if ($flag & 1) {
				# Auf den Hashcode verlassen wir uns *nicht*.
				$seq = DoFn("select acct.seq from acct
						left join acctassoc
						    on acct.`hash`=acctassoc.`hash`
							and acctassoc.seq=acct.seq
						where   acct.kunde = $cust
						and    acct.dienst = $dien   and acct.dest = $key
						and      acct.jjmm = $jjmm   and   acct.tt  = $tt
						and    acct.quelle = $quelle and $where
						limit 1 for update");
			} else {
				$seq = DoFn("select acct.seq from acct
						left join acctassoc
						    on acct.`hash`=acctassoc.`hash`
							and acctassoc.seq=acct.seq
						where   acct.`hash`= $hash   and acct.kunde = $cust
						and    acct.dienst = $dien   and acct.dest = $key
						and      acct.jjmm = $jjmm   and   acct.tt  = $tt
						and    acct.quelle = $quelle and $where
						limit 1 for update");
			}
		}

		if(defined $seq and $flag & 1) {
			Do("delete from acctassoc where `hash`= $hash and seq = $seq");
			Do("delete from acct where `hash`= $hash and seq = $seq");
			$seq = undef;
		}

		if(defined $seq) {
			my $cnt = Do("update acct
						set acct.pakete = acct.pakete + $pkt,
						    acct.bytes = acct.bytes + $bytes
						where   acct.`hash`= $hash   and acct.seq = $seq");
			fehler "Kein acct-Datensatz geupdatet, $hash $seq"
				if not $cnt and ($pkt or $bytes);
		} else {
			$pkt = rund($pkt);
			$pkt = 1 if $pkt == 0;
			$seq = 1+DoFn("select max(seq) from acct where `hash`=$hash for update");
			Do(<<_);
	INSERT INTO acct SET
		`hash`    = $hash,
		seq       = $seq,
		kunde     = $cust,
		jjmm      = $jjmm,
		tt        = $tt,
		dienst    = $dien,
		dest      = $key,
		bytes     = $bytes,
		pakete    = $pkt,
		quelle    = $quelle
_
			Do("insert into acctassoc set `hash`=$hash,seq=$seq,info=${\qquote $txt}") if defined $txt;
		}
	};
	wantarray ? ($hash,$seq) : $hash;
}

sub nice_intervall($;$) {
	my ($s,$flag) = @_;
	$flag ||= 0;

	use integer;
	my $res = "";
	if($s < 0) {
		$res = "-";
		$s = -$s;
	} elsif($s == 0) {
		if ($flag & 1) {
			return "0 Sec";
		} else {
			return "-";
		}
	}

	my $sec = $s%60; $s /= 60;  # /
	my $min = $s%60; $s /= 60;  # /
	my($hr,$tg);
	if($flag & 2) {
		$hr = $s;
	} else {
		$hr  = $s%24; $s /= 24;  # /
		$tg  = $s;
	}
	if($tg)  { $res .= "$tg Tag".(($tg>1)?"e":""); }
	if($hr)  { $res .= " " if $res ne ""; $res .= "$hr Std";  }
	if($min) { $res .= " " if $res ne ""; $res .= "$min Min"; }
	if($sec) { $res .= " " if $res ne ""; $res .= "$sec Sec";  }
	$res;
}

sub iso_intervall($;$) {
	my ($s,$flag) = @_;
	$flag=0 if not defined $flag;
	use integer;

	return ($flag&8)?"-":undef if not defined $s;

	my $res = "";
	if($s < 0) { 
		$s = -$s;
		$res = "-";
	}
	my($sec,$min,$hr,$tg);
	if($flag&2) {
		($sec,$min,$hr) = localtime($s);
	} else {
		$sec = $s%60; $s /= 60;  # /
		$min = $s%60; $s /= 60;  # /
		if($flag & 4) {
			$hr = $s;
		} else {
			$hr  = $s%24; $s /= 24;  # /
			$tg  = $s;
		}
	}
	if($tg) { $res  .= "${tg}d "; }
	$res .= sprintf "%02d:%02d",$hr,$min;
	$res .= sprintf ":%02d",$sec if ($sec and not $flag&16) or $flag&1;
	$res;
}

# my @klist;
sub unterkunden($;$$) {
	my($first,$kn,$flag) = @_;
  unless (defined $flag) { $flag = 0; }
	my $kl;
	$kl = sub {
		my @kl;
		my($knd) = @_;
		push(@kl,$knd);
		DoSelect {
            push( @kl, $kl->( $_[0] ) );
		} "select id from kunde where kunde = $knd "
        . ( ($flag & 1)
          ? ' AND ( kunde.ende IS NULL OR kunde.ende >= UNIX_TIMESTAMP(NOW()) )'
          : '' );
		@kl;
	};
	$kn = "kunde" if $kn eq "";
#	if($first != 1) {
		wantarray ? $kl->($first) : "( $kn = ".join(" or $kn = ",$kl->($first))." )";
#	} else {
#		@klist = kl(1) unless @klist;
#		wantarray ? @klist : "( $kn = ".join(" or $kn = ",@klist)." )";
#	}
}

sub oberkunden($;$) {
	my($first,$kn) = @_;
	my @kl;
	while($first) {
		push(@kl,$first);
		$first = DoFn("select kunde from kunde where id = $first");
	}
	$kn = "kunde" if $kn eq "";
	wantarray ? @kl : "( $kn = ".join(" or $kn = ",@kl)." )";
}

sub oberkunde($;$) {
	my($kunde,$flag) = @_;
	$flag ||= 0;

	my($ober,$flg) = DoFn("select kunde,flags from kunde where id = $kunde");
	my $eigene_re = bignum(1)<<find_descr("kunde","eigene_re");

	while(1) {
		last unless $ober;
		last if $flag&1 and $flg&$eigene_re;
		$kunde = $ober;
		($ober,$flg) = DoFn("select kunde,flags from kunde where id = $kunde");
	}
	$kunde;
}

sub ist_unterkunde($;$) {
	my($knd,$base) = @_;
	$base = 1 unless defined $base;
	do {
		return 1 if $knd == $base;
		$knd = DoFn("select kunde from kunde where id = $knd");
	} while $knd;
	0;
}

sub resellerkunden($) {
	my($knd) = @_;
	my @res;
	push(@res,unterkunden($knd));
	DoSelect {
		my($kd) = @_;
		push(@res,unterkunden($kd));
	} "select id from kunde where reseller=$knd";

	my %res;
	grep { not $res{$_}++ } @res;
}

sub ist_intern($;$) {
	my($adr,$base) = @_;

	my $puny = puny_encode(lc $adr, 4);

	if ( my $kunde = DoFn("SELECT kunde FROM person WHERE email = ${\qquote $puny}") ||
	                 DoFn("SELECT person.kunde FROM mailassoc, person WHERE mailassoc.person = person.id AND mailassoc.email = ${\qquote $puny}") ) {
		return ist_unterkunde($kunde,$base);
	}

	$adr =~ s/^.*\@//;
	$adr = puny_encode(lc $adr, 4);
	pl: while($adr =~ /\.(.+)/) {
		my $nadr = $1;
		if ( my $kunde = DoFn("SELECT kunde FROM domainkunde WHERE domain = ${\qquote $nadr}" ||
		                 DoFn("SELECT person.kunde FROM mailassoc, person WHERE mailassoc.person = person.id AND mailaddoc.email = ${\qquote $nadr}") ) < 0 ) {
			return ist_unterkunde($kunde,$base);
		}
		$adr = $nadr;
	}
	undef;
}

sub flag_names($;$$) {
	my($flags,$domain,$oflags) = @_;
	my @names;
	my $w_a = wantarray;
	if(defined $oflags) {
		$oflags = 0+$oflags; # könnte ein String "0" sein, wird ansonsten
		$flags ^= $oflags;   # <== hier falsch behandelt ("0" ^ "0" == "\0").
	}
	return $w_a?():"-" if $flags == 0;
	$domain = "dienst" if $domain eq "";
	my $rs = "";
	my $b1=bignum(1);
	DoSelect {
		my($desc,$bla) = @_;
		return unless ($b1 << $desc) & $flags;
		$flags &=~ ($b1 << $desc);

		$rs = "" if $w_a;
		if(defined $oflags) {
			$rs .= (($b1 << $desc) & $oflags) ? "-" : "+";
			$oflags &=~ ($b1 << $desc);
		}
		$rs .= "$bla";
		$rs .= "," unless $w_a;
		push(@names,$rs) if $w_a;
	} "select descr.descr,descr.bla from descr,descr_typ where descr.typ = descr_typ.id and descr_typ.name = '${\quote $domain}'";

	if($flags) { # Reste ohne Namen?
		if(defined $oflags) {
			if($flags & $oflags) {
				$rs = "" if $w_a;
				$rs .= sprintf("+0x%x,",$flags & $oflags) if $flags & $oflags;
				$rs .= "," unless $w_a;
				push(@names,$rs) if $w_a;
			}
			if($flags & ~$oflags) {
				$rs = "" if $w_a;
				$rs .= sprintf("-0x%x,",$flags & ~$oflags) if $flags & ~$oflags;
				$rs .= "," unless $w_a;
				push(@names,$rs) if $w_a;
			}
		} else {
			$rs = "" if $w_a;
			$rs .= sprintf("0x%x",$flags);
			$rs .= "," unless $w_a;
			push(@names,$rs) if $w_a;
		}
	}
	return @names if $w_a;
	chop $rs;
	$rs;
}

my $gdienst;

## Mailadresse => Kunde

sub find_kunde_by_mail($) {
	my($adr) = @_;
	my $pers = get_person($adr,"mail,?email",1);
	return DoFn("SELECT kunde FROM person WHERE id = $pers") if $pers;

	# hilfsweise Suche anhand Domain oder Teilen davon:
	$adr = puny_encode(lc $adr, 4);
	while($adr =~ /\.(.+)/) {
		my $nadr = $1;
		my $kunde = DoFn "SELECT kunde FROM domainkunde WHERE domain=${\qquote $adr} ORDER BY domainkunde.beginn DESC LIMIT 1";
		$kunde = DoFn("SELECT person.kunde FROM mailassoc,person WHERE mailassoc.email = ${\qquote $adr} and mailassoc.person=person.id")
			unless $kunde;
        return $kunde if $kunde;
		$adr = $nadr;
	}
    0;
}

my %kname;
my %lkname;
sub name_kunde($;$) {
	my($kunde,$flag) = @_;
	$flag ||= 0;
	return "-" unless $kunde;
	delete $kname{$kunde} if $flag&1;
	delete $lkname{$kunde} if $flag&1;
	if($flag & 2) {
		return $lkname{$kunde} if exists $lkname{$kunde};
		my $res = DoFn("select person.name from person,kunde where kunde.id=$kunde and person.id=kunde.hauptperson");
		if(defined $res) {
			$lkname{$kunde} = $res;
			return $res;
		}
	}
	return $kname{$kunde} if exists $kname{$kunde};
	my $res = DoFn("select name from kunde where id = $kunde");
	$kname{$kunde} = $res;
	return $res;
}


my @nummern = (
	"null", "eins", "zwei", "drei", "vier",
	"fünf", "sechs", "sieben", "acht", "neun",
	"zehn", "elf", "zwölf", "dreizehn", "vierzehn",
	"fünfzehn", "sechzehn", "siebzehn", "achtzehn", "neunzehn",
	"zwanzig", "einundzwanzig", "zweiundzwanzig", "dreiundzwanzig", 
	"vierundzwanzig", "fünfundzwanzig", "sechsundzwanzig", 
	"siebenundzwanzig", "achtundzwanzig", "neunundzwanzig", 
	"dreißig", "einunddreißig", "zweiunddreißig", "dreiunddreißig", 
	"vierunddreißig", "fünfunddreißig", "sechsunddreißig", 
	"siebenunddreißig", "achtunddreißig", "neununddreißig", 
	"vierzig", "einundvierzig", "zweiundvierzig", "dreiundvierzig", 
	"vierundvierzig", "fünfundvierzig", "sechsundvierzig", 
	"siebenundvierzig", "achtundvierzig", "neunundvierzig", 
	"fünfzig", "einundfünfzig", "zweiundfünfzig", "dreiundfünfzig", 
	"vierundfünfzig", "fünfundfünfzig", "sechsundfünfzig", 
	"siebenundfünfzig", "achtundfünfzig", "neunundfünfzig", 
	"sechzig", "einundsechzig", "zweiundsechzig", "dreiundsechzig", 
	"vierundsechzig", "fünfundsechzig", "sechsundsechzig", 
	"siebenundsechzig", "achtundsechzig", "neunundsechzig", 
	"siebzig", "einundsiebzig", "zweiundsiebzig", "dreiundsiebzig", 
	"vierundsiebzig", "fünfundsiebzig", "sechsundsiebzig", 
	"siebenundsiebzig", "achtundsiebzig", "neunundsiebzig", 
	"achtzig", "einundachtzig", "zweiundachtzig", "dreiundachtzig", 
	"vierundachtzig", "fünfundachtzig", "sechsundachtzig", 
	"siebenundachtzig", "achtundachtzig", "neunundachtzig", 
	"neunzig", "einundneunzig", "zweiundneunzig", "dreiundneunzig", 
	"vierundneunzig", "fünfundneunzig", "sechsundneunzig", 
	"siebenundneunzig", "achtundneunzig", "neunundneunzig" );

sub zeit_klartext($$) {
	my($hour,$min) = @_;
	my $stunde = $nummern[$hour];
	my $minute = $nummern[$min];

	$stunde = "ein" if $hour == 1;
	$minute = ""    if $min  == 0;

	return "$stunde Uhr $minute";
}

sub this_date($) {
	my($date) = @_;
	my($sec,$min,$hour,$day,$mon,$year) = localtime($date);
    timelocal(0,0,0,$day,$mon,$year);
}

sub this_time($) {
	my($date) = @_;
	my($sec,$min,$hour,$day,$mon,$year) = localtime($date);
    ($hour*60+$min)*60+$sec;
}

sub enddate ($;@) {
    my($yy,$mm,$dd,$h,$m,$s) = @_;
	my $add = 0;
    if($yy !~ /^\d{4}$/) { # textuelle Angabe
		my $datum = $yy;
        return undef unless $datum =~ s/^(\d\d\d\d)(?:\-?)(\d\d?)(?:\-(\d\d?))?\b//;
        $datum = " $mm" if $mm;
        ($yy,$mm,$dd) = ($1,$2,$3);
        if ($datum =~ /^\s*(\d\d)(?:\:(\d\d)(?:\:(\d\d))?)?/) {
            $h = $1; $m = $2; $s = $3;
        }
    }
    $yy -= 1900; 
	if($dd) {
		$mm--;
		$add = 24*60*60;
	} elsif($mm) {
		$dd=1; if($mm==12) { $yy++; $mm=0; }
	} else {
		$yy++; $mm=0; $dd=1;
	}
	if(defined $s) { $add += 1; }
	elsif(defined $m) { $add += 60; }
	elsif(defined $h) { $add += 60*60; }
	$add -= 24*60*60 if defined $h;

    $dd = 1 if $dd < 1;
    $dd = 27 if $dd > 31;
    timelocal($s,$m,$h,$dd,$mm,$yy) + $add;
}


sub rangedate($;$) {
	my($beginn,$ende) = @_;
	unless($ende) {
		if($beginn =~ s/\s*-\s*(\d{4}\b.*)//) {
			$ende = $1;
		} else {
			$ende = $beginn;
		}
	}
	( unixdate($beginn), enddate($ende) );
}

sub get_handle($$;$$$) {
	my($pers,$reg,$res,$set,$setreg) = @_;
	my $status;
	foreach my $reg(split(/,/,$reg)) {
		my $nic;
		if($reg =~ /^\d+$/) {
			$nic = $reg;
		} else {
			$nic = find_descr("nic",$reg);
		}
		# Trennung von Variablen-Deklaration und bedingter
		# Initialisierung wichtig, vgl. RT#244828
		my($hdl,$status);
		($hdl,$status) = DoFn("select handle,status from nic where person = $pers and nic = $nic") if defined $nic;
		if(defined($status) and $status < find_descr("handlestatus","OK_NIC")) {
			problem $res,"Handle unvollständig beantragt",$pers,$nic,$hdl,$reg,get_descr("handlestatus",$status);
			$hdl = undef;
			$set = undef;
		}
		return $hdl if defined $hdl;
	}
	if($set) {
		unless($setreg) {
			$setreg = $reg;
			$setreg =~ s/,.*//;
		}
		my $nic;
		if($setreg =~ /^\d+$/) {
			$nic = $setreg;
		} else {
			$nic = find_descr("nic",$setreg);
			return undef unless defined $nic;
		}
		
		Do("insert into nic set person=$pers, nic=$nic, handle='${\quote $set}'");
		return $set;
	}
	undef;
}


{

my $num = 0;
my @a;
my $lastperson = 0;
my $proc;
my $gkunde;
my $gflags;
my %gskip;

{
	my %artHotline;

	sub is_hotline(;$) {
		DoSelect { $artHotline{ +shift } = undef }
		'SELECT id FROM stunden_art WHERE flags & (1<<'
		  . find_descr( stunden_art => hotline => 1 ) . ')'
		  unless keys %artHotline;

		if   (@_) { exists $artHotline{ +shift } }
		else      { keys %artHotline }
	}
}

sub dsb_flush1() {
	my $a = pop @a;
	return if $a->[2] == 0 and $a->[$#$a];
	return if defined $gkunde and $a->[4] != $gkunde;

	$num++;
	$a->[$#$a]++;
	&$proc(@$a);
}

sub dsb_add(@) {
	my(@d) = @_;

	if ( is_hotline( $d[8] ) ) {
		return if defined $gskip{$d[0]};

		if($gflags&1) { # kumuliere Viertelstundenpausen
			while(1) {
				my($mid,$mbeg,$mdau,$mkunde) = DoFn <<_;
	select id,beginn,dauer,kunde
	  from stunden
	 WHERE ${\ in_list( art => '', is_hotline() ) }
	   and beginn>=$d[1]+$d[2]
	   and beginn<=$d[1]+$d[2]+15*60
	   and person=$d[3]
	 order by beginn,dauer desc limit 1
_
				last unless defined $mid;
				$gskip{$mid}=1;
				$d[2] = $mbeg-$d[1] +$mdau;
				$d[4] = undef if defined $d[4] and $d[4] != $mkunde;
			}
		}
		if($gflags&2) { # erste halbe Stunde wegwerfen
			return if $d[2] <= 30*60;
			$d[1] += 30*60;
			$d[2] -= 30*60;
		}
	}

	# 1=Beginn 2=Dauer 3=Person 5=Faktor 8=Art
	while(@a) {
		my $la = $a[$#a];
		if($la->[1] > $d[1]) { # Altlast
			push(@a,\@d);
			return;
		}
		if($la->[1] + $la->[2] <= $d[1]) { ## keine Überschneidung
			dsb_flush1();
			if (   ( $gflags & ( 4 | 1 | 2 ) ) == 4
				&& is_hotline( $d[8] )
				&& is_hotline( $la->[8] ) )
			{
				next if @a;
				my $startPause = $la->[1]+$la->[2];
				my $dauerPause = $d[1] - $startPause;

				&$proc(undef,$startPause,$dauerPause, $d[3],undef,$d[5], undef)
					if $dauerPause < 15*60;
			}
			next;
		}
		# Überschneidung!
		my $restdauer = $la->[2];
		$la->[2] = $d[1]-$la->[1]; ## kann 0 sein, wenn gleiche Startzeiten
		$restdauer -= $la->[2]+$d[2]; # Restzeit
		dsb_flush1(); # $la ist jetzt draußen
		if($restdauer) { # Restzeit
			$la->[1] = $d[1]+$d[2];
			$la->[2] = $restdauer;
			push(@a,$la);
		}
		last;
	}
	push(@a,\@d);
}

sub dsb_flush {
	dsb_flush1() while @a;
}

sub stunden_bereich($$$$$&) {
	my($person,$kunde,$beginn,$ende,$flags,$sub) = @_;
	$flags=0 unless defined $flags;
	$proc = $sub;
	$num = 0;
	@a = ();
	%gskip = ();

	$ende += $beginn if $ende < $beginn;

	my $sel = "( stunden.beginn >= $beginn AND stunden.beginn + stunden.dauer <= $ende and stunden.beginn < $ende)";
	$sel .= " AND stunden.person = $person" if defined $person;
	# $sel .= " AND stunden.kunde = $kunde" if defined $kunde;
	$gkunde=get_kunde($kunde);
	$gflags=$flags;
	$sel .= ' AND ' . in_list( 'stunden.art', '', is_hotline() )
	  if $flags & 4;

	DoSelect {
		# print "Res: ".$_[0]." ".isodate($_[1])." ".iso_intervall($_[1]%(24*60*60))."-".iso_intervall(($_[1]+$_[2])%(24*60*60))."\n";
		if($lastperson != $_[3]) {
			dsb_flush();
			$lastperson = $_[3];
		}
		push @_, my $out_flag;
		dsb_add(@_,0);
	} <<_;
	SELECT    stunden.id,           # 0
	          stunden.beginn,       # 1
	          stunden.dauer,        # 2
	          stunden.person,       # 3
	          stunden.kunde,        # 4
	          stunden_art.faktor,   # 5
	          stunden.infotext,     # 6
	          stunden.ticket,       # 7
	          stunden.art,          # 8
	          stunden_art.flags,    # 9
	          stunden_art.name      # 10
			                        # 11: Output-Flag
	FROM      stunden
	JOIN      stunden_art ON stunden.art    = stunden_art.id
	WHERE     $sel
	ORDER BY  person, beginn, dauer DESC
_

	dsb_flush();
	$num;
}

}


sub update_stunden_cache(;$$$)
{
	my($beginn,$ende,$pers) = @_;
	my @add_sql;
	push @add_sql, "AND stunden.beginn >= $beginn" if $beginn;
	push @add_sql, "AND stunden.beginn < $ende" if $ende;
	push @add_sql, "AND stunden.person = $pers" if $pers;
	my %zcache;
	DoSelect {
		my ( $sid, $pers, $kund, $beg, $dau ) = @_;
		stunden_bereich $pers, $kund, $beg, $dau, 0,
			sub {
				my ( $id, $be, $dauer, $pe, $ku, $faktor ) = @_;
				return unless $sid == $id;
				$zcache{$id} += $dauer * $faktor / 100;
			};
		} <<_;
			SELECT id, person, kunde, beginn, dauer
			FROM   stunden
			WHERE stunden.zeit is NULL and stunden.dauer > 0
			@add_sql
_
	while(my($id,$zeit) = each %zcache) {
		Do("update stunden set zeit=$zeit, timestamp=timestamp where id=$id");
	}

	# (fehlerhafte) Einträge mit Dauer=0 werden
	# von stunden_bereich() übersprungen
	my $n = Do("update stunden set zeit=0, timestamp=timestamp where zeit is null and dauer=0 @add_sql");
	Db::log_trace("Stunden mit Dauer=0 gefunden ($n) (@add_sql)") if $n;
}


sub time4ticket {
	my ($ticket, $beginn, $ende, $kunden, $ohne_kunden ) = @_;
	fehler qq("$ticket" ist keine gültige Ticket-ID.) unless $ticket =~ /^\d+\z/;
	defined() && !/^\d+\z/ and fehler qq("$_" ist kein gültiger Unix-Timestamp.)
		for $beginn, $ende;

	require noris::Ticket::API;
	my $api = noris::Ticket::API::get_pooled_connection();

	my @tick;
	{
		my $res = $api->select_tickets(attributes => ["ticket_number"], query => {merge_root => $ticket});
		$res->foreach_row(sub {
			push @tick,$_[0];
		});
	}
	return undef, undef unless @tick;

    my ( $zeit_ohne_faktor, $zeit_mit_faktor );
	DoTrans {
		if ( my ( $min_beginn, $max_beginn ) = DoFn(<<_) ) {
	SELECT MIN(stunden.beginn), MAX(stunden.beginn)
	FROM   stunden
	WHERE  ${\ in_list("ticket","",@tick) }
_

			my @add_sql;
			push @add_sql, "AND stunden.beginn >= $beginn" if $beginn;
			push @add_sql, "AND stunden.beginn < $ende" if $ende;
			push @add_sql, 'AND ' . in_list( 'stunden.kunde', '', @$kunden )
			  if $kunden && @$kunden;
			push @add_sql, 'AND ' . in_list( 'stunden.kunde', NOT => @$ohne_kunden )
			  if $ohne_kunden && @$ohne_kunden;
			( $zeit_mit_faktor, $zeit_ohne_faktor ) = DoFn(<<_);
			SELECT SUM(stunden.zeit), SUM(stunden.zeit/stunden_art.faktor)*100
			FROM   stunden, stunden_art
			WHERE  ${\ in_list("stunden.ticket","",@tick)} 
			AND stunden_art.id = stunden.art
			@add_sql
_
			$zeit_ohne_faktor =~ s/\.0+\z// if defined $zeit_ohne_faktor;
		}
	} 2;

	$zeit_ohne_faktor||0, $zeit_mit_faktor||0;
}

sub time4kunde {
	my ($kunde, $beginn, $ende ) = @_;
	my $kid = get_kunde($kunde);
	fehler qq("$kunde" ist kein gültiger Kunde.) unless $kid;
	defined() && !/^\d+\z/ and fehler qq("$_" ist kein gültiger Unix-Timestamp.)
		for $beginn, $ende;

	my ( $zeit_ohne_faktor, $zeit_mit_faktor );

	DoTrans {
		my @add_sql;
		push @add_sql, "AND stunden.beginn >= $beginn" if $beginn;
		push @add_sql, "AND stunden.beginn < $ende" if $ende;
		( $zeit_mit_faktor, $zeit_ohne_faktor ) = DoFn(<<_);
			SELECT SUM(stunden.zeit), SUM(stunden.zeit/stunden_art.faktor)*100
			FROM   stunden, stunden_art
			WHERE  stunden.kunde = $kid   @add_sql
			AND stunden.zeit is NOT NULL
			AND stunden_art.id = stunden.art
_
	} 2;

	$zeit_ohne_faktor =~ s/\.0+\z// if defined $zeit_ohne_faktor;
	$zeit_ohne_faktor||0, $zeit_mit_faktor||0;
}

sub time4kunden {
	my ($beginn, $ende, $limit ) = @_;
	defined() && !/^\d+\z/ and fehler qq("$_" ist kein gültiger Unix-Timestamp.)
		for $beginn, $ende;
	fehler qq("$limit" ist keine positive Zahl)
		if defined $limit and $limit !~ /^\d+\z/;

	my @res;
	DoTrans {
		my @add_sql;
		push @add_sql, "AND stunden.beginn >= $beginn" if $beginn;
		push @add_sql, "AND stunden.beginn < $ende" if $ende;
		push @add_sql, "GROUP BY stunden.kunde";
		push @add_sql, "HAVING tmf > 0";
		push @add_sql, "ORDER BY tmf desc";
		push @add_sql, "LIMIT $limit" if $limit;
		DoSelect {
			push @res, [ @_ ];
		} <<_ ;
			SELECT stunden.kunde, SUM(stunden.zeit) as tof, SUM(stunden.zeit/stunden_art.faktor)*100 as tmf
			FROM   stunden, stunden_art
			WHERE stunden.zeit is NOT NULL
			AND stunden_art.id = stunden.art
			@add_sql
_
	} 2;

	@res;
}

sub flush_std_cache($$$) {
	my($person,$start,$dauer) = @_;

	DoTrans {
		# Ohne Überschneidung liegt entweder das Ende von A vor (oder auf) dem
		# Anfang von B, oder umgekehrt.
		Do("update stunden set zeit=NULL,timestamp=timestamp where beginn+dauer>$start and $start+$dauer>beginn and person=$person and beginn>$start-50*3600");
	
		# da person+beginn ein Index ist, dient die letzte Klausel dazu, zu
		# verhindern, dass die DB *alle* älteren Datensätze durchsucht, nur um
		# festzustellen, dass keiner 5 Jahre am Stück gearbeitet hat
		# ... auch wenn es sich für manche hier so anfühlen mag.
	
		DoSeq("stdupdate");

		$start = date_start($start);
		update_stunden_cache($start,date_add_ymd($start, 0,0,1), $person);
	} 2;
}


use MIME::Parser;
use Mail::Address;
use Dbase::IP;

my $parser = MIME::Parser->new;
fehler "No parser" unless $parser;
$parser->output_to_core(1);

sub _decode {
	my $self = shift;

	my ($tag, $i, @decoded);
	foreach $tag ($self->tags) {
		@decoded = map { Encode::decode("mime-header", $_)
				} $self->get_all($tag);
		for ($i = 0; $i < @decoded; $i++) {
			$self->replace($tag, $decoded[$i], $i);
		}
	}
	$self->{MH_Decoded} = 1;
}

sub mail2mime($) {
	my($msg) = @_;
	fehler "Mail ist leer" if not defined $msg or (not ref $msg and length($msg) < 30);

	# nur wenn noch kein Mail- bzw. MIME-Objekt
	$msg = &$msg() if ref $msg eq "CODE";
	unless(UNIVERSAL::isa($msg,"Mail::Internet")) {
		if(ref $msg and (UNIVERSAL::isa($msg,"GLOB") or UNIVERSAL::isa($msg,"IO::Handle"))) { # Datei
			$msg = eval { $parser->parse($msg); };
		} else {
			$msg = eval { $parser->parse_data($msg); };
		}
		fehler "Mail nicht parsbar",($@ || $parser->last_error)
			if $@ or $parser->last_error;
	}

	_decode($msg->head);
	$msg;
}

# Baue aus einem "Ding" eine Mail.

sub mime2mail($;$@) {
	my($mail,$from,@to) = @_;

	# Code? Aufrufen.
	($mail,$from,@to) = &$mail() if not @to and UNIVERSAL::isa($mail, "CODE");

	# Array-Ref: Aufsplitten.
	if(UNIVERSAL::isa($mail, "ARRAY")) {
		if(@to) {
			($mail) = @$mail;
		} else {
			($mail,$from,@to) = @$mail;
		}
	}

	# Mail als Code? Aufrufen.
	$mail = &$mail($from,@to) if @to and UNIVERSAL::isa($mail, "CODE");

	# Mail in Datei? Lesen.
	if(ref $mail and (UNIVERSAL::isa($mail,"GLOB") or UNIVERSAL::isa($mail,"IO::Handle"))) { # Datei
		local $/ = undef;
		$mail = <$mail>;
	}

	# Sonstiges? RAUS.
	fehler "Nicht erkennbar",$mail
		if ref $mail and not UNIVERSAL::isa($mail,"Mail::Internet");

	# keine Adressen verlangt? Mail zurückliefern.
	return (ref $mail ? $mail->as_string : $mail) unless wantarray;

	# Keine Adressen angegeben? Aus der Mail extrahieren.
	my $mh = mail2mime($mail)->head;  # evtl. no-op

	unless(defined $from) {
		$from = $mh->get("from");
		if(defined $from and $from ne "") {
			my @from = Mail::Address->parse($from) unless @to;
			$from = $from[0]->address if @from == 1;
		}
		if(not defined $from or $from eq "") { # Suche nach Sender
			$from = $mh->get("sender");
			if(defined $from and $from ne "") {
				$from = (Mail::Address->parse($from))[0];
				$from = $from->address if $from;
			}
		}
		# Default
		$from = "$ENV{'USER'}\@$MAILDOM" if not defined $from or $from eq "";
	}
	unless (@to) {
		foreach my $r(qw(to cc bcc)) {
			my $adr;
			if(defined $mh->get("x-$WDESCR-$r")) {
				$adr = $mh->get("x-$WDESCR-$r");
			} else {
				$adr = $mh->get($r);
			}
			next if not defined $adr;
			my @adr = Mail::Address->parse($adr);
			push(@to,map { $_->address() } @adr);
		}
	}

	# Mail als Text
	$mail = $mail->as_string if ref $mail;

	# $from wurde oben in jedem Fall gesetzt
	($mail,$from,@to);
}

sub charset(;$) {
	my($utf8flag) = @_;
	$utf8flag = $Db::utf8 unless defined $utf8flag;
	return $utf8flag ? "utf-8" : "iso-8859-15";
}

sub mimeheader(;$$$) {
	my($fh,$utf8flag,$contenttype) = @_;
	$utf8flag = $Db::utf8 unless defined $utf8flag;
	$contenttype = "text/plain" unless defined $contenttype;
	my $charset = charset($utf8flag);
	my $mime = <<_;
Mime-Version: 1.0
Content-Type: $contenttype; charset="$charset"
Content-Transfer-Encoding: 8bit
Content-Disposition: inline
_
	if(defined $fh) {
		$utf8flag ? utf8modus($fh) : latinmodus($fh);
		print $fh $mime;
	} else {
		return $mime;
	}
}

sub sendmail($$@) {
	my($mail,$from,@to) = @_;

	($mail,$from,@to) = mime2mail($mail,$from,@to);
	return problem "Keine Adressaten angegeben.\n" unless @to;

	my $add;
	my $real;
	{ no warnings 'once';
	  $real = $Db::no_mail;
	}
	$real = $ENV{'NO_MAIL'} unless $real;
	if($real) {
		$add = <<END;
X-Real-From: $from
X-Real-To: @to
END
		$from = $real;
		@to = ($real);
	}
	if(ref $Db::test_mails) { # Testsystem: Dbase::KundeTest
		push(@{$Db::test_mails}, [$mail,$from,@to]);
	} elsif(in_test() < 3) {
		defined( my $pid = open my $fh, '|-' ) or return problem "open('|-'): $!\n";
		if ($pid) {
			binmodus($fh);
		}
		else {
			exec qw(/usr/sbin/sendmail -oi -f), $from, @to or die $!;
		}
	
		my($hdr,$bdy)=split(/\n\n/,$mail,2);
		mimeheader($fh) unless $hdr =~ /^Content-Type:/mi;
		print $fh $add if defined $add;
		# $hdr =~ s/(\S*[^[:ascii:]]\S*)/ encode_mimewords($1) /eg;
		$hdr .= "\n\n";
		$hdr =~ s/\s*\n\s+/ /g;
		$hdr =~ s/^(\S+):\s+(.*)/ "$1: " . encode_mimewords($2) /egm;
		$hdr .= "\n\n" unless substr($hdr,-2) eq "\n\n";
		print $fh $hdr,$bdy;
        close $fh
          or return problem(
            explain_child_error('Externes Programm zum Mail-Versand') );
	}
	scalar @to;
}


sub rwdiff() {
	return 0 if Dbase::db_handle->{no_write};
	my $rtime = DoFn("select readonly id from nextid where name = 'lastupdate'");
	my $wtime = DoFn("select writeonly id from nextid where name = 'lastupdate'");
	$wtime-$rtime;
}


sub stufung($) {
	my($step) = @_;
	return 0 unless $step;
	return ($step/1024/1024/1024)." GB" unless $step%(1024*1024*1024);
	return ($step/1024/1024)." MB" unless $step%(1024*1024);
	return ($step/1024)." KB" unless $step%(1024);
	return ($step/1000/1000/1000)." G" unless $step%(1000*1000*1000);
	return ($step/1000/1000)." M" unless $step%(1000*1000);
	return ($step/1000)." k" unless $step%(1000);
	$step;
}

sub stufung_in($) {
	my($step) = @_;
	return undef unless $step;
	return $1*1024*1024*1024 if $step =~ /^(\d+)\s?gb$/i;
	return $1*1024*1024 if $step =~ /^(\d+)\s?mb$/i;
	return $1*1024 if $step =~ /^(\d+)\s?kb$/i;
	return $1*1000*1000*1000 if $step =~ /^(\d+)\s?g$/i;
	return $1*1000*1000 if $step =~ /^(\d+)\s?m$/i;
	return $1*1000 if $step =~ /^(\d+)\s?k$/i;
	return undef unless $step =~ /^\d+$/;
	$step;
}

no warnings 'qw';
my @voc = qw(a e i u o);
my @kon = qw(q w r t z p s d f g h k x c v b n m);
my @num = qw(2 3 4 5 6 7 8 9 + - # $ % ! ? & = *);
sub gen_passwd($) {
	my($len) = @_;
	$len=8 unless $len;
	my $pw = "";
#	if($len and not int(rand(3))) {
#		$pw .= $num[int(rand(0+@num))];
#		$len--;
#	}
	while($len > 1) {
		$pw .= $kon[int(rand(0+@kon))];
		$pw .= $voc[int(rand(0+@voc))];

		$len -= 2;
	}
	$pw .= $num[int(rand(0+@num))] if $len;
	$pw;
}


sub print_komma($$;$$$) {
	my($val,$min,$max,$all,$space) = @_;
	$all="" unless defined $all;
	$val = sprintf "%${all}.${max}f",$val;
	$max -= $min;
	$val =~ s/(0{1,$max})$/$space ? (" "x length($1)) : ""/e if $max;
	$val;
}

sub flatten(&@) {
	my $line = shift;
	my $addline;
	$addline = sub {
		my $data = "";
		my ($text,$pref) = @_;
		if(not ref $text) {
			$text =~ s/\r*\n*$//s;
			$data .= "$pref " if $pref ne "";
			$data .= "$text\n";
			&$line($data);
		} elsif(UNIVERSAL::isa($text,"SCALAR")) {
			&$addline($$text,$pref);
		} elsif(UNIVERSAL::isa($text, "ARRAY")) {
			foreach my $tex(@$text) {
				&$addline($tex,$pref);
			}
		} elsif(UNIVERSAL::isa($text, "HASH")) {
			foreach (my ($key,$tex) = each %$text) {
				&$addline($tex,"${pref}${key}::");
			}
		} elsif(UNIVERSAL::isa($text, "GLOB")) {
			while(<$text>) {
				&$addline($_,$pref);
			}
		} else {
			$data .= "$pref " if $pref ne "";
			$data .= "** Unknown: $text\n";
			&$line($data);
		}
	};
	foreach my $d(@_) {
		&$addline($d);
	}
}


my $euro=unixdate("2002-01-01");
sub in_euro($) {
	my($dt)= @_;
	return ($dt >= $euro);
}

sub rund($;$) {
	my($num,$shift) = @_;
	my $sign;
	my $rshift = 0;

	if($num < 0) {
		$sign = -1;
		$num = -$num;
	} else {
		$sign = 1;
	}

	if($shift) {
		if($shift < 0) {
			$shift = -$shift;
			$rshift = $shift;
		}
		if(ref $num and $num->isa("Math::BigInt")) {
			$num *= bignum(10);
			$shift += 1;

			$num += 5*(bignum(10)**($shift-1));
		} else {
			$num += 0.500000000001*(10**$shift);
		}
		$num /= 10**$shift;
	} elsif(not ref $num) {
		$num += 0.500000000001;
	}

	$num = int($num) unless ref $num and $num->isa("Math::BigInt");
	$num->bfround(0) if ref $num and $num->isa("Math::BigFloat");
	$num *= 10**$rshift if $rshift;

	$num * $sign;
}

sub preis(\$;$) {
	my($pr,$k) = @_;
	$k=3 unless defined $k;
	my $p = $$pr;

	return undef unless $p =~ s/^(-)?(\d*)(?:[\.\,](\d+))?$/$2/e;
	return undef if $p eq "" and $3 eq "";
	$p = 0 if $p eq "";

	my $sign = $1;
	$p += $3/10**length($3) if $3;

	$p = rund($p*10**$k);
	$p = -$p if $sign;
	$$pr = $p;
	$p;
}

sub no_crlf($;$) {
	my($txt,$rep) = @_;
	$rep = " " unless defined $rep;
	return "" unless defined $txt;

	$txt =~ s/\n/$rep/g;
	$txt;
}

my $msgid_seq=0;
sub message_id() {
	sprintf("%x-%x-%x\@%s", DoTime,$$,++$msgid_seq,$MAILDOM);
}


my $gendienst;
my %isd_cache;
sub is_dienst($$) {
	my($x,$y)=@_;

	return 0 if not defined $x;

	$gendienst=find_dienst("general") unless defined $gendienst;
	$x=find_dienst($x) unless $x =~ /^\d+$/;
	$y=find_dienst($y) unless $y =~ /^\d+$/;
	my $px=$x;

	while(1) {
		return $isd_cache{$px}{$y}=1 if $x == $y;
		return $isd_cache{$px}{$y}=$isd_cache{$x}{$y}
			if exists $isd_cache{$x}{$y};
		my $xx = DoFn("select berechne from tarifeq where dienst=$x");
		return $isd_cache{$px}{$y}=0 if not defined $xx or $xx==$x;
		$x = $xx;
	}
}

my $use_bignum;
if($BIGNUM >= 0) {
	$use_bignum = $BIGNUM;
} elsif($Config{'use64bitint'} eq "define" or
       $Config{'use64bits'} eq "define") {
	$use_bignum = 0;
} else {
	$use_bignum = 1;
}

sub bignum(;$) {
	return $use_bignum unless @_;

	my($num) = @_;
	return Math::BigInt->new($num) if $use_bignum;
	return $num;
}

sub aufzaehlung {
	my %option = ( trenner => ', ' );
	{
		require Scalar::Util and Scalar::Util->import('reftype')
		  unless defined &reftype;
		my $reftype = reftype($_[0]);
		if ( defined $reftype && $reftype eq 'HASH' ) {
			my $options = shift;
			@option{ keys %$options } = values %$options;
		}
	}
	$option{letzter_trenner} =
	  defined $option{bindewort} ? " $option{bindewort} " : ' und '
	  unless defined $option{letzter_trenner};
	@_ < 2 ? "@_" : join $option{letzter_trenner},
	  join( $option{trenner}, @_[ 0 .. $#_ - 1 ] ), $_[-1];
}

my $PUNY="xn--";

sub puny_decode($;$) {
	my($dom,$flag)=@_;
	$dom = lc $dom;
	$flag=0 if not defined $flag;

	my $prefix = '';
	$dom =~ s/^([\w-]+!|.*\@)// and $prefix = $1 if $flag & 4;

	$dom=join(".",map { substr($_,0,length($PUNY)) eq $PUNY ? decode_punycode(substr($_,length($PUNY))) : $_ } split /\./, $dom);

	$dom=decode_anything($dom); # was auch immmer decode_punycode() ausspuckt
	$prefix=decode_anything($prefix); # sicherheitshalber
	$prefix . $dom;
}


sub puny_encode($;$) {
	my($dom,$flag)=@_;
	$dom = lc $dom;
	$flag=0 if not defined $flag;

	my $prefix = '';
	$dom = decode_anything($dom);
	$dom =~ s/^([\w-]+!|.*\@)// and $prefix = $1 if $flag & 4;

	return $prefix . join(".",map { /[^-0-9a-z]/ ? $PUNY . encode_punycode($_) : $_ } split /\./, $dom);
}

sub is_holiday(;$$) {
	my $time    = shift || DoTime;
	my $profile = shift;

	unless ( ref $profile ) {
		require noris::Date::Calendar::Profiles;
		$profile = 'DE-BY-noris' unless defined $profile;
		no warnings 'once';
		$profile = $noris::Date::Calendar::Profiles::Profiles->{$profile}
		  or return -1;
	}

	my @date = (localtime $time)[5,4,3];
	$date[0] += 1900;
	$date[1] ++;
	require Date::Calendar;
	my $calendar = Date::Calendar->new($profile) or return -1;
	if    ( $calendar->is_full(@date) ) { 1  }
	elsif ( $calendar->is_half(@date) ) { 2  }
	else                                { '' }
}

sub in_period($;$) {
	my($period,$time) = @_;
	$time = DoTime unless defined $time;
	if ( $period =~ s/\B([+-])(\S+)\b\s*// ) {
		my $pm      = $1;
		no warnings 'numeric';
		( my $is_holiday = is_holiday($time, my $profile = $2) ) == -1 and return -1;
		return '' if $is_holiday ? $pm eq '-' : $pm eq '+';
	}
	require Time::Period;
	Time::Period::inPeriod($time,$period);
}

sub content($) {
    my $value = shift;
    defined $value && $value =~ /\S/;
}

sub def_or_minus($) {
    my $value = shift;
	$value = "-" unless defined $value;
    $value;
}


sub addr_from_block($@) {
	my($bits,@frar)=@_;
	splice(@frar,0,$bits);
	foreach my $block(@frar) {
		next unless $block;
		my $adr = $block->bitmask($bits);
		return $adr;
	}
	return undef;
}

# ## Diese Funktion liefert zurück, wie lange der Betreffende aktuell arbeiten
# ## müsste, damit er am Monatsende genau sein Soll erreicht.
my $artPause;
# sub arbeitszeit_heute($) {
# 	my($person) = @_;
#     my( $y, $m, $d ) = isodate( DoTime() );
# 	my $ym = $y*100+$m;
# 	my $days = Days_in_Month($y,$m);
# 	my $soll = DoFn("select soll from persomonat where monat=$ym and person=$person")
# 	        || DoFn("select soll from persomonat where monat=$ym and person is null");
# 	return undef unless defined $soll;
# 
# 	$artPause = DoFn("select id from stunden_art where name ='Pause'")
# 		unless defined $artPause;
# 
# 	my($uest,$zuletzt,$tagstunden) = DoFn("select ueberstunde,letzter,tagstunden from perso where person=$person");
# 	my $ym_last = ($m==1) ? ($y-1)*100+12 : $y*100+$m-1;
# 
# 	# Berücksichtige Überstunden nur, wenn der letzte Monat
# 	# abgeschlossen wurde (weil der Eintrag ansonsten nicht aktuell ist)
# 	$soll -= $uest if $zuletzt == $ym_last;
# 
# 	# Urlaub etc. wird als Achtstundenblock eingetragen
# 	if(defined $tagstunden) {
# 		$tagstunden /= 1000;
# 	} else {
# 		$tagstunden = 8;
# 	}
# 
# 	# Summiere die Arbeitstage bis dato / für den Gesamtmonat
# 	my $work = 0;
# 	my $done = 0;
# 	for(1..$days) {
# 		next if is_holiday( unixdate($y,$m,$_) );
# 		my $arbeit = ($m != 11 || $_ != 24 && $_ != 31 || 0.5);
# 		$work += $arbeit;
# 		next if $_ >= $d;
# 		$done += $arbeit;
# 	}
# 
# 	# Ziehe die bis exklusiv heute gearbeiteten Stunden vom Soll ab
# 	stunden_bereich( $person,undef,unixdate($y,$m,1),unixdate($y,$m,$d),1|2, sub {
# 		my(undef,undef,$dauer,undef,undef,$faktor,undef,undef,$art,$flg) = @_;
# 		if(test_flag("stunden_art","keine_arbeit",$flg) and $art != $artPause) {
# 			$soll -= $dauer*$tagstunden/8;
# 		} elsif($art != $artPause) {
# 			$soll -= $dauer*$faktor/100;
# 		}
# 	});
# 
# 	# Sag nix, wenn das Monatssoll bereits überschritten ist
# 	return undef if $soll < 0;
# 
# 	# Verteile den Soll auf die Resttage.
# 	return $soll / ($work-$done);
# }

## Diese Funktion liefert zurück, wie viele Monatsstunden der Betreffende
#bis inkl. heute hätte arbeiten sollen.
sub soll_bis_heute($) {
	my($person) = @_;
    my( $y, $m, $d ) = isodate( DoTime() );
	my $ym = $y*100+$m;
	my $days = Days_in_Month($y,$m);

	$artPause = DoFn("select id from stunden_art where name ='Pause'")
		unless defined $artPause;

	my($uest,$zuletzt,$tagstunden) = DoFn("select ueberstunde,letzter,tagstunden from perso where person=$person");
	my $ym_last = ($m==1) ? ($y-1)*100+12 : $y*100+$m-1;

	if(defined $tagstunden) {
		$tagstunden /= 1000;
	} else {
		$tagstunden = 8;
	}

	# Summiere die Arbeitstage bis dato / für den Gesamtmonat
	my $work = 0;
	my $done = 0;
	for(1..$days) {
		next if is_holiday( unixdate($y,$m,$_) );
		my $arbeit = ($m != 12 || $_ != 24 && $_ != 31 || 0.5);
		$work += $arbeit;
		next if $_ > $d;
		$done += $arbeit;
	}

	return $tagstunden * 3600 * $done;
}

sub if_defined($$;&) {
    die "if_defined() darf nur im Listenkontext aufgerufen werden.\n"
      unless wantarray;
    my ( $name, $value, $sub ) = @_;
    return unless defined $value;
    $name => $sub ? $sub->($value) : $value;
}

sub rufnummernliste($) {
    my ($rufnummern) = @_;
    my @liste = map {
        my %rufnummer;
        ++$rufnummer{ +{qw(< in > out)}->{$1} } while s/^([<>])//;
        %rufnummer = ( in => 1, out => 1 ) unless keys %rufnummer;

        # DIN 5008 sieht laut
        # <http://de.wikipedia.org/wiki/Rufnummer#Schreibweisen>
        # einen Bindestrich vor der Durchwahl vor.
        s/([^ ]+ [^ ]+ [^ ]+) /$1-/;
        $rufnummer{din5008} = $_;
        \%rufnummer
    } split /\|/, $rufnummern;
    if   (wantarray) { @liste }
    else             { \@liste }
}

sub chomped($) {
	my $string = shift;
	chomp $string;
	$string;
}

sub DoSelect_flaglist($&$;&&) {
	my($dname,$proc,$sel, $selproc,$prproc) = @_;
	my $descr = DoFn "select id from descr_typ where name=${\qquote $dname}";
	die "Deskriptor '$dname' nicht gefunden\n" unless $descr;
	my @data;
	my $bits = 0;
	my @ltr;

	# Liste der Buchstaben holen
	DoSelect {
		my($id,$dis,$bla) = @_;
		push(@ltr, [1<<$id,$dis,$bla]);
	} "select descr,idchar,bla from descr where typ=$descr and idchar is not null order by bla";

	# Daten holen und tatsächlich verwendete Bits merken
	DoSelect {
		return if defined $selproc and not &$selproc(@_);
		$bits |= $_[0];
		push(@data,[@_]);
	} $sel;

	# die nirgends verwendeten Buchstaben ausfiltern
	@ltr = grep { $_->[0] & $bits} @ltr;

	# Bitmaske nicht mitliefern
	my @lltr = map {[$_->[1], $_->[2]]} @ltr;
	&$prproc(@lltr) if defined $prproc;

	# Daten zurückliefern
	foreach my $r(@data) {
		my $bits = $r->[0];
		my $str = "";
		foreach my $b(@ltr) {
			$str .= ($b->[0] & $bits) ? $b->[1] : "-";
		}
		$r->[0] = $str;
		&$proc(@$r);
	}
	return @lltr;
}

sub gen_flaglist($@) {
	my($head,@flags) = @_;
	
# a  bla     <- gen_flaglist(1, ...)
# |b fasel
# -- xxx     <- eigentlicher Output
# a- xxx
# -b xxx
# -- xxx
# |b fasel   <- gen_flaglist(0, ...)
# a  bla

	my $indent = 0;
	my $res = "";
	if($head) {
		foreach my $k(@flags) {
			$res .= "|"x$indent. $k->[0]. " "x(@flags-$indent+1). $k->[1]. "\n";
			$indent++;
		}
	} else {
		foreach my $k(reverse @flags) {
			$res .= "|"x(@flags-$indent-1). $k->[0]. " "x($indent+1). $k->[1]. "\n";
			$indent++;
		}
	}
	$res;
}

sub print_flaglist($$@) {
	my($out,$head,@flags) = @_;
	print $out gen_flaglist($head,@flags);
}

sub umlaut_e($) {
	my ($text) = @_;

	$text =~ s/Ä/Ae/g;    $text =~ s/ä/ae/g;
	$text =~ s/Ö/Oe/g;    $text =~ s/ö/oe/g;
	$text =~ s/Ü/Ue/g;    $text =~ s/ü/ue/g;
	$text =~ s/ß/ss/g;

	$text;
}

sub explain_child_error(;$$$) {
    my $program     = @_ ? shift : 'Externes Programm';
	my $child_error = @_ ? shift: $?;
	my $errno       = @_ ? shift: $!;
	"$program "
	  . (
		$child_error == -1 ? "konnte nicht ausgeführt werden: $errno"
		: (
			$child_error & 127
			? 'wurde durch Signal ' . ( $child_error & 127 )
			  . ' beendet'
			  . ( ( $child_error & 128 ) != 0 && ' und warf einen Core-Dump' )
			: 'hatte Exit-Status ' . ( $child_error >> 8 )
		  )
		  . '.'
	  );
}

sub get_rz_from_ipkunde($) {
    my $ipk = shift;
    my ( $rack, $ei, $hst ) = DoFn <<_;
SELECT hardware.rack, hardware.enthalten_in, hardware.standort
  FROM ipkunde JOIN hardware ON ipkunde.id = hardware.ip
 WHERE ipkunde.id = $ipk
   AND ( ipkunde.ende IS NULL OR ipkunde.ende >= UNIX_TIMESTAMP(NOW()) )
   AND ( hardware.ende IS NULL OR hardware.ende >= UNIX_TIMESTAMP(NOW()) )
_

    return if !defined $rack and !defined $ei and !defined $hst;

    if ( defined $hst ) {
        my ($rzid) = DoFn "SELECT id FROM rz WHERE standort = $hst";
        return $rzid if defined $rzid;
    }

    while ( !defined $rack ) {
        if (defined $ei) {
            ( $rack, $ei, $hst ) = DoFn <<_;
SELECT rack, enthalten_in, standort
  FROM hardware
 WHERE id = $ei
   AND ( ende IS NULL OR ende >= UNIX_TIMESTAMP(NOW()) )
_
        }

        if ( defined $hst ) {
            my ($rzid) = DoFn "SELECT id FROM rz WHERE standort = $hst";
            return $rzid if defined $rzid;
        }

        return if !defined $rack and !defined $ei;
    }

    return unless defined $rack;
    my ($rz) = DoFn "SELECT rz.id FROM rz JOIN rack ON rack.rz = rz.id WHERE rack.id = $rack";
    return $rz;
}

1;
