#!/usr/bin/perl -w

# Source: @RPM_PACKAGE_VERSION@-@RPM_PACKAGE_RELEASE@

use utf8;
use strict;
use warnings;

BEGIN {
    unshift( @INC, ( $ENV{'POPHOME'} || '@POPHOME@' ) . '/lib' )
      unless $ENV{'KUNDE_NO_PERLPATH'};
}

select(STDERR);
$| = 1;
select(STDOUT);

use Dbase::Help qw(Do DoSeq DoT unixtime unixdate);
use Dbase::Globals;
use Loader qw(line_in);
use Umlaut qw(binmodus textmodus);
use noris::Table;

sub Usage() {
    my $vers = '@RPM_PACKAGE_VERSION@-@RPM_PACKAGE_RELEASE@';

    textmodus( \*STDERR );
    die <<END;
Usage: $0 -- führt eine Datenbankabfrage durch
          -v      variabel breite Spalten (Default: Tab)
          -b      binär arbeiten
          -n      mit Namen der Spalten
          -i      mit Spaltennummern
          -m      auch Datenfelder mit Newlines ausgeben
          -d      interpretiere gewisse Zahlen (im Ergebnis) als Unixdatum
          -D      interpretiere JJJJ-MM-TT (im Befehl) _nicht_ als Unixdatum
          -s NAME verwende diese Datenbank
          QUERY   was zu tun ist

Version: $vers
END
}

use Getopt::Std;
use vars qw(
  $opt_b
  $opt_d
  $opt_D
  $opt_h
  $opt_i
  $opt_m
  $opt_n
  $opt_s
  $opt_v
  $opt_w
);
getopts("bdDhimns:vw") or Usage;
Usage if $opt_h;

Dbase::Help->import(':readonly') unless $opt_w;

=head1 Name

db -- führt eine Datenbankabfrage durch

=head1 Zusammenfassung

    db -v     variabel breite Spalten
       -n     mit Spaltennamen
	   -i     mit Zeilennummern
	   -m     auch Datenfelder mit Newlines ausgeben
	          (Per Default werden diese durch "<LÄNGE>" abgekürzt.)
	   -d     interpretiere gewisse Zahlen als Unixdatum
	   -D     interpretiere JJJJ-MM-TT _nicht_ als Unixdatum
       QUERY  die Abfrage
	      UK(X) Unterkundenliste
		        where UK(1) => where (kunde = X or kunde = ...)
	      UK(X,Y) dito, s/kunde/Y
		  2000-12-31 Datum wird nach unixtime übersetzt

=head1 Beschreibung

Führt eine Datenbankabfrage durch.

SQL-Kenntnisse sind hierfür sehr empfehlenswert...

=head2 Optionen

=over 4

=item C<-i>

gibt Zeilennummern in der ersten Spalte aus.

=item C<-n>

gibt Spaltenüberschriften in der ersten Zeile aus.

=item C<-h>

Kurzhilfe.

=item C<-v>

variabel breite SPalten (für den längsten Wert).

Default: ein Tab pro Spalte.

Vorsicht: Da hierfür das gesamte Resultat eingelesen werden muß,
kann es im Extremfall zu Speicherüberlauf kommen.

=item C<-w>

Verwende auch für C<SELECT>-Befehle die write-only-Datenbank.

=item C<Query> ...

Das auszuführende SQL-Statement.

Un der Query werden die folgenden Spezialitäten automatisch übersetzt:

=over 4

=item YYYY-MM-DD

zu unixtime (Sekunden),

=item UK(X)

zu C<(kunde = X or kunde = ...)>

Dies ist die Unterkundenliste von Kunde X, in einer für SQL verwertbaren
Form.

=item UK(X,Y)

zu C<(Y = X or Y = ...)>

dito, mit anderem Feldnamen.

=back

Ist keine Query angegeben, werden die Abfragen zeilenweise von I<STDIN>
gelesen.

=back

=head1 Daten

Das Programm greift auf folgende Datenbestände zu:

=head2 C<@POPCONFIG@>

Konfigurationsdatei (Datenbankname).

=head2 I<Datenbank>

logischerweise...

=cut

my $rdr;

if (@ARGV) {
    $rdr = sub {
        return undef unless @ARGV;
        my $cmd = "@ARGV";
        @ARGV = ();
        return $cmd;
    };

}
else {
    $rdr = sub {
        line_in("Db> ");
      }
}

my $d1 = 1;
my $d2 = 0;
if ($opt_d) {
    $d1 = unixdate( 1995, 1, 1 );
    $d2 = time + 10 * 356 * 24 * 3600;
}

{
    my $sn = 1;

    sub normalize_rows {
        my $row = shift;

        foreach my $j ( 0 .. $#$row ) {
            my $dat = $row->[$j];

            # Zahl als Zeit interpretieren?
            $row->[$j] = $dat = isotime( $dat, 1 )
              if defined $dat
              and $dat =~ /^\d+$/
              and $dat >= $d1
              and $dat <= $d2;

            # wenn Text mit Linefeeds: <Länge> drucken
            # ansonsten Inhalt oder <NULL>
			$row->[$j] = defined $dat
			  ? ( !$opt_m && $dat =~ /\n/ )
			  ? '<' . ( length($dat) ) . '>'
			  : $dat

			  # historisch gewachsen, vgl. Ticket #10035160:
			  : $opt_v ? '<NULL>' : 'NULL';
        }
        unshift( @$row, $sn++ ) if $opt_i;
        return $row;
    }
}

while (1) {

    my $cmd = &$rdr;
    last unless defined $cmd;
    $cmd =~ s/;\s*$//;
    next if $cmd =~ /^\s*$/;

    $cmd =~
      s/\b(\d{4}-\d{2}-\d{2}(?:\s+\d{2}:\d{2}(?:\d{2})?)?)\b/unixtime($1)/eg
      unless $opt_D;
    $cmd =~ s/\bUK\((\d+,(\S+))\)/unterkunden($1,$2)/eg;
    $cmd =~ s/\bUK\((\d+)\)/unterkunden($1)/eg;
    my $res;

    my @args = ();
    push( @args, DB          => $opt_s ) if $opt_s;
    push( @args, FieldHeader => 1 )      if $opt_n;

    my $db = new Dbase(@args);
    if ($opt_b) {
        binmodus( \*STDOUT );
        $db->{'binary'} = 1;
        foreach my $var (qw(client connection results database server)) {
            $db->Do("set character_set_$var=binary");
        }
    }

    if ( $cmd =~ /^\s*(?:select|show|explain)\b/is ) {
        if ($opt_v) {
            $res = $db->DoT($cmd);
            if ( ref $res ) {
                my $table = noris::Table->new();

                if ($opt_n) {
                    my @titel = @{ shift @$res };
                    unshift( @titel, ' 0' ) if $opt_i;
                    $table->titel(@titel);
                }
                foreach my $i (@$res) {
                    $table->daten( @{ normalize_rows($i) } );
                }
                $table->drucken( empty_is_ok => 1 );
            }
        }
        else {
            $res = $db->DoT(
                $cmd,
                sub {
                    print join( "\t", @{ normalize_rows( $_[0] ) } ) . "\n";
                }
            );
        }
        if ( not ref $res and $res !~ /^\d+$/ ) {
            print STDERR "Fehler: $res\n";
            exit 1;
        }
    }
    elsif ( $cmd =~ s/^seq\s+//i ) {
        $res = DoSeq $cmd;
        print STDERR "Fehler: $res\n", exit 1 if $res < 0;
        print "$res\n";
    }
    else {
        $res = Do $cmd;
        print STDERR "$res\n", exit 1 if $res < 0;
        print "$res\n" if $opt_v;
    }
}
