"""
6. Specifying ordering

Specify default ordering for a model using the ``ordering`` attribute, which
should be a list or tuple of field names. This tells Django how to order
queryset results.

If a field name in ``ordering`` starts with a hyphen, that field will be
ordered in descending order. Otherwise, it'll be ordered in ascending order.
The special-case field name ``"?"`` specifies random order.

The ordering attribute is not required. If you leave it off, ordering will be
undefined -- not random, just undefined.

This also applies for models that have a relationship to another model, you
can use the name of the related field just like you'd any other field to
specify ordering of the model instances in the ``Meta.ordering`` attribute or
the ``order_by`` queryset method.

Another feature is that you can even order by a field of the related model,
for this you use a notation similar to the one used in field lookup, that is
``relationname__fieldname`` (that's a double-underscore).
"""

from django.db import models

class Article(models.Model):
    headline = models.CharField(max_length=100)
    pub_date = models.DateTimeField()
    class Meta:
        ordering = ('-pub_date', 'headline')

    def __unicode__(self):
        return self.headline

class ReaderLetter(models.Model):
    article = models.ForeignKey(Article, null=True)
    reader_name = models.CharField(maxlength=100)
    date_sent = models.DateTimeField()
    class Meta:
        ordering = ('reader_name',)

    def __str__(self):
    	if self.article is None:
            return self.reader_name
	else:
            return "%s, %s" % (self.reader_name, self.article)

class EditorResponse(models.Model):
    reader_letter = models.ForeignKey(ReaderLetter)
    editor_name = models.CharField(maxlength=100)
    class Meta:
        ordering = ('editor_name',)

    def __str__(self):
        return "%s, %s" % (self.editor_name, self.reader_letter)

class Reporter(models.Model):
    name = models.CharField(maxlength=30)
    articles = models.ManyToManyField(Article)
    class Meta:
        ordering = ('articles__headline',)

    def __str__(self):
        r = "%s" % (self.name)
        if self.articles.count():
            r += '; articles: %s' % ', '.join([str(a) for a in self.articles.all()])
        return r

__test__ = {'API_TESTS':"""
# Create a couple of Articles.
>>> from datetime import datetime
>>> a1 = Article(headline='Article 1', pub_date=datetime(2005, 7, 26))
>>> a1.save()
>>> a2 = Article(headline='Article 2', pub_date=datetime(2005, 7, 27))
>>> a2.save()
>>> a3 = Article(headline='Article 3', pub_date=datetime(2005, 7, 27))
>>> a3.save()
>>> a4 = Article(headline='Article 4', pub_date=datetime(2005, 7, 28))
>>> a4.save()

# By default, Article.objects.all() orders by pub_date descending, then
# headline ascending.
>>> Article.objects.all()
[<Article: Article 4>, <Article: Article 2>, <Article: Article 3>, <Article: Article 1>]

# Override ordering with order_by, which is in the same format as the ordering
# attribute in models.
>>> Article.objects.order_by('headline')
[<Article: Article 1>, <Article: Article 2>, <Article: Article 3>, <Article: Article 4>]
>>> Article.objects.order_by('pub_date', '-headline')
[<Article: Article 1>, <Article: Article 3>, <Article: Article 2>, <Article: Article 4>]

# Use the 'stop' part of slicing notation to limit the results.
>>> Article.objects.order_by('headline')[:2]
[<Article: Article 1>, <Article: Article 2>]

# Use the 'stop' and 'start' parts of slicing notation to offset the result list.
>>> Article.objects.order_by('headline')[1:3]
[<Article: Article 2>, <Article: Article 3>]

# Getting a single item should work too:
>>> Article.objects.all()[0]
<Article: Article 4>

# Use '?' to order randomly. (We're using [...] in the output to indicate we
# don't know what order the output will be in.
>>> Article.objects.order_by('?')
[...]

# Create some reader letters.
# First the ones sent as comments about some article.
>>> rl1 = ReaderLetter.objects.create(reader_name='Reader D', article=a1, date_sent=(datetime(2005, 8, 5)))
>>> rl2 = ReaderLetter.objects.create(reader_name='Reader D', article=a2, date_sent=(datetime(2005, 8, 4)))
>>> rl3 = ReaderLetter.objects.create(reader_name='Reader C', article=a4, date_sent=(datetime(2005, 8, 3)))
>>> rl4 = ReaderLetter.objects.create(reader_name='Reader B', article=a4, date_sent=(datetime(2005, 8, 2)))
>>> rl5 = ReaderLetter.objects.create(reader_name='Reader A', article=None, date_sent=(datetime(2005, 8, 1)))

# By default ReaderLetter.objects.all() orders by the name
# of the reader that wrote it ascending.
>>> ReaderLetter.objects.all()
[<ReaderLetter: Reader A>, <ReaderLetter: Reader B, Article 4>, <ReaderLetter: Reader C, Article 4>, <ReaderLetter: Reader D, Article 1>, <ReaderLetter: Reader D, Article 2>]

# Composite ordering using the related article headline descending as one of the components
>>> ReaderLetter.objects.order_by('reader_name', '-article__headline')
[<ReaderLetter: Reader B, Article 4>, <ReaderLetter: Reader C, Article 4>, <ReaderLetter: Reader D, Article 2>, <ReaderLetter: Reader D, Article 1>]

>>> er1 = EditorResponse.objects.create(editor_name='Editor A', reader_letter=rl1)
>>> er2 = EditorResponse.objects.create(editor_name='Editor B', reader_letter=rl2)
>>> er3 = EditorResponse.objects.create(editor_name='Editor B', reader_letter=rl3)

>>> EditorResponse.objects.all()
[<EditorResponse: Editor A, Reader D, Article 1>, <EditorResponse: Editor B, Reader D, Article 2>, <EditorResponse: Editor B, Reader C, Article 4>]

# Order just by the relationship field, this should order by the natural
# ordering of the ReaderLetter model (i.e. as dictated by its Meta.ordering),
# but right now is ordering by the ReaderLetter PK.
>>> EditorResponse.objects.order_by('reader_letter')
[<EditorResponse: Editor A, Reader D, Article 1>, <EditorResponse: Editor B, Reader D, Article 2>, <EditorResponse: Editor B, Reader C, Article 4>]

# Order by another field of the related ReaderLetter object.
>>> EditorResponse.objects.order_by('reader_letter__reader_name', 'reader_letter__article')
[<EditorResponse: Editor B, Reader C, Article 4>, <EditorResponse: Editor A, Reader D, Article 1>, <EditorResponse: Editor B, Reader D, Article 2>]

# The ordering specification can span models through relationships too.
# Order by the PK of the article related to the reader letter related
# to the editor response, descending.
>>> EditorResponse.objects.order_by('-reader_letter__article')
[<EditorResponse: Editor B, Reader C, Article 4>, <EditorResponse: Editor B, Reader D, Article 2>, <EditorResponse: Editor A, Reader D, Article 1>]

# Order by the article publication date, ascending.
>>> EditorResponse.objects.order_by('reader_letter__article__pub_date')
[<EditorResponse: Editor A, Reader D, Article 1>, <EditorResponse: Editor B, Reader D, Article 2>, <EditorResponse: Editor B, Reader C, Article 4>]

# Composite ordering with one of the components being a specification that
# spans a FK relationship
>>> EditorResponse.objects.order_by('-editor_name', 'reader_letter__article')
[<EditorResponse: Editor B, Reader D, Article 2>, <EditorResponse: Editor B, Reader C, Article 4>, <EditorResponse: Editor A, Reader D, Article 1>]

# Test many-to-many relationships
>>> r1 = Reporter.objects.create(name='John')
>>> r2 = Reporter.objects.create(name='Jane')
>>> r3 = Reporter.objects.create(name='Paul')

>>> r1.articles = [a1, a2]
>>> r2.articles = [a4, a3]
>>> r3.articles = [a4, a2]

"""}
