=head1 NAME

File::ShLock - lock against multiple execution

=head1 SYNOPSIS

    use File::ShLock
   
    $lock = new File::ShLock("NAME");
    $lock = new File::ShLock(name => "NAME", basedir => "/var/lock",
                             pid => $$);

    exit $lock unless ref $lock;
   
    ### locked section

    $lock = undef;

=head1 DESCRIPTION

This module implements lock files.

The lock files typically reside in C</var/lock> and contain the process
ID of the program which locked the resource in question.

=cut

#------------------------------------------------------------------------------
#
# End of POD
#
#------------------------------------------------------------------------------

package File::ShLock;
require 5.002;
use strict;

use File::Basename;
use Errno qw(EEXIST ESRCH);
use IO::File;

use vars qw( $VERSION );
$VERSION = '0.001';

#------------------------------------------------------------------------------

=head1 CONSTRUCTOR

   $lock = new File::ShLock("name");

Creates a new lock with the given name.

   $lock = new File::ShLock(name => "name",
                            pid => $pid,
                            basedir => "/var/lock");

If C<name> is an absolute path, the lock will be named by appending
directory inode information to the file's name.


=cut

#------------------------------------------------------------------------------

sub new
{
    my $class    = shift;
    my %options;
    my $lock = IO::File->new;

    if (@_ == 1) {
	%options = (name => $_[0]);
    } elsif(@_ % 1) {
	return "3 Usage: File::ShLock->new(NAME) or (name=>NAME)";
    } else {
    	%options = @_;
	return "3 Usage: File::ShLock->new(NAME) or (name=>NAME)"
	    unless defined $options{name};
    }

    my $lockfile = $options{name};
    my $basedir = $options{basedir} || "/var/lock";
    my $pid = $options{pid} || $$;
    return "2 process $pid must be running: $!" if not kill(0,$pid) and $! == ESRCH;
    if (dirname($lockfile) ne ".") {
	my($dev,$ino) = (stat dirname $lockfile)[0,1];
	$lockfile = sprintf("%s/%s.%x_%x",$basedir,$lockfile,$dev,$ino);
    } else {
	$lockfile = "$basedir/$lockfile";
    }
    $options{lockfile} = $lockfile;

    my $ltmp = sprintf("%s/LTMP.%x", $basedir,$$);
    unlink($ltmp);
    $lock->open("> $ltmp") or return "2 could not open temp link: $!";
    print $lock $pid,"\n" or unlink($ltmp),return "2 could not write temp link: $!";;
    $lock->close or unlink($ltmp),return "2 could not close temp link: $!";;
    unless(link($ltmp,$lockfile)) {
	unlink($ltmp), return "2 could not link lock: $!" if $! != EEXIST;
	$lock->open($lockfile) or unlink($ltmp), return "2 could not open lock: $!";
	my $pid; defined($pid = <$lock>) or unlink($ltmp), return "2 could not read lock: $!";
	$pid =~ /^(\d+)$/  and $pid = $1; # un-taint
	$lock->close;
	unlink($ltmp),return 1 if kill(0,$pid) or $! != ESRCH;

### the lock is dead. This code is tricky.
#process A                        process B
#
#link(temp,lock) => fail
#                                link(temp,lock) => fail
#  check_lock(lock) => fail
#                                  check_lock(lock) => fail
#    unlink(lock)
#link(temp,lock) => SUCCESS
#                                    unlink(lock) <= deletes the wrong file
#                                link(temp,lock) => SUCCESS
### OOPS

	unlink($lockfile) and sleep 15;

#process A                        process B
#
#link(temp,lock) => fail
#                                link(temp,lock) => fail
#  check_lock(lock) => fail
#                                  check_lock(lock) => fail
#    unlink(lock) and sleep 15
#SLEEP                               unlink(lock) => fail => no sleep
#SLEEP                           link(temp,lock) => SUCCESS
#SLEEP
#link(temp,lock) => fail

    	link($ltmp,$lockfile) or unlink($ltmp),return 1; ## try again (once)

#We don't retry more than once. That might or might not cause more subtle
#intercommunications problems. So we'll repair the lock next time around.

    }
    unlink($ltmp);
    bless \%options, $class;
}

sub DESTROY {
    my $self = shift;
    unlink $self->{lockfile};
}

1;
