package noris::CGI::Mail;

$VERSION = 20000621.0; # by Martin H. Sluka <sluka@noris.net>

use strict;
use vars qw($VERSION $errstr $LF $Validate %Validate);

$LF       = "\n";
$Validate = 1;
%Validate = (to=>1, cc=>1, bcc=>1);

use Email::Valid;

sub send {
  undef $errstr;
  my @addresses;
  $errstr = 'First argument to send() must be a hash reference.', return
    unless ref(my $header = shift) eq 'HASH';
  for (keys %$header) { # not each, 'cause we may use delete()
    $header->{$_} = join ', ', @{$header->{$_}} if ref $header->{$_} eq 'ARRAY';
    push @addresses, map $_->address, Mail::Address->parse($header->{$_})
      if defined $Validate{lc()};
    $header->{From} = delete $header->{$_} if $_ ne 'From' && lc eq 'from'
  }
  $header->{From} = $ENV{SERVER_ADMIN}
    unless $header->{From} and
           !defined $Validate ||
           Email::Valid->address( -address=>$header->{From},
                                  $Validate ? (-mxcheck=>1) : ()
                                );
  $errstr = 'No recipient addresses found.', return unless @addresses;
  if ($Validate) {
    require Mail::Address;
    for (@addresses) {
      $errstr = "<$_> is no valid e-mail address.", return
        unless Email::Valid->address(-address=>$_, -mxcheck=>1)
    }
  }
  {
    my $fileno = select STDOUT;
    local $| = 1;
    print '';
    select $fileno
  }
  defined(my $pid = open MAIL, '|-') or $errstr = "fork(): $!", return;
  exec qw(/usr/sbin/sendmail -f), $ENV{SERVER_ADMIN}, @addresses
    or $errstr = "sendmail: $!", return
    unless $pid;
  while(my($field, $content) = each %$header) {
    $field   =~ s/[:\s]//g;
    $content =~ y/\n//d;
    print MAIL "$field: $content\n" and next;
    $errstr = "print(MAIL): $!";
    close MAIL
  }
  print MAIL "\n";
  local $\ = $LF if defined $LF;
  for (@_) {
    my $line = $_;
    chomp $line if defined $LF;
    print MAIL $line and next;
    $errstr = "print(MAIL): $!";
    close MAIL;
    return
  }
  close MAIL or $errstr = "sendmail: $!", return;
  @addresses
}

42;

=cut

=head1 NAME

noris::CGI::Mail -- send mail from within a CGI script
 
=head1 SYNOPSIS

    use noris::CGI::Mail;

    noris::CGI::Mail::send { To      => 'info@your-domain.de',
                             Subject => 'Anfrage aus dem WWW'
                           }
                           @body
      or warn "Error sending mail: $noris::CGI::Mail::errstr\n";

=head1 DESCRIPTION

Using the mail() function of this module you can conveniently
send mail from within your CGI script.

The module will automatically set your ServerAdmin's e-mail
address as envelope-From.

The function returns the list of reciepients' addresses,
if the mail was sent successfully.
Otherwise,
it sets C<$noris::CGI::Mail::errstr> and returns L<C<undef>|perlfunc/undef>
(or an empty list if called in a list context).

=head1 OPTIONS

You can alternate noris::CGI::Mail::send()'s behaviour by setting the
following global variables (defaults in brackets):

=over 4

=item C<$noris::CGI::Mail::LF> [ \n ]

The content of this variable will be appended to each "line"
(that is, each element of the C<@body>) on output.
Unless unL<defined|perlfunc/defined>,
each "line" will be L<chomp()ed|perlfunc/chomp> before that happens.

=item C<$noris::CGI::Mail::Validate> [ 1 ]

If L<defined|perlfun/defined>, adresses contained in the From, To, Cc and Bcc
headers will be scanned using L<Email::Valid|Email::Valid>.
If true (in Perl's sense), will also check for the existance of MX records.

=back

=head1 AUTHOR

Martin H. Sluka <sluka@noris.net>
for noris network GmbH, 2000-03-04

=cut
