use Test::More tests => 8;
use strict;
use warnings;
binmode STDOUT, ':encoding(UTF-8)';
use Encode qw/encode_utf8/;

# test everything with two strings, one in the
# codepoing range 128..255, and one >= 256
# this is because Perl special-cases codepoints >= 256
# to always encode as UTF-8 if no IO layer is given
#
# test cases:
# U+00C4 LATIN CAPITAL LETTER A WITH DIAERESIS
# U+20AC EURO SIGN

# encode a byte string as hex pairs
sub bytes2hex {
    unpack 'H*', shift;
}

# encode a codepoint as hex pairs
sub code2hex {
    bytes2hex encode_utf8 chr shift;
}

my $use = 'use noris::IO::AutoEncoding qw/:auto/;';
SKIP : {
    require encoding;
    skip('non-UTF locale', 8) unless encoding::_get_locale_encoding() eq 'UTF-8';
    skip('Windows', 8) if $^O eq 'MSWin32';
    is bytes2hex(`$^X -e '$use print chr 0xC4'`),
       code2hex(0xC4), 'auto-encoded output (STDOUT, <256)';
    
    is bytes2hex(`$^X -e '$use print chr 0x20AC'`),
       code2hex(0x20AC), 'auto-encoded output (STDOUT, >256)';
    
    is bytes2hex(`$^X -e '$use print STDERR chr 0xC4' 2>&1`),
       code2hex(0xC4), 'auto-encoded output (STDERR, <256)';
    
    is bytes2hex(`$^X -e '$use print STDERR chr 0x20AC' 2>&1`),
       code2hex(0x20AC), 'auto-encoded output (STDERR, >256)';
    
    my $input = code2hex(0x20AC);
    $input =~ s/(..)/\\x$1/g;

    # the shell built-in echo command sometimes does not interpret -e
    # correctly, so use the binary instead;
    my $echo = `which echo`;
    chomp $echo;
    is `$echo -n -e "$input"| $^X -e '$use print ord <>'`, 0x20AC,
        'auto-decoded input (>256)';

    $input = code2hex(0xEC);
    $input =~ s/(..)/\\x$1/g;
    is `$echo -n -e "$input"| $^X -e '$use print ord <>'`, 0xEC,
        'auto-decoded input (<256)';

    my $input_bytes = encode_utf8 chr 0xC4;
    is `$^X -e '$use print ord \$ARGV[0]' $input_bytes`, 0xC4,
        ':auto decodes @ARGV (<256)';
    $input_bytes = encode_utf8 chr 0x20AC;
    is `$^X -e '$use print ord \$ARGV[0]' $input_bytes`, 0x20AC,
        ':auto decodes @ARGV (>256)';
}

